/*
Copyright (c) 1995-2023 held by the author(s).  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer
      in the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the Web3D Consortium (https://www.web3D.org)
      nor the names of its contributors may be used to endorse or
      promote products derived from this software without specific
      prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

package org.web3d.x3d.jsail.fields;
import org.web3d.x3d.jsail.*;
import org.web3d.x3d.jsail.Core.*;
import org.web3d.x3d.sai.InvalidFieldValueException;
// import org.web3d.x3d.sai.*;


/**
 * This utility class provides a concrete implementation corresponding to SFMatrix4d X3D field type.
 * 
 * <br><br>
 *
 * <i>Warning:</i> this is an abstract interface that cannot be instantiated as a concrete object.
 * Java programmers typically only need to use concrete objects provided by the <code>org.web3d.x3d.jsail</code> classes.

 * <br>
 * <i>Package hint:</i>  This specification class is defined by the X3D Java Language Binding Specification for the Scene Authoring Interface (SAI).
 * SFMatrix4d specifies a 4x4 matrix of double-precision floating point numbers, organized in row-major fashion. Warning: comma characters within singleton values do not pass strict XML validation.
 * <br><br>
 * Related field object: {@link MFMatrix4d}
 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix4d">X3D Tooltips: type SFMatrix4d</a>
 * 
 * @author Don Brutzman and Roy Walmsley
 * @see <a href="https://www.web3d.org/documents/specifications/19777-2/V3.3/Part2/abstracts.html#X3DFieldTypes" target="_blank">SAI Java Specification: B.4.11 X3DFieldTypes</a>
 * @see <a href="https://www.web3d.org/documents/specifications/19775-2/V3.3/Part02/dataRef.html#SAIFieldType" target="blank">SAI Abstract Specification: 5.2.15 SAIFieldType</a>
 * @see <a href="https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-IS.proof/Part01/fieldsDef.html#SFMatrix4dAndMFMatrix4d" target="blank">X3D Abstract Specification: SFMatrix4dAndMFMatrix4d</a>
 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html" target="_blank">X3D Tooltips</a>
 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html#field"      target="_blank">X3D Tooltips: field</a>
 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html#fieldValue" target="_blank">X3D Tooltips: fieldValue</a>
 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html" target="_blank">X3D Scene Authoring Hints</a>
 */
public class SFMatrix4d extends X3DConcreteField implements org.web3d.x3d.sai.SFMatrix4d
{
	/** String constant <i>NAME</i> provides name of this element: <i>SFMatrix4d</i> */
	public static final String NAME = "SFMatrix4d";

	/** Default value for this field type contains values of <i>1 0 0 0 0 1 0 0 0 0 1 0 0 0 0 1</i>.
	 * @see <a href="https://docs.oracle.com/javase/tutorial/java/nutsandbolts/datatypes.html" target="_blank">Java Tutorials: Primitive Data Types</a>
     */
	public static final double[] DEFAULT_VALUE = {1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0}; // X3DUOM default value: 1 0 0 0 0 1 0 0 0 0 1 0 0 0 0 1

	/** Default string value for this field type is "1 0 0 0 0 1 0 0 0 0 1 0 0 0 0 1". */
	public static final String DEFAULT_VALUE_STRING = "1 0 0 0 0 1 0 0 0 0 1 0 0 0 0 1";

	/** Whether or not this field type is an array (<i>false</i>) 
      * @return true if array type */
	public static final boolean isArray()
    {
        return false;
    }

	/** Default tuple size for this field type is <i>16</i> (i.e. number of component values making up a single-field SF object). */
	public static final int TUPLE_SIZE = 16;

	/** Default regular expression (regex) pattern for this field type is <i>\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){15}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*</i>
	 * @see #PATTERN
	 * @see #matches
	 * @see #matches(String)
	 * @see #validate
	 * @see <a href="https://www.web3d.org/specifications/X3dRegularExpressions.html#SFMatrix4d" target="_blank">X3D Regular Expressions: SFMatrix4d</a>
	 */
	public static final String REGEX = "\\s*(([+-]?((0|[1-9][0-9]*)(\\.[0-9]*)?|\\.[0-9]+)([Ee][+-]?[0-9]+)?)\\s+){15}([+-]?((0|[1-9][0-9]*)(\\.[0-9]*)?|\\.[0-9]+)([Ee][+-]?[0-9]+)?)\\s*";

	/** Precompiled regular expression (regex) pattern for this field type using default <em>REGEX</em> value.
	 * @see REGEX
	 * @see #matches
	 * @see #matches(String)
	 * @see #validate
	 * @see <a href="https://docs.oracle.com/javase/8/docs/api/java/util/regex/Pattern.html">java.util.regex.Pattern</a>
	 */
	public static final java.util.regex.Pattern PATTERN = java.util.regex.Pattern.compile(REGEX);
                    
    /** Flag to minimize excessive regex warnings for this field type, helpful if memory limitations are present. */
    private static boolean priorRegexStackOverflowFound = false;

	// Member value declaration is encapsulated and private, using preferred Java types for concretes library
	private double[] SFMatrix4d = java.util.Arrays.copyOf(DEFAULT_VALUE, DEFAULT_VALUE.length); // must be separate copy

	/**
	 * Constructor for SFMatrix4d performs value initialization.
	 */
	public SFMatrix4d()
	{
		initialize();
	}

	/**
	 * Initialization for SFMatrix4d applies default initial value.
	 * Static initializer also provided to verify that the regex pattern compiles and matches that default value.
	 * @see <a href="https://stackoverflow.com/questions/2704857/how-to-check-if-a-given-regex-is-valid">Stack Overflow: How to check if a given Regex is valid?</a>
	 * @see #DEFAULT_VALUE
	 */
	@Override
	public final void initialize()
	{
		SFMatrix4d = java.util.Arrays.copyOf(DEFAULT_VALUE, DEFAULT_VALUE.length); // must be separate copy
	}
	// static initializer
	static {
		try {
            // verify pattern is compiling OK, report exception if not
			java.util.regex.Pattern.compile(REGEX);
				
			// verify DEFAULT_VALUE matches pattern OK, report if not
			if ((new SFMatrix4d(DEFAULT_VALUE)).matches() == false)
			{
				System.out.println("SFMatrix4d.initialize() problem: failed to match DEFAULT_VALUE=" + java.util.Arrays.toString(DEFAULT_VALUE));
			}
        }
		catch (java.util.regex.PatternSyntaxException exception)
		{
			System.out.println("Exception in SFMatrix4d initialization testing, regex pattern compilation failure.");
			System.out.println("REGEX=\"" + REGEX + "\"");
			System.out.println(exception.getDescription());
        }
	}

	/**
	 * Determine whether two objects contain equal values.
	 * @param comparisonSFMatrix4d field type to compare
	 * @return true if equivalent, false otherwise
	 */
	public boolean equals(SFMatrix4d comparisonSFMatrix4d)
	{
        return (getPrimitiveValue() == comparisonSFMatrix4d.getPrimitiveValue());
	}
	/**
	 * Validate current value via get/set comparison tests
	 * @return empty string if get/set testing passes, warning otherwise
	 * @see #validate
	 * @see #validateRegex()
	 */
	public final String validate()
	{
        // validation via get/set checks and comparison
        SFMatrix4d tempHold = new SFMatrix4d(getPrimitiveValue());
        if  (!this.equals(tempHold))
             return "SFMatrix4d validate() error: failed get/set round-trip test";
        else return "";
        
    }
	/**
	 * Validate current value via regular expression (regex) check of current object's toString() value, reporting errors only if found.
	 * @see #REGEX
	 * @see #PATTERN
	 * @see #matches
	 * @see #matches(String)
	 * @see #validate
	 * @see #validate()
	 * @see <a href="https://docs.oracle.com/javase/8/docs/api/java/util/regex/Pattern.html">java.util.regex.Pattern</a>
	 * @return empty string if PATTERN matches, warning otherwise
	 */
	public final String validateRegex()
	{
        try
        {
            java.util.regex.Matcher m = PATTERN.matcher(toString());
            if (!m.matches())
            {
			String errorNotice = "[error] Regular expression (regex) failure, new SFMatrix4d PATTERN mismatch (\"" + toString() + "\")";
                return errorNotice;
            }
		}
        catch (java.lang.StackOverflowError soe) // regex processing required too much memory for String version of this attribute
        {
            String validationMessage = "java.lang.StackOverflow warning: regular expression (regex) unable to evaluate large corresponding String value for SFMatrix4d";
            validationResult.append(validationMessage).append("\n");
            System.out.println ("[exception] " + validationMessage);
            return validationMessage;
		}
        return "";
	}
	/**
	 * Test PATTERN match via regular expression (regex) check of current object's toString() value.
	 * @see #REGEX
	 * @see #PATTERN
	 * @see #matches
	 * @see #matches(String)
	 * @see #validate
	 * @see <a href="https://docs.oracle.com/javase/8/docs/api/java/util/regex/Pattern.html">java.util.regex.Pattern</a>
	 * @return true if PATTERN matches, false otherwise
	 */
	public final boolean matches()
	{
        try
        {
            java.util.regex.Matcher m = PATTERN.matcher(toString());
            return m.matches();
		}
        catch (java.lang.StackOverflowError soe) // regex processing required too much memory for String version of this attribute
        {
            String validationMessage = "java.lang.StackOverflow warning: regular expression (regex) unable to evaluate large corresponding String value for SFMatrix4d";
            validationResult.append(validationMessage).append("\n");
            System.out.println ("[exception] " + validationMessage);
            return false;
		}
	}
	/**
	 * Test PATTERN match with regular expression (regex) of provided value.
	 * @param value String to check against regex pattern for successful match
	 * @see #REGEX
	 * @see #PATTERN
	 * @see #matches
	 * @see #matches(String)
	 * @see #validate
	 * @see <a href="https://docs.oracle.com/javase/8/docs/api/java/util/regex/Pattern.html">java.util.regex.Pattern</a>
	 * @return true if PATTERN matches, false otherwise
	 */
	public static final boolean matches(String value)
	{
		java.util.regex.Matcher m = PATTERN.matcher(value);
		return m.matches();
	}

	/**
	 * Utility accessor for SFMatrix4d using String value (which must pass parsing validation checks).
	 * <br ><br >
	 * <i>Warning:</i> this method is not 
	 * <i><a href="https://en.wikipedia.org/wiki/Type_safety" target="_blank">type safe</a></i>
	 * at compile time!  Best practice for X3DJSAIL programmers is to use strongly typed methods instead, in order to avoid potential for run-time errors.
	 * @see X3DLoaderDOM#toX3dModelInstance(org.w3c.dom.Node)
	 * @param newValue is new value to assign, if empty then assign DEFAULT_VALUE
	 * @return {@link MFMatrix4d} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public SFMatrix4d setValueByString (String newValue) throws InvalidFieldValueException
	{
		if (newValue == null)
			newValue = new String(); // Principle of Least Astonishment (POLA) #1
			// https://en.wikipedia.org/wiki/Principle_of_least_astonishment

		if (!matches(newValue)) // regex test
		{
			String errorNotice = "[error] Regular expression (regex) failure, new SFMatrix4d(" + newValue + ")";
			validationResult.append(errorNotice).append("\n");
		}
		if (newValue.trim().isEmpty())
		{
			SFMatrix4d = java.util.Arrays.copyOf(DEFAULT_VALUE, DEFAULT_VALUE.length); // must be separate copy
			return this;
		}
		try
		{
			String[] newValues = newValue.replace(","," ").trim().split("\\s+");
			if (newValues.length != 16)
			{
				String errorNotice = "illegal number of values in initialization string";
				validationResult.append(errorNotice).append("\n");
				throw new InvalidFieldValueException(errorNotice);
			}
			SFMatrix4d [0] = Double.parseDouble(newValues[0]);
			SFMatrix4d [1] = Double.parseDouble(newValues[1]);
			SFMatrix4d [2] = Double.parseDouble(newValues[2]);
			SFMatrix4d [3] = Double.parseDouble(newValues[3]);
			SFMatrix4d [4] = Double.parseDouble(newValues[4]);
			SFMatrix4d [5] = Double.parseDouble(newValues[5]);
			SFMatrix4d [6] = Double.parseDouble(newValues[6]);
			SFMatrix4d [7] = Double.parseDouble(newValues[7]);
			SFMatrix4d [8] = Double.parseDouble(newValues[8]);
			SFMatrix4d [9] = Double.parseDouble(newValues[9]);
			SFMatrix4d[10] = Double.parseDouble(newValues[10]);
			SFMatrix4d[11] = Double.parseDouble(newValues[11]);
			SFMatrix4d[12] = Double.parseDouble(newValues[12]);
			SFMatrix4d[13] = Double.parseDouble(newValues[13]);
			SFMatrix4d[14] = Double.parseDouble(newValues[14]);
			SFMatrix4d[15] = Double.parseDouble(newValues[15]);
		}
		catch (NumberFormatException e)
		{
			String errorNotice = "new SFMatrix4d(" + newValue + ") " + e.getMessage();
			validationResult.append(errorNotice).append("\n");
			throw new InvalidFieldValueException (errorNotice);
		}
		return this;
	}

	/**
	 * Constructor to copy an SFMatrix4d value as initial value for this new field object.
	 * @param newValue The newValue to apply
	 */
	public SFMatrix4d(SFMatrix4d newValue)
	{
		if (newValue == null)
		{
			SFMatrix4d = java.util.Arrays.copyOf(DEFAULT_VALUE, DEFAULT_VALUE.length); // must be separate copy
			return;
		}
		SFMatrix4d = newValue.getPrimitiveValue();
	}

	/**
	 * Constructor for SFMatrix4d using a corresponding Java primitive double[] array as new initial value.
	 * @param newValue is new value to assign
	 * <i>Warning:</i> newValue array length must correspond to tuple size for base type SFMatrix4d tuple size of <i>16</i>.
	 * setContainerFieldOverride(containerFieldName); // apply checksConcreteField#getTupleSize(String)
	 */
	public SFMatrix4d (double[] newValue)
	{
		if      (newValue == null)
			     newValue = java.util.Arrays.copyOf(DEFAULT_VALUE, DEFAULT_VALUE.length); // must be separate copy
		else if (newValue.length != 16)
		{
			String errorNotice = "Illegal SFMatrix4d newValue array length=" + newValue.length +
				", must equal 16 or else be empty (newValue=" + toString(newValue) + ")";
			validationResult.append(errorNotice).append("\n");
			throw new InvalidFieldValueException(errorNotice);
		}
		
		SFMatrix4d = newValue;
	}

	/**
	 * Static utility method to provide String representation of a correctly typed input value.
	 * @param value The value to convert to a String
	 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html#type">X3D Tooltips: type</a>
	 * @see SFDouble#stripTrailingZeroes(double)
	 * @return String version of the provided value, with trailing zeroes and decimal points omitted.
	 */
	public static String toString (double[] value)
	{
		StringBuilder result = new StringBuilder();
		for (int i=0; i < value.length; i++)
		{
			if  (ConfigurationProperties.isStripTrailingZeroes())
				 result.append(org.web3d.x3d.jsail.fields.SFDouble.stripTrailingZeroes(value[i])).append(" ");
			else result.append(value[i]).append(" ");
		}
		return result.toString().trim();
	}

	/**
	 * Provides current value of the field as a Java primitive type.
	 * @return current value
	 */
	public double[] getPrimitiveValue()
	{
		return SFMatrix4d;
	}
	/**
	 * Provides current value as a String.
	 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix4d">X3D Tooltips: type SFMatrix4d</a>
	 * @see org.web3d.x3d.jsail.fields.SFDouble#stripTrailingZeroes(double)
	 * @return String version of the provided value, with trailing zeroes and decimal points omitted.
	 */
	/* @Override */
	public String toString()
	{
		StringBuilder result = new StringBuilder();
		for (int i = 0; i < SFMatrix4d.length; i++)
		{
			if  (ConfigurationProperties.isStripTrailingZeroes())
				 result.append(org.web3d.x3d.jsail.fields.SFDouble.stripTrailingZeroes(SFMatrix4d[i]));
			else result.append(SFMatrix4d[i]);
			if (i < SFMatrix4d.length - 1)
				result.append(" ");
		}
		return result.toString();
	}
	/**
	 * Apply an array of primitive values to this field.
	 * @param newValue The newValue to apply
	 * @return {@link SFMatrix4d} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public SFMatrix4d setValue(double[] newValue)
	{
		if (newValue == null)
		{
			SFMatrix4d = java.util.Arrays.copyOf(DEFAULT_VALUE, DEFAULT_VALUE.length); // must be separate copy
			return this;
		}
        if (newValue.length != 16)
		{
			String errorNotice = "Illegal array newValue=" + newValue + " must base have length 16 for setValue() on SFMatrix4d";
			validationResult.append(errorNotice).append("\n");
			throw new InvalidFieldValueException(errorNotice);
		}
		SFMatrix4d = newValue;
		return this;
	}
	/* @Override */
	public void setIdentity()
	{
		SFMatrix4d = org.web3d.x3d.jsail.fields.SFMatrix4d.DEFAULT_VALUE;
	}

	/* @Override */
	public void set(int row, int column)
	{
		// TODO
	}

	/* @Override */
	public float get(int row, int column) // TODO SAI interface problem, needs to be double
	{
		if ((row < 0) || (row > 4) || (column < 0) || (column > 4))
		{
			String errorNotice = "Illegal SFMatrix4d[row=" + row + ",column=" + column + "] access, " +
				"all values must be in numeric range [0..4]";
			validationResult.append(errorNotice).append("\n");
			throw new InvalidFieldValueException(errorNotice);
		}
		return (float)SFMatrix4d[3*row + column]; // TODO typing
	}

	/* @Override */
	public void setTransform(org.web3d.x3d.sai.SFVec3d translation,
		org.web3d.x3d.sai.SFRotation rotation,
		org.web3d.x3d.sai.SFVec3d scale,
		org.web3d.x3d.sai.SFRotation scaleOrientation,
		org.web3d.x3d.sai.SFVec3d center)
	{
		// TODO
	}

	/* @Override */
	public void getTransform(org.web3d.x3d.sai.SFVec3d translation,
		org.web3d.x3d.sai.SFRotation rotation,
		org.web3d.x3d.sai.SFVec3d scale)
	{
		// TODO
	}

	/* @Override */
	public org.web3d.x3d.sai.Matrix4 inverse()
	{
		return null; // TODO
	}

	/* @Override */
	public org.web3d.x3d.sai.Matrix4 transpose()
	{
		return null; // TODO
	}

	/* @Override */
	public org.web3d.x3d.sai.Matrix4 multiplyLeft (org.web3d.x3d.sai.Matrix4 matrix3x3)
	{
		return null; // TODO
	}

	/* @Override */
	public org.web3d.x3d.sai.Matrix4 multiplyRight(org.web3d.x3d.sai.Matrix4 matrix3x3)
	{
		return null; // TODO
	}

	/* @Override */
	public org.web3d.x3d.sai.Matrix4 multiplyRowVector(org.web3d.x3d.sai.SFVec3d vector3d) // TODO SFVec4d ?
	{
		return null; // TODO
	}

	/* @Override */
	public org.web3d.x3d.sai.Matrix4 multiplyColVector(org.web3d.x3d.sai.SFVec3d vector3d) // TODO SFVec4d ?
	{
		return null; // TODO
	}

	/**
	 * Provide double array for this field type.
	 * @return Array of doubles in this field array.
	 */
	public double[] toDoubleArray()
	{
		return SFMatrix4d;
	}

	/**
	 * Apply an SFMatrix4d value to this field.
	 * @param newValue The newValue to apply
	 * @return {@link SFMatrix4d} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public SFMatrix4d setValue(SFMatrix4d newValue)
	{
		if (newValue == null)
		{
			SFMatrix4d = java.util.Arrays.copyOf(DEFAULT_VALUE, DEFAULT_VALUE.length); // must be separate copy
			return this;
		}
		SFMatrix4d = newValue.getPrimitiveValue();
		return this;
	}								

	/**
	 * Multiply scaleFactor times all values in this field type.
	 * @param scaleFactor scalar value for vector multiplication
	 * @return {@link SFMatrix4d} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public SFMatrix4d multiply(double scaleFactor)
	{
		for (int i = 0; i < SFMatrix4d.length; i++)
		{
			SFMatrix4d[i] *= scaleFactor;
		}
		return this;
	}

	/**
	 * Determine whether current value matches DEFAULT_VALUE
	 * @see #DEFAULT_VALUE
	 * @return whether current value matches DEFAULT_VALUE
	 */
    public boolean isDefaultValue()
    {
        return java.util.Arrays.equals(SFMatrix4d, DEFAULT_VALUE);
    }
}
