/*
Copyright (c) 1995-2023 held by the author(s).  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer
      in the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the Web3D Consortium (https://www.web3D.org)
      nor the names of its contributors may be used to endorse or
      promote products derived from this software without specific
      prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

package org.web3d.x3d.jsail.Core;
import java.util.*;
// import org.web3d.x3d.sai.*;      // making sure #2
// import org.web3d.x3d.sai.Core.*;  // making sure #2
import org.web3d.x3d.sai.InvalidFieldValueException;  // making sure #2
import org.web3d.x3d.jsail.fields.*; // making sure #4
// import org.web3d.x3d.jsail.Core.*; // interfaces for this component
import org.web3d.x3d.jsail.Grouping.*;
import org.web3d.x3d.jsail.Navigation.*;
// import org.web3d.x3d.jsail.Shape.*;
// import org.web3d.x3d.jsail.Networking.*;
// import org.web3d.x3d.jsail.Core.*;
import org.web3d.x3d.jsail.*; // again making sure #4

import org.web3d.x3d.jsail.Core.*;
// import org.web3d.x3d.jsail.Core.*; // interfaces for this component
import org.web3d.x3d.jsail.Grouping.*;
import org.web3d.x3d.jsail.Navigation.*;
// import org.web3d.x3d.jsail.Shape.*;
// import org.web3d.x3d.jsail.Networking.*;
// import org.web3d.x3d.jsail.Core.*;
import org.web3d.x3d.jsail.*; // again making sure #4

import org.web3d.x3d.jsail.Core.*;
// import org.web3d.x3d.jsail.Core.*; // interfaces for this component
import org.web3d.x3d.jsail.Grouping.*;
import org.web3d.x3d.jsail.Navigation.*;
// import org.web3d.x3d.jsail.Shape.*;
// import org.web3d.x3d.jsail.Networking.*;
// import org.web3d.x3d.jsail.Core.*;
import org.web3d.x3d.jsail.*; // again making sure #4

import org.web3d.x3d.jsail.Core.*;
import java.util.Arrays;

/**
 * ProtoInstance can override field default values via fieldValue initializations. Non-recursive nested ProtoInstance and ProtoDeclare statements are allowed within a ProtoDeclare.
 * 
 * <br><br>
 *
 * <i>Warning:</i> this is an abstract interface that cannot be instantiated as a concrete object.
 * Java programmers typically only need to use concrete objects provided by the <code>org.web3d.x3d.jsail</code> classes.
 * <i>X3D statement tooltip</i>:  [X3D statement] ProtoInstance creates an instance node of a locally or externally declared prototype definition. ProtoDeclare/ExternProtoDeclare definitions are abstract, corresponding ProtoInstance nodes are concrete. ProtoInstance nodes typically contain fieldValue statements to override default field values defined in the original ProtoDeclare statement.
Hint: the node type of a ProtoInstance exactly matches the first node in the corresponding ProtoDeclare/ProtoBody declaration.
Warning: be sure to correctly set the containerField value to match the intended field in the parent node.
Warning: do not mismatch node types when placing a ProtoInstance in a scene graph. Most validation tools will not catch this error.
Hint: scene authors can override default initializations of prototype fields by including contained &amp;lt;fieldValue&amp;gt; elements.
Hint: Nested ProtoInstance and ProtoDeclare statements are allowed within a ProtoDeclare/ProtoBody.
Hint: if contained within a ProtoDeclare, then a ProtoInstance node can also contain IS/connect statements for other defined fields.
Warning: a ProtoInstance node may not be instantiated inside its own ProtoDeclare or ExternProtoDeclare declaration (i.e. recursive prototypes are illegal).
Warning: the local context of each ProtoInstance node must match the node type of the first node in the original ProtoDeclare declaration. 
Warning: ProtoInstance substitution for a Metadata* node might not validate. Workaround: put metadata-typed ProtoInstance nodes as contained values within a parent MetadataSet node.
Hint: X3D Scene Authoring Hints, Inlines and Prototypes &lt;a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#InlinesPrototypes" target="_blank"&gt;https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#InlinesPrototypes&lt;/a&gt;
Hint: X3D Architecture 4.4.4 Prototype semantics, &lt;a href="https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-IS.proof//Part01/concepts.html#PROTOdefinitionsemantics" target="_blank"&gt;https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-IS.proof//Part01/concepts.html#PROTOdefinitionsemantics&lt;/a&gt;
Examples: X3D Example Archives, X3D for Web Authors, Chapter 14 Prototypes &lt;a href="https://www.web3d.org/x3d/content/examples/X3dForWebAuthors/Chapter14Prototypes" target="_blank"&gt;https://www.web3d.org/x3d/content/examples/X3dForWebAuthors/Chapter14Prototypes&lt;/a&gt;  * <br>
 * <br>

 * <br>
 * <i>Package hint:</i>  This org.web3d.x3d.jsail concrete class is used for implementing a standalone X3D object as a <a href="https://en.wikipedia.org/wiki/Plain_old_Java_object" target="_blank">Plain Old Java Object (POJO)</a>.
 * If you are writing Java code for use inside an X3D Script node, compile separate code using only the <i>org.web3d.x3d.sai</i> package instead.
 * @author Don Brutzman and Roy Walmsley
 * @see <a href="https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-IS.proof/Part01/concepts.html#ProtoInstanceAndFieldValueStatement" target="blank">X3D Abstract Specification: ProtoInstance</a>
 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ProtoInstance" target="_blank">X3D Tooltips: ProtoInstance</a>
 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#InlinesPrototypes" target="_blank">X3D Scene Authoring Hints: InlinesPrototypes</a>
 */
public class ProtoInstance extends org.web3d.x3d.jsail.X3DConcreteNode implements org.web3d.x3d.sai.Core.X3DPrototypeInstance, org.web3d.x3d.sai.Core.X3DChildNode
{
    /** required by internal interface, empty list provided since no children array present in this class */
    private ArrayList<String> commentsList; 

    // Member value declarations are encapsulated and private, using preferred Java types for concretes library

    /* @name=fieldValue, @type=MFNode, $javaType=ArrayList<fieldValue>, $javaReferenceType=fieldValue, $javaPrimitiveType=ArrayList<fieldValue>, $normalizedMemberObjectName=fieldValueList, $isX3dStatement=true, @acceptableNodeTypes=fieldValue */ 
    private ArrayList<fieldValue> fieldValueList = new ArrayList<>(); //  X3D field type: MFNode acceptable node types: fieldValue

    /* @name=IS, @type=SFNode, $javaType=IS, $javaReferenceType=IS, $javaPrimitiveType=IS, $normalizedMemberObjectName=IS, $isX3dStatement=true, @acceptableNodeTypes=IS */ 
    private IS IS; //  X3D field type: SFNode acceptable node types: IS

    /* @name=metadata, @type=SFNode, $javaType= org.web3d.x3d.sai.Core.X3DMetadataObject, $javaReferenceType= org.web3d.x3d.sai.Core.X3DMetadataObject, $javaPrimitiveType= org.web3d.x3d.sai.Core.X3DMetadataObject, $normalizedMemberObjectName=metadata, $isX3dStatement=false, @acceptableNodeTypes=X3DMetadataObject */ 
    private  org.web3d.x3d.sai.Core.X3DMetadataObject metadata; //  X3D field type: SFNode acceptable node types: X3DMetadataObject or ProtoInstance
	private ProtoInstance metadataProtoInstance; // allowed alternative for metadata field

    /* @name=name, @type=SFString, $javaType=String, $javaReferenceType=String, $javaPrimitiveType=String, $normalizedMemberObjectName=name, $isX3dStatement=false, @acceptableNodeTypes= */ 
    private String name = new String(); //  X3D field type: SFString

					      ProtoDeclare referenceProtoDeclare;
					ExternProtoDeclare referenceExternProtoDeclare;
								
	// String constants for default field values match X3D Schema definitions

	/** String constant <i>NAME</i> provides name of this element: <i>ProtoInstance</i>. */
	@SuppressWarnings("FieldNameHidesFieldInSuperclass")
	public static final String NAME = "ProtoInstance";

	/** Provides name of this element: <i>ProtoInstance</i>
	 * @return name of this element
	 */
	/* @Override */
	public final String getElementName()
	{
		return NAME;
	}

	/** String constant <i>COMPONENT</i> defines X3D component for the <i>ProtoInstance</i> element: <i>Core</i> */
	public static final String COMPONENT = "Core";

	/** Defines X3D component for the <i>ProtoInstance</i> element: <i>Core</i>
	 * @return X3D component for this element
	 */
	/* @Override */
	public final String getComponent()
	{
		return COMPONENT;
	}

	/** Integer constant <i>LEVEL</i> provides default X3D <i>Core</i> component level for this element: <i>2</i> */
	public static final int LEVEL = 2;

	/** Provides default X3D component level for this element: <i>2</i>
	 * @return default X3D component level for this element
	 */
	/* @Override */
	public final int getComponentLevel()
	{
		return LEVEL;
	}

            // DEFAULT_VALUE declarations

	/** SFNode field named <i>IS</i> has default value <i>null</i> (Java syntax) or <i>NULL</i> (XML syntax). */
	public static final IS IS_DEFAULT_VALUE = null;

	/** SFNode field named <i>metadata</i> has default value <i>null</i> (Java syntax) or <i>NULL</i> (XML syntax). */
	public static final  org.web3d.x3d.sai.Core.X3DMetadataObject METADATA_DEFAULT_VALUE = null;

	/** SFString field named <i>name</i> has default value equal to an empty string. */
	public static final String NAME_DEFAULT_VALUE = "";

	/** Indicate type corresponding to given fieldName.
	 * @param fieldName name of field in this X3D statement
	 * @see ConfigurationProperties#ERROR_UNKNOWN_FIELD_TYPE
	 * @return X3D type (SFvec3f etc.), otherwise ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE if not recognized
	 */
	/* @Override */		
	public String getFieldType(String fieldName)
	{
		String result;

		switch (fieldName)
		{
			// String constants for exact field type values matching X3D Schema definitions,
			// thus avoiding spelling errors and allowing type-matching checks
											
			case "containerField":
				result = "SFString";
				break;
			case "fieldValue":
				result = "MFNode";
				break;
			case "IS":
				result = "SFNode";
				break;
			case "metadata":
				result = "SFNode";
				break;
			case "name":
				result = "SFString";
				break;
			case "DEF":
				result = "SFString";
				break;
			case "USE":
				result = "SFString";
				break;
			case "class":
				result = "SFString";
				break;
			case "id":
				result = "SFString";
				break;
			case "style":
				result = "SFString";
				break;
			default:
			{
				// if fieldName has a prefix "set_" prepended (or a suffix "_changed" appended) then try again by stripping it and recursing once
				if      (fieldName.trim().startsWith("set_"))
						 result = getFieldType(fieldName.trim().substring(4)); // after "set_"
				else if (fieldName.trim().endsWith("_changed"))
						 result = getFieldType(fieldName.trim().substring(0, fieldName.length() - 8)); // before "_changed"
				else     result = ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE; // unique return value avoids mistaken matches
			}
		}
		// now check author-defined fields
		field fieldDeclaration = null;
		if      (      hasProtoDeclare())
			fieldDeclaration =       referenceProtoDeclare.findFieldByName(fieldName);
		else if (hasExternProtoDeclare())
			fieldDeclaration = referenceExternProtoDeclare.findFieldByName(fieldName);
		if (fieldDeclaration != null)
			result = fieldDeclaration.getType(); // found it!
		return result;
	}

	/** Indicate accessType corresponding to given fieldName.
	 * @param fieldName name of field in this X3D statement
	 * @see ConfigurationProperties#ERROR_UNKNOWN_FIELD_ACCESSTYPE
	 * @return X3D accessType (inputOnly etc.), otherwise ConfigurationProperties.ERROR_UNKNOWN_FIELD_ACCESSTYPE if not recognized
	 */
	/* @Override */
	public String getAccessType(String fieldName)
	{
		String result; // set by following checks
		switch (fieldName)
		{
			// String constants for field accessType values matching X3D Schema definitions,
			// thus avoiding spelling errors and allowing accessType-matching checks
			case "fieldValue":
				result = "inputOutput";
				break;
			case "IS":
				result = "inputOutput";
				break;
			case "metadata":
				result = "inputOutput";
				break;
			case "name":
				result = "inputOutput";
				break;
			case "DEF":
				result = "inputOutput";
				break;
			case "USE":
				result = "inputOutput";
				break;
			case "class":
				result = "inputOutput";
				break;
			case "id":
				result = "inputOutput";
				break;
			case "style":
				result = "inputOutput";
				break;
			default:
			{
				// if user has added a prefix "set_" or suffix "_changed" then try again by stripping it and recursing once
				if      (fieldName.trim().startsWith("set_"))
						 result = getAccessType(fieldName.trim().substring(4)); // after "set_"
				else if (fieldName.trim().endsWith("_changed"))
						 result = getAccessType(fieldName.trim().substring(0, fieldName.length() - 8)); // before "_changed"
				else     result = ConfigurationProperties.ERROR_UNKNOWN_FIELD_ACCESSTYPE; // unique return value avoids mistaken matches
			}
		}
		// now check author-defined fields
		field fieldDeclaration = null;
		if      (       hasProtoDeclare())
			fieldDeclaration =       referenceProtoDeclare.findFieldByName(fieldName);
		else if (hasExternProtoDeclare())
			fieldDeclaration = referenceExternProtoDeclare.findFieldByName(fieldName);
		if (fieldDeclaration != null)
			result = fieldDeclaration.getAccessType(); // found it!
		return result;
	}

	/** containerField describes typical field relationship of a node to its parent.
	 * Usage is not ordinarily needed when using this API, default value is provided for informational purposes.
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#containerField">X3D Scene Authoring Hints: containerField</a>
	 * @see <a href="https://www.web3d.org/specifications/X3DUOM.html">X3D Unified Object Model (X3DUOM)</a>
	 */
	public static final String containerField_DEFAULT_VALUE = "children"; // type xs:NMTOKEN
	/** containerField describes typical field relationship of a node to its parent.
	 * Usage is not ordinarily needed when using this API, default value is provided for informational purposes.
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#containerField">X3D Scene Authoring Hints: containerField</a>
	 * @see <a href="https://www.web3d.org/specifications/X3DUOM.html">X3D Unified Object Model (X3DUOM)</a>
     * @return default containerField value            
	 */
    /* @Override */
    public final String getContainerFieldDefault()
    {
        return containerField_DEFAULT_VALUE;
    }

	/** Constructor for ProtoInstance to initialize member variables with default values. */
	public ProtoInstance()
	{
		super(); // constructor invocation and corresponding initialize()
		initialize();
	}

	/** Initialize all member variables to default values. */
	@Override
	public final void initialize()
	{
		super.initialize();

        super.includesNameField();
		fieldValueList = new ArrayList<>();
		IS = null; // clear out any prior node
		metadata = null;              // clear out any prior node
		metadataProtoInstance = null; // clear out any prior node
		name = NAME_DEFAULT_VALUE;

		commentsList = new ArrayList<>(); // instantiate, initialize

		fieldValueList = new ArrayList<>(); // instantiate, initialize
	}

	// ==== Accessor methods: strongly typed get/set methods for compile-time strictness

	/**
	 * Provide array of fieldValue results (using an array consisting of properly typed nodes or ProtoInstances) from inputOutput MFNode field <i>fieldValue</i>.
	 * <br><br>
	 * <i>Tooltip:</i>  Include fieldValue statements if this ProtoInstance overrides default values in any of the original field declarations.  * <br>

	 * <br><br>
	 * <i>Warning:</i> according to X3D Unified Object Model (X3DUOM), acceptable node types are limited to fieldValue.
	 * @see org.web3d.x3d.jsail.Core.fieldValue
	 * @return value of fieldValue field
	 */
	// ($isInterface=false, $isException=false, $isServiceInterface=false, $isX3dStatement=true, $isClassX3dStatement=true, preceding-sibling::Inheritance=, $baseType=X3DPrototypeInstance, $additionalInheritanceBaseType=X3DChildNode, $thisClassName=ProtoInstance, $CamelCaseName=FieldValue, $memberObjectName=fieldValue, $memberObjectName=fieldValueList)
	public ArrayList<fieldValue> getFieldValueList()
	{
		return fieldValueList;
	}

	/**
	 * Accessor method to assign fieldValue array (using an array consisting of properly typed nodes or ProtoInstances) to inputOutput MFNode field <i>fieldValue</i>.
	 * <br><br>
	 * <i>Tooltip:</i>  Include fieldValue statements if this ProtoInstance overrides default values in any of the original field declarations.  * <br>

	 * <br><br>
	 * <i>Note:</i> according to X3D Unified Object Model (X3DUOM), acceptable node types are limited to fieldValue.
	 * @param newValue is new value for the fieldValue field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public ProtoInstance setFieldValueList(ArrayList<fieldValue> newValue)
	{
		// set-newValue-validity-checks #0.a
		fieldValueList = newValue;
		for (fieldValue arrayElement : fieldValueList)
		{
			((X3DConcreteElement) arrayElement).setParent(this); // parentTest11
		}
		return this;
	}

	/**
	 * Set single child fieldValue node, replacing prior array of existing nodes (if any).
	 * <br><br>
	 * <i>Note:</i> according to X3D Unified Object Model (X3DUOM), acceptable node types are restricted to fieldValue.
	 * @param newValue is new node for the fieldValue field (restricted to fieldValue)
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public ProtoInstance setFieldValue(fieldValue newValue)
	{
		if (newValue == null)
		{
			clearFieldValue(); // reset newValueNullClearsFieldReturnThis
			return this;
		}
		for (fieldValue element : fieldValueList) // 
			((X3DConcreteElement) element).clearParent(); // remove references to facilitate Java memory management
		clearFieldValue(); // reset
		fieldValueList.add(newValue);
		((X3DConcreteElement) newValue).setParent(this); // parentTest7

		return this;
}
	/**
	 * Utility method to clear MFNode value of fieldValueList field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
setAttribute method invocations).
	 */
	public ProtoInstance clearFieldValue()
	{
		for (fieldValue element : fieldValueList)
			((X3DConcreteElement) element).clearParent(); // remove references to facilitate Java memory management
		fieldValueList.clear(); // reset MF field
		return this;
	}

	/**
	 * Indicate whether an object is available for inputOutput MFNode field <i>fieldValueList</i>.
	 * @return whether a concrete statement array or CommentsBlock is available.
	 * @see #getFieldValueList()
	 */
	public boolean hasFieldValue()
	{
		return (fieldValueList.isEmpty() == false);
	}
	/**
	 * Provide IS instance (using a properly typed node) from inputOutput SFNode field <i>IS</i>.
	 * @return value of IS field
	 */
	// ($isInterface=false, $isException=false, $isServiceInterface=false, $isX3dStatement=true, $isClassX3dStatement=true, preceding-sibling::Inheritance=, $baseType=X3DPrototypeInstance, $additionalInheritanceBaseType=X3DChildNode, $thisClassName=ProtoInstance, $CamelCaseName=IS, $memberObjectName=IS, $memberObjectName=IS)
	public IS getIS()
	{
		return IS;
	}

	/**
	 * Accessor method to assign IS instance (using a properly typed node) to inputOutput SFNode field <i>IS</i>.
	 * @param newValue is new value for the IS field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public ProtoInstance setIS(IS newValue)
	{
		// set-newValue-validity-checks #0.a
		IS = newValue;
		if (newValue != null)
		{
			((X3DConcreteElement) IS).setParent(this); // parentTest15.1
		}
		return this;
	}

	/**
	 * Utility method to clear SFNode value of IS field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
setAttribute method invocations).	 */
	public ProtoInstance clearIS()
	{
		((X3DConcreteElement) IS).clearParent(); // remove references to facilitate Java memory management
		IS = null; // reset SFNode field
		return this;
	}

	/**
	 * Indicate whether an object is available for inputOutput SFNode field <i>IS</i>.
	 * @return whether a concrete statement or CommentsBlock is available.
	 * @see #getIS()
	 */
	public boolean hasIS()
	{
		return (IS != null);
	}
	/**
	 * Provide  org.web3d.x3d.sai.Core.X3DMetadataObject instance (using a properly typed node) from inputOutput SFNode field <i>metadata</i>.
	 * @see #getMetadataProtoInstance()
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#Metadata" target="_blank">X3D Scene Authoring Hints: Metadata Nodes</a>
	 * @return value of metadata field
	 */
	// ($isInterface=false, $isException=false, $isServiceInterface=false, $isX3dStatement=false, $isClassX3dStatement=true, preceding-sibling::Inheritance=, $baseType=X3DPrototypeInstance, $additionalInheritanceBaseType=X3DChildNode, $thisClassName=ProtoInstance, $CamelCaseName=Metadata, $memberObjectName=metadata, $memberObjectName=metadata)

	public  org.web3d.x3d.sai.Core.X3DMetadataObject getMetadata()
	{
		return metadata;
	}

	/**
	 * Accessor method to assign  org.web3d.x3d.sai.Core.X3DMetadataObject instance (using a properly typed node) to inputOutput SFNode field <i>metadata</i>.
	 * @see #setMetadata(ProtoInstance)
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#Metadata" target="_blank">X3D Scene Authoring Hints: Metadata Nodes</a>
	 * @param newValue is new value for the metadata field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */

	public ProtoInstance setMetadata( org.web3d.x3d.sai.Core.X3DMetadataObject newValue)
	{
		// set-newValue-validity-checks #0.a
		metadata = newValue;
		if (newValue != null)
		{
			((X3DConcreteElement) metadata).setParent(this); // parentTest15.1
			// change X3D4 MetadataSet default 'value' back to 'metadata' if parent is not another Metadata* node */
			if (metadata instanceof org.web3d.x3d.jsail.Core.MetadataSet)
{
				((X3DConcreteNode) metadata).setContainerFieldOverride("metadata");
}
		}
		if (metadataProtoInstance != null)
		{
			metadataProtoInstance.setParent(null); // housekeeping, clear prior object
			metadataProtoInstance = null;
		}
		return this;
	}

	/**
	 * Utility method to clear SFNode value of metadata field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
setAttribute method invocations).	 */
	public ProtoInstance clearMetadata()
	{
		((X3DConcreteElement) metadata).clearParent(); // remove references to facilitate Java memory management
		metadata = null; // reset SFNode field
		return this;
	}
	/**
	 * Assign ProtoInstance (using a properly typed node) to inputOutput SFNode field <i>metadata</i>.
	 * @see #setMetadata( org.web3d.x3d.sai.Core.X3DMetadataObject)
	 * @param newValue is new value for the metadata field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	/* @Override */
	public ProtoInstance setMetadata(ProtoInstance newValue)
	{
		if (metadataProtoInstance != null)
		{
			((X3DConcreteElement) metadataProtoInstance).setParent(null); // parentTest15.5
	    }
		metadataProtoInstance = newValue;
		if (newValue != null)
		{
			((X3DConcreteElement) metadataProtoInstance).setParent(this); // parentTest15.6
            metadataProtoInstance.setContainerField("metadata");
	    }
		if (metadata != null)
		{
			((X3DConcreteElement) metadata).setParent(null); // housekeeping, clear prior object
			metadata = null;
		}
	    return this;
	}

		/** Private utility method to access SFNode ProtoInstance field **/
		private ProtoInstance getMetadataProtoInstance()
		{
			return metadataProtoInstance;
		}

	/**
	 * Indicate whether an object is available for inputOutput SFNode field <i>metadata</i>.
	 * @return whether a properly typed node or ProtoInstance or CommentsBlock is available.
	 * @see #getMetadata()
	 * @see #getMetadataProtoInstance()
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#Metadata" target="_blank">X3D Scene Authoring Hints: Metadata Nodes</a>
	 */
	public boolean hasMetadata()
	{
		return (metadata != null) || (metadataProtoInstance != null);
	}
	/**
	 * Provide String value from inputOutput SFString field named <i>name</i>.
	 * <br><br>
	 * <i>Tooltip:</i>  name of the prototype node being instanced. Hint: must match name of corresponding ProtoDeclare or ExternProtoDeclare in this scene. Warning: name must be specified unless this ProtoInstance is a USE node. Do not include name with ProtoInstance USE since it is duplicative (i.e. name is already defined in the DEF instance) and this approach avoids potential name mismatches. Hint: well-defined names can simplify design and debugging through improved author understanding. Hint: X3D Scene Authoring Hints, Naming Conventions &lt;a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#NamingConventions" target="_blank"&gt;https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#NamingConventions&lt;/a&gt;  * <br>

	 * @return value of name field
	 */
	// ($isInterface=false, $isException=false, $isServiceInterface=false, $isX3dStatement=false, $isClassX3dStatement=true, preceding-sibling::Inheritance=, $baseType=X3DPrototypeInstance, $additionalInheritanceBaseType=X3DChildNode, $thisClassName=ProtoInstance, $CamelCaseName=Name, $memberObjectName=name, $memberObjectName=name)

	public String getName()
	{
		return name;
	}

	/**
	 * Accessor method to assign String value to inputOutput SFString field named <i>name</i>.
	 * <br><br>
	 * <i>Tooltip:</i>  name of the prototype node being instanced. Hint: must match name of corresponding ProtoDeclare or ExternProtoDeclare in this scene. Warning: name must be specified unless this ProtoInstance is a USE node. Do not include name with ProtoInstance USE since it is duplicative (i.e. name is already defined in the DEF instance) and this approach avoids potential name mismatches. Hint: well-defined names can simplify design and debugging through improved author understanding. Hint: X3D Scene Authoring Hints, Naming Conventions &lt;a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#NamingConventions" target="_blank"&gt;https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#NamingConventions&lt;/a&gt;  * <br>

	 * <br><br>@see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#NamingConventions" target="_blank">X3D Scene Authoring Hints: Naming Conventions</a>
	 * @param newValue is new value for the name field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */

	public final ProtoInstance setName(String newValue)
	{
		// set-newValue-validity-checks #0.a
		if (newValue == null)
			newValue = new String();
		// Check that newValue parameter meets naming requirements before assigning to ProtoInstance name
		if (!newValue.isEmpty() && !org.web3d.x3d.jsail.fields.SFString.isNMTOKEN(newValue))
		{
			throw new org.web3d.x3d.sai.InvalidFieldValueException("ProtoInstance name newValue='" + newValue + "'" + 
				" has illegal name value, cannot be empty and must be defined with valid NMTOKEN name string" + 
				" (with legal characters and no embedded whitespace).");
		}

        if (!newValue.isEmpty() && !org.web3d.x3d.jsail.fields.SFString.meetsX3dInteroperabilityNamingConventions(newValue))
		{
			if (ConfigurationProperties.isConsoleOutputVerbose())
                          System.out.println("[warning] ProtoInstance name newValue='" + newValue + "'" + 
				" has name value that does not meet X3D naming conventions.");
		}
		name = newValue;
		return this;
	}

	/**
	 * Assign typed object value to SFString name field, similar to {@link #setName(String)}.
	 * @param newValue is new value for the name field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public ProtoInstance setName(SFString newValue)
	{
			// set-newValue-validity-checks #1 gets handled by set-primitive method
			setName(newValue.getPrimitiveValue());
			return this;
	}
	/**
	 * Provide String value from inputOutput SFString field named <i>DEF</i>.
	 * <br><br>
	 * <i>Tooltip:</i>  DEF defines a unique ID name for this node, referenceable by other nodes. Hint: descriptive DEF names improve clarity and help document a model. Hint: well-defined names can simplify design and debugging through improved author understanding. Hint: X3D Scene Authoring Hints, Naming Conventions &lt;a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#NamingConventions" target="_blank"&gt;https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#NamingConventions&lt;/a&gt;  * <br>

	 * @return value of DEF field
	 */
	// ($isInterface=false, $isException=false, $isServiceInterface=false, $isX3dStatement=false, $isClassX3dStatement=true, preceding-sibling::Inheritance=, $baseType=X3DPrototypeInstance, $additionalInheritanceBaseType=X3DChildNode, $thisClassName=ProtoInstance, $CamelCaseName=DEF, $memberObjectName=DEF, $memberObjectName=DEF)

	public String getDEF()
	{
		// override abstract method in X3DConcreteNode
		return super.getDEF();
	}

	/**
	 * Accessor method to assign String value to inputOutput SFString field named <i>DEF</i>.
	 * <br><br>
	 * <i>Tooltip:</i>  DEF defines a unique ID name for this node, referenceable by other nodes. Hint: descriptive DEF names improve clarity and help document a model. Hint: well-defined names can simplify design and debugging through improved author understanding. Hint: X3D Scene Authoring Hints, Naming Conventions &lt;a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#NamingConventions" target="_blank"&gt;https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#NamingConventions&lt;/a&gt;  * <br>

	 * <br><br>
	 *  Note that setting the DEF value clears the USE value.
	 * @param newValue is new value for the DEF field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */

	public final ProtoInstance setDEF(String newValue)
	{
		// set-newValue-validity-checks #0.a
		if (newValue == null)
			newValue = new String();
		// Check that newValue parameter meets naming requirements before assigning to ProtoInstance DEF
		if (!newValue.isEmpty() && !org.web3d.x3d.jsail.fields.SFString.isNMTOKEN(newValue))
		{
			throw new org.web3d.x3d.sai.InvalidFieldValueException("ProtoInstance DEF newValue='" + newValue + "'" + 
				" has illegal name value, cannot be empty and must be defined with valid NMTOKEN name string" + 
				" (with legal characters and no embedded whitespace).");
		}
		setConcreteUSE(""); // ensure that no previous USE value remains
		setConcreteDEF(newValue); // private superclass methods
		return this;
	}

	/**
	 * Assign typed object value to SFString DEF field, similar to {@link #setDEF(String)}.
	 * @param newValue is new value for the DEF field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public ProtoInstance setDEF(SFString newValue)
	{
			// set-newValue-validity-checks #1 gets handled by set-primitive method
			setDEF(newValue.getPrimitiveValue());
			return this;
	}
	/**
	 * Provide String value from inputOutput SFString field named <i>USE</i>.
	 * <br><br>
	 * <i>Tooltip:</i>  USE means reuse an already DEF-ed node ID, excluding all child nodes and all other attributes (except for containerField, which can have a different value). Hint: USE references to previously defined DEF geometry (instead of duplicating nodes) can improve performance. Warning: do NOT include any child nodes, a DEF attribute, or any other attribute values (except for containerField) when defining a USE attribute. Warning: each USE value must match a corresponding DEF value that is defined earlier in the scene. Warning: do not define a name field in a USE node, the correct name can already be found in the original ProtoInstance DEF node.  * <br>

	 * @return value of USE field
	 */
	// ($isInterface=false, $isException=false, $isServiceInterface=false, $isX3dStatement=false, $isClassX3dStatement=true, preceding-sibling::Inheritance=, $baseType=X3DPrototypeInstance, $additionalInheritanceBaseType=X3DChildNode, $thisClassName=ProtoInstance, $CamelCaseName=USE, $memberObjectName=USE, $memberObjectName=USE)

	public String getUSE()
	{
		// override abstract method in X3DConcreteNode
		return super.getUSE();
	}

	/**
	 * Accessor method to assign String value to inputOutput SFString field named <i>USE</i>.
	 * <br><br>
	 * <i>Tooltip:</i>  USE means reuse an already DEF-ed node ID, excluding all child nodes and all other attributes (except for containerField, which can have a different value). Hint: USE references to previously defined DEF geometry (instead of duplicating nodes) can improve performance. Warning: do NOT include any child nodes, a DEF attribute, or any other attribute values (except for containerField) when defining a USE attribute. Warning: each USE value must match a corresponding DEF value that is defined earlier in the scene. Warning: do not define a name field in a USE node, the correct name can already be found in the original ProtoInstance DEF node.  * <br>

	 * <br><br>
	 * <i>Note:</i> each <code>USE</code> node is still an independent object, with the <code>USE</code> value matching the <code>DEF</code> value in the preceding object. 
	 * <br><br>
	 * <i>Warning:</i> invoking the <code>setUSE()</code> method on this node resets all other fields to their default values (except for containerField) and also releases all child nodes.<br><br>
	 * <i>Warning:</i> no other operations can be performed to modify a USE node other than setting an alternate containerField value.
	 * @param newValue is new value for the USE field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */

	public final ProtoInstance setUSE(String newValue)
	{
		// set-newValue-validity-checks #0.a
		if (newValue == null)
			newValue = new String();
		// Check that newValue parameter meets naming requirements before assigning to ProtoInstance USE
		if (!newValue.isEmpty() && !org.web3d.x3d.jsail.fields.SFString.isNMTOKEN(newValue))
		{
			throw new org.web3d.x3d.sai.InvalidFieldValueException("ProtoInstance USE newValue='" + newValue + "'" + 
				" has illegal name value, cannot be empty and must be defined with valid NMTOKEN name string" + 
				" (with legal characters and no embedded whitespace).");
		}
		setConcreteDEF(""); // ensure that no previous DEF value remains
		setConcreteUSE(newValue); // private superclass method
		return this;
	}

	/**
	 * Assign typed object value to SFString USE field, similar to {@link #setUSE(String)}.
	 * @param newValue is new value for the USE field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public ProtoInstance setUSE(SFString newValue)
	{
			// set-newValue-validity-checks #1 gets handled by set-primitive method
			setUSE(newValue.getPrimitiveValue());
			return this;
	}
	/**
	 * Provide String value from inputOutput SFString field named <i>class</i>.
	 * @return value of class field
	 */
	// ($isInterface=false, $isException=false, $isServiceInterface=false, $isX3dStatement=false, $isClassX3dStatement=true, preceding-sibling::Inheritance=, $baseType=X3DPrototypeInstance, $additionalInheritanceBaseType=X3DChildNode, $thisClassName=ProtoInstance, $CamelCaseName=CssClass, $memberObjectName=cssClass, $memberObjectName=cssClass)

	public String getCssClass()
	{
		// override abstract method in X3DConcreteNode
		return super.getCssClass();
	}

	/**
	 * Accessor method to assign String value to inputOutput SFString field named <i>class</i>.
	 * @param newValue is new value for the class field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */

	public final ProtoInstance setCssClass(String newValue)
	{
		// set-newValue-validity-checks #0.a
		if (newValue == null)
			newValue = new String(); // Principle of Least Astonishment (POLA) #5
			// https://en.wikipedia.org/wiki/Principle_of_least_astonishment
		setConcreteCssClass(newValue); // private superclass method
		return this;
	}

	/**
	 * Assign typed object value to SFString cssClass field, similar to {@link #setCssClass(String)}.
	 * @param newValue is new value for the class field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public ProtoInstance setCssClass(SFString newValue)
	{
			// set-newValue-validity-checks #1 gets handled by set-primitive method
			setCssClass(newValue.getPrimitiveValue());
			return this;
	}
	/**
	 * Provide String value from inputOutput SFString field named <i>id</i>.
	 * @return value of id field
	 */
	// ($isInterface=false, $isException=false, $isServiceInterface=false, $isX3dStatement=false, $isClassX3dStatement=true, preceding-sibling::Inheritance=, $baseType=X3DPrototypeInstance, $additionalInheritanceBaseType=X3DChildNode, $thisClassName=ProtoInstance, $CamelCaseName=HtmlID, $memberObjectName=HtmlID, $memberObjectName=HtmlID)
	public String getHtmlID()
	{
		// override abstract method in X3DConcreteNode
		return super.getHtmlID();
	}

	/**
	 * Accessor method to assign String value to inputOutput SFString field named <i>id</i>.
	 * @param newValue is new value for the id field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public final ProtoInstance setHtmlID(String newValue)
	{
		// set-newValue-validity-checks #0.a
		if (newValue == null)
			newValue = new String(); // Principle of Least Astonishment (POLA) #5
			// https://en.wikipedia.org/wiki/Principle_of_least_astonishment
		setConcreteHtmlID(newValue); // private superclass method
		return this;
	}

	/**
	 * Assign typed object value to SFString HtmlID field, similar to {@link #setHtmlID(String)}.
	 * @param newValue is new value for the id field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public ProtoInstance setHtmlID(SFString newValue)
	{
			// set-newValue-validity-checks #1 gets handled by set-primitive method
			setHtmlID(newValue.getPrimitiveValue());
			return this;
	}
	/**
	 * Provide String value from inputOutput SFString field named <i>style</i>.
	 * @return value of style field
	 */
	// ($isInterface=false, $isException=false, $isServiceInterface=false, $isX3dStatement=false, $isClassX3dStatement=true, preceding-sibling::Inheritance=, $baseType=X3DPrototypeInstance, $additionalInheritanceBaseType=X3DChildNode, $thisClassName=ProtoInstance, $CamelCaseName=CssStyle, $memberObjectName=cssClass, $memberObjectName=cssClass)
	public String getCssStyle()
	{
		// override abstract method in X3DConcreteNode
		return super.getCssStyle();
	}

	/**
	 * Accessor method to assign String value to inputOutput SFString field named <i>style</i>.
	 * @param newValue is new value for the style field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public final ProtoInstance setCssStyle(String newValue)
	{
		// set-newValue-validity-checks #0.a
		if (newValue == null)
			newValue = new String(); // Principle of Least Astonishment (POLA) #5
			// https://en.wikipedia.org/wiki/Principle_of_least_astonishment
		setConcreteCssStyle(newValue); // private superclass method
		return this;
	}

	/**
	 * Assign typed object value to SFString cssClass field, similar to {@link #setCssStyle(String)}.
	 * @param newValue is new value for the style field.
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public ProtoInstance setCssStyle(SFString newValue)
	{
			// set-newValue-validity-checks #1 gets handled by set-primitive method
			setCssStyle(newValue.getPrimitiveValue());
			return this;
	}

	// Additional utility methods for this class ==============================

	/**
	 * Assign a USE reference to another DEF node of same node type, similar to {@link #setUSE(String)}.
	 * <br ><br >
	 * <i>Warning:</i> note that the <code>setUSE()</code> method on this node resets all other fields to their default values (except for containerField) and also releases all child nodes.
	 * <br><br>
	 * <i>Warning:</i> no other operations can be performed to modify a USE node other than setting an alternate containerField value.
	 * @param DEFnode must have a DEF value defined
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public ProtoInstance setUSE(ProtoInstance DEFnode)
	{
		if (DEFnode.getDEF().isEmpty())
		{
			String errorNotice = "setUSE(DEFnode) invoked on ProtoInstance" +
				" that has no DEF name defined, thus a copy cannot be referenced as a USE node";
			validationResult.append(errorNotice).append("\n");
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
		setUSE(DEFnode.getDEF());
		return this;
	}
	/**
	 * Utility constructor that assigns ProtoInstance name after initializing member variables with default values.
	 * @param prototypeName initial name for this ProtoInstance statement
	 */
	public ProtoInstance (String prototypeName)
	{
		initialize();
		setName(prototypeName); // apply checks
	}

	/**
	 * Utility constructor that assigns ProtoInstance prototypeName and DEF after initializing member variables with default values.
	 * @param prototypeName initial name for this ProtoInstance statement
	 * @param DEFlabel unique DEF name for this X3D node
	 */
	public ProtoInstance (String prototypeName, String DEFlabel)
	{
		initialize();
		setName(prototypeName); // apply checks
		setDEF (DEFlabel);      // apply checks
	}

	/**
	 * Utility method that provides ProtoInstance containerField value
	 * <i>Hint:</i> ProtoInstance containerField is important for determining parent-child node relationships.
	 * @see org.web3d.x3d.jsail.X3DConcreteNode#setContainerFieldOverride(String)
	 * @return containerField value
	 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ProtoInstance.containerField">X3D Tooltips: ProtoInstance.containerField</a>
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#containerField">X3D Scene Authoring Hints: containerField</a>
	 */
	public String getContainerField()
	{
		return getContainerFieldOverride();
	}

	/**
	 * Utility method that assigns ProtoInstance containerField value
	 * <i>Hint:</i> ProtoInstance containerField is important for determining parent-child node relationships.
	 * @param containerFieldName overriding value
	 * @see org.web3d.x3d.jsail.X3DConcreteNode#setContainerFieldOverride(String)
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ProtoInstance.containerField">X3D Tooltips: ProtoInstance.containerField</a>
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#containerField">X3D Scene Authoring Hints: containerField</a>
	 */
	public ProtoInstance setContainerField(String containerFieldName)
	{
		setContainerFieldOverride(containerFieldName);
		return this;
	}

	/**
	 * Utility method to find original ProtoInstance name if this ProtoInstance is a USE node (which does not include name).
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public String getOriginalName()
	{
		if (!getName().isEmpty())
		    return getName();

		String          originalName = getName();
		X3DConcreteNode originalProtoInstanceNode = null;
		if (originalName.isEmpty())
		{
            if (findAncestorScene() != null)
                 originalProtoInstanceNode = findAncestorScene().findNodeByDEF(getUSE());
         
			if (!ConfigurationProperties.isCreationConnectionValidationExceptionAllowed())
            {
                if (originalProtoInstanceNode == null)
                {
                    String errorNotice = ConfigurationProperties.ERROR_NODE_NOT_FOUND + 
                            ": unable to locate reference ProtoInstance DEF='" + getUSE() + "' for ProtoInstance USE='" + getUSE() + "'";
                    validationResult.append(errorNotice);
                    throw new org.web3d.x3d.sai.InvalidProtoException(errorNotice); // report error
                }
                else if (!(originalProtoInstanceNode instanceof ProtoInstance))
                {
                    String errorNotice = ConfigurationProperties.ERROR_NODE_NOT_FOUND + 
                            ": unable to locate correct node reference, ProtoInstance USE='" + getUSE() + "' mistakenly matches " + originalProtoInstanceNode.getClass() + " DEF='" + getUSE() + "'";
                    validationResult.append(errorNotice);
                    throw new org.web3d.x3d.sai.InvalidProtoException(errorNotice); // report error
                }
                originalName = ((ProtoInstance)originalProtoInstanceNode).getName();
			}
		}
		return originalName;
	}

	/**
	 * Determine whether a corresponding ProtoDeclare with same name is connected in this scene graph.
	 * @see #hasExternProtoDeclare
	 * @see #getProtoDeclare
	 * @see #getExternProtoDeclare
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public boolean hasProtoDeclare()
	{
		// check for corresponding declaration
		if  (findAncestorScene() == null)
			 return false;
		// ProtoInstance USE nodes don't include name, so utility method getOriginalName() helps here
		X3DConcreteElement matchingDeclaration = null;
        if (!getOriginalName().isEmpty())
             matchingDeclaration = findAncestorScene().findElementByNameValue(getOriginalName(), ProtoDeclare.NAME); 
		if ((matchingDeclaration != null) && (matchingDeclaration instanceof org.web3d.x3d.jsail.Core.ProtoDeclare))
		{
			 referenceProtoDeclare = (ProtoDeclare) matchingDeclaration;
			 return true;
		}
		else return false;
	}
		 
	/**
	 * Determine whether a corresponding ExternProtoDeclare with same name is connected in this scene graph.
	 * @see #hasProtoDeclare
	 * @see #getProtoDeclare
	 * @see #getExternProtoDeclare
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public boolean hasExternProtoDeclare()
	{
		// check for corresponding declaration
		if  (findAncestorScene() == null)
			 return false;
		// ProtoInstance USE nodes don't include name, so utility method getOriginalName() helps here
		X3DConcreteElement matchingDeclaration = null;
        if  (!getOriginalName().isEmpty())
             matchingDeclaration = findAncestorScene().findElementByNameValue(getOriginalName(), ExternProtoDeclare.NAME); 
		if ((matchingDeclaration != null) && (matchingDeclaration instanceof org.web3d.x3d.jsail.Core.ExternProtoDeclare))
		{
			 referenceExternProtoDeclare = (ExternProtoDeclare) matchingDeclaration;
			 return true;
		}
		else return false;
	}

	/**
	 * Provide corresponding ProtoDeclare with same name if connected in this scene graph.
	 * @see #hasProtoDeclare
	 * @see #hasExternProtoDeclare
	 * @see #getExternProtoDeclare
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public ProtoDeclare getProtoDeclare()
	{
		if  (hasProtoDeclare())
			 return referenceProtoDeclare;
		else return null;
	}
	/**
	 * Provide corresponding ExternProtoDeclare with same name if connected in this scene graph.
	 * @see #hasProtoDeclare
	 * @see #hasExternProtoDeclare
	 * @see #getProtoDeclare
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public ExternProtoDeclare getExternProtoDeclare()
	{
		if  (hasExternProtoDeclare())
			 return referenceExternProtoDeclare;
		else return null;
	}

	/**
	 * This method helps determine node type of ProtoInstance by inspecting first node within corresponding ProtoDeclare ProtoBody.
	 * @see <a href="https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-IS.proof/Part01/concepts.html#PROTOdefinitionsemantics">X3D Abstract Specification: 4.4.4.3 PROTO definition semantics</a>
	 * @see ProtoDeclare#getNodeType()
	 * @see ProtoBody#getNodeType()
	 * @see ExternProtoDeclare#getNodeType()
	 * @see Scene
		 
	 * @return name of X3D node corresponding to node type for this ProtoInstance statement
	 */
	public String getNodeType()
	{
		// check for corresponding declaration
		String errorNotice = new String();
		if (findAncestorScene() == null)
		{
			if (!ConfigurationProperties.isCreationConnectionValidationExceptionAllowed())
			{
				errorNotice = ConfigurationProperties.ERROR_UNKNOWN_PROTOINSTANCE_NODE_TYPE + ": " +
							   "ProtoInstance name='" + getOriginalName() + "' must first be connected to Scene scene graph in order to get checked.";
				validationResult.append(errorNotice).append("\n");
				return ConfigurationProperties.ERROR_NOT_CONNECTED_TO_SCENE_GRAPH + "_PrototypeNotFound"; // node type not found
			}
			else return errorNotice;
		}
		// ProtoInstance USE nodes don't include name, so utility method getOriginalName() helps here
		X3DConcreteElement matchingDeclaration = null;
		if ((findAncestorScene() != null) && !getOriginalName().isEmpty())
        {
            // first try ProtoDeclare
			matchingDeclaration = findAncestorScene().findElementByNameValue(getOriginalName(), ProtoDeclare.NAME); 
            if ((matchingDeclaration != null) && (matchingDeclaration instanceof org.web3d.x3d.jsail.Core.ProtoDeclare))
            {
                if (getContainerFieldOverride().isEmpty()) // fall back to use ProtoDeclare default value since actual value not provided in source model
                {
					String computedContainerField = ((ProtoDeclare) matchingDeclaration).getContainerFieldDefault();
                    setContainerFieldOverride(computedContainerField);

                    if (!computedContainerField.equals(ProtoInstance.containerField_DEFAULT_VALUE))
					{
						String warningNotice = ConfigurationProperties.WARNING_MESSAGE + ": " + 
							"ProtoInstance name='" + getOriginalName() + "' DEF='" + getDEF() + "' USE='" + getUSE() + 
							"' is missing containerField relationship to parent node, assuming " +
							"containerField='" + getContainerFieldOverride() + "' from initial node in corresponding ProtoDeclare. " +
							"Need to fix ProtoInstance definition in model source.";
						validationResult.append(warningNotice).append("\n");
						System.out.println(warningNotice);
					}
                }
                return ((ProtoDeclare) matchingDeclaration).getNodeType();
            }
            // next try ExternProtoDeclare
            matchingDeclaration = findAncestorScene().findElementByNameValue(getOriginalName(), ExternProtoDeclare.NAME);
            if ((matchingDeclaration != null) && (matchingDeclaration instanceof org.web3d.x3d.jsail.Core.ExternProtoDeclare))
            {
                // no way to determine default containerField
                return ((ExternProtoDeclare) matchingDeclaration).getNodeType();
            }
        }
        if  (ConfigurationProperties.isCreationConnectionValidationExceptionAllowed())
             return "";
        else
        {
            errorNotice = ConfigurationProperties.ERROR_UNKNOWN_PROTOINSTANCE_NODE_TYPE + ": ";
            if  (hasUSE())
                 errorNotice += "ProtoInstance USE='" + getUSE();
            else errorNotice += "ProtoInstance name='" + getName();
            errorNotice += "' has no corresponding ProtoDeclare or ExternProtoDeclare to provide type.";
            validationResult.append(errorNotice).append("\n");
            return errorNotice; // node type not found
        }
        

	}

	/**
	 * Add comment as String to contained commentsList.
	 * @param newComment initial value
	 * @see <a href="https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-IS.proof/Part01/components/core.html#Organization">X3D Architecture, clause 7.2.5.1 Organization</a>
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	/* @Override */
	public ProtoInstance addComments (String newComment)
	{
		commentsList.add(newComment);
		return this;
	}
	/**
	 * Add comments as String[] array to contained commentsList.
	 * @param newComments array of comments
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
setAttribute method invocations).
	 */
	/* @Override */
	public ProtoInstance addComments (String[] newComments)
	{
		commentsList.addAll(Arrays.asList(newComments));
		return this;
	}
	/**
	 * Add CommentsBlock to contained commentsList.
	 * @param newCommentsBlock block of comments to add
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
setAttribute method invocations).
	 */
	/* @Override */
	public ProtoInstance addComments (CommentsBlock newCommentsBlock)
	{
		commentsList.addAll(newCommentsBlock.toStringList());
		return this;
	}
	/**
	 * Utility method to adjust field synonyms
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#fieldNameChanges">X3D Scene Authoring Hints: X3D4 Field name changes for improved consistency</a>
     */
    private void handleFieldSynonyms()
    {
            // no synonyms to handle
	}
		
	/**
	 * Recursive method to provide X3D string serialization of this model subgraph, utilizing XML encoding and conforming to X3D Canonical Form.
	 * @param indentLevel number of levels of indentation for this element
	 * @see X3D#FILE_EXTENSION_X3D
	 * @see X3D#FILE_EXTENSION_XML
	 * @see X3D#toStringXML()
	 * @see X3D#toFileXML(String)
	 * @see X3D#toFileX3D(String)
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-1/V3.3/Part01/X3D_XML.html">X3D XML Encoding</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-3/V3.3/Part03/concepts.html#X3DCanonicalForm" target="blank">X3D Compressed Binary Encoding: X3D Canonical Form</a>
	 * @see <a href="https://www.web3d.org/x3d/tools/canonical/doc/x3dTools.htm">X3D Canonicalization (C14N) Tool</a>
	 * @return X3D string
	 */
	@Override
	public String toStringX3D(int indentLevel)
	{
		boolean hasAttributes = true; // TODO check for non-default attribute values
		boolean      hasChild = (IS != null) || (!fieldValueList.isEmpty()) || (IS != null) || (metadata != null) || (metadataProtoInstance != null) || !commentsList.isEmpty();
		handleFieldSynonyms(); // adjust containerField values for X3D3 differences, if any
		StringBuilder indent = new StringBuilder();
		int  indentIncrement = ConfigurationProperties.getIndentIncrement();
		char indentCharacter = ConfigurationProperties.getIndentCharacter();
		for (int i = 0; i < indentLevel; i++)
			indent.append(indentCharacter); // level of indentation for this level

		StringBuilder stringX3D = new StringBuilder();

		stringX3D.append(indent).append("<ProtoInstance"); // start opening tag
		if (hasAttributes)
		{
			if (!getDEF().equals(DEF_DEFAULT_VALUE) && !hasUSE())
			{
				stringX3D.append(" DEF='").append(SFString.toString(getDEF())).append("'");
			}
			
			if (!getUSE().equals(USE_DEFAULT_VALUE))
			{
				stringX3D.append(" USE='").append(SFString.toString(getUSE())).append("'");
			}
			
			// note special case for ProtoInstance, do not include name field if USE is present
			if (!getName().equals(NAME_DEFAULT_VALUE) && !hasUSE())
			{
				stringX3D.append(" name='").append(SFString.toString(getName())).append("'");
			}
			
            // containerField is critical for ProtoInstance relationship to parent node
            if (!getContainerFieldOverride().isEmpty() && !getContainerFieldOverride().equals(this.getContainerFieldDefault())) // output if not default ProtoInstance containerField
            {
                    stringX3D.append(" containerField='").append(getContainerFieldOverride()).append("'");
            }

            // test attributes

			if ((!getCssClass().equals(CLASS_DEFAULT_VALUE) || !ConfigurationProperties.getStripDefaultAttributes()) /* isSingleValueType */)
			{
				stringX3D.append(" class='").append(new SFString(getCssClass()).toStringX3D()).append("'");
			}
			if ((!getHtmlID().equals(ID_DEFAULT_VALUE) || !ConfigurationProperties.getStripDefaultAttributes()) /* isSingleValueType */)
			{
				stringX3D.append(" id='").append(new SFString(getHtmlID()).toStringX3D()).append("'");
			}
			if ((!getCssStyle().equals(STYLE_DEFAULT_VALUE) || !ConfigurationProperties.getStripDefaultAttributes()) /* isSingleValueType */)
			{
				stringX3D.append(" style='").append(new SFString(getCssStyle()).toStringX3D()).append("'");
			}
		}
		if ((hasChild)) // has contained node(s), comment(s), IS/connect and/or source code
		{
			stringX3D.append(">").append("\n"); // finish opening tag

			// recursively iterate over child elements
			if (!commentsList.isEmpty())
			{
				CommentsBlock commentsBlock = new CommentsBlock(commentsList);
				stringX3D.append(commentsBlock.toStringX3D(indentLevel + indentIncrement));
			}
			if      (metadata != null)
			{
				stringX3D.append(((X3DConcreteElement)metadata).toStringX3D(indentLevel + indentIncrement));
			}
			else if (metadataProtoInstance != null)
			{
					 stringX3D.append(((X3DConcreteElement)metadataProtoInstance).toStringX3D(indentLevel + indentIncrement));
			}
			if      (IS != null)
			{
				stringX3D.append(((X3DConcreteElement)IS).toStringX3D(indentLevel + indentIncrement));
			}
			for (fieldValue element : fieldValueList)
			{
				stringX3D.append(((X3DConcreteElement)element).toStringX3D(indentLevel + indentIncrement));
			}
			stringX3D.append(indent).append("</ProtoInstance>").append("\n"); // finish closing tag
		}
		else
		{
			stringX3D.append("/>").append("\n"); // otherwise finish singleton tag
		}
		return stringX3D.toString();
	}
// ($name=ProtoInstance, isX3dStatement=true, isClassX3dStatement=true, $hasChild=true, $hasAttributes=true
	/**
	 * Recursive method to provide ClassicVRML string serialization.
	 * @param indentLevel number of levels of indentation for this element
	 * @see X3D#FILE_EXTENSION_CLASSICVRML
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dResources.html#VRML">X3D Resources: Virtual Reality Modeling Language (VRML) 97</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-2/V3.3/Part02/X3D_ClassicVRML.html">Extensible 3D (X3D) encodings Part 2: Classic VRML encoding</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-2/V3.3/Part02/grammar.html">Extensible 3D (X3D) encodings Part 2: Classic VRML encoding, Annex A: Grammar</a>
	 * @return ClassicVRML string
	 */
	@Override     
	public String toStringClassicVRML(int indentLevel)
	{
		StringBuilder stringClassicVRML = new StringBuilder();
		boolean hasAttributes = true; // TODO further refinement
		boolean      hasChild = (IS != null) || (!fieldValueList.isEmpty()) || (!fieldValueList.isEmpty()) || (!fieldValueList.isEmpty()) || (IS != null) || (!fieldValueList.isEmpty()) || (metadata != null) || (metadataProtoInstance != null) || !commentsList.isEmpty();
        if (!serializingVRML97output)
            handleFieldSynonyms(); // adjust containerField values for X3D3 differences, if any
		StringBuilder  indent = new StringBuilder();
		char  indentCharacter = ConfigurationProperties.getIndentCharacter();
		int   indentIncrement = ConfigurationProperties.getIndentIncrement();
		for (int i = 0; i < indentLevel; i++)
			indent.append(indentCharacter); // level of indentation for this level

		if (!getDEF().equals(DEF_DEFAULT_VALUE))
		{
			stringClassicVRML.append("DEF ").append(SFString.toString(getDEF())).append(" ");
		}
		if (!getUSE().equals(USE_DEFAULT_VALUE))
		{
			 stringClassicVRML.append("USE ").append(SFString.toString(getUSE())).append("\n");
		}
		else // only have further output if not a USE node
		{
			if (getName().isEmpty())
				 stringClassicVRML.append("NoNameFoundError");
			else stringClassicVRML.append(getName());
			stringClassicVRML.append(" { "); // define ProtoInstance node name, fieldValue content follows

			if (hasAttributes)
			{
				if (!getCssClass().equals(CLASS_DEFAULT_VALUE) || !ConfigurationProperties.getStripDefaultAttributes())
				{
					stringClassicVRML.append(" # class ").append("\"").append(SFString.toString(getCssClass())).append("\"").append("\n");
				}
				if (!getHtmlID().equals(ID_DEFAULT_VALUE) || !ConfigurationProperties.getStripDefaultAttributes())
				{
					stringClassicVRML.append(" # id ").append("\"").append(SFString.toString(getHtmlID())).append("\"").append("\n");
				}
				if (!getName().equals(NAME_DEFAULT_VALUE) || !ConfigurationProperties.getStripDefaultAttributes())
				{
					stringClassicVRML.append("name ").append("\"").append(SFString.toString(getName())).append("\"").append("\n");
				}
				if (!getCssStyle().equals(STYLE_DEFAULT_VALUE) || !ConfigurationProperties.getStripDefaultAttributes())
				{
					stringClassicVRML.append(" # style ").append("\"").append(SFString.toString(getCssStyle())).append("\"").append("\n");
				}
			}
		}
		if (hasChild) // has contained node(s), comment(s), IS/connect and/or source code
		{
			// recursively iterate over child elements
			if (metadata != null)
			{
				stringClassicVRML.append(indentCharacter).append("metadata").append(" "); // containerField for SFNode
				stringClassicVRML.append(((X3DConcreteElement) metadata).toStringClassicVRML(indentLevel + indentIncrement));
				stringClassicVRML.append(indent); // end SFNode
			}
			else if (metadataProtoInstance != null)
			{
				stringClassicVRML.append(indentCharacter).append("metadata").append(" "); // containerField for SFNode
				stringClassicVRML.append(((X3DConcreteElement) metadataProtoInstance).toStringClassicVRML(indentLevel + indentIncrement));
				stringClassicVRML.append(indent); // end SFNode ProtoInstance
			}
			if (IS != null)
			{
				stringClassicVRML.append(((X3DConcreteElement) IS).toStringClassicVRML(indentLevel)); // end X3D statement
			}
			for (fieldValue element : fieldValueList)
			{
				stringClassicVRML.append(((X3DConcreteElement)element).toStringClassicVRML(indentLevel));
			}
			if (!commentsList.isEmpty())
			{
				CommentsBlock commentsBlock = new CommentsBlock(commentsList);
				stringClassicVRML.append(commentsBlock.toStringClassicVRML(indentLevel));
			}
		}
		stringClassicVRML.append(indent).append("}").append("\n"); // finish closing tag
		return stringClassicVRML.toString();
	}

    
    private boolean serializingVRML97output = false; // whether to avoid handling field name synonyms
                                
	/**
	 * Recursive method to provide VRML97 string serialization.
	 * @param indentLevel number of levels of indentation for this element
	 * @see X3D#FILE_EXTENSION_VRML97
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dResources.html#VRML">X3D Resources: Virtual Reality Modeling Language (VRML) 97</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/14772/V2.0/index.html">Virtual Reality Modeling Language (VRML) 97 specification</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/14772-1/V2.1/index.html">VRML 97 v2.1 Amendment</a>
	 * @return VRML97 string
	 */
	@Override
	public String toStringVRML97(int indentLevel)
	{
        serializingVRML97output = true;
        String result = toStringClassicVRML(indentLevel);
        serializingVRML97output = false;
                                
		return result;
	}

	/**
	 * Recursive method to provide object reference to node or statement by name attribute, if found as part of this element or in a contained element.
	 * Elements with name fields include meta, Metadata* nodes, field/fieldValue, ProtoDeclare/ExternProtoDeclare/ProtoInstance, CAD and HAnim nodes.
	 * <br ><br >
	 * <i>Warning:</i> first start with findAncestorScene() to check entire scene graph, or findAncestorX3D() to check entire model document.
	 * <br ><br >
	 * <i>Warning:</i> more than one element may be found that has the same name, this method does not handle that case.
	 * @see #findNodeByDEF(String)
	 * @see X3DConcreteElement#hasAncestorScene()
	 * @see org.web3d.x3d.jsail.X3DConcreteElement#findAncestorX3D()
	 * @param nameValue is value of the name field being searched for in this element and child elements(if any)
	 * @return object reference to found element, null otherwise
	 */
	/* @Override */
	public X3DConcreteElement findElementByNameValue(String nameValue)
	{
		return findElementByNameValue(nameValue, ""); // empty string is wildcard, any element match is allowed
	}
								
	/**
	 * Recursive method to provide object reference to node or statement by name attribute, if found as part of this element or in a contained element.
	 * Elements with name fields include meta, Metadata* nodes, field/fieldValue, ProtoDeclare/ExternProtoDeclare/ProtoInstance, CAD and HAnim nodes.
	 * <br ><br >
	 * <i>Warning:</i> first start with findAncestorScene() to check entire scene graph, or findAncestorX3D() to check entire model document.
	 * <br ><br >
	 * <i>Warning:</i> more than one element may be found that has the same name, this method does not handle that case.
	 * @see #findNodeByDEF(String)
	 * @see X3DConcreteElement#hasAncestorScene()
	 * @see org.web3d.x3d.jsail.X3DConcreteElement#findAncestorX3D()
	 * @param nameValue is value of the name field being searched for in this element and child elements(if any)
	 * @param elementName identifies the element of interest (meta MetadataString ProtoDeclare CADassembly ProtoInstance HAnimHumanoid etc.)
	 * @return object reference to found element, null otherwise
	 */
	/* @Override */
	public X3DConcreteElement findElementByNameValue(String nameValue, String elementName)
	{
		if ((nameValue == null) || nameValue.isEmpty())
		{
			String errorNotice = "findElementByNameValue(\"\", " + elementName + ") cannot use empty string to find a name attribute";
			validationResult.append(errorNotice).append("\n");
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
		if (elementName == null)
			elementName = "";
		if (getName().equals(nameValue) &&
			(elementName.isEmpty() || elementName.equals(getElementName())))
			return this;
		X3DConcreteElement referenceElement;
		for (fieldValue element : fieldValueList)
		{
			if (element instanceof X3DConcreteElement)
			{
				referenceElement = ((X3DConcreteElement) element).findElementByNameValue(nameValue, elementName);
				if ((referenceElement != null))
					return referenceElement;
			}
		}
		if (IS != null)
		{
			referenceElement = ((X3DConcreteElement) IS).findElementByNameValue(nameValue, elementName);
			if (referenceElement != null)
				return referenceElement;
		}
		if (metadata != null)
		{
			referenceElement = ((X3DConcreteElement) metadata).findElementByNameValue(nameValue, elementName);
			if (referenceElement != null)
				return referenceElement;
		}
		return null; // not found
	}
	/**
	 * Recursive method to determine whether node or statement with given name attribute is found, meaning both objects are attached to same scene graph.
	 * @param nameValue is value of the name field being searched for in this element and child elements(if any)
	 * @param elementName identifies the element of interest (meta MetadataString ProtoDeclare CADassembly ProtoInstance HAnimHumanoid etc.)
	 * @see #findElementByNameValue(String, String)
	 * @return whether node is found
	 */
	public boolean hasElementByNameValue(String nameValue, String elementName)
	{
		return (findElementByNameValue(nameValue, elementName) != null);
	}

	/**
	 * Recursive method to provide object reference to node by DEF, if found as this node or in a contained node.
	 * <br ><br >
	 * <i>Warning:</i> first start with findAncestorScene() to check entire scene graph, or findAncestorX3D() to check entire model document.
	 * <br ><br >
	 * <i>Warning:</i> more than one element may be found that has the same DEF, this method does not handle that case.
	 * @see #findElementByNameValue(String)
	 * @see X3DConcreteElement#hasAncestorScene()
	 * @see org.web3d.x3d.jsail.X3DConcreteElement#findAncestorX3D()
	 * @param DEFvalue is value of the name field being searched for in this element and child elements(if any)
	 * @return object reference to found node, null otherwise
	 */
	/* @Override */
	public X3DConcreteNode findNodeByDEF(String DEFvalue)
	{
		if ((DEFvalue == null) || DEFvalue.isEmpty())
		{
			String errorNotice = "findNodeByDEF(\"\") cannot use empty string to find a name";
			validationResult.append(errorNotice).append("\n");
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
		if (getDEF().equals(DEFvalue))
			return this;
		X3DConcreteNode referenceNode;
		if (IS != null)
		{
			referenceNode = ((X3DConcreteElement) IS).findNodeByDEF(DEFvalue);
			if (referenceNode != null)
				return referenceNode;
		}
		if (metadata != null)
		{
			referenceNode = ((X3DConcreteElement) metadata).findNodeByDEF(DEFvalue);
			if (referenceNode != null)
				return referenceNode;
		}
		if (metadataProtoInstance != null)
		{
			referenceNode = ((X3DConcreteElement) metadataProtoInstance).findNodeByDEF(DEFvalue);
			if (referenceNode != null)
				return referenceNode;
		}
		return null; // not found
	}
	/**
	 * Recursive method to determine whether node with DEFvalue is found, meaning both objects are attached to same scene graph.
	 * @param DEFvalue is value of the name field being searched for in this element and child elements(if any)
	 * @see #findNodeByDEF(String)
	 * @return whether node is found
	 */
	public boolean hasNodeByDEF(String DEFvalue)
	{
		return (findNodeByDEF(DEFvalue) != null);
	}

		
	/**
	 * Recursive method to validate this element plus all contained nodes and statements, 
	 * using both datatype-specification value checks and regular expression (regex) checking of corresponding string values.
	 * @return validation results (if any)
	 */
	@Override
	public String validate()
	{
		validationResult = new StringBuilder(); // prepare for updated results

                getNodeType(); // initialize ProtoInstance containerField, if missing and available in corresponding ProtoDeclare

		setName(getName()); // exercise field checks, simple types
        
        if (getName().isEmpty() && !hasUSE())
        {
            String errorNotice = "ProtoInstance missing name field, which is usually required (since this ProtoInstance is not a USE node).";
            validationResult.append(errorNotice);
            throw new org.web3d.x3d.sai.InvalidProtoException(errorNotice); // report error
        }
	if (!getName().isEmpty() && hasUSE())
        {
            String errorNotice = "ProtoInstance has name field, which is not included in a ProtoInstance USE node.";
            validationResult.append(errorNotice);
            throw new org.web3d.x3d.sai.InvalidProtoException(errorNotice); // report error
        }
        String originalName = getOriginalName(); // call once here, for efficiency and also to allow separate changes

        // check for presence of corresponding ProtoDeclare/ExternProtoDeclare having same name, report if missing
        if (ConfigurationProperties.isDebugModeActive() && hasProtoDeclare())
        {
                if  (hasUSE())
                     validationResult.append("ProtoInstance ").append(originalName).append(" USE='").append(getUSE()).append("' has corresponding ProtoDeclare").append("\n");
                else validationResult.append("ProtoInstance ").append(getName())   .append(" DEF='").append(getDEF()).append("' has corresponding ProtoDeclare").append("\n");
        }
        else if (ConfigurationProperties.isDebugModeActive() && hasExternProtoDeclare())
        {
                if  (hasUSE())
                     validationResult.append("ProtoInstance ").append(originalName).append(" USE='").append(getUSE()).append("' has corresponding ExternProtoDeclare (but node type is unconfirmed)").append("\n");
                else validationResult.append("ProtoInstance ").append(getName())   .append(" DEF='").append(getDEF()).append("' has corresponding ExternProtoDeclare (but node type is unconfirmed)").append("\n");
        }
        else if (hasProtoDeclare() && hasExternProtoDeclare())
        {
                String errorNotice = ConfigurationProperties.ERROR_UNKNOWN_PROTOINSTANCE_NODE_TYPE +
                                "ProtoInstance " + getName() + " has both corresponding ProtoDeclare and ExternProtoDeclare";
                validationResult.append(errorNotice);
                throw new org.web3d.x3d.sai.InvalidProtoException(errorNotice); // report error
        }
        if (getNodeType().startsWith(ConfigurationProperties.ERROR_UNKNOWN_PROTOINSTANCE_NODE_TYPE))
        {
            String errorNotice = getNodeType();
            validationResult.append(errorNotice);
            throw new org.web3d.x3d.sai.InvalidProtoException(errorNotice); // report error
        }
        if (getNodeType().startsWith(ConfigurationProperties.ERROR_UNKNOWN_PROTOINSTANCE_NODE_TYPE))
        {
            String errorNotice = getNodeType();
            validationResult.append(errorNotice);
            throw new org.web3d.x3d.sai.InvalidProtoException(errorNotice); // report error
        }
            // TODO check for legal containerField among allowed getContainerFieldAlternateValues() for given nodeType

		if (!hasUSE()) // be careful! setting DEF via setDEF() method will reset USE value
			setDEF(getDEF()); // exercise field checks, simple types
        
		if (hasUSE()) // be careful! setting USE via setUSE() method resets all attributes to default values and wipes out all children
			setUSE(getUSE()); // exercise field checks, simple types
        
		setCssClass(getCssClass()); // exercise field checks, simple types
        
		setHtmlID(getHtmlID()); // exercise field checks, simple types
        
		setCssStyle(getCssStyle()); // exercise field checks, simple types
        
		for (fieldValue element : fieldValueList) // MFNode
		{
			try {
                            ((X3DConcreteElement) element).validate(); // exercise field checks, MFNode element
                            validationResult.append(((X3DConcreteElement) element).getValidationResult());
                        }
                        catch (Exception e)
                        {
                            String exceptionMessage = "[exception] during validation: " + e.getMessage();
                            validationResult.append(exceptionMessage);
                        //  System.err.println(exceptionMessage);
                            e.printStackTrace();
                        }
		}
		setFieldValueList(getFieldValueList()); // also test getter/setter validation, which also includes checks for acceptableNodeTypes (if applicable)

		if (hasUSE() && hasFieldValue()) // test USE restrictions
		{
			String errorNotice = "ProtoInstance USE='" + getUSE() + "' is not allowed to have contained MFNode fieldValue";
			validationResult.append(errorNotice);
			throw new InvalidFieldValueException(errorNotice); // report error		
		}
		
		if (IS != null)
		{
			setIS(getIS());
			((X3DConcreteElement) IS).validate(); // exercise field checks, SFNode
			validationResult.append(((X3DConcreteElement) IS).getValidationResult());
		}
		if (hasUSE() && hasIS()) // test USE restrictions
		{
			String errorNotice = "ProtoInstance USE='" + getUSE() + "' is not allowed to have contained SFNode IS";
			validationResult.append(errorNotice);
			throw new InvalidFieldValueException(errorNotice); // report error
		}
		
		if (hasUSE() && !commentsList.isEmpty())// test USE restrictions
		{
			String errorNotice = "ProtoInstance USE='" + getUSE() + "' is not allowed to have contained comments";
			validationResult.append(errorNotice);
			throw new InvalidFieldValueException(errorNotice); // report error		
		}

		if (metadata != null)
		{
			setMetadata(getMetadata());
			((X3DConcreteElement) metadata).validate(); // exercise field checks, SFNode
			validationResult.append(((X3DConcreteElement) metadata).getValidationResult());
		}
		if (hasUSE() && hasMetadata()) // test USE restrictions
		{
			String errorNotice = "ProtoInstance USE='" + getUSE() + "' is not allowed to have contained SFNode metadata";
			validationResult.append(errorNotice);
			throw new InvalidFieldValueException(errorNotice); // report error
		}
		
		if (hasUSE() && !commentsList.isEmpty())// test USE restrictions
		{
			String errorNotice = "ProtoInstance USE='" + getUSE() + "' is not allowed to have contained comments";
			validationResult.append(errorNotice);
			throw new InvalidFieldValueException(errorNotice); // report error		
		}

		if (getIS() != null)
		{
			if (getIS().getConnectList().isEmpty())
			{
				String errorNotice = "IS statement present, but contains no connect statements";
				validationResult.append(errorNotice).append("\n");
				throw new org.web3d.x3d.sai.InvalidProtoException(errorNotice); // report error
			}				
			// TODO also check that this node has ancestor ProtoBody, and that a field with same name also exists, so that IS is legal
		}
		return validationResult.toString();
	}

	/**
	 * Add fieldValue
	 * @param newFieldValue is fieldValue to add
	 * @return {@link ProtoInstance} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
setAttribute method invocations).
	 */
	public ProtoInstance addFieldValue (fieldValue newFieldValue)
	{
		String errorNotice = "Illegal addFieldValue() invocation for ProtoInstance name='" + getName() +
					"' fieldValue name='" + newFieldValue.getName() + "', ";
        if (newFieldValue == null)
        {
            errorNotice += "provided null fieldValue";
            validationResult.append(errorNotice).append("\n");
            throw new org.web3d.x3d.sai.InvalidFieldException(errorNotice);
        }
		for (fieldValue priorFieldValue : fieldValueList) // check for field with duplicate name
		{
			if (priorFieldValue.getName().equals(newFieldValue.getName()))
			{
				errorNotice += "trying to add fieldValue with duplicate name='" + newFieldValue.getName() + "'";
				validationResult.append(errorNotice).append("\n");
				throw new InvalidFieldValueException(errorNotice);
			}
		}
		String fieldValueValidationResult = newFieldValue.validate();
		if (!ConfigurationProperties.isCreationConnectionValidationExceptionAllowed() &&
			!fieldValueValidationResult.trim().isEmpty())
		{
			errorNotice += "newFieldValue.validation() failure:\n   " + fieldValueValidationResult;
			validationResult.append(errorNotice).append("\n");
			throw new org.web3d.x3d.sai.InvalidFieldException(errorNotice);
		}
		fieldValueList.add(newFieldValue);
		newFieldValue.setParent(this);
		return this;
	}
	
	/**
	 * Find contained fieldValue statement by name, if present.
	 * @param fieldValueName is name attribute for fieldValue of interest
	 * @return fieldValue reference of interest, null otherwise
	 */
	public fieldValue findFieldValueByName (String fieldValueName)
	{
		for (fieldValue element : fieldValueList)
		{
			 if (element.getName().equalsIgnoreCase(fieldValueName))
				 return element;
		}
		return null;
	}
	/**
	 * Remove a fieldValue from contained fieldValueList
	 * @param fieldValueToRemove matching fieldValue of interest
	 * @see #removeFieldValue(String)
	 * @return whether fieldValue was removed
	 */
	public boolean removeFieldValue (fieldValue fieldValueToRemove)
	{
		String errorNotice = "Illegal removeFieldValue() invocation for ProtoInstance DEF='" + getDEF() + "', ";
        if (fieldValueToRemove == null)
        {
            errorNotice += "provided null fieldValue";
            validationResult.append(errorNotice).append("\n");
            throw new org.web3d.x3d.sai.InvalidFieldException(errorNotice);
        }
		int  fieldValueIndex = fieldValueList.indexOf(fieldValueToRemove);
		if  (fieldValueIndex >= 0)
		{
			 fieldValue removedfieldValue = fieldValueList.remove(fieldValueIndex);
			 return (removedfieldValue != null);
		}
		else return false;
	}
	/**
	 * Remove a fieldValue by name from contained fieldValueList
	 * @param fieldValueName matching fieldValue of interest
	 * @see #removeFieldValue(fieldValue)
	 * @return whether fieldValue was removed
	 */
	public boolean removeFieldValue (String fieldValueName)
	{
		String errorNotice = "Illegal removeFieldValue() invocation for ProtoInstance DEF='" + getDEF() + "', ";
        if ((fieldValueName == null) || fieldValueName.isEmpty())
        {
            errorNotice += "provided null or empty fieldValueName string";
            validationResult.append(errorNotice).append("\n");
            throw new org.web3d.x3d.sai.InvalidFieldException(errorNotice);
        }
		fieldValue fieldValueToRemove = findFieldValueByName (fieldValueName);
		if (fieldValueToRemove != null)
			return removeFieldValue (fieldValueToRemove);
		else return false;
	}
	/**
	 * Clear all fieldValues
	 */
	public void clearFieldValues()
	{
		fieldValueList.clear();
	}
	/**
	 * Determine if fieldValue statement is present.
	 * @param fieldValueName is name attribute for fieldValue of interest
	 * @return whether fieldValue is found
	 */
	public boolean hasFieldValue (String fieldValueName)
	{
		return (findFieldValueByName(fieldValueName) != null);
	}
}
