/*
Web3D Consortium Open-Source License for Models and Software

Copyright (c) 1995-2025 held by the author(s).  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer
      in the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the Web3D Consortium (https://www.web3D.org)
      nor the names of its contributors may be used to endorse or
      promote products derived from this software without specific
      prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

package org.web3d.x3d.jsail;

import java.util.Arrays;
import org.web3d.x3d.sai.Core.*;  // making sure #3
import org.web3d.x3d.jsail.*;
import org.web3d.x3d.jsail.Core.*;


/**
 * Topmost abstract parent class for concrete X3D nodes and statements, containing common methods and member variables.
 * 
 * <br><br>

 * @author Don Brutzman and Roy Walmsley
 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html" target="_blank">X3D Tooltips</a>
 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html" target="_blank">X3D Scene Authoring Hints</a>
 */
public abstract class X3DConcreteElement
{
	/** String constant <i>NAME</i> provides name of this element; overridden by implementing class. */
	protected static final String NAME = ""; // must be overridden

            // DEFAULT_VALUE declarations

	// Protected member value declarations are encapsulated and private, for internal library use only

	private X3DConcreteElement parent = null; // X3D node or statement

	/**
	 * Provide object reference to parent X3D node or statement, if any.
	 * This reference is named "parentObject" rather than "parent" to avoid potential name collision with any X3D field named "parent".
	 * @return object reference to parent X3D node or statement, otherwise null if none
	 */
	public X3DConcreteElement getParent()
	{
		return parent;
	}
	/**
	 * Whether parent object exists.
	 * This reference is named "parentObject" rather than "parent" to avoid potential name collision with any X3D field named "parent".
	 * @return true if found
	 */
	public boolean hasParent()
	{
		return (parent != null);
	}

	/**
	 * Package-internal method to set parent object reference.
	 * @param newParentObject object reference to parent node or X3D statement that contains this node
	 */
	public void setParent(X3DConcreteElement newParentObject)
	{
		parent = newParentObject;
	}

	/**
	 * Package-protected internal method to clear local reference to parent object, if any.
	 */
	public void clearParent()
	{
		setParent(null);
	}

	/**
	 * Find object reference to ancestor Scene element, assumes this object is an attached child.
	 * @see X3DConcreteElement#hasAncestorScene()
	 * @see #hasAncestorX3D()
	 * @see X3DConcreteElement#findAncestorX3D()
	 * @return ancestor Scene child reference if attached, otherwise null
	 */
	public Scene findAncestorScene()
	{
		if ((this instanceof org.web3d.x3d.sai.Core.X3DNode) && ((org.web3d.x3d.sai.Core.X3DNode)this) instanceof org.web3d.x3d.jsail.Core.Scene)
			return (Scene)((org.web3d.x3d.sai.Core.X3DNode)this);
		X3DConcreteElement element = this.getParent();
		while (element != null)
		{
			if (element instanceof org.web3d.x3d.jsail.Core.Scene)
				 return (Scene)element;
			else element = element.getParent(); // walk up the tree to top, then back down to Scene
		}
		return null; // not found
	}
	/**
	 * Determine whether ancestor Scene element is found, meaning this object is an attached child.
	 * @see X3DConcreteElement#hasAncestorScene()
	 * @see #hasAncestorX3D()
	 * @see X3DConcreteElement#findAncestorX3D()
	 * @return whether ancestor Scene node is found
	 */
	public boolean hasAncestorScene()
	{
		return (findAncestorScene() != null);
	}

	/**
	 * Find object reference to ancestor element (i.e. node or statement), if this object is an attached child.
	 * @param ancestorElementName elementName of ancestor of interest (e.g. first Transform)
	 * @return ancestor node reference if attached and found, otherwise null
	 */
	public X3DConcreteElement findAncestorElementByName(String ancestorElementName)
	{
		if (this.getElementName().equals(ancestorElementName))
			return this;
		X3DConcreteElement element = this.getParent();
		while (element != null)
		{
			if (element.getElementName().equals(ancestorElementName))
				 return element;
			else element = element.getParent(); // walk up the tree to top
		}
		return null; // not found
	}
	/**
	 * Determine whether ancestor element (i.e. node or statement) is found, meaning this object is an attached child.
	 * @param ancestorElementName elementName of ancestor of interest (e.g. first Transform)
	 * @see #findAncestorProtoBody()
	 * @return whether ancestor element is found
	 */
	public boolean hasAncestorElementByName(String ancestorElementName)
	{
		return (findAncestorElementByName(ancestorElementName) != null);
	}

	/**
	 * Find object reference to ancestor ProtoBody, if this node or statement is a child.
	 * @see #hasAncestorProtoBody()
	 * @return ancestor ProtoBody reference if attached and found, otherwise null
	 */
	public ProtoBody findAncestorProtoBody()
	{
		return (ProtoBody) findAncestorElementByName(ProtoBody.NAME);
	}
	/**
	 * Determine whether ancestor ProtoBody element is found, meaning this object is an attached child.
	 * @see #findAncestorProtoBody()
	 * @return whether ancestor ProtoBody element is found
	 */
	public boolean hasAncestorProtoBody()
	{
		return (findAncestorProtoBody() != null);
	}

	/**
	 * Find object reference to ancestor X3D element, if this node or statement is part of an X3D object.
	 * @see #hasAncestorX3D()
	 * @see X3DConcreteElement#hasAncestorScene()
	 * @see X3DConcreteElement#hasAncestorScene()
	 * @return ancestor X3D reference if attached, otherwise null
	 */
	public X3D findAncestorX3D()
	{
		if (((org.web3d.x3d.sai.Core.X3DNode)this) instanceof org.web3d.x3d.jsail.Core.X3D)
			return (X3D)((org.web3d.x3d.sai.Core.X3DNode)this);
		X3DConcreteElement element = this.getParent();
		while (element != null)
		{
			if (element instanceof org.web3d.x3d.jsail.Core.X3D)
				 return (X3D)element;
			else element = element.getParent(); // walk up the tree to top
		}
		return null; // not found
	}
	/**
	 * Determine whether ancestor X3D element is found, meaning this object is an attached child.
	 * @see X3DConcreteElement#findAncestorX3D()
	 * @see X3DConcreteElement#hasAncestorScene()
	 * @see X3DConcreteElement#hasAncestorScene()
	 * @return whether ancestor X3D element is found
	 */
	public boolean hasAncestorX3D()
	{
		return (findAncestorX3D() != null);
	}

	/** Provides name of this element.
	 * @return name of this element
	 */
	abstract public String getElementName(); // must be overridden (static methods cannot be abstract)

	/** Defines X3D component for this element.
	 * @return X3D component for this element
	 */
	abstract public String getComponent(); // must be overridden (static methods cannot be abstract)

	/** Provides default X3D component level for this element
	 * @return default X3D component level for this element
	 */
	abstract public int getComponentLevel(); // must be overridden (static methods cannot be abstract)

	/** Indicate type corresponding to given fieldName.
	 * @param fieldName name of field in this X3D statement
	 * @see ConfigurationProperties#ERROR_UNKNOWN_FIELD_TYPE
	 * @return X3D type (SFvec3f etc.), otherwise ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE if not recognized
	 */
	abstract public String getFieldType(String fieldName); // must be overridden

	/** Indicate accessType corresponding to given fieldName.
	 * @param fieldName name of field in this X3D statement
	 * @see ConfigurationProperties#ERROR_UNKNOWN_FIELD_TYPE
	 * @return X3D accessType (inputOnly etc.), otherwise ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE if not recognized
	 */
	abstract public String getAccessType(String fieldName); // must be overridden

	/**
	 * Recursive method to provide object reference to node or statement by name attribute, if found as part of this element or in a contained element.
	 * Elements with name fields include meta, Metadata* nodes, field/fieldValue, ProtoDeclare/ExternProtoDeclare/ProtoInstance, CAD and HAnim nodes.
	 * <br ><br >
	 * <i>Warning:</i> first start with findAncestorScene() to check entire scene graph, or findAncestorX3D() to check entire model document.
	 * <br ><br >
	 * @see #findNodeByDEF(String)
	 * @see X3DConcreteElement#hasAncestorScene()
	 * @see org.web3d.x3d.jsail.X3DConcreteElement#findAncestorX3D()
	 * @see meta
	 * @see org.web3d.x3d.jsail.Core.MetadataString
	 * <i>Warning:</i> more than one element may be found that has the same name, this method does not handle that case.
	 * @param nameValue is value of the name field being searched for in this element and child elements(if any)
	 * @param elementName identifies the element of interest (meta MetadataString ProtoDeclare CADassembly ProtoInstance HAnimHumanoid etc.)
	 * @return object reference to found element, null otherwise
	 */
	abstract public X3DConcreteElement findElementByNameValue(String nameValue, String elementName); // required interface

	/**
	 * Recursive method to provide object reference to node or statement by name attribute, if found as part of this element or in a contained element.
	 * Elements with name fields include meta, Metadata* nodes, field/fieldValue, ProtoDeclare/ExternProtoDeclare/ProtoInstance, CAD and HAnim nodes.
	 * <br ><br >
	 * <i>Warning:</i> first start with findAncestorScene() to check entire scene graph, or findAncestorX3D() to check entire model document.
	 * <br ><br >
	 * <i>Warning:</i> more than one element may be found that has the same name, this method does not handle that case.
	 * @see #findNodeByDEF(String)
	 * @see X3DConcreteElement#hasAncestorScene()
	 * @see org.web3d.x3d.jsail.X3DConcreteElement#findAncestorX3D()
	 * @param nameValue is value of the name field being searched for in this element and child elements(if any)
	 * @return object reference to found element, null otherwise
	 */
	abstract public X3DConcreteElement findElementByNameValue(String nameValue); // required interface

	/**
	 * Recursive method to provide object reference to node by DEF, if found as this node or in a contained node.
	 * <br ><br >
	 * <i>Warning:</i> first start with findAncestorScene() to check entire scene graph, or findAncestorX3D() to check entire model document.
	 * <br ><br >
	 * <i>Warning:</i> more than one element may be found that has the same DEF, this method does not handle that case.
	 * @see #findElementByNameValue(String)
	 * @see X3DConcreteElement#hasAncestorScene()
	 * @see org.web3d.x3d.jsail.X3DConcreteElement#findAncestorX3D()
	 * @param DEFvalue is value of the name field being searched for in this element and child elements(if any)
	 * @return object reference to found node, null otherwise
	 */
	abstract public X3DConcreteNode findNodeByDEF(String DEFvalue); // required interface

	/** Provide fully qualified package for a given element object, including className.  Helpful for reflection.
	 * @param className X3D statement or node of interest
	 * @return fully qualified package for className
	 */
	static public String getPackageName(String className)
	{
		if (className.contains("Object"))
		    className = className.substring(0,className.indexOf("Object"));
		switch (className)
		{
			case "AcousticProperties":
				return "org.web3d.x3d.jsail.Shape.AcousticProperties";
			case "Analyser":
				return "org.web3d.x3d.jsail.Sound.Analyser";
			case "Anchor":
				return "org.web3d.x3d.jsail.Networking.Anchor";
			case "Appearance":
				return "org.web3d.x3d.jsail.Shape.Appearance";
			case "Arc2D":
				return "org.web3d.x3d.jsail.Geometry2D.Arc2D";
			case "ArcClose2D":
				return "org.web3d.x3d.jsail.Geometry2D.ArcClose2D";
			case "AudioClip":
				return "org.web3d.x3d.jsail.Sound.AudioClip";
			case "AudioDestination":
				return "org.web3d.x3d.jsail.Sound.AudioDestination";
			case "Background":
				return "org.web3d.x3d.jsail.EnvironmentalEffects.Background";
			case "BallJoint":
				return "org.web3d.x3d.jsail.RigidBodyPhysics.BallJoint";
			case "Billboard":
				return "org.web3d.x3d.jsail.Navigation.Billboard";
			case "BiquadFilter":
				return "org.web3d.x3d.jsail.Sound.BiquadFilter";
			case "BlendedVolumeStyle":
				return "org.web3d.x3d.jsail.VolumeRendering.BlendedVolumeStyle";
			case "BooleanFilter":
				return "org.web3d.x3d.jsail.EventUtilities.BooleanFilter";
			case "BooleanSequencer":
				return "org.web3d.x3d.jsail.EventUtilities.BooleanSequencer";
			case "BooleanToggle":
				return "org.web3d.x3d.jsail.EventUtilities.BooleanToggle";
			case "BooleanTrigger":
				return "org.web3d.x3d.jsail.EventUtilities.BooleanTrigger";
			case "BoundaryEnhancementVolumeStyle":
				return "org.web3d.x3d.jsail.VolumeRendering.BoundaryEnhancementVolumeStyle";
			case "BoundedPhysicsModel":
				return "org.web3d.x3d.jsail.ParticleSystems.BoundedPhysicsModel";
			case "Box":
				return "org.web3d.x3d.jsail.Geometry3D.Box";
			case "BufferAudioSource":
				return "org.web3d.x3d.jsail.Sound.BufferAudioSource";
			case "CADAssembly":
				return "org.web3d.x3d.jsail.CADGeometry.CADAssembly";
			case "CADFace":
				return "org.web3d.x3d.jsail.CADGeometry.CADFace";
			case "CADLayer":
				return "org.web3d.x3d.jsail.CADGeometry.CADLayer";
			case "CADPart":
				return "org.web3d.x3d.jsail.CADGeometry.CADPart";
			case "CartoonVolumeStyle":
				return "org.web3d.x3d.jsail.VolumeRendering.CartoonVolumeStyle";
			case "ChannelMerger":
				return "org.web3d.x3d.jsail.Sound.ChannelMerger";
			case "ChannelSelector":
				return "org.web3d.x3d.jsail.Sound.ChannelSelector";
			case "ChannelSplitter":
				return "org.web3d.x3d.jsail.Sound.ChannelSplitter";
			case "Circle2D":
				return "org.web3d.x3d.jsail.Geometry2D.Circle2D";
			case "ClipPlane":
				return "org.web3d.x3d.jsail.Rendering.ClipPlane";
			case "CollidableOffset":
				return "org.web3d.x3d.jsail.RigidBodyPhysics.CollidableOffset";
			case "CollidableShape":
				return "org.web3d.x3d.jsail.RigidBodyPhysics.CollidableShape";
			case "Collision":
				return "org.web3d.x3d.jsail.Navigation.Collision";
			case "CollisionCollection":
				return "org.web3d.x3d.jsail.RigidBodyPhysics.CollisionCollection";
			case "CollisionSensor":
				return "org.web3d.x3d.jsail.RigidBodyPhysics.CollisionSensor";
			case "CollisionSpace":
				return "org.web3d.x3d.jsail.RigidBodyPhysics.CollisionSpace";
			case "Color":
				return "org.web3d.x3d.jsail.Rendering.Color";
			case "ColorChaser":
				return "org.web3d.x3d.jsail.Followers.ColorChaser";
			case "ColorDamper":
				return "org.web3d.x3d.jsail.Followers.ColorDamper";
			case "ColorInterpolator":
				return "org.web3d.x3d.jsail.Interpolation.ColorInterpolator";
			case "ColorRGBA":
				return "org.web3d.x3d.jsail.Rendering.ColorRGBA";
			case "ComposedCubeMapTexture":
				return "org.web3d.x3d.jsail.CubeMapTexturing.ComposedCubeMapTexture";
			case "ComposedShader":
				return "org.web3d.x3d.jsail.Shaders.ComposedShader";
			case "ComposedTexture3D":
				return "org.web3d.x3d.jsail.Texturing3D.ComposedTexture3D";
			case "ComposedVolumeStyle":
				return "org.web3d.x3d.jsail.VolumeRendering.ComposedVolumeStyle";
			case "Cone":
				return "org.web3d.x3d.jsail.Geometry3D.Cone";
			case "ConeEmitter":
				return "org.web3d.x3d.jsail.ParticleSystems.ConeEmitter";
			case "Contact":
				return "org.web3d.x3d.jsail.RigidBodyPhysics.Contact";
			case "Contour2D":
				return "org.web3d.x3d.jsail.NURBS.Contour2D";
			case "ContourPolyline2D":
				return "org.web3d.x3d.jsail.NURBS.ContourPolyline2D";
			case "Convolver":
				return "org.web3d.x3d.jsail.Sound.Convolver";
			case "Coordinate":
				return "org.web3d.x3d.jsail.Rendering.Coordinate";
			case "CoordinateChaser":
				return "org.web3d.x3d.jsail.Followers.CoordinateChaser";
			case "CoordinateDamper":
				return "org.web3d.x3d.jsail.Followers.CoordinateDamper";
			case "CoordinateDouble":
				return "org.web3d.x3d.jsail.Rendering.CoordinateDouble";
			case "CoordinateInterpolator":
				return "org.web3d.x3d.jsail.Interpolation.CoordinateInterpolator";
			case "CoordinateInterpolator2D":
				return "org.web3d.x3d.jsail.Interpolation.CoordinateInterpolator2D";
			case "Cylinder":
				return "org.web3d.x3d.jsail.Geometry3D.Cylinder";
			case "CylinderSensor":
				return "org.web3d.x3d.jsail.PointingDeviceSensor.CylinderSensor";
			case "Delay":
				return "org.web3d.x3d.jsail.Sound.Delay";
			case "DirectionalLight":
				return "org.web3d.x3d.jsail.Lighting.DirectionalLight";
			case "DISEntityManager":
				return "org.web3d.x3d.jsail.DIS.DISEntityManager";
			case "DISEntityTypeMapping":
				return "org.web3d.x3d.jsail.DIS.DISEntityTypeMapping";
			case "Disk2D":
				return "org.web3d.x3d.jsail.Geometry2D.Disk2D";
			case "DoubleAxisHingeJoint":
				return "org.web3d.x3d.jsail.RigidBodyPhysics.DoubleAxisHingeJoint";
			case "DynamicsCompressor":
				return "org.web3d.x3d.jsail.Sound.DynamicsCompressor";
			case "EaseInEaseOut":
				return "org.web3d.x3d.jsail.Interpolation.EaseInEaseOut";
			case "EdgeEnhancementVolumeStyle":
				return "org.web3d.x3d.jsail.VolumeRendering.EdgeEnhancementVolumeStyle";
			case "ElevationGrid":
				return "org.web3d.x3d.jsail.Geometry3D.ElevationGrid";
			case "EspduTransform":
				return "org.web3d.x3d.jsail.DIS.EspduTransform";
			case "ExplosionEmitter":
				return "org.web3d.x3d.jsail.ParticleSystems.ExplosionEmitter";
			case "Extrusion":
				return "org.web3d.x3d.jsail.Geometry3D.Extrusion";
			case "FillProperties":
				return "org.web3d.x3d.jsail.Shape.FillProperties";
			case "FloatVertexAttribute":
				return "org.web3d.x3d.jsail.Shaders.FloatVertexAttribute";
			case "Fog":
				return "org.web3d.x3d.jsail.EnvironmentalEffects.Fog";
			case "FogCoordinate":
				return "org.web3d.x3d.jsail.EnvironmentalEffects.FogCoordinate";
			case "FontStyle":
				return "org.web3d.x3d.jsail.Text.FontStyle";
			case "ForcePhysicsModel":
				return "org.web3d.x3d.jsail.ParticleSystems.ForcePhysicsModel";
			case "Gain":
				return "org.web3d.x3d.jsail.Sound.Gain";
			case "GeneratedCubeMapTexture":
				return "org.web3d.x3d.jsail.CubeMapTexturing.GeneratedCubeMapTexture";
			case "GeoCoordinate":
				return "org.web3d.x3d.jsail.Geospatial.GeoCoordinate";
			case "GeoElevationGrid":
				return "org.web3d.x3d.jsail.Geospatial.GeoElevationGrid";
			case "GeoLocation":
				return "org.web3d.x3d.jsail.Geospatial.GeoLocation";
			case "GeoLOD":
				return "org.web3d.x3d.jsail.Geospatial.GeoLOD";
			case "GeoMetadata":
				return "org.web3d.x3d.jsail.Geospatial.GeoMetadata";
			case "GeoOrigin":
				return "org.web3d.x3d.jsail.Geospatial.GeoOrigin";
			case "GeoPositionInterpolator":
				return "org.web3d.x3d.jsail.Geospatial.GeoPositionInterpolator";
			case "GeoProximitySensor":
				return "org.web3d.x3d.jsail.Geospatial.GeoProximitySensor";
			case "GeoTouchSensor":
				return "org.web3d.x3d.jsail.Geospatial.GeoTouchSensor";
			case "GeoTransform":
				return "org.web3d.x3d.jsail.Geospatial.GeoTransform";
			case "GeoViewpoint":
				return "org.web3d.x3d.jsail.Geospatial.GeoViewpoint";
			case "Group":
				return "org.web3d.x3d.jsail.Grouping.Group";
			case "HAnimDisplacer":
				return "org.web3d.x3d.jsail.HAnim.HAnimDisplacer";
			case "HAnimHumanoid":
				return "org.web3d.x3d.jsail.HAnim.HAnimHumanoid";
			case "HAnimJoint":
				return "org.web3d.x3d.jsail.HAnim.HAnimJoint";
			case "HAnimMotion":
				return "org.web3d.x3d.jsail.HAnim.HAnimMotion";
			case "HAnimSegment":
				return "org.web3d.x3d.jsail.HAnim.HAnimSegment";
			case "HAnimSite":
				return "org.web3d.x3d.jsail.HAnim.HAnimSite";
			case "ImageCubeMapTexture":
				return "org.web3d.x3d.jsail.CubeMapTexturing.ImageCubeMapTexture";
			case "ImageTexture":
				return "org.web3d.x3d.jsail.Texturing.ImageTexture";
			case "ImageTexture3D":
				return "org.web3d.x3d.jsail.Texturing3D.ImageTexture3D";
			case "IndexedFaceSet":
				return "org.web3d.x3d.jsail.Geometry3D.IndexedFaceSet";
			case "IndexedLineSet":
				return "org.web3d.x3d.jsail.Rendering.IndexedLineSet";
			case "IndexedQuadSet":
				return "org.web3d.x3d.jsail.CADGeometry.IndexedQuadSet";
			case "IndexedTriangleFanSet":
				return "org.web3d.x3d.jsail.Rendering.IndexedTriangleFanSet";
			case "IndexedTriangleSet":
				return "org.web3d.x3d.jsail.Rendering.IndexedTriangleSet";
			case "IndexedTriangleStripSet":
				return "org.web3d.x3d.jsail.Rendering.IndexedTriangleStripSet";
			case "Inline":
				return "org.web3d.x3d.jsail.Networking.Inline";
			case "IntegerSequencer":
				return "org.web3d.x3d.jsail.EventUtilities.IntegerSequencer";
			case "IntegerTrigger":
				return "org.web3d.x3d.jsail.EventUtilities.IntegerTrigger";
			case "IsoSurfaceVolumeData":
				return "org.web3d.x3d.jsail.VolumeRendering.IsoSurfaceVolumeData";
			case "KeySensor":
				return "org.web3d.x3d.jsail.KeyDeviceSensor.KeySensor";
			case "Layer":
				return "org.web3d.x3d.jsail.Layering.Layer";
			case "LayerSet":
				return "org.web3d.x3d.jsail.Layering.LayerSet";
			case "Layout":
				return "org.web3d.x3d.jsail.Layout.Layout";
			case "LayoutGroup":
				return "org.web3d.x3d.jsail.Layout.LayoutGroup";
			case "LayoutLayer":
				return "org.web3d.x3d.jsail.Layout.LayoutLayer";
			case "LinePickSensor":
				return "org.web3d.x3d.jsail.Picking.LinePickSensor";
			case "LineProperties":
				return "org.web3d.x3d.jsail.Shape.LineProperties";
			case "LineSet":
				return "org.web3d.x3d.jsail.Rendering.LineSet";
			case "ListenerPointSource":
				return "org.web3d.x3d.jsail.Sound.ListenerPointSource";
			case "LoadSensor":
				return "org.web3d.x3d.jsail.Networking.LoadSensor";
			case "LocalFog":
				return "org.web3d.x3d.jsail.EnvironmentalEffects.LocalFog";
			case "LOD":
				return "org.web3d.x3d.jsail.Navigation.LOD";
			case "Material":
				return "org.web3d.x3d.jsail.Shape.Material";
			case "Matrix3VertexAttribute":
				return "org.web3d.x3d.jsail.Shaders.Matrix3VertexAttribute";
			case "Matrix4VertexAttribute":
				return "org.web3d.x3d.jsail.Shaders.Matrix4VertexAttribute";
			case "MetadataBoolean":
				return "org.web3d.x3d.jsail.Core.MetadataBoolean";
			case "MetadataDouble":
				return "org.web3d.x3d.jsail.Core.MetadataDouble";
			case "MetadataFloat":
				return "org.web3d.x3d.jsail.Core.MetadataFloat";
			case "MetadataInteger":
				return "org.web3d.x3d.jsail.Core.MetadataInteger";
			case "MetadataSet":
				return "org.web3d.x3d.jsail.Core.MetadataSet";
			case "MetadataString":
				return "org.web3d.x3d.jsail.Core.MetadataString";
			case "MicrophoneSource":
				return "org.web3d.x3d.jsail.Sound.MicrophoneSource";
			case "MotorJoint":
				return "org.web3d.x3d.jsail.RigidBodyPhysics.MotorJoint";
			case "MovieTexture":
				return "org.web3d.x3d.jsail.Texturing.MovieTexture";
			case "MultiTexture":
				return "org.web3d.x3d.jsail.Texturing.MultiTexture";
			case "MultiTextureCoordinate":
				return "org.web3d.x3d.jsail.Texturing.MultiTextureCoordinate";
			case "MultiTextureTransform":
				return "org.web3d.x3d.jsail.Texturing.MultiTextureTransform";
			case "NavigationInfo":
				return "org.web3d.x3d.jsail.Navigation.NavigationInfo";
			case "Normal":
				return "org.web3d.x3d.jsail.Rendering.Normal";
			case "NormalInterpolator":
				return "org.web3d.x3d.jsail.Interpolation.NormalInterpolator";
			case "NurbsCurve":
				return "org.web3d.x3d.jsail.NURBS.NurbsCurve";
			case "NurbsCurve2D":
				return "org.web3d.x3d.jsail.NURBS.NurbsCurve2D";
			case "NurbsOrientationInterpolator":
				return "org.web3d.x3d.jsail.NURBS.NurbsOrientationInterpolator";
			case "NurbsPatchSurface":
				return "org.web3d.x3d.jsail.NURBS.NurbsPatchSurface";
			case "NurbsPositionInterpolator":
				return "org.web3d.x3d.jsail.NURBS.NurbsPositionInterpolator";
			case "NurbsSet":
				return "org.web3d.x3d.jsail.NURBS.NurbsSet";
			case "NurbsSurfaceInterpolator":
				return "org.web3d.x3d.jsail.NURBS.NurbsSurfaceInterpolator";
			case "NurbsSweptSurface":
				return "org.web3d.x3d.jsail.NURBS.NurbsSweptSurface";
			case "NurbsSwungSurface":
				return "org.web3d.x3d.jsail.NURBS.NurbsSwungSurface";
			case "NurbsTextureCoordinate":
				return "org.web3d.x3d.jsail.NURBS.NurbsTextureCoordinate";
			case "NurbsTrimmedSurface":
				return "org.web3d.x3d.jsail.NURBS.NurbsTrimmedSurface";
			case "OpacityMapVolumeStyle":
				return "org.web3d.x3d.jsail.VolumeRendering.OpacityMapVolumeStyle";
			case "OrientationChaser":
				return "org.web3d.x3d.jsail.Followers.OrientationChaser";
			case "OrientationDamper":
				return "org.web3d.x3d.jsail.Followers.OrientationDamper";
			case "OrientationInterpolator":
				return "org.web3d.x3d.jsail.Interpolation.OrientationInterpolator";
			case "OrthoViewpoint":
				return "org.web3d.x3d.jsail.Navigation.OrthoViewpoint";
			case "OscillatorSource":
				return "org.web3d.x3d.jsail.Sound.OscillatorSource";
			case "PackagedShader":
				return "org.web3d.x3d.jsail.Shaders.PackagedShader";
			case "ParticleSystem":
				return "org.web3d.x3d.jsail.ParticleSystems.ParticleSystem";
			case "PeriodicWave":
				return "org.web3d.x3d.jsail.Sound.PeriodicWave";
			case "PhysicalMaterial":
				return "org.web3d.x3d.jsail.Shape.PhysicalMaterial";
			case "PickableGroup":
				return "org.web3d.x3d.jsail.Picking.PickableGroup";
			case "PixelTexture":
				return "org.web3d.x3d.jsail.Texturing.PixelTexture";
			case "PixelTexture3D":
				return "org.web3d.x3d.jsail.Texturing3D.PixelTexture3D";
			case "PlaneSensor":
				return "org.web3d.x3d.jsail.PointingDeviceSensor.PlaneSensor";
			case "PointEmitter":
				return "org.web3d.x3d.jsail.ParticleSystems.PointEmitter";
			case "PointLight":
				return "org.web3d.x3d.jsail.Lighting.PointLight";
			case "PointPickSensor":
				return "org.web3d.x3d.jsail.Picking.PointPickSensor";
			case "PointProperties":
				return "org.web3d.x3d.jsail.Shape.PointProperties";
			case "PointSet":
				return "org.web3d.x3d.jsail.Rendering.PointSet";
			case "Polyline2D":
				return "org.web3d.x3d.jsail.Geometry2D.Polyline2D";
			case "PolylineEmitter":
				return "org.web3d.x3d.jsail.ParticleSystems.PolylineEmitter";
			case "Polypoint2D":
				return "org.web3d.x3d.jsail.Geometry2D.Polypoint2D";
			case "PositionChaser":
				return "org.web3d.x3d.jsail.Followers.PositionChaser";
			case "PositionChaser2D":
				return "org.web3d.x3d.jsail.Followers.PositionChaser2D";
			case "PositionDamper":
				return "org.web3d.x3d.jsail.Followers.PositionDamper";
			case "PositionDamper2D":
				return "org.web3d.x3d.jsail.Followers.PositionDamper2D";
			case "PositionInterpolator":
				return "org.web3d.x3d.jsail.Interpolation.PositionInterpolator";
			case "PositionInterpolator2D":
				return "org.web3d.x3d.jsail.Interpolation.PositionInterpolator2D";
			case "PrimitivePickSensor":
				return "org.web3d.x3d.jsail.Picking.PrimitivePickSensor";
			case "ProgramShader":
				return "org.web3d.x3d.jsail.Shaders.ProgramShader";
			case "ProjectionVolumeStyle":
				return "org.web3d.x3d.jsail.VolumeRendering.ProjectionVolumeStyle";
			case "ProtoInstance":
				return "org.web3d.x3d.jsail.Core.ProtoInstance";
			case "ProximitySensor":
				return "org.web3d.x3d.jsail.EnvironmentalSensor.ProximitySensor";
			case "QuadSet":
				return "org.web3d.x3d.jsail.CADGeometry.QuadSet";
			case "ReceiverPdu":
				return "org.web3d.x3d.jsail.DIS.ReceiverPdu";
			case "Rectangle2D":
				return "org.web3d.x3d.jsail.Geometry2D.Rectangle2D";
			case "RigidBody":
				return "org.web3d.x3d.jsail.RigidBodyPhysics.RigidBody";
			case "RigidBodyCollection":
				return "org.web3d.x3d.jsail.RigidBodyPhysics.RigidBodyCollection";
			case "ScalarChaser":
				return "org.web3d.x3d.jsail.Followers.ScalarChaser";
			case "ScalarDamper":
				return "org.web3d.x3d.jsail.Followers.ScalarDamper";
			case "ScalarInterpolator":
				return "org.web3d.x3d.jsail.Interpolation.ScalarInterpolator";
			case "ScreenFontStyle":
				return "org.web3d.x3d.jsail.Layout.ScreenFontStyle";
			case "ScreenGroup":
				return "org.web3d.x3d.jsail.Layout.ScreenGroup";
			case "Script":
				return "org.web3d.x3d.jsail.Scripting.Script";
			case "SegmentedVolumeData":
				return "org.web3d.x3d.jsail.VolumeRendering.SegmentedVolumeData";
			case "ShadedVolumeStyle":
				return "org.web3d.x3d.jsail.VolumeRendering.ShadedVolumeStyle";
			case "ShaderPart":
				return "org.web3d.x3d.jsail.Shaders.ShaderPart";
			case "ShaderProgram":
				return "org.web3d.x3d.jsail.Shaders.ShaderProgram";
			case "Shape":
				return "org.web3d.x3d.jsail.Shape.Shape";
			case "SignalPdu":
				return "org.web3d.x3d.jsail.DIS.SignalPdu";
			case "SilhouetteEnhancementVolumeStyle":
				return "org.web3d.x3d.jsail.VolumeRendering.SilhouetteEnhancementVolumeStyle";
			case "SingleAxisHingeJoint":
				return "org.web3d.x3d.jsail.RigidBodyPhysics.SingleAxisHingeJoint";
			case "SliderJoint":
				return "org.web3d.x3d.jsail.RigidBodyPhysics.SliderJoint";
			case "Sound":
				return "org.web3d.x3d.jsail.Sound.Sound";
			case "SpatialSound":
				return "org.web3d.x3d.jsail.Sound.SpatialSound";
			case "Sphere":
				return "org.web3d.x3d.jsail.Geometry3D.Sphere";
			case "SphereSensor":
				return "org.web3d.x3d.jsail.PointingDeviceSensor.SphereSensor";
			case "SplinePositionInterpolator":
				return "org.web3d.x3d.jsail.Interpolation.SplinePositionInterpolator";
			case "SplinePositionInterpolator2D":
				return "org.web3d.x3d.jsail.Interpolation.SplinePositionInterpolator2D";
			case "SplineScalarInterpolator":
				return "org.web3d.x3d.jsail.Interpolation.SplineScalarInterpolator";
			case "SpotLight":
				return "org.web3d.x3d.jsail.Lighting.SpotLight";
			case "SquadOrientationInterpolator":
				return "org.web3d.x3d.jsail.Interpolation.SquadOrientationInterpolator";
			case "StaticGroup":
				return "org.web3d.x3d.jsail.Grouping.StaticGroup";
			case "StreamAudioDestination":
				return "org.web3d.x3d.jsail.Sound.StreamAudioDestination";
			case "StreamAudioSource":
				return "org.web3d.x3d.jsail.Sound.StreamAudioSource";
			case "StringSensor":
				return "org.web3d.x3d.jsail.KeyDeviceSensor.StringSensor";
			case "SurfaceEmitter":
				return "org.web3d.x3d.jsail.ParticleSystems.SurfaceEmitter";
			case "Switch":
				return "org.web3d.x3d.jsail.Grouping.Switch";
			case "TexCoordChaser2D":
				return "org.web3d.x3d.jsail.Followers.TexCoordChaser2D";
			case "TexCoordDamper2D":
				return "org.web3d.x3d.jsail.Followers.TexCoordDamper2D";
			case "Text":
				return "org.web3d.x3d.jsail.Text.Text";
			case "TextureBackground":
				return "org.web3d.x3d.jsail.EnvironmentalEffects.TextureBackground";
			case "TextureCoordinate":
				return "org.web3d.x3d.jsail.Texturing.TextureCoordinate";
			case "TextureCoordinate3D":
				return "org.web3d.x3d.jsail.Texturing3D.TextureCoordinate3D";
			case "TextureCoordinate4D":
				return "org.web3d.x3d.jsail.Texturing3D.TextureCoordinate4D";
			case "TextureCoordinateGenerator":
				return "org.web3d.x3d.jsail.Texturing.TextureCoordinateGenerator";
			case "TextureProjector":
				return "org.web3d.x3d.jsail.TextureProjection.TextureProjector";
			case "TextureProjectorParallel":
				return "org.web3d.x3d.jsail.TextureProjection.TextureProjectorParallel";
			case "TextureProperties":
				return "org.web3d.x3d.jsail.Texturing.TextureProperties";
			case "TextureTransform":
				return "org.web3d.x3d.jsail.Texturing.TextureTransform";
			case "TextureTransform3D":
				return "org.web3d.x3d.jsail.Texturing3D.TextureTransform3D";
			case "TextureTransformMatrix3D":
				return "org.web3d.x3d.jsail.Texturing3D.TextureTransformMatrix3D";
			case "TimeSensor":
				return "org.web3d.x3d.jsail.Time.TimeSensor";
			case "TimeTrigger":
				return "org.web3d.x3d.jsail.EventUtilities.TimeTrigger";
			case "ToneMappedVolumeStyle":
				return "org.web3d.x3d.jsail.VolumeRendering.ToneMappedVolumeStyle";
			case "TouchSensor":
				return "org.web3d.x3d.jsail.PointingDeviceSensor.TouchSensor";
			case "Transform":
				return "org.web3d.x3d.jsail.Grouping.Transform";
			case "TransformSensor":
				return "org.web3d.x3d.jsail.EnvironmentalSensor.TransformSensor";
			case "TransmitterPdu":
				return "org.web3d.x3d.jsail.DIS.TransmitterPdu";
			case "TriangleFanSet":
				return "org.web3d.x3d.jsail.Rendering.TriangleFanSet";
			case "TriangleSet":
				return "org.web3d.x3d.jsail.Rendering.TriangleSet";
			case "TriangleSet2D":
				return "org.web3d.x3d.jsail.Geometry2D.TriangleSet2D";
			case "TriangleStripSet":
				return "org.web3d.x3d.jsail.Rendering.TriangleStripSet";
			case "TwoSidedMaterial":
				return "org.web3d.x3d.jsail.Shape.TwoSidedMaterial";
			case "UniversalJoint":
				return "org.web3d.x3d.jsail.RigidBodyPhysics.UniversalJoint";
			case "UnlitMaterial":
				return "org.web3d.x3d.jsail.Shape.UnlitMaterial";
			case "Viewpoint":
				return "org.web3d.x3d.jsail.Navigation.Viewpoint";
			case "ViewpointGroup":
				return "org.web3d.x3d.jsail.Navigation.ViewpointGroup";
			case "Viewport":
				return "org.web3d.x3d.jsail.Layering.Viewport";
			case "VisibilitySensor":
				return "org.web3d.x3d.jsail.EnvironmentalSensor.VisibilitySensor";
			case "VolumeData":
				return "org.web3d.x3d.jsail.VolumeRendering.VolumeData";
			case "VolumeEmitter":
				return "org.web3d.x3d.jsail.ParticleSystems.VolumeEmitter";
			case "VolumePickSensor":
				return "org.web3d.x3d.jsail.Picking.VolumePickSensor";
			case "WaveShaper":
				return "org.web3d.x3d.jsail.Sound.WaveShaper";
			case "WindPhysicsModel":
				return "org.web3d.x3d.jsail.ParticleSystems.WindPhysicsModel";
			case "WorldInfo":
				return "org.web3d.x3d.jsail.Core.WorldInfo";
			case "component":
				return "org.web3d.x3d.jsail.Core.component";
			case "connect":
				return "org.web3d.x3d.jsail.Core.connect";
			case "EXPORT":
				return "org.web3d.x3d.jsail.Networking.EXPORT";
			case "ExternProtoDeclare":
				return "org.web3d.x3d.jsail.Core.ExternProtoDeclare";
			case "field":
				return "org.web3d.x3d.jsail.Core.field";
			case "fieldValue":
				return "org.web3d.x3d.jsail.Core.fieldValue";
			case "head":
				return "org.web3d.x3d.jsail.Core.head";
			case "IMPORT":
				return "org.web3d.x3d.jsail.Networking.IMPORT";
			case "IS":
				return "org.web3d.x3d.jsail.Core.IS";
			case "meta":
				return "org.web3d.x3d.jsail.Core.meta";
			case "ProtoBody":
				return "org.web3d.x3d.jsail.Core.ProtoBody";
			case "ProtoDeclare":
				return "org.web3d.x3d.jsail.Core.ProtoDeclare";
			case "ProtoInterface":
				return "org.web3d.x3d.jsail.Core.ProtoInterface";
			case "ROUTE":
				return "org.web3d.x3d.jsail.Core.ROUTE";
			case "Scene":
				return "org.web3d.x3d.jsail.Core.Scene";
			case "unit":
				return "org.web3d.x3d.jsail.Core.unit";
			case "X3D":
				return "org.web3d.x3d.jsail.Core.X3D";
			default:
				return "UnknownClassName_" + className;
		}
	}


	/** the <i>id</i> attribute on each X3D node is considered a unique identifier when used as part of an encompassing HTML/DOM context.
	 * @see <a href="https://www.w3.org/Style/CSS">W3C Cascading Style Sheets</a> */
	private String htmlID = ID_DEFAULT_VALUE;

	/** The <i>class</i> field is a space-separated list of classes, reserved for use by CSS cascading stylesheets.
	 * @see <a href="https://www.w3.org/Style/CSS">W3C Cascading Style Sheets</a> */
	private String cssClass = CLASS_DEFAULT_VALUE;

	/** The <i>style</i> field provides an inline block of CSS for element styling, reserved for use by CSS cascading stylesheets.
	 * @see <a href="https://www.w3.org/Style/CSS">W3C Cascading Style Sheets</a> */
	private String cssStyle = STYLE_DEFAULT_VALUE;

	/** SFString field named <i>id</i> for html has default value equal to an empty string. */
	public static final String ID_DEFAULT_VALUE = "";

	/** SFString field named <i>class</i> for CSS has default value equal to an empty string. */
	public static final String CLASS_DEFAULT_VALUE = "";

	/** SFString field named <i>style</i> for CSS has default value equal to an empty string. */
	public static final String STYLE_DEFAULT_VALUE = "";

	/** Initialize all member variables to default values. */
	public void initialize()
	{
		setParent(null);
		// no super to initialize, we are at the top of the hierarchy
          htmlID =    ID_DEFAULT_VALUE;
		cssClass = CLASS_DEFAULT_VALUE;
		cssStyle = STYLE_DEFAULT_VALUE;
	}

	/** Protected internal superclass method to keep cssClass private, scene authors should use method setCssClass(newValue) instead.
	 * This attribute is only functional if the X3D model is loaded within an HTML page.
	 * <i>Tooltip:</i> The class attribute is a space-separated list of classes, reserved for use by CSS cascading stylesheets.
	 * @see <a href="https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/htmlGuidelines.html#CSS">X3D Architecture Annex L - HTML authoring guidelines, CSS considerations</a>
	 * @see <a href="https://www.w3.org/Style/CSS">W3C Cascading Style Sheets</a>
	 * @see <a href="https://www.w3.org/TR/css-2018">W3C CSS Snapshot</a>
	 * @see <a href="https://en.wikibooks.org/wiki/XML_-_Managing_Data_Exchange/XSLT_and_Style_Sheets">Wikibooks: XML - Managing Data Exchange/XSLT and Style Sheets</a>
	 * @param newValue is new value for the class field.
	 */
	protected void setConcreteCssClass(String newValue)
	{
		if (newValue == null)
			newValue = new String(); // Principle of Least Astonishment (POLA) #4
			// https://en.wikipedia.org/wiki/Principle_of_least_astonishment
		cssClass = newValue;
	}
	/** Protected internal superclass method to keep HTML id private, scene authors should use method setHtmlID(newValue) instead.
	 * This attribute is only functional if the X3D model is loaded within an HTML page.
	 * <i>Tooltip:</i> the id attribute on each X3D node is considered a unique identifier when used as part of an encompassing HTML/DOM context.
	 * @see <a href="https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/htmlGuidelines.html#CSS">X3D Architecture Annex L - HTML authoring guidelines, CSS considerations</a>
     * <br />
     * <i>Warning:</i> the id attribute is in a distinct separate namespace from DEF identifiers and thus not applicable for USE nodes, ROUTE statements, or Script references.
	 * @see <a href="https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/htmlGuidelines.html#ContentDefinitionPagePresentation">X3D Architecture Annex L - HTML authoring guidelines, L.3.1 Content definition and page presentation</a>
	 * @see <a href="https://www.w3.org/TR/html52">HTML 5.2</a> W3C Recommendation
	 * @see <a href="https://www.w3.org/DOM/DOMTR">Document Object Model (DOM) Technical Reports</a>
	 * @see <a href="https://en.wikibooks.org/wiki/XML_-_Managing_Data_Exchange/XSLT_and_Style_Sheets">Wikibooks: XML - Managing Data Exchange/XSLT and Style Sheets</a>
	 * @param newValue is new value for the id field.
	 */
	protected void setConcreteHtmlID(String newValue)
	{
		if (newValue == null)
			newValue = new String(); // Principle of Least Astonishment (POLA) #4
			// https://en.wikipedia.org/wiki/Principle_of_least_astonishment
		htmlID = newValue;
	}
	/** Protected internal superclass method to keep cssStyles private, scene authors should use method setCssStyle(newValue) instead.
	 * This attribute is only functional if the X3D model is loaded within an HTML page.
	 * <i>Tooltip:</i> The style attribute is a space-separated list of classes, reserved for use by CSS cascading stylesheets.
	 * @see <a href="https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/htmlGuidelines.html#CSS">X3D Architecture Annex L - HTML authoring guidelines, CSS considerations</a>
	 * @see <a href="https://www.w3.org/Style/CSS">W3C Cascading Style Sheets</a>
	 * @see <a href="https://www.w3.org/TR/css-2018">W3C CSS Snapshot</a>
	 * @see <a href="https://en.wikibooks.org/wiki/XML_-_Managing_Data_Exchange/XSLT_and_Style_Sheets">Wikibooks: XML - Managing Data Exchange/XSLT and Style Sheets</a>
	 * @param newValue is new value for the style field.
	 */
	protected void setConcreteCssStyle(String newValue)
	{
		if (newValue == null)
			newValue = new String(); // Principle of Least Astonishment (POLA) #4
			// https://en.wikipedia.org/wiki/Principle_of_least_astonishment
		cssStyle = newValue;
	}
	/**
	 * Provide String value from inputOutput SFString field named <i>id</i>.
	 * <br><br>
	 * <i>Tooltip:</i> the id attribute on each X3D node is considered a unique identifier when used as part of an encompassing HTML/DOM context.The class field is a space-separated list of classes, reserved for use by CSS cascading stylesheets.
	 * @see <a href="https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/htmlGuidelines.html#CSS">X3D Architecture Annex L - HTML authoring guidelines, CSS considerations</a>
	 * @see <a href="https://www.w3.org/Style/CSS">W3C Cascading Style Sheets</a>
	 * @see <a href="https://www.w3.org/TR/css-2018">W3C CSS Snapshot</a>
	 * @see <a href="https://en.wikibooks.org/wiki/XML_-_Managing_Data_Exchange/XSLT_and_Style_Sheets">Wikibooks: XML - Managing Data Exchange/XSLT and Style Sheets</a>
	 * @return value of class field
	 */
	public String getHtmlID()
	{
		return htmlID;
	}
	/**
	 * Provide String value from inputOutput SFString field named <i>class</i>.
	 * <br><br>
	 * <i>Tooltip:</i> The class field is a space-separated list of classes, reserved for use by CSS cascading stylesheets.
	 * @see <a href="https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/htmlGuidelines.html#CSS">X3D Architecture Annex L - HTML authoring guidelines, CSS considerations</a>
	 * @see <a href="https://www.w3.org/Style/CSS">W3C Cascading Style Sheets</a>
	 * @see <a href="https://www.w3.org/TR/css-2018">W3C CSS Snapshot</a>
	 * @see <a href="https://en.wikibooks.org/wiki/XML_-_Managing_Data_Exchange/XSLT_and_Style_Sheets">Wikibooks: XML - Managing Data Exchange/XSLT and Style Sheets</a>
	 * @return value of class field
	 */
	public String getCssClass()
	{
		return cssClass;
	}
	/**
	 * Provide String value from inputOutput SFString field named <i>style</i>.
	 * <br><br>
	 * <i>Tooltip:</i> The style field provides an inline block of CSS for element styling, reserved for use by CSS cascading stylesheets.
	 * @see <a href="https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/htmlGuidelines.html#CSS">X3D Architecture Annex L - HTML authoring guidelines, CSS considerations</a>
	 * @see <a href="https://www.w3.org/Style/CSS">W3C Cascading Style Sheets</a>
	 * @see <a href="https://www.w3.org/TR/css-2018">W3C CSS Snapshot</a>
	 * @see <a href="https://en.wikibooks.org/wiki/XML_-_Managing_Data_Exchange/XSLT_and_Style_Sheets">Wikibooks: XML - Managing Data Exchange/XSLT and Style Sheets</a>
	 * @return value of style field
	 */
	public String getCssStyle()
	{
		return cssStyle;
	}

	/** Results log of local validation. */
	protected StringBuilder validationResult = new StringBuilder();

	/**
	 * Add comment as String to contained commentsList.
	 * @param newComment initial value
	 * @return <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public abstract X3DConcreteElement addComments (String newComment);

	/**
	 * Debug support: adding empty comment as CommentsBlock to children field has no effect.
	 * @return <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public X3DConcreteElement addComments()
	{
		return this;
	}

	/**
	 * Add comments as String[] array to contained commentsList.
	 * @param newComments array of comments
	 * @return <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public abstract X3DConcreteElement addComments (String[] newComments);

	/**
	 * Add CommentsBlock to element
	 * @param newCommentsBlock block of comments to add
	 * @return <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public abstract X3DConcreteElement addComments (CommentsBlock newCommentsBlock);

	/** Get output of results from prior validation, if any
	 * @return validation results (if any)
	 */
	public String getValidationResult()
	{
		return validationResult.toString();
	}

	/**
	 * Recursive method to provide X3D string serialization of this model subgraph, utilizing XML encoding and conforming to X3D Canonical Form.
	 * @see X3D#FILE_EXTENSION_X3D
	 * @see X3D#FILE_EXTENSION_XML
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-1/V3.3/Part01/X3D_XML.html">X3D XML Encoding</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-3/V3.3/Part03/concepts.html#X3DCanonicalForm" target="blank">X3D Compressed Binary Encoding: X3D Canonical Form</a>
	 * @see <a href="https://www.web3d.org/x3d/tools/canonical/doc/x3dTools.htm">X3D Canonicalization (C14N) Tool</a>
	 * @return X3D string
	 */
	public String toStringX3D()
	{
		return toStringX3D(0); // apply next method with initial indentation level 0
	}

	/**
	 * Recursive method to provide X3D string serialization of this model subgraph, utilizing XML encoding and conforming to X3D Canonical Form.
	 * @param indentLevel number of levels of indentation for this element
	 * @see X3D#FILE_EXTENSION_X3D
	 * @see X3D#FILE_EXTENSION_XML
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-1/V3.3/Part01/X3D_XML.html">X3D XML Encoding</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-3/V3.3/Part03/concepts.html#X3DCanonicalForm" target="blank">X3D Compressed Binary Encoding: X3D Canonical Form</a>
	 * @see <a href="https://www.web3d.org/x3d/tools/canonical/doc/x3dTools.htm">X3D Canonicalization (C14N) Tool</a>
	 * @return X3D string
	 */
	abstract public String toStringX3D(int indentLevel); // must be overridden

	/**
	 * Recursive method to provide ClassicVRML string serialization.
	 * @see X3D#FILE_EXTENSION_CLASSICVRML
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dResources.html#VRML">X3D Resources: Virtual Reality Modeling Language (VRML) 97</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-2/V3.3/Part02/X3D_ClassicVRML.html">Extensible 3D (X3D) encodings Part 2: Classic VRML encoding</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-2/V3.3/Part02/grammar.html">Extensible 3D (X3D) encodings Part 2: Classic VRML encoding, Annex A: Grammar</a>
	 * @return ClassicVRML string
	 */
	public String toStringClassicVRML()
	{
		return toStringClassicVRML(0); // apply next method with initial indentation level 0
	}

	/**
	 * Recursive method to provide ClassicVRML string serialization.
	 * @param indentLevel number of levels of indentation for this element
	 * @see X3D#FILE_EXTENSION_CLASSICVRML
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dResources.html#VRML">X3D Resources: Virtual Reality Modeling Language (VRML) 97</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-2/V3.3/Part02/X3D_ClassicVRML.html">Extensible 3D (X3D) encodings Part 2: Classic VRML encoding</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-2/V3.3/Part02/grammar.html">Extensible 3D (X3D) encodings Part 2: Classic VRML encoding, Annex A: Grammar</a>
	 * @return ClassicVRML string
	 */
	abstract public String toStringClassicVRML(int indentLevel); // must be overridden

	/**
	 * Recursive method to provide VRML97 string serialization.
	 * @see X3D#FILE_EXTENSION_VRML97
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dResources.html#VRML">X3D Resources: Virtual Reality Modeling Language (VRML) 97</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/14772/V2.0/index.html">Virtual Reality Modeling Language (VRML) 97 specification</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/14772-1/V2.1/index.html">VRML 97 v2.1 Amendment</a>
	 * @return VRML97 string
	 */
	public String toStringVRML97()
	{
		return toStringVRML97(0); // apply next method with initial indentation level 0
	}

	/**
	 * Recursive method to provide VRML97 string serialization.
	 * @param indentLevel number of levels of indentation for this element
	 * @see X3D#FILE_EXTENSION_VRML97
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dResources.html#VRML">X3D Resources: Virtual Reality Modeling Language (VRML) 97</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/14772/V2.0/index.html">Virtual Reality Modeling Language (VRML) 97 specification</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/14772-1/V2.1/index.html">VRML 97 v2.1 Amendment</a>
	 * @return VRML97 string
	 */
	abstract public String toStringVRML97(int indentLevel); // must be overridden

	/**
	 * Recursive method to determine whether this element is valid.
	 * @return true if validate() has no results
	 */
	public boolean isValid()
    {
        return validate().isEmpty();
    }

	/**
	 * Recursive method to validate this element plus all contained nodes and statements,
	 * using both datatype-specification value checks and regular expression (regex) checking of corresponding string values.
         * This method must be overridden, ensuring that a concrete method is provided for each class.
	 * @return validation results (if any)
	 */
	abstract public String validate();
}
