/*
Web3D Consortium Open-Source License for Models and Software

Copyright (c) 1995-2025 held by the author(s).  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer
      in the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the Web3D Consortium (https://www.web3D.org)
      nor the names of its contributors may be used to endorse or
      promote products derived from this software without specific
      prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

package org.web3d.x3d.jsail.Core;
import java.util.*;
// import org.web3d.x3d.sai.*;      // making sure #2
// import org.web3d.x3d.sai.Core.*;  // making sure #2
import org.web3d.x3d.sai.InvalidFieldValueException;  // making sure #2
import org.web3d.x3d.jsail.*; // again making sure #2

import java.util.Arrays;
import org.web3d.x3d.sai.Core.*;  // making sure #3
import org.web3d.x3d.jsail.*;
import org.web3d.x3d.jsail.Core.*;

import org.web3d.x3d.jsail.fields.*; // making sure #4
import org.web3d.x3d.jsail.Grouping.*;
import org.web3d.x3d.jsail.Navigation.*;
// import org.web3d.x3d.jsail.Shape.*;
// import org.web3d.x3d.jsail.Networking.*;
// import org.web3d.x3d.jsail.Core.*;
import org.web3d.x3d.jsail.*; // again making sure #4

import org.web3d.x3d.jsail.Core.*;
import java.util.Arrays;

/**
 * Functional summary: a fieldValue statement re-initializes the default value of a field in a ProtoInstance. Each fieldValue statement can contain either attribute-value or node content. This concrete class represents an X3D fieldValue statement.
 * 
 * <br><br>
 * <i>fieldValue statement tooltip</i>: A fieldValue statement re-initializes the default value of a field in a parent ProtoInstance. Note that a field statement with a matching name must be already defined by the corresponding ProtoDeclare or ExternProtoDeclare statement.
 * <ul>
 *  <li> <i>Hint:</i> first add a parent ProtoInstance node before defining a new fieldValue. </li> 
 *  <li> <i>Warning:</i> fieldValue content type must match the content type defined in the corresponding ProtoDeclare or ExternProtoDeclare field statement. </li> 
 *  <li> <i>Hint:</i>  in addition to defining a simple-type value attribute, a fieldValue statement can alternatively hold initializing SFNode/MFNode values as contained node(s). </li> 
 * </ul>
 * @author Don Brutzman and Roy Walmsley
 * @see <a href="https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/concepts.html#ProtoInstanceAndFieldValueStatement" target="blank">X3D Abstract Specification: fieldValue</a>
 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html#fieldValue" target="_blank">X3D Tooltips: fieldValue</a>
 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html" target="_blank">X3D Scene Authoring Hints</a>
 */
public class fieldValue extends org.web3d.x3d.jsail.X3DConcreteStatement
{
    // Member value declarations are encapsulated and private, using preferred Java types for concretes library

    /* @name=children, @type=MFNode, $javaType=ArrayList< org.web3d.x3d.sai.Core.X3DNode>, $javaReferenceType=org.web3d.x3d.sai.Core.X3DNode, $javaPrimitiveType=ArrayList< org.web3d.x3d.sai.Core.X3DNode>, $normalizedMemberObjectName=children, $isX3dStatement=false, @acceptableNodeTypes=X3DNode */ 
    private ArrayList<org.web3d.x3d.sai.Core.X3DNode> children = new ArrayList<>(); //  X3D field type: MFNode acceptable node types: X3DNode or ProtoInstance

    /* @name=name, @type=SFString, $javaType=String, $javaReferenceType=String, $javaPrimitiveType=String, $normalizedMemberObjectName=name, $isX3dStatement=false, @acceptableNodeTypes= */ 
    private String name = new String(); //  X3D field type: SFString

    /* @name=value, @type=SFString, $javaType=String, $javaReferenceType=String, $javaPrimitiveType=String, $normalizedMemberObjectName=value, $isX3dStatement=false, @acceptableNodeTypes= */ 
    private String value = new String(); //  X3D field type: SFString or MFString

    /** Default value for this field type is an empty array. */
    private static final boolean[] DEFAULT_VALUE_BOOLEAN = new boolean[0]; // initialize as empty array

    /** boolean array, typed value holder */
    private boolean[] valueArrayBoolean;

    /** Default value for this field type is an empty array. */
    private static final int[] DEFAULT_VALUE_INTEGER = new int[0]; // initialize as empty array

    /** int array, typed value holder */
    private int[] valueArrayInteger;

    /** Default value for this field type is an empty array. */
    private static final float[] DEFAULT_VALUE_FLOAT = new float[0]; // initialize as empty array

    /** float array, typed value holder */
    private float[] valueArrayFloat;

    /** Default value for this field type is an empty array. */
    private static final double[] DEFAULT_VALUE_DOUBLE = new double[0]; // initialize as empty array

    /** double array, typed value holder */
    private double[] valueArrayDouble;


	// String constants for default field values match X3D Schema definitions

	/** String constant <i>NAME</i> provides name of this element: <i>fieldValue</i>. */
	@SuppressWarnings("FieldNameHidesFieldInSuperclass")
	public static final String NAME = "fieldValue";

	/** Provides name of this element: <i>fieldValue</i>
	 * @return name of this element
	 */
	@Override
	public final String getElementName()
	{
		return NAME;
	}

	/** String constant <i>COMPONENT</i> defines X3D component for the <i>fieldValue</i> element: <i>Core</i> */
	public static final String COMPONENT = "Core";

	/** Defines X3D component for the <i>fieldValue</i> element: <i>Core</i>
	 * @return X3D component for this element
	 */
	/* @Override */
	public final String getComponent()
	{
		return COMPONENT;
	}

	/** Integer constant <i>LEVEL</i> provides default X3D <i>Core</i> component level for this element: <i>2</i> */
	public static final int LEVEL = 2;

	/** Provides default X3D component level for this element: <i>2</i>
	 * @return default X3D component level for this element
	 */
	/* @Override */
	public final int getComponentLevel()
	{
		return LEVEL;
	}

            // DEFAULT_VALUE declarations

	/** SFString field named <i>name</i> has default value equal to an empty string. */
	public static final String NAME_DEFAULT_VALUE = "";

	/** SFString field named <i>value</i> has default value equal to an empty string. */
	public static final String VALUE_DEFAULT_VALUE = "";

	/** Indicate type corresponding to given fieldName.
	 * @param fieldName name of field in this X3D statement
	 * @see ConfigurationProperties#ERROR_UNKNOWN_FIELD_TYPE
	 * @return X3D type (SFvec3f etc.), otherwise ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE if not recognized
	 */
	/* @Override */
	public String getFieldType(String fieldName)
	{
		String result;

		switch (fieldName)
		{
			// String constants for exact field type values matching X3D Schema definitions,
			// thus avoiding spelling errors and allowing type-matching checks

			case "containerField":
				result = "SFString";
				break;
			case "children":
				result = "MFNode";
				break;
			case "class":
				result = "SFString";
				break;
			case "id":
				result = "SFString";
				break;
			case "name":
				result = "SFString";
				break;
			case "style":
				result = "SFString";
				break;
			case "value":
				result = "SFString";
				break;
			default:
			{
				// if fieldName has a prefix "set_" prepended (or a suffix "_changed" appended) then try again by stripping it and recursing once
				if      (fieldName.trim().startsWith("set_"))
						 result = getFieldType(fieldName.trim().substring(4)); // after "set_"
				else if (fieldName.trim().endsWith("_changed"))
						 result = getFieldType(fieldName.trim().substring(0, fieldName.length() - 8)); // before "_changed"
				else     result = ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE; // unique return value avoids mistaken matches
			}
		}
		return result;
	}

	/** Indicate accessType corresponding to given fieldName.
	 * @param fieldName name of field in this X3D statement
	 * @see ConfigurationProperties#ERROR_UNKNOWN_FIELD_ACCESSTYPE
	 * @return X3D accessType (inputOnly etc.), otherwise ConfigurationProperties.ERROR_UNKNOWN_FIELD_ACCESSTYPE if not recognized
	 */
	/* @Override */
	public String getAccessType(String fieldName)
	{
		String result; // set by following checks
		switch (fieldName)
		{
			// String constants for field accessType values matching X3D Schema definitions,
			// thus avoiding spelling errors and allowing accessType-matching checks
			case "children":
				result = "inputOutput";
				break;
			case "class":
				result = "inputOutput";
				break;
			case "id":
				result = "inputOutput";
				break;
			case "name":
				result = "inputOutput";
				break;
			case "style":
				result = "inputOutput";
				break;
			case "value":
				result = "inputOutput";
				break;
			default:
			{
				// if user has added a prefix "set_" or suffix "_changed" then try again by stripping it and recursing once
				if      (fieldName.trim().startsWith("set_"))
						 result = getAccessType(fieldName.trim().substring(4)); // after "set_"
				else if (fieldName.trim().endsWith("_changed"))
						 result = getAccessType(fieldName.trim().substring(0, fieldName.length() - 8)); // before "_changed"
				else     result = ConfigurationProperties.ERROR_UNKNOWN_FIELD_ACCESSTYPE; // unique return value avoids mistaken matches
			}
		}
		return result;
	}

	/** Constructor for fieldValue to initialize member variables with default values. */
	public fieldValue()
	{
		super(); // constructor invocation and corresponding initialize()
		initialize();
	}

	/** Initialize all member variables to default values. */
	@Override
	public final void initialize()
	{
		super.initialize();

		children = new ArrayList<>();
		name = NAME_DEFAULT_VALUE;
		value = VALUE_DEFAULT_VALUE;

		valueArrayBoolean = java.util.Arrays.copyOf(DEFAULT_VALUE_BOOLEAN, DEFAULT_VALUE_BOOLEAN.length); // must be separate copy
		valueArrayInteger = java.util.Arrays.copyOf(DEFAULT_VALUE_INTEGER, DEFAULT_VALUE_INTEGER.length); // must be separate copy
		valueArrayFloat   = java.util.Arrays.copyOf(DEFAULT_VALUE_FLOAT,   DEFAULT_VALUE_FLOAT.length  ); // must be separate copy
		valueArrayDouble  = java.util.Arrays.copyOf(DEFAULT_VALUE_DOUBLE,  DEFAULT_VALUE_DOUBLE.length ); // must be separate copy
	}

	// ==== Accessor methods: strongly typed get/set methods for compile-time strictness

	/**
	 * Provide array of  org.web3d.x3d.sai.Core.X3DNode results (using an array consisting of properly typed nodes or ProtoInstances) from inputOutput MFNode field <i>children</i>.
	 * <br><br>
	 * <i>Tooltip:</i> [X3DNode] If this field definition has type SFNode or MFNode, then initialization node (or nodes) of any appropriate type may be provided as children of the field definition.
 * <ul>
 *  <li> <i>Warning:</i> do not include a value attribute when providing such child node(s). </li> 
 *  <li> <i>Hint:</i> X3D Architecture, Concepts, 4.4.2 Object model, <a href="https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/concepts.html#Objectmodel" target="_blank">https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/concepts.html#Objectmodel</a> </li> 
 *  <li> <i>Hint:</i>  X3D Architecture, Concepts, 4.4.4 Prototype semantics, <a href="https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/concepts.html#PrototypeSemantics" target="_blank">https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/concepts.html#PrototypeSemantics</a> </li> 
 * </ul>
	 * <br><br>
	 * <i>Warning:</i> according to X3D Unified Object Model (X3DUOM), acceptable node types are limited to X3DNode.
	 * @see org.web3d.x3d.sai.Core.X3DNode
	 * @return value of children field
	 */
	// ($isInterface=false, $isException=false, $isServiceInterface=false, $isX3dStatement=false, $isClassX3dStatement=true, preceding-sibling::Inheritance=, $baseType=, $additionalInheritanceBaseType=, $thisClassName=fieldValue, $CamelCaseName=Children, $memberObjectName=children, $memberObjectName=children)
	public ArrayList< org.web3d.x3d.sai.Core.X3DNode> getChildren()
	{
		return children;
	}

	/**
	 * Accessor method to assign  org.web3d.x3d.sai.Core.X3DNode array (using an array consisting of properly typed nodes or ProtoInstances) to inputOutput MFNode field <i>children</i>.
	 * <br><br>
	 * <i>Tooltip:</i> [X3DNode] If this field definition has type SFNode or MFNode, then initialization node (or nodes) of any appropriate type may be provided as children of the field definition.
 * <ul>
 *  <li> <i>Warning:</i> do not include a value attribute when providing such child node(s). </li> 
 *  <li> <i>Hint:</i> X3D Architecture, Concepts, 4.4.2 Object model, <a href="https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/concepts.html#Objectmodel" target="_blank">https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/concepts.html#Objectmodel</a> </li> 
 *  <li> <i>Hint:</i>  X3D Architecture, Concepts, 4.4.4 Prototype semantics, <a href="https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/concepts.html#PrototypeSemantics" target="_blank">https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/concepts.html#PrototypeSemantics</a> </li> 
 * </ul>
	 * <br><br>
	 * <i>Note:</i> according to X3D Unified Object Model (X3DUOM), acceptable node types are limited to X3DNode.
	 * @param newValue is new value for the children field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue setChildren(ArrayList< org.web3d.x3d.sai.Core.X3DNode> newValue)
	{
		// set-newValue-validity-checks #0.a
		children = newValue;
		for (org.web3d.x3d.sai.Core.X3DNode arrayElement : children)
		{
			((X3DConcreteElement) arrayElement).setParent(this); // parentTest11
		}
		return this;
	}

	/**
	 * Add single node to existing array of children nodes (if any).
	 * <br><br>
	 * <i>Note:</i> according to X3D Unified Object Model (X3DUOM), acceptable node types are limited to X3DNode and ProtoInstance.
	 * @param newValue is new value to be appended the children field.	 */
	public void addChildren(org.web3d.x3d.sai.Core.X3DNode newValue)
	{
		if (newValue == null) return; // newValueNullReturnSelf
		children.add(newValue);
		((X3DConcreteElement) newValue).setParent(this); // parentTest2
	}
	/**
	 * Add single ProtoInstance (with appropriate node type and containerField value) to MFNode array for children field.
	 * <br><br>
	 * <i>Note:</i> according to X3D Unified Object Model (X3DUOM), acceptable node types are limited to ShaderPart and ProtoInstance.
	 * @param newValue is new value to be appended the parts field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).	 */
	public fieldValue addChildren(ProtoInstance newValue)
	{
		if (newValue == null) return this; // newValueNullReturnThis
		children.add(newValue);
		((X3DConcreteElement) newValue).setParent(this); // parentTest16
		return this;
	}

	/**
	 * Utility method to add single child element to contained list of existing children nodes (if any).
	 * @param newValue is new node value to be appended the children field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	*/
	public fieldValue addChild(org.web3d.x3d.sai.Core.X3DNode newValue)
	{
		if (newValue == null) return this; // newValueNullReturnThis

		children.add((org.web3d.x3d.sai.Core.X3DNode)newValue);
		((X3DConcreteElement) newValue).setParent(this); // parentTest3
//      if  (newValue instanceof ProtoInstance)
//           ((ProtoInstance) newValue).setContainerField("children");

		return this;
	}
	/**
	 * Add array of children nodes to array of existing nodes (if any).
	 * <br><br>
	 * <i>Note:</i> according to X3D Unified Object Model (X3DUOM), acceptable node types are limited to X3DNode.
	 * @param newValue is new value array to be appended the children field.
	 */
	public void addChildren(org.web3d.x3d.sai.Core.X3DNode[] newValue)
	{
		if (newValue == null) return; // newValueNullReturnSelf
		for (int i = 0; i < newValue.length; i++)
		{
			if  ((newValue[i] instanceof org.web3d.x3d.sai.Core.X3DNode) || (newValue[i] instanceof ProtoInstance))
			{
				children.add(newValue[i]);
				((X3DConcreteElement) newValue[i]).setParent(this); // parentTest6
				if (newValue[i] instanceof ProtoInstance)
					((ProtoInstance) newValue[i]).setContainerField("children");
			}
			else throw new org.web3d.x3d.sai.InvalidFieldValueException("org.web3d.x3d.sai.Core.X3DNode[] newValue["+i+"] is not instanceof org.web3d.x3d.sai.Core.X3DNode or ProtoInstance, array newValue=" + Arrays.toString(newValue));
		}
	}

	/**
	 * Set single children node, replacing prior array of existing nodes (if any).
	 * <br><br>
	 * <i>Note:</i> according to X3D Unified Object Model (X3DUOM), acceptable node types are restricted to X3DNode.
	 * @param newValue is new node for the children field (restricted to X3DNode)
	 */
	public void setChildren(org.web3d.x3d.sai.Core.X3DNode newValue)
	{
		if (newValue == null)
		{
			clearChildren(); // reset newValueNullClearsFieldReturnVoid
			return;
		}
		if  (newValue instanceof org.web3d.x3d.sai.Core.X3DNode)
		{
			for (org.web3d.x3d.sai.Core.X3DNode element : children)
				((X3DConcreteElement) element).clearParent(); // remove references to facilitate Java memory management
			clearChildren(); // reset
			((X3DConcreteElement) newValue).setParent(this); // parentTest8
			children.add(newValue);
		}
		else throw new org.web3d.x3d.sai.InvalidFieldValueException("org.web3d.x3d.sai.Core.X3DNode newValue is not instanceof org.web3d.x3d.sai.Core.X3DNode; newValue=" + newValue);
}
	/**
	 * Utility method to clear MFNode value of children field.  This method does not initialize withCHILDREN_DEFAULT_VALUE.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
setAttribute method invocations).
	 */
	public fieldValue clearChildren()
	{
		for (org.web3d.x3d.sai.Core.X3DNode element : children)
			((X3DConcreteElement) element).clearParent(); // remove references to facilitate Java memory management
		children.clear(); // reset MF field
		return this;
	}

	/**
	 * Indicate whether an object is available for inputOutput MFNode field <i>children</i>.
	 * @return whether a properly typed node or ProtoInstance array or CommentsBlock is available.
	 * @see #getChildren()
	 */
	public boolean hasChildren()
	{
		return (!children.isEmpty());
	}

	/**
	 * Add comment as CommentsBlock to children field
	 * @param newComment initial value
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
setAttribute method invocations).
	 */
	/* @Override */
	public fieldValue addComments (String newComment)
	{
		if (newComment == null) return this;
		children.add(new CommentsBlock (newComment));
		return this;
	}
	/**
	 * Add comments as String[] array to children field
	 * @param newComments array of comments
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
setAttribute method invocations).
	 */
	/* @Override */
	public fieldValue addComments (String[] newComments)
	{
		if (newComments == null) return this;
		children.add(new CommentsBlock (newComments));
		return this;
	}
	/**
	 * Add CommentsBlock to children field
	 * @param newCommentsBlock block of comments to add
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	/* @Override */
	public fieldValue addComments (CommentsBlock newCommentsBlock)
	{
		if (newCommentsBlock == null) return this;
		children.add(newCommentsBlock);
		return this;
	}
											/**
	 * Provide String value from inputOutput SFString field named <i>class</i>.
	 * @return value of class field
	 */
	// ($isInterface=false, $isException=false, $isServiceInterface=false, $isX3dStatement=false, $isClassX3dStatement=true, preceding-sibling::Inheritance=, $baseType=, $additionalInheritanceBaseType=, $thisClassName=fieldValue, $CamelCaseName=CssClass, $memberObjectName=cssClass, $memberObjectName=cssClass)
	public String getCssClass()
	{
		// override abstract method in X3DConcreteNode
		return super.getCssClass();
	}

	/**
	 * Accessor method to assign String value to inputOutput SFString field named <i>class</i>.
	 * @param newValue is new value for the class field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public final fieldValue setCssClass(String newValue)
	{
		// set-newValue-validity-checks #0.a
		if (newValue == null)
			newValue = new String(); // Principle of Least Astonishment (POLA) #5
			// https://en.wikipedia.org/wiki/Principle_of_least_astonishment
		setConcreteCssClass(newValue); // private superclass method
		return this;
	}

	/**
	 * Assign typed object value to SFString cssClass field, similar to {@link #setCssClass(String)}.
	 * @param newValue is new value for the class field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue setCssClass(SFString newValue)
	{
			// set-newValue-validity-checks #1 gets handled by set-primitive method
			setCssClass(newValue.getPrimitiveValue());
			return this;
	}
	/**
	 * Provide String value from inputOutput SFString field named <i>id</i>.
	 * @return value of id field
	 */
	// ($isInterface=false, $isException=false, $isServiceInterface=false, $isX3dStatement=false, $isClassX3dStatement=true, preceding-sibling::Inheritance=, $baseType=, $additionalInheritanceBaseType=, $thisClassName=fieldValue, $CamelCaseName=HtmlID, $memberObjectName=HtmlID, $memberObjectName=HtmlID)
	public String getHtmlID()
	{
		// override abstract method in X3DConcreteNode
		return super.getHtmlID();
	}

	/**
	 * Accessor method to assign String value to inputOutput SFString field named <i>id</i>.
	 * @param newValue is new value for the id field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public final fieldValue setHtmlID(String newValue)
	{
		// set-newValue-validity-checks #0.a
		if (newValue == null)
			newValue = new String(); // Principle of Least Astonishment (POLA) #5
			// https://en.wikipedia.org/wiki/Principle_of_least_astonishment
		setConcreteHtmlID(newValue); // private superclass method
		return this;
	}

	/**
	 * Assign typed object value to SFString HtmlID field, similar to {@link #setHtmlID(String)}.
	 * @param newValue is new value for the id field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue setHtmlID(SFString newValue)
	{
			// set-newValue-validity-checks #1 gets handled by set-primitive method
			setHtmlID(newValue.getPrimitiveValue());
			return this;
	}
	/**
	 * Provide String value from inputOutput SFString field named <i>name</i>.
	 * <br><br>
	 * <i>Tooltip:</i> Name of the ProtoInstance field being re-initialized (corresponding to field name already defined in ProtoDeclare or ExternProtoDeclare).
 * <ul>
 *  <li> <i>Hint:</i> well-defined names can simplify design and debugging through improved author understanding. </li> 
 *  <li> <i>Hint:</i>  X3D Scene Authoring Hints, Naming Conventions <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#NamingConventions" target="_blank">https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#NamingConventions</a> </li> 
 * </ul>
	 * @return value of name field
	 */
	// ($isInterface=false, $isException=false, $isServiceInterface=false, $isX3dStatement=false, $isClassX3dStatement=true, preceding-sibling::Inheritance=, $baseType=, $additionalInheritanceBaseType=, $thisClassName=fieldValue, $CamelCaseName=Name, $memberObjectName=name, $memberObjectName=name)
	public String getName()
	{
		return name;
	}

	/**
	 * Accessor method to assign String value to inputOutput SFString field named <i>name</i>.
	 * <br><br>
	 * <i>Tooltip:</i> Name of the ProtoInstance field being re-initialized (corresponding to field name already defined in ProtoDeclare or ExternProtoDeclare).
 * <ul>
 *  <li> <i>Hint:</i> well-defined names can simplify design and debugging through improved author understanding. </li> 
 *  <li> <i>Hint:</i>  X3D Scene Authoring Hints, Naming Conventions <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#NamingConventions" target="_blank">https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#NamingConventions</a> </li> 
 * </ul>
	 * <br><br>@see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#NamingConventions" target="_blank">X3D Scene Authoring Hints: Naming Conventions</a>
	 * @param newValue is new value for the name field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public final fieldValue setName(String newValue)
	{
		// set-newValue-validity-checks #0.a
		if (newValue == null)
			newValue = new String();
		// Check that newValue parameter meets naming requirements before assigning to fieldValue name
		if (!org.web3d.x3d.jsail.fields.SFString.isNMTOKEN(newValue))
		{
			throw new org.web3d.x3d.sai.InvalidFieldValueException("fieldValue name newValue='" + newValue + "'" + 
				" has illegal name value, cannot be empty and must be defined with valid NMTOKEN name string" + 
				" (with legal characters and no embedded whitespace).");
		}

        if (!org.web3d.x3d.jsail.fields.SFString.meetsX3dInteroperabilityNamingConventions(newValue))
		{
			if (ConfigurationProperties.isConsoleOutputVerbose())
                          System.out.println("[warning] fieldValue name newValue='" + newValue + "'" + 
				" has name value that does not meet X3D naming conventions.");
		}
		name = newValue;
		return this;
	}

	/**
	 * Assign typed object value to SFString name field, similar to {@link #setName(String)}.
	 * @param newValue is new value for the name field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue setName(SFString newValue)
	{
			// set-newValue-validity-checks #1 gets handled by set-primitive method
			setName(newValue.getPrimitiveValue());
			return this;
	}
	/**
	 * Provide String value from inputOutput SFString field named <i>style</i>.
	 * @return value of style field
	 */
	// ($isInterface=false, $isException=false, $isServiceInterface=false, $isX3dStatement=false, $isClassX3dStatement=true, preceding-sibling::Inheritance=, $baseType=, $additionalInheritanceBaseType=, $thisClassName=fieldValue, $CamelCaseName=CssStyle, $memberObjectName=cssClass, $memberObjectName=cssClass)
	public String getCssStyle()
	{
		// override abstract method in X3DConcreteNode
		return super.getCssStyle();
	}

	/**
	 * Accessor method to assign String value to inputOutput SFString field named <i>style</i>.
	 * @param newValue is new value for the style field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public final fieldValue setCssStyle(String newValue)
	{
		// set-newValue-validity-checks #0.a
		if (newValue == null)
			newValue = new String(); // Principle of Least Astonishment (POLA) #5
			// https://en.wikipedia.org/wiki/Principle_of_least_astonishment
		setConcreteCssStyle(newValue); // private superclass method
		return this;
	}

	/**
	 * Assign typed object value to SFString cssClass field, similar to {@link #setCssStyle(String)}.
	 * @param newValue is new value for the style field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue setCssStyle(SFString newValue)
	{
			// set-newValue-validity-checks #1 gets handled by set-primitive method
			setCssStyle(newValue.getPrimitiveValue());
			return this;
	}
	/**
	 * Provide String value from inputOutput SFString field named <i>value</i>.
	 * <br><br>
	 * <i>Tooltip:</i> Initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
 * <ul>
 *  <li> <i>Hint:</i> initialize SFNode/MFNode field value using contained content element(s), instead of a simple attribute value. </li> 
 *  <li> <i>Warning:</i>  do not include a value attribute when providing child node(s) for type SFNode/MFNode. </li> 
 * </ul>
	 * @return value of value field
	 */
	// ($isInterface=false, $isException=false, $isServiceInterface=false, $isX3dStatement=false, $isClassX3dStatement=true, preceding-sibling::Inheritance=, $baseType=, $additionalInheritanceBaseType=, $thisClassName=fieldValue, $CamelCaseName=Value, $memberObjectName=value, $memberObjectName=value)
	public String getValue()
	{
		String valueType = getType();
		// alternatives for return typed value
		if ((valueArrayBoolean != null) && (valueArrayBoolean.length > 0))
		{
			if      (valueType.equals(field.TYPE_SFBOOL)) return SFBool.toString(valueArrayBoolean[0]);
			else if (valueType.equals(field.TYPE_MFBOOL)) return MFBool.toString(valueArrayBoolean);
                        // ProtoInstance fieldValue does not directly contain type but a value has been found nonetheless
                        else if (valueArrayBoolean.length == 1)               return SFBool.toString(valueArrayBoolean[0]);
                        else                                                  return MFBool.toString(valueArrayBoolean);
		}
		else if ((valueArrayInteger != null) && (valueArrayInteger.length > 0))
		{
			if      (valueType.equals(field.TYPE_SFINT32)) return SFInt32.toString(valueArrayInteger[0]);
			else if (valueType.equals(field.TYPE_MFINT32)) return MFInt32.toString(valueArrayInteger);
			else if (valueType.equals(field.TYPE_SFIMAGE)) return SFImage.toString(valueArrayInteger);
			else if (valueType.equals(field.TYPE_MFIMAGE)) return MFImage.toString(valueArrayInteger);
                        // ProtoInstance fieldValue does not directly contain type but a value has been found nonetheless
                        else if (valueArrayInteger.length == 1)               return SFInt32.toString(valueArrayInteger[0]);
                        else                                                  return MFInt32.toString(valueArrayInteger);
		}
		else if ((valueArrayFloat != null) && (valueArrayFloat.length > 0))
		{
			if      (valueType.equals(field.TYPE_SFFLOAT))     return SFFloat.toString(valueArrayFloat[0]);
			else if (valueType.equals(field.TYPE_MFFLOAT))     return MFFloat.toString(valueArrayFloat);
			else if (valueType.equals(field.TYPE_SFCOLOR))     return SFColor.toString(valueArrayFloat);
			else if (valueType.equals(field.TYPE_MFCOLOR))     return MFColor.toString(valueArrayFloat);
			else if (valueType.equals(field.TYPE_SFCOLORRGBA)) return SFColorRGBA.toString(valueArrayFloat);
			else if (valueType.equals(field.TYPE_MFCOLORRGBA)) return MFColorRGBA.toString(valueArrayFloat);
			else if (valueType.equals(field.TYPE_SFROTATION))  return SFRotation.toString(valueArrayFloat);
			else if (valueType.equals(field.TYPE_MFROTATION))  return MFRotation.toString(valueArrayFloat);
			else if (valueType.equals(field.TYPE_SFVEC2F))     return SFVec2f.toString(valueArrayFloat);
			else if (valueType.equals(field.TYPE_SFVEC3F))     return SFVec3f.toString(valueArrayFloat);
			else if (valueType.equals(field.TYPE_SFVEC4F))     return SFVec4f.toString(valueArrayFloat);
			else if (valueType.equals(field.TYPE_MFVEC2F))     return MFVec2f.toString(valueArrayFloat);
			else if (valueType.equals(field.TYPE_MFVEC3F))     return MFVec3f.toString(valueArrayFloat);
			else if (valueType.equals(field.TYPE_MFVEC4F))     return MFVec4f.toString(valueArrayFloat);
			else if (valueType.equals(field.TYPE_MFMATRIX3F))  return MFMatrix3f.toString(valueArrayFloat);
			else if (valueType.equals(field.TYPE_MFMATRIX4F))  return MFMatrix4f.toString(valueArrayFloat);
                        // ProtoInstance fieldValue does not directly contain type but a value has been found nonetheless
                        else if (valueArrayFloat.length == 1)              return SFFloat.toString(valueArrayFloat[0]);
                        else                                               return MFFloat.toString(valueArrayFloat);
		}
		else if ((valueArrayDouble != null) && (valueArrayDouble.length > 0))
		{
			if      (valueType.equals(field.TYPE_SFDOUBLE))    return SFDouble.toString(valueArrayDouble[0]);
			else if (valueType.equals(field.TYPE_MFDOUBLE))    return MFDouble.toString(valueArrayDouble);
			else if (valueType.equals(field.TYPE_SFTIME))      return SFTime.toString(valueArrayDouble[0]);
			else if (valueType.equals(field.TYPE_MFTIME))      return MFTime.toString(valueArrayDouble);
			else if (valueType.equals(field.TYPE_SFVEC2D))     return SFVec2d.toString(valueArrayDouble);
			else if (valueType.equals(field.TYPE_SFVEC3D))     return SFVec3d.toString(valueArrayDouble);
			else if (valueType.equals(field.TYPE_SFVEC4D))     return SFVec4d.toString(valueArrayDouble);
			else if (valueType.equals(field.TYPE_MFVEC2D))     return MFVec2d.toString(valueArrayDouble);
			else if (valueType.equals(field.TYPE_MFVEC3D))     return MFVec3d.toString(valueArrayDouble);
			else if (valueType.equals(field.TYPE_MFVEC4D))     return MFVec4d.toString(valueArrayDouble);
			else if (valueType.equals(field.TYPE_MFMATRIX3D))  return MFMatrix3d.toString(valueArrayDouble);
			else if (valueType.equals(field.TYPE_MFMATRIX4D))  return MFMatrix4d.toString(valueArrayDouble);
                        // ProtoInstance fieldValue does not directly contain type but a value has been found nonetheless
                        else if (valueArrayDouble.length == 1)                    return SFDouble.toString(valueArrayDouble[0]);
                        else                                                      return MFDouble.toString(valueArrayDouble);
		}
		else if (value != null)
		{
			if      (valueType.equals(field.TYPE_SFSTRING) ||
					 valueType.equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE)) // matches XML default; might not be connected yet
				return new SFString(value).toString();
			else if (valueType.equals(field.TYPE_MFSTRING))
				return new MFString(value).toString();
		}
		// TODO error handling
		if (value == null)
		{
			return "";
		}
		else return value;
	}

	/**
	 * Accessor method to assign String value to inputOutput SFString field named <i>value</i>.
	 * <br><br>
	 * <i>Tooltip:</i> Initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
 * <ul>
 *  <li> <i>Hint:</i> initialize SFNode/MFNode field value using contained content element(s), instead of a simple attribute value. </li> 
 *  <li> <i>Warning:</i>  do not include a value attribute when providing child node(s) for type SFNode/MFNode. </li> 
 * </ul>
	 * @param newValue is new value for the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue setValue(String newValue)
	{
		// set-newValue-validity-checks #0.a
		if (newValue == null)
			newValue = new String(); // Principle of Least Astonishment (POLA) #5
			// https://en.wikipedia.org/wiki/Principle_of_least_astonishment
		// check for legal type
		String valueType = getType();
		if (valueType.equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE))
			valueType = field.TYPE_SFSTRING; // same as default found in file encodings
			// TODO must recheck during run-time validation to ensure fully constructed OK

		if ((valueType.equals(field.TYPE_SFNODE) || valueType.equals(field.TYPE_MFNODE)) &&
			(newValue.length() > 0))
		{
			throw new org.web3d.x3d.sai.InvalidProtoException("field name='" + getName() + "' with type='" + valueType +
					"' cannot have any simple-type value (newValue='" + newValue + "').  Use setChildren() method instead.");
		}
		else if (valueType.equals(field.TYPE_SFSTRING) || valueType.equals(field.TYPE_MFSTRING))
		{
			value = newValue;
		}
		else if (valueType.equals(field.TYPE_SFBOOL))
		{
			valueArrayBoolean = new MFBool().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFCOLOR))
		{
			valueArrayFloat = new MFColor().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFCOLORRGBA))
		{
			valueArrayFloat = new MFColorRGBA().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFDOUBLE))
		{
			valueArrayDouble = new MFDouble().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFFLOAT))
		{
			valueArrayFloat = new MFFloat().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFIMAGE))
		{
			valueArrayInteger = new MFImage().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFINT32))
		{
			valueArrayInteger = new MFInt32().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFMATRIX3D))
		{
			valueArrayDouble = new MFMatrix3d().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFMATRIX3F))
		{
			valueArrayFloat = new MFMatrix3f().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFMATRIX4D))
		{
			valueArrayDouble = new MFMatrix4d().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFMATRIX4F))
		{
			valueArrayFloat = new MFMatrix4f().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFROTATION))
		{
			valueArrayFloat = new MFRotation().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFTIME))
		{
			valueArrayDouble = new MFTime().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFVEC2D))
		{
			valueArrayDouble = new MFVec2d().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFVEC2F))
		{
			valueArrayFloat = new MFVec2f().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFVEC3D))
		{
			valueArrayDouble = new MFVec3d().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFVEC3F))
		{
			valueArrayFloat = new MFVec3f().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFVEC4D))
		{
			valueArrayDouble = new MFVec4d().setValueByString(newValue).getPrimitiveValue();
		}
	
		else if (valueType.equals(field.TYPE_SFVEC4F))
		{
			valueArrayFloat = new MFVec4f().setValueByString(newValue).getPrimitiveValue();
		}
			value = newValue;
		return this;
	}


	/**
	 * Append SFBool to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFBool newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFBOOL) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            boolean[] newArray = new boolean[valueArrayBoolean.length + 1];
            System.arraycopy(valueArrayBoolean, 0, newArray, 0, valueArrayBoolean.length);
            newArray[valueArrayBoolean.length] = newValue.getPrimitiveValue();
            valueArrayBoolean = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFBool " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFBool to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFBool newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFBOOL) ||
             getType().equals(field.TYPE_SFBOOL) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            boolean[] newArray = new boolean[valueArrayBoolean.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayBoolean, 0, newArray, 0, valueArrayBoolean.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayBoolean = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFBool " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFColor to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFColor newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFCOLOR) ||
             getType().equals(field.TYPE_SFCOLOR) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFColor " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFColor to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFColor newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFCOLOR) ||
             getType().equals(field.TYPE_SFCOLOR) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFColor " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFColorRGBA to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFColorRGBA newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFCOLORRGBA) ||
             getType().equals(field.TYPE_SFCOLORRGBA) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFColorRGBA " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFColorRGBA to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFColorRGBA newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFCOLORRGBA) ||
             getType().equals(field.TYPE_SFCOLORRGBA) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFColorRGBA " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFDouble to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFDouble newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFDOUBLE) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            double[] newArray = new double[valueArrayDouble.length + 1];
            System.arraycopy(valueArrayDouble, 0, newArray, 0, valueArrayDouble.length);
            newArray[valueArrayDouble.length] = newValue.getPrimitiveValue();
            valueArrayDouble = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFDouble " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFDouble to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFDouble newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFDOUBLE) ||
             getType().equals(field.TYPE_SFDOUBLE) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            double[] newArray = new double[valueArrayDouble.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayDouble, 0, newArray, 0, valueArrayDouble.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayDouble = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFDouble " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFFloat to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFFloat newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFFLOAT) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + 1];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            newArray[valueArrayFloat.length] = newValue.getPrimitiveValue();
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFFloat " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFFloat to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFFloat newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFFLOAT) ||
             getType().equals(field.TYPE_SFFLOAT) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFFloat " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFImage to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFImage newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFIMAGE) ||
             getType().equals(field.TYPE_SFIMAGE) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            int[] newArray = new int[valueArrayInteger.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayInteger, 0, newArray, 0, valueArrayInteger.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayInteger = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFImage " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFImage to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFImage newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFIMAGE) ||
             getType().equals(field.TYPE_SFIMAGE) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            int[] newArray = new int[valueArrayInteger.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayInteger, 0, newArray, 0, valueArrayInteger.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayInteger = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFImage " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFInt32 to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFInt32 newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFINT32) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            int[] newArray = new int[valueArrayInteger.length + 1];
            System.arraycopy(valueArrayInteger, 0, newArray, 0, valueArrayInteger.length);
            newArray[valueArrayInteger.length] = newValue.getPrimitiveValue();
            valueArrayInteger = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFInt32 " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFInt32 to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFInt32 newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFINT32) ||
             getType().equals(field.TYPE_SFINT32) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            int[] newArray = new int[valueArrayInteger.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayInteger, 0, newArray, 0, valueArrayInteger.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayInteger = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFInt32 " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFMatrix3d to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFMatrix3d newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFMATRIX3D) ||
             getType().equals(field.TYPE_SFMATRIX3D) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            double[] newArray = new double[valueArrayDouble.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayDouble, 0, newArray, 0, valueArrayDouble.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayDouble = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFMatrix3d " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFMatrix3d to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFMatrix3d newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFMATRIX3D) ||
             getType().equals(field.TYPE_SFMATRIX3D) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            double[] newArray = new double[valueArrayDouble.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayDouble, 0, newArray, 0, valueArrayDouble.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayDouble = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFMatrix3d " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFMatrix3f to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFMatrix3f newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFMATRIX3F) ||
             getType().equals(field.TYPE_SFMATRIX3F) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFMatrix3f " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFMatrix3f to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFMatrix3f newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFMATRIX3F) ||
             getType().equals(field.TYPE_SFMATRIX3F) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFMatrix3f " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFMatrix4d to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFMatrix4d newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFMATRIX4D) ||
             getType().equals(field.TYPE_SFMATRIX4D) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            double[] newArray = new double[valueArrayDouble.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayDouble, 0, newArray, 0, valueArrayDouble.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayDouble = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFMatrix4d " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFMatrix4d to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFMatrix4d newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFMATRIX4D) ||
             getType().equals(field.TYPE_SFMATRIX4D) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            double[] newArray = new double[valueArrayDouble.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayDouble, 0, newArray, 0, valueArrayDouble.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayDouble = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFMatrix4d " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFMatrix4f to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFMatrix4f newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFMATRIX4F) ||
             getType().equals(field.TYPE_SFMATRIX4F) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFMatrix4f " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFMatrix4f to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFMatrix4f newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFMATRIX4F) ||
             getType().equals(field.TYPE_SFMATRIX4F) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFMatrix4f " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFRotation to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFRotation newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFROTATION) ||
             getType().equals(field.TYPE_SFROTATION) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFRotation " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFRotation to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFRotation newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFROTATION) ||
             getType().equals(field.TYPE_SFROTATION) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFRotation " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFString to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFString newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFSTRING) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            if (newValue != null)
                   value += newValue.getPrimitiveValue();
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFString " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFString to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFString newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFSTRING) || getType().equals(field.TYPE_MFSTRING) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
                if (newValue != null)
                   value += newValue.getPrimitiveValue();
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFString " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFTime to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFTime newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFTIME) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            double[] newArray = new double[valueArrayDouble.length + 1];
            System.arraycopy(valueArrayDouble, 0, newArray, 0, valueArrayDouble.length);
            newArray[valueArrayDouble.length] = newValue.getPrimitiveValue();
            valueArrayDouble = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFTime " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFTime to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFTime newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFTIME) ||
             getType().equals(field.TYPE_SFTIME) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            double[] newArray = new double[valueArrayDouble.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayDouble, 0, newArray, 0, valueArrayDouble.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayDouble = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFTime " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFVec2d to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFVec2d newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFVEC2D) ||
             getType().equals(field.TYPE_SFVEC2D) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            double[] newArray = new double[valueArrayDouble.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayDouble, 0, newArray, 0, valueArrayDouble.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayDouble = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFVec2d " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFVec2d to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFVec2d newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFVEC2D) ||
             getType().equals(field.TYPE_SFVEC2D) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            double[] newArray = new double[valueArrayDouble.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayDouble, 0, newArray, 0, valueArrayDouble.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayDouble = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFVec2d " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFVec2f to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFVec2f newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFVEC2F) ||
             getType().equals(field.TYPE_SFVEC2F) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFVec2f " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFVec2f to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFVec2f newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFVEC2F) ||
             getType().equals(field.TYPE_SFVEC2F) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFVec2f " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFVec3d to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFVec3d newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFVEC3D) ||
             getType().equals(field.TYPE_SFVEC3D) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            double[] newArray = new double[valueArrayDouble.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayDouble, 0, newArray, 0, valueArrayDouble.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayDouble = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFVec3d " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFVec3d to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFVec3d newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFVEC3D) ||
             getType().equals(field.TYPE_SFVEC3D) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            double[] newArray = new double[valueArrayDouble.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayDouble, 0, newArray, 0, valueArrayDouble.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayDouble = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFVec3d " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFVec3f to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFVec3f newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFVEC3F) ||
             getType().equals(field.TYPE_SFVEC3F) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFVec3f " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFVec3f to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFVec3f newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFVEC3F) ||
             getType().equals(field.TYPE_SFVEC3F) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFVec3f " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFVec4d to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFVec4d newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFVEC4D) ||
             getType().equals(field.TYPE_SFVEC4D) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            double[] newArray = new double[valueArrayDouble.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayDouble, 0, newArray, 0, valueArrayDouble.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayDouble = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFVec4d " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFVec4d to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFVec4d newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFVEC4D) ||
             getType().equals(field.TYPE_SFVEC4D) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            double[] newArray = new double[valueArrayDouble.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayDouble, 0, newArray, 0, valueArrayDouble.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayDouble = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFVec4d " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append SFVec4f to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(SFVec4f newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_SFVEC4F) ||
             getType().equals(field.TYPE_SFVEC4F) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(SFVec4f " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append MFVec4f to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(MFVec4f newValue)
	{
        // set-newValue-validity-checks #TODO

        if ( getType().equals(field.TYPE_MFVEC4F) ||
             getType().equals(field.TYPE_SFVEC4F) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            // https://stackoverflow.com/questions/2843366/how-to-add-new-elements-to-an-array
            // arrays cannot be dynamically resized, must create new array of proper length
            float[] newArray = new float[valueArrayFloat.length + newValue.getPrimitiveValue().length];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            System.arraycopy(newValue.getPrimitiveValue(), 0, newArray, 0, newValue.getPrimitiveValue().length);
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(MFVec4f " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }

	/**
	 * Append boolean to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(boolean newValue)
	{
            if ( getType().equals(field.TYPE_MFBOOL) ||
                (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
            {
                boolean[] newArray = new boolean[valueArrayBoolean.length + 1];
                System.arraycopy(valueArrayBoolean, 0, newArray, 0, valueArrayBoolean.length);
                newArray[valueArrayBoolean.length] = newValue;
                valueArrayBoolean = newArray;
            }
            else
            {
                String errorNotice = "*** Illegal type: cannot appendValue(boolean " + newValue + ") when fieldValue type='" + getType() + "'";
                validationResult.append(errorNotice).append("\n");
                throw new InvalidFieldValueException(errorNotice); // report error
            }
            return this;
        }
	/**
	 * Append integer to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(int newValue)
	{
        if ( getType().equals(field.TYPE_MFINT32) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            int[] newArray = new int[valueArrayInteger.length + 1];
            System.arraycopy(valueArrayInteger, 0, newArray, 0, valueArrayInteger.length);
            newArray[valueArrayInteger.length] = newValue;
            valueArrayInteger = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(int " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }
	/**
	 * Append float to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(float newValue)
	{
        if ( getType().equals(field.TYPE_MFFLOAT) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            float[] newArray = new float[valueArrayFloat.length + 1];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            newArray[valueArrayFloat.length] = newValue;
            valueArrayFloat = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(float " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }
	/**
	 * Append double to fieldValue for <i>value</i> attribute.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(double newValue)
	{
        if ( getType().equals(field.TYPE_MFDOUBLE) || getType().equals(field.TYPE_MFTIME) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            double[] newArray = new double[valueArrayDouble.length + 1];
            System.arraycopy(valueArrayDouble, 0, newArray, 0, valueArrayDouble.length);
            newArray[valueArrayDouble.length] = newValue;
            valueArrayDouble = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(double " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }
	/**
	 * Append String (converted if necessary) to fieldValue for <i>value</i> attribute.
	 * For type SFString, the single string gets longer.
	 * For type MFString, another string is added to the list.
	 * For other MF types, another value converted from the string is added to the list.
	 * <br><br>
	 * <i>Tooltip:</i> fieldValue contains initial value for this field, which overrides default initialization value defined in original ProtoDeclare field.
	 * @param newValue is new value to append to the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue appendValue(String newValue)
	{
        // set-newValue-validity-checks #TODO
        if ( getType().equals(field.TYPE_SFSTRING) ||
            (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
        {
            value += newValue; // append string
        }
        else if (getType().equals(field.TYPE_MFSTRING))
        {
            if (!newValue.startsWith(" "))
                value += " ";
            if (!newValue.trim().startsWith("\""))
                value += "\""; // wrap quote
            value += newValue; // append string
            if (!newValue.trim().endsWith("\""))
                value += "\""; // wrap quote
        }
        else if (getType().equals(field.TYPE_MFBOOL))
        {
            boolean[] newArray = new boolean[valueArrayBoolean.length + 1];
            System.arraycopy(valueArrayBoolean, 0, newArray, 0, valueArrayBoolean.length);
            newArray[valueArrayBoolean.length] = new SFBool().setValueByString(newValue).getValue(); // type conversion
            valueArrayBoolean = newArray;
        }
        else if (getType().equals(field.TYPE_MFINT32))
        {
            int[] newArray = new int[valueArrayInteger.length + 1];
            System.arraycopy(valueArrayInteger, 0, newArray, 0, valueArrayInteger.length);
            newArray[valueArrayInteger.length] = new SFInt32().setValueByString(newValue).getValue(); // type conversion
            valueArrayInteger = newArray;
        }
        else if (getType().equals(field.TYPE_MFFLOAT))
        {
            float[] newArray = new float[valueArrayFloat.length + 1];
            System.arraycopy(valueArrayFloat, 0, newArray, 0, valueArrayFloat.length);
            newArray[valueArrayFloat.length] = new SFFloat().setValueByString(newValue).getValue(); // type conversion
            valueArrayFloat = newArray;
        }
        else if (getType().equals(field.TYPE_MFDOUBLE) || getType().equals(field.TYPE_MFTIME))
        {
            double[] newArray = new double[valueArrayDouble.length + 1];
            System.arraycopy(valueArrayDouble, 0, newArray, 0, valueArrayDouble.length);
            newArray[valueArrayDouble.length] = new SFDouble().setValueByString(newValue).getValue(); // type conversion
            valueArrayDouble = newArray;
        }
        else
        {
            String errorNotice = "*** Illegal type: cannot appendValue(String " + newValue + ") when fieldValue type='" + getType() + "'";
            validationResult.append(errorNotice).append("\n");
            throw new InvalidFieldValueException(errorNotice); // report error
        }
        return this;
    }
	/**
	 * Assign typed object MFString value to fieldValue, similar to {@link #setValue(String)}.
	 * @param newValue is new value for the value field.
	 * @return {@link field} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue setValue(MFString newValue)
	{
            if ( getType().equals(field.TYPE_MFSTRING) ||
                (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
            {
                if (newValue != null)
                    value += newValue.getPrimitiveValue();
                return this;
            }
            else
            {
                String errorNotice = "*** Illegal type: cannot appendValue(MFString " + newValue + ") when fieldValue type='" + getType() + "'";
                validationResult.append(errorNotice).append("\n");
                throw new InvalidFieldValueException(errorNotice); // report error
            }
	}
	/**
	 * Assign typed object String[] array value to fieldValue, similar to {@link #setValue(String)}.
	 * @param newValue is new value for the value field.
	 * @return {@link field} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue setValue(String[] newValue)
	{
            if ( getType().equals(field.TYPE_MFSTRING) ||
                (getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
            {
                if (newValue != null)
                {
                    for (int i = 0; i < newValue.length; i++)
                    {
                        String currentString = newValue[i];
                        if (!currentString.startsWith("\""))
                        {
                             value += "\"" + newValue[i];
                        }
                        else
                        {
                            value += newValue[i];
                        }
                        if (!currentString.endsWith("\""))
                        {
                            value += "\"";
                        }
                        if (i < newValue.length - 1)
                                value += " ";
                    }
                    return this;
                }
                else
                {
                    value = null; // empty array vice singleton empty string
                    return this;
                }
            }
            else
            {
                String errorNotice = "*** Illegal type: cannot appendValue(MFString " + newValue + ") when fieldValue type='" + getType() + "'";
                validationResult.append(errorNotice).append("\n");
                throw new InvalidFieldValueException(errorNotice); // report error
            }
	}
	/**
	 * Assign typed object SFString value field, similar to {@link #setValue(String)}.
	 * @param newValue is new value for the value field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue setValue(SFString newValue)
	{
			// set-newValue-validity-checks #1 gets handled by set-primitive method
			setValue(newValue.getPrimitiveValue());
			return this;
	}

	// Additional utility methods for this class ==============================

	/**
	 * Utility constructor that also assigns fieldValueName
	 * @param fieldValueName unique DEF name for this X3D node
	 */
	public fieldValue (String fieldValueName)
	{
		initialize();
		setName(fieldValueName);
	}
	/**
	 * Utility constructor that also assigns fieldValueName, defaultValue
	 * @param fieldValueName unique name for this field
	 * @param defaultValue string version of defaultValue for this field
	 */
	public fieldValue (String fieldValueName, String defaultValue)
	{
		initialize();
		setName (fieldValueName);
	    setValue(defaultValue);
	}
        // special utilities for fieldValue IS

        private IS IS; // SFNode acceptable node types: IS

	/**
	 * Provide IS instance (using a properly typed node) from inputOutput SFNode field <i>IS</i>.
	 * @return value of IS field
	 */
	public IS getIS()
	{
		return IS;
	}

	/**
	 * Accessor method to assign IS instance (using a properly typed node) to inputOutput SFNode field <i>IS</i>.
	 * @param newValue is new value for the IS field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public fieldValue setIS(IS newValue)
	{
            // set-newValue-validity-checks #0.d
            IS = newValue;
            if (newValue != null)
            {
                    ((X3DConcreteElement) IS).setParent(this); // parentTest15.2
            }
            return this;
	}

	/**
	 * Utility method to clear SFNode value of IS field.
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
setAttribute method invocations).	 */
	public fieldValue clearIS()
	{
		((X3DConcreteElement) IS).clearParent(); // remove references to facilitate Java memory management
		IS = null; // reset SFNode field
		return this;
	}

	/**
	 * Indicate whether an object is available for inputOutput SFNode field <i>IS</i>.
	 * @return whether an IS statement is available.
	 * @see #getIS()
	 */
	public boolean hasIS()
	{
		return (IS != null);
	}

	/** Utility method to indicate accessType corresponding to given fieldValue name.
	 * @see ConfigurationProperties#ERROR_UNKNOWN_FIELD_ACCESSTYPE
	 * @return X3D accessType (inputOnly etc.), otherwise ConfigurationProperties.ERROR_UNKNOWN_FIELD_ACCESSTYPE if not recognized
	 */
	/* @Override */
	public String getAccessType()
	{
            return getAccessType(getName());
	}

	/**
	 * Indicate whether a node or statement is found in inputOutput MFNode field <i>children</i>.
	 * @return whether a node or statement is found; ignores CommentsBlock.
	 * @see #getChildren()
	 */
	public boolean hasChildrenElements()
	{
		if (children.isEmpty())
			return false; // nothing found
		for (org.web3d.x3d.sai.Core.X3DNode node : children)
		{
			if (!(node instanceof CommentsBlock))
				return true;
		}
		return false; // nothing but CommentsBlock found
	}

	/**
	 * Utility method to adjust field synonyms
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#fieldNameChanges">X3D Scene Authoring Hints: X3D4 Field name changes for improved consistency</a>
     */
    private void handleFieldSynonyms()
    {
            // no synonyms to handle
	}

	/**
	 * Recursive method to provide X3D string serialization of this model subgraph, utilizing XML encoding and conforming to X3D Canonical Form.
	 * @param indentLevel number of levels of indentation for this element
	 * @see X3D#FILE_EXTENSION_X3D
	 * @see X3D#FILE_EXTENSION_XML
	 * @see X3D#toStringXML()
	 * @see X3D#toFileXML(String)
	 * @see X3D#toFileX3D(String)
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-1/V3.3/Part01/X3D_XML.html">X3D XML Encoding</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-3/V3.3/Part03/concepts.html#X3DCanonicalForm" target="blank">X3D Compressed Binary Encoding: X3D Canonical Form</a>
	 * @see <a href="https://www.web3d.org/x3d/tools/canonical/doc/x3dTools.htm">X3D Canonicalization (C14N) Tool</a>
	 * @return X3D string
	 */
	@Override
	public String toStringX3D(int indentLevel)
	{
		boolean hasAttributes = true; // TODO check for non-default attribute values
		boolean      hasChild = (!children.isEmpty());
		handleFieldSynonyms(); // adjust containerField values for X3D3 differences, if any
		StringBuilder indent = new StringBuilder();
		int  indentIncrement = ConfigurationProperties.getIndentIncrement();
		char indentCharacter = ConfigurationProperties.getIndentCharacter();
		for (int i = 0; i < indentLevel; i++)
			indent.append(indentCharacter); // level of indentation for this level

		StringBuilder stringX3D = new StringBuilder();

		stringX3D.append(indent).append("<fieldValue"); // start opening tag
		if (hasAttributes)
		{
			if (!getName().equals(NAME_DEFAULT_VALUE))
			{
				stringX3D.append(" name='").append(SFString.toString(getName())).append("'");
			}
			
            // test attributes

			if ((!getCssClass().equals(CLASS_DEFAULT_VALUE) || !ConfigurationProperties.getStripDefaultAttributes()) /* isSingleValueType */)
			{
				stringX3D.append(" class='").append(new SFString(getCssClass()).toStringX3D()).append("'");
			}
			if ((!getHtmlID().equals(ID_DEFAULT_VALUE) || !ConfigurationProperties.getStripDefaultAttributes()) /* isSingleValueType */)
			{
				stringX3D.append(" id='").append(new SFString(getHtmlID()).toStringX3D()).append("'");
			}
			if ((!getCssStyle().equals(STYLE_DEFAULT_VALUE) || !ConfigurationProperties.getStripDefaultAttributes()) /* isSingleValueType */)
			{
				stringX3D.append(" style='").append(new SFString(getCssStyle()).toStringX3D()).append("'");
			}
			if ((!getValue().equals(VALUE_DEFAULT_VALUE) || !ConfigurationProperties.getStripDefaultAttributes()) /* isSingleValueType */)
			{
				stringX3D.append(" value='").append(new SFString(getValue()).toStringX3D()).append("'");
			}
		}
		if ((hasChild)) // has contained node(s), comment(s), IS/connect and/or source code
		{
			stringX3D.append(">").append("\n"); // finish opening tag

			// recursively iterate over child elements
			for (org.web3d.x3d.sai.Core.X3DNode element : children)
			{
				stringX3D.append(((X3DConcreteElement)element).toStringX3D(indentLevel + indentIncrement));
			}
			stringX3D.append(indent).append("</fieldValue>").append("\n"); // finish closing tag
		}
		else
		{
			stringX3D.append("/>").append("\n"); // otherwise finish singleton tag
		}
		return stringX3D.toString();
	}
// ($name=fieldValue, isX3dStatement=true, isClassX3dStatement=true, $hasChild=true, $hasAttributes=true
	/**
	 * Recursive method to provide ClassicVRML string serialization.
	 * @param indentLevel number of levels of indentation for this element
	 * @see X3D#FILE_EXTENSION_CLASSICVRML
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dResources.html#VRML">X3D Resources: Virtual Reality Modeling Language (VRML) 97</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-2/V3.3/Part02/X3D_ClassicVRML.html">Extensible 3D (X3D) encodings Part 2: Classic VRML encoding</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-2/V3.3/Part02/grammar.html">Extensible 3D (X3D) encodings Part 2: Classic VRML encoding, Annex A: Grammar</a>
	 * @return ClassicVRML string
	 */
	@Override
	public String toStringClassicVRML(int indentLevel)
	{
		StringBuilder stringClassicVRML = new StringBuilder();
		boolean hasAttributes = true; // TODO further refinement
		boolean      hasChild = (!children.isEmpty());
                if (!serializingVRML97output)
                    handleFieldSynonyms(); // adjust containerField values for X3D3 differences, if any
		StringBuilder  indent = new StringBuilder();
		char  indentCharacter = ConfigurationProperties.getIndentCharacter();
		int   indentIncrement = ConfigurationProperties.getIndentIncrement();
		for (int i = 0; i < indentLevel; i++)
			indent.append(indentCharacter); // level of indentation for this level

		// fieldValue type is figured out using ProtoDeclare/ExternProtoDeclare field getType()
		// fieldValue definition
		stringClassicVRML.append("\n").append(indent).append(indentCharacter).append(indentCharacter).append(name);
                String  parentName   = "fieldParentNotKnown"; // default value, just in case the follow-on determination is unsuccessful
                boolean hasISconnect = false;
                IS      parentIS     = null;

                if ((getParent() !=  null) && (getParent() instanceof X3DConcreteNode))
                {
                        parentName   = ((X3DConcreteNode)getParent()).getElementName();
                        parentIS     = ((X3DConcreteNode)getParent()).getIS();
                        hasISconnect = false;
                        if (parentIS != null)
                          for (connect element : parentIS.getConnectList())
                          {
                            if (element.getNodeField().equals(name))
                            {
                                hasISconnect = true;
                                break;
                            }
                          }
                }
                else if ((getParent() !=  null) && (getParent() instanceof X3DConcreteStatement))
                          parentName = ((X3DConcreteStatement)getParent()).getElementName();

                // first case: field IS/connect
                if (hasISconnect)
                {
                    if (parentIS != null)
                      for (connect element : parentIS.getConnectList())
                      {
                        if (element.getNodeField().equals(name))
                        {
                            stringClassicVRML.append(indentCharacter).append("IS").append(indentCharacter).append(element.getProtoField()); // found matching connect
                        }
                      }
                }
                // additional case: default value, if any is provided
		else if (((getValue().length() > 0) || getType().startsWith("MF") || getType().equals(field.TYPE_SFSTRING)) &&
                        !parentName.equals(ExternProtoDeclare.NAME) && !hasISconnect)
		{
                    // no value for EXTERNPROTODECLARE fields since default values are found in the original prototype
                    stringClassicVRML.append(" ");
                    if (getType().equals(field.TYPE_SFBOOL) &&
                             (getValue().equalsIgnoreCase("true") || getValue().equalsIgnoreCase("false")))
                    {
                        stringClassicVRML.append(getValue().toUpperCase()).append("\n"); // .append(" ### trace field initialization value #1")
                        for (int i = 0; i < (indentLevel + indentIncrement); i++)
                            stringClassicVRML.append(indentCharacter); // level of indentation for this level
                    }
                    else if (getType().equals(field.TYPE_MFBOOL))
                    {
                        String capitalizedValue = getValueMFBool().toStringClassicVRML();
                        stringClassicVRML.append("[ ").append(capitalizedValue).append(" ]").append("\n"); // .append(" ### trace field initialization value #2")
                        for (int i = 0; i < (indentLevel + indentIncrement); i++)
                            stringClassicVRML.append(indentCharacter); // level of indentation for this level
                    }
                    else if (getType().equals(field.TYPE_SFSTRING))
                    {
                         stringClassicVRML.append("\"").append(SFString.toStringClassicVRML(getValue())).append("\""); // might be empty string
                    }
//                  else if (getType().equals(field.TYPE_MFSTRING))
//                  // problematic, TODO requires parsing to separate individual strings in array and then individual escaping
//                  {
//                       stringClassicVRML.append("[ ").append(MFString.toStringClassicVRML(getValue())).append(" ]"); // might be empty array
//                  }
                    else if (getType().startsWith("MF") && !getType().equals(field.TYPE_MFNODE))
                    {
                        stringClassicVRML.append("[ ").append(getValue()).append(" ]"); // .append(" ### trace field initialization value #3")
                    }
                    else if (getValue().length() > 0)
                    {
                        stringClassicVRML.append(getValue().replaceAll("\"","\\\"")); // .append(" ### trace field initialization value #4")
                    }
		}
                else if (parentName.equals(ExternProtoDeclare.NAME))
                {
                    // no initialization value provided within ExternProtoDeclare
                }
                else if (getValue().length() > 0)
                {
                    stringClassicVRML.append(getValue()); // .append(" ### trace field initialization value #5")
                }
                if (getType().equals(field.TYPE_SFNODE) && !hasChildrenElements() &&
                    !parentName.equals(ExternProtoDeclare.NAME) && !hasISconnect &&
                    (getAccessType().equals(field.ACCESSTYPE_INITIALIZEONLY) ||
                     getAccessType().equals(field.ACCESSTYPE_INPUTOUTPUT)))
                {
                    stringClassicVRML.append(" ").append("NULL # no initialization node provided").append("\n");
                    for (int i = 0; i < (indentLevel + indentIncrement); i++)
                        stringClassicVRML.append(indentCharacter); // level of indentation for this level
                }
                // see below
//                 else if ((getType().equals(field.TYPE_SFNODE)) && hasChildrenElements())
//                 {
//                     for (int i = 0; i < getChildren().size(); i++) // expecting one element plus optional comments
//                     {
//                         stringClassicVRML.append(" ").append(((X3DConcreteElement)getChildren().get(i)).toStringClassicVRML(indentLevel));
//                     }
//                 }
                else if (getType().equals(field.TYPE_MFNODE) && !parentName.endsWith(ExternProtoDeclare.NAME) &&
                         !hasISconnect &&
                         (getAccessType().equals(field.ACCESSTYPE_INITIALIZEONLY) ||
                          getAccessType().equals(field.ACCESSTYPE_INPUTOUTPUT))) // && hasChildrenElements()
                {
                    // TODO check containerField matches?
                    // recursively iterate over child elements
                    stringClassicVRML.append(" [ ");
                    for (X3DNode element : getChildren())
                    {
                            stringClassicVRML.append(" ").append(((X3DConcreteElement)element).toStringClassicVRML(indentLevel));
                    }
                    stringClassicVRML.append(" ]").append("\n"); // .append(" ### trace field initialization value #6")
                    for (int i = 0; i < (indentLevel + indentIncrement); i++)
                        stringClassicVRML.append(indentCharacter); // level of indentation for this level
                }
                // TODO possible additional case: default value for this type, if none is provided by field in Script/ProtoInterface
                //      perhaps best to let that ambiguous case fail, so that author initialization intent is explicitly declared

		if (hasChild) // has contained node(s) (including USE), comment(s), IS/connect and/or source code
		{
                        if ((getType().equals(field.TYPE_SFNODE) && hasChildren()) ||
                            (hasChildren() && !hasChildrenElements())) // second case: comments only, no children nodes or statements
			{
                            for (org.web3d.x3d.sai.Core.X3DNode element : children) // children might include comment statements
                            {
                                stringClassicVRML.append(indentCharacter).append(((X3DConcreteElement) element).toStringClassicVRML(indentLevel + indentIncrement + indentIncrement)).append("\n");
                            }
                            stringClassicVRML.append(" ### trace field initialization value #7").append("\n"); //
//                          for (int i = 0; i < (indentLevel + indentIncrement); i++)
//                              stringClassicVRML.append(indentCharacter); // level of indentation for this level
                        }
		}
		return stringClassicVRML.toString();
	}


    private boolean serializingVRML97output = false; // whether to avoid handling field name synonyms

	/**
	 * Recursive method to provide VRML97 string serialization.
	 * @param indentLevel number of levels of indentation for this element
	 * @see X3D#FILE_EXTENSION_VRML97
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dResources.html#VRML">X3D Resources: Virtual Reality Modeling Language (VRML) 97</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/14772/V2.0/index.html">Virtual Reality Modeling Language (VRML) 97 specification</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/14772-1/V2.1/index.html">VRML 97 v2.1 Amendment</a>
	 * @return VRML97 string
	 */
	@Override
	public String toStringVRML97(int indentLevel)
	{
            serializingVRML97output = true;
            String result = toStringClassicVRML(indentLevel);
            serializingVRML97output = false;

            return result;
	}

	/**
	 * Recursive method to provide object reference to node or statement by name attribute, if found as part of this element or in a contained element.
	 * Elements with name fields include meta, Metadata* nodes, field/fieldValue, ProtoDeclare/ExternProtoDeclare/ProtoInstance, CAD and HAnim nodes.
	 * <br ><br >
	 * <i>Warning:</i> first start with findAncestorScene() to check entire scene graph, or findAncestorX3D() to check entire model document.
	 * <br ><br >
	 * <i>Warning:</i> more than one element may be found that has the same name, this method does not handle that case.
	 * @see #findNodeByDEF(String)
	 * @see X3DConcreteElement#hasAncestorScene()
	 * @see org.web3d.x3d.jsail.X3DConcreteElement#findAncestorX3D()
	 * @param nameValue is value of the name field being searched for in this element and child elements(if any)
	 * @return object reference to found element, null otherwise
	 */
	/* @Override */
	public X3DConcreteElement findElementByNameValue(String nameValue)
	{
		return findElementByNameValue(nameValue, ""); // empty string is wildcard, any element match is allowed
	}

	/**
	 * Recursive method to provide object reference to node or statement by name attribute, if found as part of this element or in a contained element.
	 * Elements with name fields include meta, Metadata* nodes, field/fieldValue, ProtoDeclare/ExternProtoDeclare/ProtoInstance, CAD and HAnim nodes.
	 * <br ><br >
	 * <i>Warning:</i> first start with findAncestorScene() to check entire scene graph, or findAncestorX3D() to check entire model document.
	 * <br ><br >
	 * <i>Warning:</i> more than one element may be found that has the same name, this method does not handle that case.
	 * @see #findNodeByDEF(String)
	 * @see X3DConcreteElement#hasAncestorScene()
	 * @see org.web3d.x3d.jsail.X3DConcreteElement#findAncestorX3D()
	 * @param nameValue is value of the name field being searched for in this element and child elements(if any)
	 * @param elementName identifies the element of interest (meta MetadataString ProtoDeclare CADassembly ProtoInstance HAnimHumanoid etc.)
	 * @return object reference to found element, null otherwise
	 */
	/* @Override */
	public X3DConcreteElement findElementByNameValue(String nameValue, String elementName)
	{
		if ((nameValue == null) || nameValue.isEmpty())
		{
			String errorNotice = "*** findElementByNameValue(\"\", " + elementName + ") cannot use empty string to find a name attribute";
			validationResult.append(errorNotice).append("\n");
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
		if (elementName == null)
			elementName = "";
		if (getName().equals(nameValue) &&
			(elementName.isEmpty() || elementName.equals(getElementName())))
			return this;
		X3DConcreteElement referenceElement;
		for ( org.web3d.x3d.sai.Core.X3DNode element : children) // MFNode
		{
			if (element instanceof X3DConcreteElement)
			{
				referenceElement = ((X3DConcreteElement) element).findElementByNameValue(nameValue, elementName);
				if (referenceElement != null)
					return referenceElement;
			}
		}
		return null; // not found
	}
	/**
	 * Recursive method to determine whether node or statement with given name attribute is found, meaning both objects are attached to same scene graph.
	 * @param nameValue is value of the name field being searched for in this element and child elements(if any)
	 * @param elementName identifies the element of interest (meta MetadataString ProtoDeclare CADassembly ProtoInstance HAnimHumanoid etc.)
	 * @see #findElementByNameValue(String, String)
	 * @return whether node is found
	 */
	public boolean hasElementByNameValue(String nameValue, String elementName)
	{
		return (findElementByNameValue(nameValue, elementName) != null);
	}

	/**
	 * Recursive method to provide object reference to node by DEF, if found as this node or in a contained node.
	 * <br ><br >
	 * <i>Warning:</i> first start with findAncestorScene() to check entire scene graph, or findAncestorX3D() to check entire model document.
	 * <br ><br >
	 * <i>Warning:</i> more than one element may be found that has the same DEF, this method does not handle that case.
	 * @see #findElementByNameValue(String)
	 * @see X3DConcreteElement#hasAncestorScene()
	 * @see org.web3d.x3d.jsail.X3DConcreteElement#findAncestorX3D()
	 * @param DEFvalue is value of the name field being searched for in this element and child elements(if any)
	 * @return object reference to found node, null otherwise
	 */
	/* @Override */
	public X3DConcreteNode findNodeByDEF(String DEFvalue)
	{
		if ((DEFvalue == null) || DEFvalue.isEmpty())
		{
			String errorNotice = "*** findNodeByDEF(\"\") cannot use empty string to find a name";
			validationResult.append(errorNotice).append("\n");
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
		X3DConcreteNode referenceNode;
		for ( org.web3d.x3d.sai.Core.X3DNode element : children) // MFNode
		{
			if (element instanceof org.web3d.x3d.jsail.X3DConcreteNode)
			{
				referenceNode = ((X3DConcreteElement) element).findNodeByDEF(DEFvalue);
				if (referenceNode != null)
					return referenceNode;
			}
		}
		return null; // not found
	}
	/**
	 * Recursive method to determine whether node with DEFvalue is found, meaning both objects are attached to same scene graph.
	 * @param DEFvalue is value of the name field being searched for in this element and child elements(if any)
	 * @see #findNodeByDEF(String)
	 * @return whether node is found
	 */
	public boolean hasNodeByDEF(String DEFvalue)
	{
		return (findNodeByDEF(DEFvalue) != null);
	}


	/**
	 * Recursive method to validate this element plus all contained nodes and statements,
	 * using both datatype-specification value checks and regular expression (regex) checking of corresponding string values.
         * TODO document specific validation methods.
	 * @return validation results (if any)
	 */
	@Override
	public String validate()
	{
		validationResult = new StringBuilder(); // prepare for updated results

		setCssClass(getCssClass()); // exercise field checks, simple types
        
		setHtmlID(getHtmlID()); // exercise field checks, simple types
        
		if (getName().isEmpty()) // required attribute
		{
			String errorNotice = ConfigurationProperties.ERROR_VALUE_NOT_FOUND +
					     ", " + NAME + " name field is required but no value found.";
			validationResult.append(errorNotice).append("\n");
			throw new InvalidFieldValueException(errorNotice); // report error
		}
		// TODO duplicated name checks in setValue() method
		setName(getName()); // exercise field checks, simple types
        
		setCssStyle(getCssStyle()); // exercise field checks, simple types
        
		if (!hasChildren()) // be careful! only check simple types if no children nodes found
			setValue(getValue()); // exercise field checks, simple types
        
		for ( org.web3d.x3d.sai.Core.X3DNode element : children) // MFNode
		{
			try {
                            ((X3DConcreteElement) element).validate(); // exercise field checks, MFNode element
                            validationResult.append(((X3DConcreteElement) element).getValidationResult());
                        }
                        catch (Exception e)
                        {
                            String exceptionMessage = "[exception] during validation: " + e.getMessage();
                            validationResult.append(exceptionMessage);
                        //  System.err.println(exceptionMessage);
                            e.printStackTrace();
                        }
		}
		setChildren(getChildren()); // also test getter/setter validation, which also includes checks for acceptableNodeTypes (if applicable)

		return validationResult.toString();
	}

	/**
	 * Discover base type of this fieldValue declaration from corresponding ProtoDeclare/ExternProtoDeclare field declaration.
	 * Note that returned value might be empty string if never set in original field object.
	 * If found, provides String value ["SFBool" | "MFBool" | "SFColor" | "MFColor" | "SFColorRGBA" | "MFColorRGBA" | "SFDouble" | "MFDouble" | "SFFloat" | "MFFloat" | "SFImage" | "MFImage" | "SFInt32" | "SFNode" | "MFNode" | "MFInt32" | "SFRotation" | "MFRotation" | "SFString" | "MFString" | "SFTime" | "MFTime" | "SFVec2d" | "MFVec2d" | "SFVec2f" | "MFVec2f" | "SFVec3d" | "MFVec3d" | "SFVec3f" | "MFVec3f" | "SFVec4d" | "MFVec4d" | "SFVec4f" | "MFVec4f" | "SFMatrix3d" | "MFMatrix3d" | "SFMatrix3f" | "MFMatrix3f" | "SFMatrix4d" | "MFMatrix4d" | "SFMatrix4f" | "MFMatrix4f"] from inputOutput SFString field named <i>type</i>.
	 * <br><br>
	 * @see org.web3d.x3d.jsail.Core.field
	 * @return type value from corresponding field declaration, otherwise "fieldValueTypeNotFound"
	 */
	public String getType()
	{
		String prototypeName;
		String fieldValueType = ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE; // default until found
		String errorNotice;

		if (getParent() == null)
		{
			errorNotice = ConfigurationProperties.ERROR_NOT_CONNECTED_TO_SCENE_GRAPH +
							": fieldValue name='" + getName() + "' is not currently connected to a ProtoInstance and thus getType() cannot be checked.";
			validationResult.append(errorNotice).append("\n");

			if (!ConfigurationProperties.isCreationConnectionValidationExceptionAllowed())
			{
				 throw new org.web3d.x3d.sai.InvalidProtoException(errorNotice); // report error
                 // ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE
			}
			else // missing type might allowably occur during object creation
			{
				return fieldValueType; // default is unknown type
            }
		}
		else prototypeName = ((ProtoInstance) getParent()).getName();

		field fieldDefinition = new field();
		Scene ancestorScene = findAncestorScene();
		if (ancestorScene == null)
		{
			errorNotice = "ProtoInstance name='" + prototypeName + "' with fieldValue  name='" + getName() + "' problem: " +
						   ConfigurationProperties.ERROR_NOT_CONNECTED_TO_SCENE_GRAPH;
			validationResult.append(errorNotice).append("\n");

			if (!ConfigurationProperties.isCreationConnectionValidationExceptionAllowed())
			{
				 throw new org.web3d.x3d.sai.InvalidProtoException(errorNotice); // report error
			}
			else // missing type might allowably occur during object creation
			{
				return fieldValueType; // default is unknown type
			}
		}
		X3DConcreteElement correspondingPrototype = ancestorScene.findElementByNameValue(prototypeName);
		if (correspondingPrototype == null)
		{
			errorNotice = "ProtoInstance name='" + prototypeName + "' does not have a corresponding " +
						  "ProtoDeclare or ExternProtoDeclare name='" + name + "' definition";
			validationResult.append(errorNotice).append("\n");
			throw new org.web3d.x3d.sai.InvalidProtoException(errorNotice); // report error
		}
		correspondingPrototype = findAncestorScene().findElementByNameValue(prototypeName, ProtoDeclare.NAME);
		if (correspondingPrototype instanceof org.web3d.x3d.jsail.Core.ProtoDeclare)
		{
			fieldDefinition = ((ProtoDeclare) correspondingPrototype).getProtoInterface().findFieldByName(name);
			if (fieldDefinition != null)
				fieldValueType = fieldDefinition.getType(); // found in scene graph
			else
			{
				errorNotice = "fieldValue name='" + name + "' does not have a corresponding ProtoDeclare field name='" +
							   name + "' definition";
				validationResult.append(errorNotice).append("\n");
				throw new org.web3d.x3d.sai.InvalidProtoException(errorNotice); // report error
			}
			return fieldValueType; // might be empty string if never set in original field
		}
		// else
		correspondingPrototype = findAncestorScene().findElementByNameValue(prototypeName, ExternProtoDeclare.NAME);
		if (correspondingPrototype instanceof org.web3d.x3d.jsail.Core.ExternProtoDeclare)
		{
			fieldDefinition = ((ExternProtoDeclare) correspondingPrototype).findFieldByName(name);
			if (fieldDefinition != null)
				fieldValueType = fieldDefinition.getType(); // found in scene graph
			else
			{
				errorNotice = "fieldValue name='" + name + "' does not have a corresponding ProtoDeclare field name='" +
							   name + "' definition";
				validationResult.append(errorNotice).append("\n");
				throw new org.web3d.x3d.sai.InvalidProtoException(errorNotice); // report error
			}
		}
		return fieldValueType; // might be empty string if never set in original field
	}

	/**
	 * If type matches, then get typed value of a SFBool fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFBool getValueSFBool()
	{
		if (getType().equals(field.TYPE_SFBOOL))
		{
			if (valueArrayBoolean.length == 0)
			{
				return new SFBool();
			}
			else if (valueArrayBoolean.length == 1)
			{
				return new SFBool(valueArrayBoolean[0]);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFBool() singleton method when field value='" +
									 valueArrayBoolean + "' has length=" + valueArrayBoolean.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFBool() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFBool fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFBool newValue)
	{
		clearValues();
		valueArrayBoolean = new boolean[1]; // fix array size
		valueArrayBoolean[0] = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFBool fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFBool getValueMFBool()
	{
		if (getType().equals(field.TYPE_MFBOOL))
		{
			return new MFBool(valueArrayBoolean);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFBool() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFBool fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFBool newValue)
	{
		clearValues();
		valueArrayBoolean = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFColor fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFColor getValueSFColor()
	{
		if (getType().equals(field.TYPE_SFCOLOR))
		{
			if (valueArrayFloat.length == 0)
			{
				return new SFColor();
			}
			else if (valueArrayFloat.length == 3)
			{
				return new SFColor(valueArrayFloat);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFColor() singleton method when field value='" +
									 valueArrayFloat + "' has length=" + valueArrayFloat.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFColor() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFColor fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFColor newValue)
	{
		clearValues();
		valueArrayFloat = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFColor fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFColor getValueMFColor()
	{
		if (getType().equals(field.TYPE_MFCOLOR))
		{
			return new MFColor(valueArrayFloat);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFColor() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFColor fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFColor newValue)
	{
		clearValues();
		valueArrayFloat = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFColorRGBA fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFColorRGBA getValueSFColorRGBA()
	{
		if (getType().equals(field.TYPE_SFCOLORRGBA))
		{
			if (valueArrayFloat.length == 0)
			{
				return new SFColorRGBA();
			}
			else if (valueArrayFloat.length == 4)
			{
				return new SFColorRGBA(valueArrayFloat);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFColorRGBA() singleton method when field value='" +
									 valueArrayFloat + "' has length=" + valueArrayFloat.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFColorRGBA() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFColorRGBA fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFColorRGBA newValue)
	{
		clearValues();
		valueArrayFloat = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFColorRGBA fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFColorRGBA getValueMFColorRGBA()
	{
		if (getType().equals(field.TYPE_MFCOLORRGBA))
		{
			return new MFColorRGBA(valueArrayFloat);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFColorRGBA() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFColorRGBA fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFColorRGBA newValue)
	{
		clearValues();
		valueArrayFloat = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFDouble fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFDouble getValueSFDouble()
	{
		if (getType().equals(field.TYPE_SFDOUBLE))
		{
			if (valueArrayDouble.length == 0)
			{
				return new SFDouble();
			}
			else if (valueArrayDouble.length == 1)
			{
				return new SFDouble(valueArrayDouble[0]);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFDouble() singleton method when field value='" +
									 valueArrayDouble + "' has length=" + valueArrayDouble.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFDouble() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFDouble fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFDouble newValue)
	{
		clearValues();
		valueArrayDouble = new double[1]; // fix array size
		valueArrayDouble[0] = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFDouble fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFDouble getValueMFDouble()
	{
		if (getType().equals(field.TYPE_MFDOUBLE))
		{
			return new MFDouble(valueArrayDouble);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFDouble() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFDouble fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFDouble newValue)
	{
		clearValues();
		valueArrayDouble = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFFloat fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFFloat getValueSFFloat()
	{
		if (getType().equals(field.TYPE_SFFLOAT))
		{
			if (valueArrayFloat.length == 0)
			{
				return new SFFloat();
			}
			else if (valueArrayFloat.length == 1)
			{
				return new SFFloat(valueArrayFloat[0]);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFFloat() singleton method when field value='" +
									 valueArrayFloat + "' has length=" + valueArrayFloat.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFFloat() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFFloat fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFFloat newValue)
	{
		clearValues();
		valueArrayFloat = new float[1]; // fix array size
		valueArrayFloat[0] = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFFloat fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFFloat getValueMFFloat()
	{
		if (getType().equals(field.TYPE_MFFLOAT))
		{
			return new MFFloat(valueArrayFloat);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFFloat() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFFloat fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFFloat newValue)
	{
		clearValues();
		valueArrayFloat = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFImage fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFImage getValueSFImage()
	{
		if (getType().equals(field.TYPE_SFIMAGE))
		{
			if (valueArrayInteger.length == 0)
			{
				return new SFImage();
			}
			else if (valueArrayInteger.length == 1)
			{
				return new SFImage(valueArrayInteger);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFImage() singleton method when field value='" +
									 valueArrayInteger + "' has length=" + valueArrayInteger.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFImage() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFImage fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFImage newValue)
	{
		clearValues();
		valueArrayInteger = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFImage fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFImage getValueMFImage()
	{
		if (getType().equals(field.TYPE_MFIMAGE))
		{
			return new MFImage(valueArrayInteger);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFImage() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFImage fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFImage newValue)
	{
		clearValues();
		valueArrayInteger = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFInt32 fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFInt32 getValueSFInt32()
	{
		if (getType().equals(field.TYPE_SFINT32))
		{
			if (valueArrayInteger.length == 0)
			{
				return new SFInt32();
			}
			else if (valueArrayInteger.length == 1)
			{
				return new SFInt32(valueArrayInteger[0]);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFInt32() singleton method when field value='" +
									 valueArrayInteger + "' has length=" + valueArrayInteger.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFInt32() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFInt32 fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFInt32 newValue)
	{
		clearValues();
		valueArrayInteger = new int[1]; // fix array size
		valueArrayInteger[0] = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFInt32 fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFInt32 getValueMFInt32()
	{
		if (getType().equals(field.TYPE_MFINT32))
		{
			return new MFInt32(valueArrayInteger);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFInt32() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFInt32 fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFInt32 newValue)
	{
		clearValues();
		valueArrayInteger = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFMatrix3d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFMatrix3d getValueSFMatrix3d()
	{
		if (getType().equals(field.TYPE_SFMATRIX3D))
		{
			if (valueArrayDouble.length == 0)
			{
				return new SFMatrix3d();
			}
			else if (valueArrayDouble.length == 9)
			{
				return new SFMatrix3d(valueArrayDouble);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFMatrix3d() singleton method when field value='" +
									 valueArrayDouble + "' has length=" + valueArrayDouble.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFMatrix3d() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFMatrix3d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFMatrix3d newValue)
	{
		clearValues();
		valueArrayDouble = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFMatrix3d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFMatrix3d getValueMFMatrix3d()
	{
		if (getType().equals(field.TYPE_MFMATRIX3D))
		{
			return new MFMatrix3d(valueArrayDouble);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFMatrix3d() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFMatrix3d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFMatrix3d newValue)
	{
		clearValues();
		valueArrayDouble = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFMatrix3f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFMatrix3f getValueSFMatrix3f()
	{
		if (getType().equals(field.TYPE_SFMATRIX3F))
		{
			if (valueArrayFloat.length == 0)
			{
				return new SFMatrix3f();
			}
			else if (valueArrayFloat.length == 9)
			{
				return new SFMatrix3f(valueArrayFloat);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFMatrix3f() singleton method when field value='" +
									 valueArrayFloat + "' has length=" + valueArrayFloat.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFMatrix3f() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFMatrix3f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFMatrix3f newValue)
	{
		clearValues();
		valueArrayFloat = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFMatrix3f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFMatrix3f getValueMFMatrix3f()
	{
		if (getType().equals(field.TYPE_MFMATRIX3F))
		{
			return new MFMatrix3f(valueArrayFloat);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFMatrix3f() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFMatrix3f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFMatrix3f newValue)
	{
		clearValues();
		valueArrayFloat = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFMatrix4d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFMatrix4d getValueSFMatrix4d()
	{
		if (getType().equals(field.TYPE_SFMATRIX4D))
		{
			if (valueArrayDouble.length == 0)
			{
				return new SFMatrix4d();
			}
			else if (valueArrayDouble.length == 16)
			{
				return new SFMatrix4d(valueArrayDouble);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFMatrix4d() singleton method when field value='" +
									 valueArrayDouble + "' has length=" + valueArrayDouble.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFMatrix4d() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFMatrix4d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFMatrix4d newValue)
	{
		clearValues();
		valueArrayDouble = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFMatrix4d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFMatrix4d getValueMFMatrix4d()
	{
		if (getType().equals(field.TYPE_MFMATRIX4D))
		{
			return new MFMatrix4d(valueArrayDouble);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFMatrix4d() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFMatrix4d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFMatrix4d newValue)
	{
		clearValues();
		valueArrayDouble = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFMatrix4f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFMatrix4f getValueSFMatrix4f()
	{
		if (getType().equals(field.TYPE_SFMATRIX4F))
		{
			if (valueArrayFloat.length == 0)
			{
				return new SFMatrix4f();
			}
			else if (valueArrayFloat.length == 16)
			{
				return new SFMatrix4f(valueArrayFloat);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFMatrix4f() singleton method when field value='" +
									 valueArrayFloat + "' has length=" + valueArrayFloat.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFMatrix4f() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFMatrix4f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFMatrix4f newValue)
	{
		clearValues();
		valueArrayFloat = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFMatrix4f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFMatrix4f getValueMFMatrix4f()
	{
		if (getType().equals(field.TYPE_MFMATRIX4F))
		{
			return new MFMatrix4f(valueArrayFloat);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFMatrix4f() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFMatrix4f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFMatrix4f newValue)
	{
		clearValues();
		valueArrayFloat = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFRotation fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFRotation getValueSFRotation()
	{
		if (getType().equals(field.TYPE_SFROTATION))
		{
			if (valueArrayFloat.length == 0)
			{
				return new SFRotation();
			}
			else if (valueArrayFloat.length == 4)
			{
				return new SFRotation(valueArrayFloat);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFRotation() singleton method when field value='" +
									 valueArrayFloat + "' has length=" + valueArrayFloat.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFRotation() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFRotation fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFRotation newValue)
	{
		clearValues();
		valueArrayFloat = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFRotation fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFRotation getValueMFRotation()
	{
		if (getType().equals(field.TYPE_MFROTATION))
		{
			return new MFRotation(valueArrayFloat);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFRotation() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFRotation fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFRotation newValue)
	{
		clearValues();
		valueArrayFloat = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFString fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFString getValueSFString()
	{
		if (getType().equals(field.TYPE_SFSTRING))
		{
			if (value.isEmpty())
			{
				return new SFString();
			}
			else
			{
				return new SFString(value);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValue() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * If type matches, then get typed value of a MFString fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFString getValueMFString()
	{
		if (getType().equals(field.TYPE_MFSTRING))
		{
			return new MFString(value);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValue() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * If type matches, then get typed value of a SFTime fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFTime getValueSFTime()
	{
		if (getType().equals(field.TYPE_SFTIME))
		{
			if (valueArrayDouble.length == 0)
			{
				return new SFTime();
			}
			else if (valueArrayDouble.length == 1)
			{
				return new SFTime(valueArrayDouble[0]);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFTime() singleton method when field value='" +
									 valueArrayDouble + "' has length=" + valueArrayDouble.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFTime() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFTime fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFTime newValue)
	{
		clearValues();
		valueArrayDouble = new double[1]; // fix array size
		valueArrayDouble[0] = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFTime fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFTime getValueMFTime()
	{
		if (getType().equals(field.TYPE_MFTIME))
		{
			return new MFTime(valueArrayDouble);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFTime() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFTime fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFTime newValue)
	{
		clearValues();
		valueArrayDouble = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFVec2d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFVec2d getValueSFVec2d()
	{
		if (getType().equals(field.TYPE_SFVEC2D))
		{
			if (valueArrayDouble.length == 0)
			{
				return new SFVec2d();
			}
			else if (valueArrayDouble.length == 2)
			{
				return new SFVec2d(valueArrayDouble);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFVec2d() singleton method when field value='" +
									 valueArrayDouble + "' has length=" + valueArrayDouble.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFVec2d() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFVec2d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFVec2d newValue)
	{
		clearValues();
		valueArrayDouble = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFVec2d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFVec2d getValueMFVec2d()
	{
		if (getType().equals(field.TYPE_MFVEC2D))
		{
			return new MFVec2d(valueArrayDouble);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFVec2d() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFVec2d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFVec2d newValue)
	{
		clearValues();
		valueArrayDouble = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFVec2f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFVec2f getValueSFVec2f()
	{
		if (getType().equals(field.TYPE_SFVEC2F))
		{
			if (valueArrayFloat.length == 0)
			{
				return new SFVec2f();
			}
			else if (valueArrayFloat.length == 2)
			{
				return new SFVec2f(valueArrayFloat);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFVec2f() singleton method when field value='" +
									 valueArrayFloat + "' has length=" + valueArrayFloat.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFVec2f() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFVec2f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFVec2f newValue)
	{
		clearValues();
		valueArrayFloat = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFVec2f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFVec2f getValueMFVec2f()
	{
		if (getType().equals(field.TYPE_MFVEC2F))
		{
			return new MFVec2f(valueArrayFloat);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFVec2f() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFVec2f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFVec2f newValue)
	{
		clearValues();
		valueArrayFloat = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFVec3d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFVec3d getValueSFVec3d()
	{
		if (getType().equals(field.TYPE_SFVEC3D))
		{
			if (valueArrayDouble.length == 0)
			{
				return new SFVec3d();
			}
			else if (valueArrayDouble.length == 3)
			{
				return new SFVec3d(valueArrayDouble);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFVec3d() singleton method when field value='" +
									 valueArrayDouble + "' has length=" + valueArrayDouble.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFVec3d() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFVec3d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFVec3d newValue)
	{
		clearValues();
		valueArrayDouble = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFVec3d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFVec3d getValueMFVec3d()
	{
		if (getType().equals(field.TYPE_MFVEC3D))
		{
			return new MFVec3d(valueArrayDouble);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFVec3d() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFVec3d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFVec3d newValue)
	{
		clearValues();
		valueArrayDouble = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFVec3f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFVec3f getValueSFVec3f()
	{
		if (getType().equals(field.TYPE_SFVEC3F))
		{
			if (valueArrayFloat.length == 0)
			{
				return new SFVec3f();
			}
			else if (valueArrayFloat.length == 3)
			{
				return new SFVec3f(valueArrayFloat);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFVec3f() singleton method when field value='" +
									 valueArrayFloat + "' has length=" + valueArrayFloat.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFVec3f() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFVec3f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFVec3f newValue)
	{
		clearValues();
		valueArrayFloat = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFVec3f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFVec3f getValueMFVec3f()
	{
		if (getType().equals(field.TYPE_MFVEC3F))
		{
			return new MFVec3f(valueArrayFloat);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFVec3f() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFVec3f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFVec3f newValue)
	{
		clearValues();
		valueArrayFloat = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFVec4d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFVec4d getValueSFVec4d()
	{
		if (getType().equals(field.TYPE_SFVEC4D))
		{
			if (valueArrayDouble.length == 0)
			{
				return new SFVec4d();
			}
			else if (valueArrayDouble.length == 4)
			{
				return new SFVec4d(valueArrayDouble);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFVec4d() singleton method when field value='" +
									 valueArrayDouble + "' has length=" + valueArrayDouble.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFVec4d() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFVec4d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFVec4d newValue)
	{
		clearValues();
		valueArrayDouble = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFVec4d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFVec4d getValueMFVec4d()
	{
		if (getType().equals(field.TYPE_MFVEC4D))
		{
			return new MFVec4d(valueArrayDouble);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFVec4d() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFVec4d fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFVec4d newValue)
	{
		clearValues();
		valueArrayDouble = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a SFVec4f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object (if values are available), otherwise returns default object
	 */
	public SFVec4f getValueSFVec4f()
	{
		if (getType().equals(field.TYPE_SFVEC4F))
		{
			if (valueArrayFloat.length == 0)
			{
				return new SFVec4f();
			}
			else if (valueArrayFloat.length == 4)
			{
				return new SFVec4f(valueArrayFloat);
			}
			else // hopefully this case cannot occur, possible internal library error
			{
				String errorNotice = "*** Invalid invocation of field.getValueSFVec4f() singleton method when field value='" +
									 valueArrayFloat + "' has length=" + valueArrayFloat.length;
				throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
			}
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueSFVec4f() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a SFVec4f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(SFVec4f newValue)
	{
		clearValues();
		valueArrayFloat = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * If type matches, then get typed value of a MFVec4f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return typed value object, if values are available
	 */
	public MFVec4f getValueMFVec4f()
	{
		if (getType().equals(field.TYPE_MFVEC4F))
		{
			return new MFVec4f(valueArrayFloat);
		}
		else
		{
			String errorNotice = "*** Invalid invocation of field.getValueMFVec4f() when declared field type='" + getType() + "'";
			throw new org.web3d.x3d.sai.InvalidFieldValueException(errorNotice);
		}
	}

	/**
	 * Set typed value of a MFVec4f fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(MFVec4f newValue)
	{
		clearValues();
		valueArrayFloat = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * Clear all values in a fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue clearValues()
	{
		value = VALUE_DEFAULT_VALUE;
		valueArrayBoolean = java.util.Arrays.copyOf(DEFAULT_VALUE_BOOLEAN, DEFAULT_VALUE_BOOLEAN.length); // must be separate copy
		valueArrayInteger = java.util.Arrays.copyOf(DEFAULT_VALUE_INTEGER, DEFAULT_VALUE_INTEGER.length); // must be separate copy
		valueArrayFloat   = java.util.Arrays.copyOf(DEFAULT_VALUE_FLOAT,   DEFAULT_VALUE_FLOAT.length  ); // must be separate copy
		valueArrayDouble  = java.util.Arrays.copyOf(DEFAULT_VALUE_DOUBLE,  DEFAULT_VALUE_DOUBLE.length ); // must be separate copy
		clearChildren(); // clear all contained children including nodes, statements and comments (if any)
		return this;
	}
	/**
	 * Set typed value of a boolean fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(boolean newValue)
	{
            if (!getType().isEmpty() &&
                !getType().equals(field.TYPE_SFBOOL) && !getType().equals(field.TYPE_MFBOOL) &&
                !(getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
            {
                String errorNotice = "*** Illegal type: cannot setValue(boolean " + newValue + ") when fieldValue type='" + getType() + "'";
                validationResult.append(errorNotice).append("\n");
                throw new InvalidFieldValueException(errorNotice); // report error
            }
            clearValues();
            valueArrayBoolean = new boolean[1]; // fix array size
            valueArrayBoolean[0] = newValue;
            return this;
	}
	/**
	 * Set typed value of a boolean[] fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(boolean[] newValue)
	{
            if (!getType().isEmpty() &&
                !getType().equals(field.TYPE_SFBOOL) && !getType().equals(field.TYPE_MFBOOL) &&
                !(getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
            {
                String errorNotice = "*** Illegal type: cannot setValue(boolean[] " + newValue + ") array when fieldValue type='" + getType() + "'";
                validationResult.append(errorNotice).append("\n");
                throw new InvalidFieldValueException(errorNotice); // report error
            }
            clearValues();
            if (newValue == null)
            {
                return this; // nothing else to do
            }
            valueArrayBoolean = newValue;
            return this;
	}
	/**
	 * Set typed value of a integer fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(int newValue)
	{
            // type promotions first
            if (getType().equals(field.TYPE_SFDOUBLE)  || getType().equals(field.TYPE_MFDOUBLE) ||
                getType().equals(field.TYPE_SFTIME)    || getType().equals(field.TYPE_MFTIME))
            {
                clearValues();
                valueArrayDouble = new double[1]; // fix array size
                valueArrayDouble[0] = (double) newValue;
                return this;
            }
            else if (!getType().isEmpty() &&
                      getType().equals(field.TYPE_SFFLOAT)   || getType().equals(field.TYPE_MFFLOAT))
            {
                clearValues();
                valueArrayFloat = new float[1]; // fix array size
                valueArrayFloat[0] = (float) newValue;
                return this;
            }
            else if (!getType().equals(field.TYPE_SFINT32) && !getType().equals(field.TYPE_MFINT32) &&
                     !(getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
            {
                String errorNotice = "*** Illegal type: cannot setValue(int " + newValue + ") when fieldValue type='" + getType() + "'";
                validationResult.append(errorNotice).append("\n");
                throw new InvalidFieldValueException(errorNotice); // report error
            }
            else // integer type
            {
		clearValues();
		valueArrayInteger = new int[1]; // fix array size
		valueArrayInteger[0] = newValue;
		return this;
            }
	}
	/**
	 * Set typed value of a integer array fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(int[] newValue)
	{
            if (!getType().isEmpty() &&
                !getType().equals(field.TYPE_SFINT32) && !getType().equals(field.TYPE_MFINT32) &&
                !getType().equals(field.TYPE_SFIMAGE) && !getType().equals(field.TYPE_MFIMAGE) &&
                !(getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
            {
                String errorNotice = "*** Illegal type: cannot setValue(int[] " + newValue + ") array when fieldValue type='" + getType() + "'";
                validationResult.append(errorNotice).append("\n");
                throw new InvalidFieldValueException(errorNotice); // report error
            }
            clearValues();
            if (newValue == null)
            {
                return this; // nothing else to do
            }
            valueArrayInteger = newValue;
            return this;
	}
	/**
	 * Set typed value of a float fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(float newValue)
	{
            // type promotions first
            if (getType().equals(field.TYPE_SFDOUBLE)  || getType().equals(field.TYPE_MFDOUBLE) ||
                getType().equals(field.TYPE_SFTIME)    || getType().equals(field.TYPE_MFTIME))
            {
                clearValues();
                valueArrayDouble = new double[1]; // fix array size
                valueArrayDouble[0] = (double) newValue;
                return this;
            }
            if (!getType().isEmpty() &&
                 getType().equals(field.TYPE_SFFLOAT)    && !getType().equals(field.TYPE_MFFLOAT) &&
                 !(getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
            {
                String errorNotice = "*** Illegal type: cannot setValue(float " + newValue + ") when fieldValue type='" + getType() + "'";
                validationResult.append(errorNotice).append("\n");
                throw new InvalidFieldValueException(errorNotice); // report error
            }
            clearValues();
            valueArrayFloat = new float[1]; // fix array size
            valueArrayFloat[0] = newValue;
            return this;
	}
	/**
	 * Set typed value of a float[] array fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(float[] newValue)
	{
            // type promotions first
            if (getType().equals(field.TYPE_SFDOUBLE)  || getType().equals(field.TYPE_MFDOUBLE) ||
                getType().equals(field.TYPE_SFTIME)    || getType().equals(field.TYPE_MFTIME))
            {
                clearValues();
                valueArrayDouble = new double[newValue.length];
                for (int i = 0; i < newValue.length; i++)
                {
                    valueArrayDouble[i] = (double) newValue[i];
                }
                return this;
            }
            if (!getType().isEmpty() &&
                !getType().equals(field.TYPE_SFCOLOR)    && !getType().equals(field.TYPE_MFCOLOR)     &&
                !getType().equals(field.TYPE_SFCOLORRGBA)&& !getType().equals(field.TYPE_MFCOLORRGBA) &&
                !getType().equals(field.TYPE_SFFLOAT)    && !getType().equals(field.TYPE_MFFLOAT)     &&
                !getType().equals(field.TYPE_SFROTATION) && !getType().equals(field.TYPE_MFROTATION)  &&
                !getType().equals(field.TYPE_SFVEC2F)    && !getType().equals(field.TYPE_MFVEC2F)     &&
                !getType().equals(field.TYPE_SFVEC3F)    && !getType().equals(field.TYPE_MFVEC3F)     &&
                !getType().equals(field.TYPE_SFVEC4F)    && !getType().equals(field.TYPE_MFVEC4F)     &&
                !getType().equals(field.TYPE_SFMATRIX3F) && !getType().equals(field.TYPE_MFMATRIX3F)  &&
                !getType().equals(field.TYPE_SFMATRIX4F) && !getType().equals(field.TYPE_MFMATRIX4F)  &&
                !(getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
            {
                String errorNotice = "*** Illegal type: cannot setValue(float[] " + newValue + ") array when fieldValue type='" + getType() + "'";
                validationResult.append(errorNotice).append("\n");
                throw new InvalidFieldValueException(errorNotice); // report error
            }
            if (!getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) &&
                (newValue.length % X3DConcreteField.getTupleSize(getType())) != 0) // tupleSize modulus check
            {
                String errorNotice = "*** illegal number of values (" + newValue.length + ")" +
                                     " in initialization string, must be multiple of " + X3DConcreteField.getTupleSize(getType());
                validationResult.append(errorNotice).append("\n");
                throw new InvalidFieldValueException(errorNotice);
            }
            clearValues();
            if (newValue == null)
            {
                return this; // nothing else to do
            }
            valueArrayFloat = newValue;
            return this;
	}
	/**
	 * Set typed value of a double fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(double newValue)
	{
            // type demotions first
            if (getType().equals(field.TYPE_SFFLOAT)    || getType().equals(field.TYPE_MFFLOAT))
            {
                clearValues();
                valueArrayFloat = new float[1]; // fix array size
                valueArrayFloat[0] = (float) newValue;
                String errorNotice = "*** (Warning) Type demotion: loss of precision may occur when setValue(double " + newValue + ") applied to fieldValue type='" + getType() + "'";
                validationResult.append(errorNotice).append("\n");
                return this;
            }
            if (!getType().isEmpty() &&
                !getType().equals(field.TYPE_SFDOUBLE)   && !getType().equals(field.TYPE_MFDOUBLE) &&
                !getType().equals(field.TYPE_SFTIME)     && !getType().equals(field.TYPE_MFTIME)   &&
                !(getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
            {
                String errorNotice = "*** Illegal type: cannot setValue(double " + newValue + ") when fieldValue type='" + getType() + "'";
                validationResult.append(errorNotice).append("\n");
                throw new InvalidFieldValueException(errorNotice); // report error
            }
            clearValues();
            valueArrayDouble = new double[1]; // fix array size
            valueArrayDouble[0] = newValue;
            return this;
	}
	/**
	 * Set typed value of a double[] array fieldValue declaration.  For general accessor use:
         * @see getValue()
	 * <i>WARNING</i>: must first <i>setType()</i> in ancestor <i>field</i> declaration to ensure validity of the proper corresponding X3D type.
	 * @param newValue is new value to assign
	 * @return {@link fieldValue} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public fieldValue setValue(double[] newValue)
	{
            // type demotions first
            if (getType().equals(field.TYPE_SFFLOAT)    || getType().equals(field.TYPE_MFFLOAT))
            {
                clearValues();
                valueArrayFloat = new float[newValue.length];
                for (int i = 0; i < newValue.length; i++)
                {
                    valueArrayFloat[i] = (float) newValue[i];
                }
                String errorNotice = "*** (Warning) Type demotion: loss of precision may occur when setValue(double[] " + newValue + ") array applied to fieldValue type='" + getType() + "'";
                validationResult.append(errorNotice).append("\n");
                return this;
            }
            if (!getType().isEmpty() &&
                !getType().equals(field.TYPE_SFDOUBLE)   && !getType().equals(field.TYPE_MFDOUBLE)   &&
                !getType().equals(field.TYPE_SFTIME)     && !getType().equals(field.TYPE_MFTIME)     &&
                !getType().equals(field.TYPE_SFFLOAT)    && !getType().equals(field.TYPE_MFFLOAT)    &&
                !getType().equals(field.TYPE_SFINT32)    && !getType().equals(field.TYPE_MFINT32)    &&
                !getType().equals(field.TYPE_SFVEC2D)    && !getType().equals(field.TYPE_MFVEC2D)    &&
                !getType().equals(field.TYPE_SFVEC3D)    && !getType().equals(field.TYPE_MFVEC3D)    &&
                !getType().equals(field.TYPE_SFVEC4D)    && !getType().equals(field.TYPE_MFVEC4D)    &&
                !getType().equals(field.TYPE_SFMATRIX3D) && !getType().equals(field.TYPE_MFMATRIX3D) &&
                !getType().equals(field.TYPE_SFMATRIX4D) && !getType().equals(field.TYPE_MFMATRIX4D) &&
                !(getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) && ConfigurationProperties.isCreationConnectionValidationExceptionAllowed()))
            {
                String errorNotice = "*** Illegal type: cannot setValue(double[] " + newValue + ") array when fieldValue type='" + getType() + "'";
                validationResult.append(errorNotice).append("\n");
                throw new InvalidFieldValueException(errorNotice); // report error
            }
            if (!getType().equals(ConfigurationProperties.ERROR_UNKNOWN_FIELD_TYPE) &&
                (newValue.length % X3DConcreteField.getTupleSize(getType())) != 0) // tupleSize modulus check
            {
                String errorNotice = "*** illegal number of values (" + newValue.length + ")" +
                                     " in initialization string, must be multiple of " + X3DConcreteField.getTupleSize(getType());
                validationResult.append(errorNotice).append("\n");
                throw new InvalidFieldValueException(errorNotice);
            }
            clearValues();
            if (newValue == null)
            {
                return this; // nothing else to do
            }
            valueArrayDouble = newValue;
            return this;
	}
}
