/*
Web3D Consortium Open-Source License for Models and Software

Copyright (c) 1995-2025 held by the author(s).  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer
      in the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the Web3D Consortium (https://www.web3D.org)
      nor the names of its contributors may be used to endorse or
      promote products derived from this software without specific
      prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

package org.web3d.x3d.jsail;

import org.web3d.x3d.jsail.Core.*;
import org.web3d.x3d.sai.InvalidFieldValueException;
import org.web3d.x3d.sai.X3DException;

import java.io.*;
import java.util.*;

/**
 * Utility class that enables developers to set custom configuration properties when using X3D Java SAI Library (X3DJSAIL). Note use of static methods.
 * 
 * <br><br>
 *
 * <i>Warning:</i> this is an abstract interface that cannot be instantiated as a concrete object.
 * Java programmers typically only need to use concrete objects provided by the <code>org.web3d.x3d.jsail</code> classes.


* <a href="../../../../../X3DJSAIL.html#property" target="blank">Utility methods</a>
and
<a href="../../../../../X3DJSAIL.html#CommandLine" target="blank">command-line support</a>
are available to load Java .property files, such as
<a href="../../../../../X3DJSAIL.properties.template">X3DJSAIL.properties.template</a> for user preferences.
* <p>Output serialization support is provided for indentation,
<a href="https://www.web3d.org/documents/specifications/19776-3/V3.3/Part03/concepts.html#X3DCanonicalForm" target="blank">X3D Compressed Binary Encoding: X3D Canonical Form</a>,
showing default attribute values, and other custom settings.</p>
	 * @see <a href="https://www.web3d.org/x3d/tools/canonical/doc/x3dTools.htm">X3D Canonicalization (C14N) Tool</a>
 * @see <a href="https://docs.oracle.com/javase/tutorial/java/javaOO/classvars.html">Java Tutorials: Understanding (Static) Class Members</a>
 * @see <a href="https://docs.oracle.com/javase/tutorial/ext/basics/spi.html#singleton">Java Tutorials: The Singleton Design Pattern</a>
 * 
 * @author Don Brutzman and Roy Walmsley
 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html" target="_blank">X3D Scene Authoring Hints</a>
 */
public class ConfigurationProperties
{
        /** Default constructor */
        public ConfigurationProperties()
        {
            // avoid warning
        }

	// singleton pattern for property values

        // ==========================================================================================

	/** Default character-count increment for serializing scene output. */
	public static final int indentIncrement_DEFAULT = 2;

	private static int indentIncrement = indentIncrement_DEFAULT; // static initialization

	/** Whitespace character for indenting when serializing scene output. */
	public static final char indentCharacter_SPACE = ' ';

	/** Alternative whitespace character for indenting when serializing scene output. */
	public static final char indentCharacter_TAB   = '\t';

	/** Default character for indenting when serializing scene output, initial value is indentCharacter_SPACE. */
	public static final char indentCharacter_DEFAULT = indentCharacter_SPACE;

	private static char indentCharacter = indentCharacter_DEFAULT; // static initialization

	/** Default XML document encoding, used in the XML document declaration appearing in the first line of an XML file.
	 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html#X3D">X3D Tooltips: X3D</a>
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html#Validation" target="_blank">X3D Scene Authoring Hints: Validation of X3D Scenes using DTD and XML Schema</a>
	*/
	public static final String XML_ENCODING_DECLARATION_DEFAULT = "UTF-8"; // this must be exact!

	// TODO add additional encoding declarations as string constants, along with mutatable configuration property.

	/** Error message if configuration of X3DJSAIL is incorrect: CLASSPATH missing jar, or other error.
	 */
	public static final String ERROR_CONFIGURATION_X3DJSAIL = "ERROR_CONFIGURATION_X3DJSAIL";

	/** Warning message if configuration of X3DJSAIL is incorrect: properties file missing, or other error.
	 */
	public static final String WARNING_CONFIGURATION_X3DJSAIL = "WARNING_CONFIGURATION_X3DJSAIL";

	/** Warning message if model information is unexpected or missing.
	 */
	public static final String WARNING_MESSAGE = "WARNING_MESSAGE";

	/** Error message if an illegal value is provided as a method parameter.
	 */
	public static final String ERROR_ILLEGAL_VALUE = "ERROR_ILLEGAL_VALUE";

	/** Error message if an element is not currently connected to an existing scene graph object, and necessary checking  for model validation is not possible.
		Example: <i>field</i> or <i>fieldValue</i> creation may fail validation() if the field types cannot be checked in a
		corresponding <i>ProtoDeclare</i> or <i>ExternProtoDeclare</i>.
	 */
	public static final String ERROR_NOT_CONNECTED_TO_SCENE_GRAPH = "ERROR_NOT_CONNECTED_TO_SCENE_GRAPH";

	/** Error message if a field is required but no value is found.
	 */
	public static final String ERROR_VALUE_NOT_FOUND = "ERROR_VALUE_NOT_FOUND";

	/** Error message if a node is required but no reference is found.
	 */
	public static final String ERROR_NODE_NOT_FOUND = "ERROR_NODE_NOT_FOUND";

	/** Error message if incorrect field accessType value encountered.
	 * @see org.web3d.x3d.jsail.Core.field#getAccessType()
	 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html#accessType">X3D Tooltips: accessType</a>
	 */
	public static final String ERROR_UNKNOWN_FIELD_ACCESSTYPE = "ERROR_UNKNOWN_FIELD_ACCESSTYPE";

	/** Error message if incorrect field type value encountered.
	 * @see org.web3d.x3d.jsail.Core.field#getType()
	 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html#type">X3D Tooltips: type</a>
	 */
	public static final String ERROR_UNKNOWN_FIELD_TYPE = "ERROR_UNKNOWN_FIELD_TYPE";

	/** Error message if node type of ProtoDeclare is not found.
	 * @see ProtoDeclare
	 * @see ExternProtoDeclare
	 * @see ProtoInstance
	 * @see <a href="https://www.web3d.org/documents/specifications/19775-2/V3.3/Part02/abstracts.html#InvalidProtoException">X3D SAI: B.7.13 InvalidProtoException</a>
	 */
	public static final String ERROR_UNKNOWN_PROTODECLARE_NODE_TYPE = "ERROR_UNKNOWN_PROTODECLARE_NODE_TYPE"; // not defined in X3D Java SAI

	/** Error message if node type of ExternProtoDeclare is not found.
	 * @see ExternProtoDeclare
	 * @see ProtoDeclare
	 * @see ProtoInstance
	 * @see <a href="https://www.web3d.org/documents/specifications/19775-2/V3.3/Part02/abstracts.html#InvalidProtoException">X3D SAI: B.7.13 InvalidProtoException</a>
	 */
	public static final String ERROR_UNKNOWN_EXTERNPROTODECLARE_NODE_TYPE = "ERROR_UNKNOWN_EXTERNPROTODECLARE_NODE_TYPE"; // not defined in X3D Java SAI

	/** Error message if node type of ProtoInstance is not found.
	 * @see ProtoInstance
	 * @see ProtoDeclare
	 * @see ExternProtoDeclare
	 * @see <a href="https://www.web3d.org/documents/specifications/19775-2/V3.3/Part02/abstracts.html#InvalidProtoException">X3D SAI: B.7.13 InvalidProtoException</a>
	 */
	public static final String ERROR_UNKNOWN_PROTOINSTANCE_NODE_TYPE = "ERROR_UNKNOWN_PROTOINSTANCE_NODE_TYPE"; // not defined in X3D Java SAI

	/** Warning message if a ProtoInstance corresponding to a given ProtoDeclare or ExternProtoDeclare is not found.
	 * @see ProtoInstance
	 * @see ProtoDeclare
	 * @see ExternProtoDeclare
	 */
	public static final String WARNING_PROTOINSTANCE_NOT_FOUND = "WARNING_PROTOINSTANCE_NOT_FOUND";

        // ==========================================================================================

	/** Default mode for debugging results, initial value is false. */
	public static boolean debugModeActive  = false;

	/** Whether SFImage pixel output values are in hexadecimal format when serializing scene output, initial value is <i>true</i>. */
	public  static final boolean SFImagePixelOutputHexadecimal_DEFAULT   = true;

	private static boolean SFImagePixelOutputHexadecimal = SFImagePixelOutputHexadecimal_DEFAULT;

	/** Whether to strip default attribute values when serializing scene output, initial value is <i>true;</i>. */
	public  static final boolean stripDefaultAttributes_DEFAULT   = true;

	private static boolean stripDefaultAttributes = stripDefaultAttributes_DEFAULT; // static initialization

	/** Set whether to allow partial output if validation exception occurs when serializing scene output, initial value is <i>false</i>. */
	public  static final boolean validationExceptionAllowed_DEFAULT   = false;

	private static boolean validationExceptionAllowed = validationExceptionAllowed_DEFAULT; // static initialization

	/** Set whether to allow partial output if validation exception occurs when creating an object, initial value is <i>true</i>. */
	public  static final boolean creationConnectionValidationExceptionAllowed_DEFAULT   = true;

	private static boolean creationConnectionValidationExceptionAllowed = creationConnectionValidationExceptionAllowed_DEFAULT;

	/** Set whether to normalize whitespace in comments, which can aid consistency in canonicalization and security; default value is <i>true</i>.
	 * @see ConfigurationProperties#setNormalizeCommentWhitespace(boolean)
	 * @see ConfigurationProperties#isNormalizeCommentWhitespace()
         */
        public  static final boolean normalizeCommentWhitespace_DEFAULT   = true;

        private static boolean normalizeCommentWhitespace = normalizeCommentWhitespace_DEFAULT; // static initialization // static initialization

	/** Whether to allow overwriting previously existing files, initial value is <i>true</i>. */
	public  static final boolean overwriteExistingFiles_DEFAULT   = true;

        /** Whether to allow overwriting previously existing files, initial value is <i>true</i>. */
	private static boolean overwriteExistingFiles = overwriteExistingFiles_DEFAULT; // static initialization

        /** Whether to show console output while processing X3D model, initial value is <i>true</i>. */
	public  static final boolean consoleOutputVerbose_DEFAULT   = true;

        /** Whether to show console output while processing X3D model, initial value is <i>true</i>. */
        private static boolean consoleOutputVerbose = consoleOutputVerbose_DEFAULT; // static initialization

	// ==========================================================================================

	/** X3DJSAIL name
	 * @see <a href="https://www.web3d.org/specifications/java/X3DJSAIL.html">https://www.web3d.org/specifications/java/X3DJSAIL.html</a> */
	public static final String NAME_X3DJSAIL     = "X3D Java Scene Access Interface Library (X3DJSAIL)";

	/** Date this version of X3DJSAIL was autogenerated: "22 June 2025".
	 * @see <a href="https://www.web3d.org/specifications/java/X3DJSAIL.html">https://www.web3d.org/specifications/java/X3DJSAIL.html</a> */
	public static final String VERSION_DATE = "22 June 2025";

	/** X3DJSAIL documentation page
	 * @see <a href="https://www.web3d.org/specifications/java/X3DJSAIL.html">https://www.web3d.org/specifications/java/X3DJSAIL.html</a> */
	public static final String URL_X3DJSAIL     = "https://www.web3d.org/specifications/java/X3DJSAIL.html";

	/** X3D Resources documentation page
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dResources.html">https://www.web3d.org/x3d/content/examples/X3dResources.html</a> */
	public static final String URL_X3DRESOURCES = "https://www.web3d.org/x3d/content/examples/X3dResources.html";

	/** X3D Scene Authoring Hints documentation pages
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html">X3D Scene Authoring Hints</a> */
	public static final String URL_X3DSCENEAUTHORINGHINTS = "https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html";

	/** X3D Regular Expressions (regexes) documentation pages
     * @see <a href="https://www.web3d.org/specifications/X3dRegularExpressions.html">X3D Regular Expressions (regexes)</a> */
	public static final String URL_X3DREGEXES = "https://www.web3d.org/specifications/X3dRegularExpressions.html";

	/** X3D Tooltips documentation pages
	 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html">https://www.web3d.org/x3d/tooltips/X3dTooltips.html</a>
	 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dResources.html#Tooltips">X3D Resources: Tooltips (multiple languages)</a> */
	public static final String URL_X3DTOOLTIPS = "https://www.web3d.org/x3d/tooltips/X3dTooltips.html";

	/** X3D Unified Object Model (X3DUOM) documentation page
	 * @see <a href="https://www.web3d.org/specifications/X3DUOM.html">X3D Unified Object Model (X3DUOM)</a> */
	public static final String URL_X3DUOM = "https://www.web3d.org/specifications/X3DUOM.html";

	/** XSLT stylesheet to create pretty-print HTML documentation page from X3D scene: <i>../lib/stylesheets/X3dToXhtml.xslt</i>
	 * @see <a href="../../../../../../lib/stylesheets/X3dToXhtml.xslt" target="_blank">X3dToXhtml.xslt</a>
	 * @see <a href="../../../../../../examples/HelloWorldProgramOutput.html" target="_blank">examples/HelloWorldProgramOutput.html</a>
	 */
	public static final String STYLESHEET_HTML_DOCUMENTATION   = "X3dToXhtml.xslt";

	/** XSLT stylesheet to create VRML97 model from X3D scene: <i>../lib/stylesheets/X3dToVrml.xslt</i>
	 * @see <a href="../../../../../../lib/stylesheets/X3dToVrml97.xslt" target="_blank">X3dToVrml97.xslt</a>
	 * @see <a href="../../../../../../examples/HelloWorldProgramOutput.wrl" target="_blank">examples/HelloWorldProgramOutput.wrl</a>
	 */
	public static final String STYLESHEET_VRML97   = "X3dToVrml97.xslt";

	/** XSLT stylesheet to create VRML97 model from X3D scene: <i>../lib/stylesheets/X3dToX3dvClassicVrmlEncoding.xslt</i>
	 * @see <a href="../../../../../../lib/stylesheets/X3dToX3dvClassicVrmlEncoding.xslt" target="_blank">X3dToX3dvClassicVrmlEncoding.xslt</a>
	 * @see <a href="../../../../../../examples/HelloWorldProgramOutput.x3dv" target="_blank">examples/HelloWorldProgramOutput.x3dv</a>
	 */
	public static final String STYLESHEET_X3DV_CLASSICVRML   = "X3dToX3dvClassicVrmlEncoding.xslt";

	/** XSLT stylesheet to create model meta information as markdown: <i>../lib/stylesheets/X3dModelMetaToMarkdown.xslt</i>
	 * @see <a href="../../../../../../lib/stylesheets/X3dModelMetaToMarkdown.xslt" target="_blank">X3dModelMetaToMarkdown.xslt</a>
	 * @see <a href="../../../../../../examples/HelloWorldProgramOutput.md" target="_blank">examples/HelloWorldProgramOutput.md</a>
	 */
	public static final String STYLESHEET_MODEL_META_TO_MARKDOWN   = "X3dModelMetaToMarkdown.xslt";

	/** XSLT stylesheet to create X3D-Tidy cleaned-up version of X3D scene: <i>../lib/stylesheets/X3dToXhtml.xslt</i>
	 * @see <a href="https://www.web3d.org/x3d/stylesheets/X3dTidy.html">X3D Tidy for Scene Cleanup, Corrections and Modifications</a>
	 * @see <a href="../../../../../../lib/stylesheets/X3dTidy.xslt" target="_blank">X3dTidy.xslt</a>
	 * @see <a href="../../../../../../examples/HelloWorldProgramOutputTidy.x3d" target="_blank">examples/HelloWorldProgramOutputTidy.x3d</a>
	 */
	public static final String STYLESHEET_X3DTIDY   = "X3dTidy.xslt";

	/** XSLT stylesheet to create Extrusion node cross sections in SVG from X3D scene: <i>../lib/stylesheets/X3dExtrusionCrossSectionToSvg.xslt</i> */
	public static final String STYLESHEET_ExtrusionCrossSectionSVG   = "X3dExtrusionCrossSectionToSvg.xslt";

	/** XSLT stylesheet to create X3DOM XHTML page or X3DOM HTML page from X3D scene: <i>../lib/stylesheets/X3dToX3domX_ITE.xslt</i>
	 * @see <a href="../../../../../../lib/stylesheets/X3dToX3domX_ITE.xslt" target="_blank">X3dToX3domX_ITE.xslt</a>
	 * @see <a href="../../../../../../examples/HelloWorldProgramOutputX_ITE.html" target="_blank">examples/HelloWorldProgramOutputX3dom.html</a>
	 */
	public static final String STYLESHEET_X3DOM    = "X3dToX3domX_ITE.xslt";

	/** XSLT stylesheet to create X_ITE XHTML page or X_ITE HTML page from X3D scene: <i>../lib/stylesheets/X3dToX3domX_ITE.xslt</i>
            TODO disambiguation needed?
	 * @see <a href="https://create3000.de/x_ite" target="_blank">X_ITE open-source X3D player</a>
	 * @see <a href="../../../../../../lib/stylesheets/X3dToX3domX_ITE.xslt" target="_blank">X3dToX3domX_ITE.xslt</a>
	 * @see <a href="../../../../../../examples/HelloWorldProgramOutputX_ITE.html" target="_blank">examples/HelloWorldProgramOutputX_ITE.html</a>
	 */
	public static final String STYLESHEET_X_ITE    = "X3dToX3domX_ITE.xslt";

	/** XSLT stylesheet Cobweb (now X_ITE)
	 * @see <a href="https://create3000.de/x_ite" target="_blank">X_ITE open-source X3D player</a>
	 */
	@Deprecated
	public static final String STYLESHEET_COBWEB   = "X3dToX3domX_ITE.xslt";

	/** XSLT stylesheet to create Java source code (using X3DJSAIL library) from X3D scene: <i>../lib/stylesheets/X3dToJava.xslt</i>.
	 * TODO: documentation.
	 * @see <a href="../../../../../../lib/stylesheets/X3dToJava.xslt" target="_blank">X3dToJava.xslt</a>
	 * @see <a href="../../../../../../examples/HelloWorldProgramOutput.java" target="_blank">examples/HelloWorldProgramOutput.java</a>
	 */
	public static final String STYLESHEET_JAVA   = "X3dToJava.xslt";

	/** XSLT stylesheet to create Javascript (ECMAScript) source code (using X3DJSAIL library) from X3D scene: <i>../lib/stylesheets/X3dToNodeJS.xslt</i>.
	 * TODO: documentation.
	 * @see <a href="../../../../../../lib/stylesheets/X3dToNodeJS.xslt" target="_blank">X3dToNodeJS.xslt</a>
	 * @see <a href="../../../../../../examples/HelloWorldProgramOutput.java" target="_blank">examples/HelloWorldProgramOutput.java</a>
	 */
	public static final String STYLESHEET_JAVASCRIPT   = "X3dToNodeJS.xslt";

	/** XSLT stylesheet to create JSON encoding from X3D scene: <i>../lib/stylesheets/X3dToJson.xslt</i>
	 * @see <a href="https://www.web3d.org/wiki/index.php/X3D_JSON_Encoding">X3D JSON Encoding</a>
	 * @see <a href="https://www.web3d.org/x3d/stylesheets/X3dToJson.html">X3D to JSON Stylesheet Converter</a>
	 * @see <a href="../../../../../../lib/stylesheets/X3dToJson.xslt" target="_blank">X3dToJson.xslt</a>
	 * @see <a href="../../../../../../examples/HelloWorldProgramOutput.json" target="_blank">examples/HelloWorldProgramOutput.json</a>
	 */
	public static final String STYLESHEET_JSON   = "X3dToJson.xslt";

	/** XSLT stylesheet to create Python source from X3D scene: <i>../lib/stylesheets/X3dToPython.xslt</i>
	 * @see <a href="../../../../../../lib/stylesheets/X3dToPython.xslt" target="_blank">X3dToPython.xslt</a>
	 * @see <a href="../../../../../../examples/HelloWorldProgramOutput.py" target="_blank">examples/HelloWorldProgramOutput.py</a>
	 */
	public static final String STYLESHEET_PYTHON   = "X3dToPython.xslt";

    /**
     * Check whether stylesheet is supported by X3DJSAIL.
     * @param stylesheetName XSLT stylesheet to apply
     * @return whether stylesheet is supported by X3DJSAIL
     */
    public static boolean isStylesheetSupported(String stylesheetName)
    {
        return stylesheetName.equalsIgnoreCase(ConfigurationProperties.STYLESHEET_HTML_DOCUMENTATION) ||
               stylesheetName.equalsIgnoreCase(ConfigurationProperties.STYLESHEET_JAVA) ||
               stylesheetName.equalsIgnoreCase(ConfigurationProperties.STYLESHEET_JAVASCRIPT) ||
               stylesheetName.equalsIgnoreCase(ConfigurationProperties.STYLESHEET_JSON) ||
               stylesheetName.equalsIgnoreCase(ConfigurationProperties.STYLESHEET_PYTHON) ||
               stylesheetName.equalsIgnoreCase(ConfigurationProperties.STYLESHEET_MODEL_META_TO_MARKDOWN) ||
               stylesheetName.equalsIgnoreCase(ConfigurationProperties.STYLESHEET_ExtrusionCrossSectionSVG) ||
               stylesheetName.equalsIgnoreCase(ConfigurationProperties.STYLESHEET_VRML97) ||
               stylesheetName.equalsIgnoreCase(ConfigurationProperties.STYLESHEET_X3DV_CLASSICVRML) ||
               stylesheetName.equalsIgnoreCase(ConfigurationProperties.STYLESHEET_X3DOM) ||
               stylesheetName.equalsIgnoreCase(ConfigurationProperties.STYLESHEET_X_ITE) ||
               stylesheetName.equalsIgnoreCase(ConfigurationProperties.STYLESHEET_X3DTIDY);
    }
    /**
     * Determine expected output filename extension based on stylesheet conversion.
     * @param stylesheetName XSLT stylesheet being applied
     * @return expected output filename extension
     */
    public static String getExpectedOutputFileExtension(String stylesheetName)
    {
        String expectedFileNameExtension = "";
        if (stylesheetName.equals(ConfigurationProperties.STYLESHEET_ExtrusionCrossSectionSVG))
        {
            expectedFileNameExtension = X3D.FILE_EXTENSION_SVG;
        }
        else if (stylesheetName.equals(ConfigurationProperties.STYLESHEET_HTML_DOCUMENTATION))
        {
            expectedFileNameExtension = X3D.FILE_EXTENSION_HTML;
        }
        else if (stylesheetName.equals(ConfigurationProperties.STYLESHEET_X3DTIDY))
        {
            expectedFileNameExtension = X3D.FILE_EXTENSION_X3D;
        }
        else if (stylesheetName.equals(ConfigurationProperties.STYLESHEET_VRML97))
        {
            expectedFileNameExtension = X3D.FILE_EXTENSION_VRML97;
        }
        else if (stylesheetName.equals(ConfigurationProperties.STYLESHEET_X3DV_CLASSICVRML))
        {
            expectedFileNameExtension = X3D.FILE_EXTENSION_CLASSICVRML;
        }
        else if (stylesheetName.equals(ConfigurationProperties.STYLESHEET_JAVASCRIPT))
        {
            expectedFileNameExtension = X3D.FILE_EXTENSION_JAVASCRIPT;
        }
        else if (stylesheetName.equals(ConfigurationProperties.STYLESHEET_JAVA))
        {
            expectedFileNameExtension = X3D.FILE_EXTENSION_JAVA;
        }
        else if (stylesheetName.equals(ConfigurationProperties.STYLESHEET_JSON))
        {
            expectedFileNameExtension = X3D.FILE_EXTENSION_JSON;
        }
        else if (stylesheetName.equals(ConfigurationProperties.STYLESHEET_PYTHON))
        {
            expectedFileNameExtension = X3D.FILE_EXTENSION_PYTHON;
        }
        else if (stylesheetName.equals(ConfigurationProperties.STYLESHEET_X3DOM))
        {
            expectedFileNameExtension = X3D.FILE_EXTENSION_HTML;
        }
        else if (stylesheetName.equals(ConfigurationProperties.STYLESHEET_X_ITE))
        {
            expectedFileNameExtension = X3D.FILE_EXTENSION_HTML;
        }
        else if (stylesheetName.equals(ConfigurationProperties.STYLESHEET_COBWEB))
        {
            expectedFileNameExtension = X3D.FILE_EXTENSION_HTML;
        }
        else if (stylesheetName.equals(ConfigurationProperties.STYLESHEET_MODEL_META_TO_MARKDOWN))
        {
            expectedFileNameExtension = X3D.FILE_EXTENSION_MARKDOWN;
        }
        if (expectedFileNameExtension.isEmpty())
		{
			throw new org.web3d.x3d.sai.X3DException("getExpectedOutputFileExtension(" + stylesheetName + ") stylesheetName not recognized;" +
				" (see ConfigurationProperties for allowed choices)");
		}
        return expectedFileNameExtension;
    }

    // ==========================================================================================

	/** List of officially released X3DJSAIL jar files.X3D
	 */
        // https://stackoverflow.com/questions/21696784/how-to-declare-an-arraylist-with-values
        public static final ArrayList<String> X3DJSAIL_JAR_RELEASE_VERSIONS =
            new ArrayList<>(Arrays.asList("X3DJSAIL.4.0.classes.jar", "X3DJSAIL.4.0.full.jar","X3DJSAIL.3.3.classes.jar", "X3DJSAIL.3.3.full.jar"));

    // ==========================================================================================

	/** Whether to delete intermediate files generated as part of various transformations, this property can be helpful for debugging. */
	private static final boolean deleteIntermediateFiles_DEFAULT = true;

	/** Whether to delete intermediate files generated as part of various transformations, this property can be helpful for debugging. */
	private static boolean deleteIntermediateFiles = deleteIntermediateFiles_DEFAULT;

	/** Indicate whether to delete intermediate files generated as part of various transformations, this property method can be helpful can be helpful for debugging.
	 * @return configuration setting whether intermediate files are deleted
	 */
	public static final boolean isDeleteIntermediateFiles()
	{
		return deleteIntermediateFiles;
	}
	/** Set whether to delete intermediate files generated as part of various transformations, this property method can be helpful can be helpful for debugging.
	 * @param newValue is new value to assign
	 */
	public static final void setDeleteIntermediateFiles(boolean newValue)
	{
		deleteIntermediateFiles = newValue;
	}
    // ==========================================================================================

	/** Whether to omit trailing zeros from floating-point or double-precision output. */
	private static final boolean stripTrailingZeroes_DEFAULT = true;

	/** Whether to omit trailing zeros from floating-point or double-precision output. */
	private static boolean stripTrailingZeroes = stripTrailingZeroes_DEFAULT;

	/** Indicates whether trailing zeros are omitted from output of floating-point or double-precision values, this property can be helpful for debugging.
	 * @return configuration setting whether trailing zeros are stripped
	 */
	public static final boolean isStripTrailingZeroes()
	{
		return stripTrailingZeroes;
	}
	/** Set whether to omit trailing zeros from floating-point or double-precision output, this property can be helpful for debugging.
	 * @param newValue is new value to assign
	 */
	public static final void setStripTrailingZeroes(boolean newValue)
	{
		stripTrailingZeroes = newValue;
	}
    // ==========================================================================================

	/** XSLT transformation engine: SAXON (default).
	 * @see <a href="https://saxon.sourceforge.net/#F9.7HE">Saxon-HE 9.7</a>
	 * @see <a href="https://sourceforge.net/projects/saxon/files">Saxon distribution</a>
	 * @see <a href="https://www.saxonica.com/documentation/index.html#!using-xsl/embedding">Saxonica &gt; Saxon &gt; Using XSLT &gt; Invoking XSLT from an application</a>
	 */
	public static final String XSLT_ENGINE_SAXON = "SAXON9HE";

	/** XSLT transformation engine: native Java.
	 * @see <a href="https://docs.oracle.com/javase/tutorial/jaxp/xslt/transformingXML.html">Java Tutorials: Transforming XML Data with XSLT</a>
	 * @see <a href="https://docs.oracle.com/javase/tutorial/jaxp/examples/xslt_samples.zip">Java Tutorials: Transforming XML Data with XSLT, sample files</a>
	 * @see <a href="https://docs.oracle.com/javase/tutorial/essential/io/file.html#textfiles">Buffered I/O Methods for Text Files</a>
	 */
	public static final String XSLT_ENGINE_NATIVE_JAVA = "NATIVE_JAVA";

	/** XSLT transformation engine setting: default <i>XSLT_ENGINE_SAXON</i>. */
	private static String xsltEngine = XSLT_ENGINE_SAXON;

	/** Get preference for XSLT transformation engine to use: {@link XSLT_ENGINE_SAXON} or {@link XSLT_ENGINE_NATIVE_JAVA}.
	 * @return String constant regarding current configuration: XSLT_ENGINE_SAXON (default) or XSLT_ENGINE_NATIVE_JAVA
	 */
	public static final String getXsltEngine()
	{
		return xsltEngine;
	}
	/** Set preference for XSLT transformation engine to use: {@link XSLT_ENGINE_SAXON} or {@link XSLT_ENGINE_NATIVE_JAVA}.
	 * @param newValue is new value to assign */
	public static final void setXsltEngine(String newValue)
	{
		if (newValue.equals(XSLT_ENGINE_SAXON) || newValue.equals(XSLT_ENGINE_NATIVE_JAVA))
			xsltEngine = newValue;
		else
		{
			String errorNotice = "*** Invalid setXsltEngine(String newValue) invocation, newValue='" + newValue +
								 "', legal values are ConfigurationProperties.XSLT_ENGINE_SAXON or ConfigurationProperties.XSLT_ENGINE_NATIVE_JAVA";
//			validationResult.append(errorNotice).append("\n");
			throw new InvalidFieldValueException(errorNotice);
		}
	}
    // ==========================================================================================

	/** EXI transformation engine: OpenEXI Nagasena.
	 * <i>Warning:</i> testing in progress.
	 * @see <a href="https://openexi.sourceforge.net">OpenEXI Nagasena</a>
	 * @see <a href="https://openexi.sourceforge.net/tutorial">Nagasena Tutorial</a>
	 * @see <a href="https://www.youtube.com/watch?v=Rig2z9veUv0">Video: OpenEXI, A Quick Introduction</a>
	 * @see ConfigurationProperties#getExiEngine()
	 * @see ConfigurationProperties#setExiEngine(String)
	 * @see ConfigurationProperties#EXI_ENGINE_EXIFICIENT
	 */
	public static final String EXI_ENGINE_OPENEXI = "OPENEXI";

	/** XSLT transformation engine: EXIficient (default).
	 * @see <a href="https://github.com/EXIficient">EXIficient project page</a>
	 * @see <a href="https://github.com/EXIficient/exificient/blob/master/README.md">EXIficient README</a>
	 * @see ConfigurationProperties#getExiEngine()
	 * @see ConfigurationProperties#setExiEngine(String)
	 * @see ConfigurationProperties#EXI_ENGINE_OPENEXI
	 */
	public static final String EXI_ENGINE_EXIFICIENT = "EXIFICIENT";

	/** EXI transformation engine setting: default <i>EXI_ENGINE_EXIFICIENT</i>. */
	private static String exiEngine = EXI_ENGINE_EXIFICIENT;

	/** Get preference for XSLT transformation engine to use: {@link EXI_ENGINE_EXIFICIENT} or {@link EXI_ENGINE_OPENEXI}.
	 * @return String constant regarding current configuration: EXI_ENGINE_EXIFICIENT} (default) or EXI_ENGINE_OPENEXI
	 * @see ConfigurationProperties#EXI_ENGINE_EXIFICIENT
	 * @see ConfigurationProperties#EXI_ENGINE_OPENEXI
	 * @see ConfigurationProperties#setExiEngine(String)
	 */
	public static final String getExiEngine()
	{
		return exiEngine;
	}
	/** Set preference for EXI transformation engine to use: {@link EXI_ENGINE_EXIFICIENT} or {@link EXI_ENGINE_OPENEXI}.
	 * @param newValue is new value to assign
	 * @see ConfigurationProperties#EXI_ENGINE_EXIFICIENT
	 * @see ConfigurationProperties#EXI_ENGINE_OPENEXI
	 * @see ConfigurationProperties#getExiEngine()
     */
	public static final void setExiEngine(String newValue)
	{
		if (newValue.equals(EXI_ENGINE_EXIFICIENT) || newValue.equals(EXI_ENGINE_OPENEXI))
			exiEngine = newValue;
		else
		{
			String errorNotice = "*** Invalid setExiEngine(String newValue) invocation, newValue='" + newValue +
								 "', legal values are ConfigurationProperties.EXI_ENGINE_EXIFICIENT or ConfigurationProperties.EXI_ENGINE_OPENEXI";
//			validationResult.append(errorNotice).append("\n");
			throw new InvalidFieldValueException(errorNotice);
		}
	}

    // ==========================================================================================

	/** Initialize this ConfigurationProperties instance to default values. */
	public static final void initialize()
	{
		indentIncrement               = indentIncrement_DEFAULT;
		indentCharacter               = indentCharacter_DEFAULT;
		SFImagePixelOutputHexadecimal = SFImagePixelOutputHexadecimal_DEFAULT;
		stripDefaultAttributes        = stripDefaultAttributes_DEFAULT;
		validationExceptionAllowed    = validationExceptionAllowed_DEFAULT;
		deleteIntermediateFiles       = deleteIntermediateFiles_DEFAULT;
		stripTrailingZeroes	      = stripTrailingZeroes_DEFAULT;
		normalizeCommentWhitespace    = normalizeCommentWhitespace_DEFAULT;
		overwriteExistingFiles        = overwriteExistingFiles_DEFAULT;
		consoleOutputVerbose          = consoleOutputVerbose_DEFAULT;
		setExiEngine (EXI_ENGINE_EXIFICIENT);
		setXsltEngine(XSLT_ENGINE_SAXON);
	}

	/** Default name of properties file: <code>X3DJSAIL.properties</code>
	 * @see ConfigurationProperties#getPropertiesFileName()
	 * @see ConfigurationProperties#setPropertiesFileName(String)
	 * @see ConfigurationProperties#loadProperties()
	 */
	public static final String PROPERTIES_FILENAME_DEFAULT = "X3DJSAIL.properties";

	/** Name of properties file. */
	private static String propertiesFileName = PROPERTIES_FILENAME_DEFAULT;

	/** Set name of properties file (optional directory path plus file name).
	 * @see ConfigurationProperties#PROPERTIES_FILENAME_DEFAULT
	 * @see ConfigurationProperties#getPropertiesFileName()
	 * @see ConfigurationProperties#loadProperties()
	 * @param fileName new name of properties file to load and parse
	 */
	public static void setPropertiesFileName(String fileName)
	{
		propertiesFileName = fileName;
	}

	/** Get name of current properties file (optional directory path plus file name).
	 * @see ConfigurationProperties#PROPERTIES_FILENAME_DEFAULT
	 * @see ConfigurationProperties#setPropertiesFileName(String)
	 * @see ConfigurationProperties#loadProperties()
	 * @return name of properties file to load and parse
	 */
	public static String getPropertiesFileName()
	{
		return propertiesFileName;
	}

	/** Update settings in this ConfigurationProperties instance using values in property file.
	 * @see ConfigurationProperties#PROPERTIES_FILENAME_DEFAULT
	 * @see ConfigurationProperties#getPropertiesFileName()
	 * @see ConfigurationProperties#setPropertiesFileName(String)
	 * @see <a href="https://docs.oracle.com/javase/tutorial/essential/environment/properties.html">Java Tutorials: Properties</a>
	 * @see <a href="https://docs.oracle.com/javase/8/docs/api/java/util/Properties.html">Javadoc: java.util.Properties</a>
	 */
	public static void loadProperties()
	{
		// create and load default properties
		Properties loadedProperties = new Properties();
		try {
			File propertiesFile = new File(getPropertiesFileName());
			if (!propertiesFile.exists())
			{
				System.out.println (WARNING_CONFIGURATION_X3DJSAIL + ": " + getPropertiesFileName() + " properties file not found");
			}
			FileInputStream propertiesFileInputStream = new FileInputStream(propertiesFile);
			loadedProperties.load(propertiesFileInputStream);
			propertiesFileInputStream.close();
		}
		catch (IOException ioe)
		{
            System.out.println (ERROR_CONFIGURATION_X3DJSAIL + ": " + ioe.getMessage());
            ioe.printStackTrace(); // further diagnosis needed
		}
		System.out.print (getPropertiesFileName() + " includes " + loadedProperties.size());
		if (loadedProperties.isEmpty())
			System.out.println (" properties");
		else if (loadedProperties.size() == 1)
			System.out.println (" property:");
		else
			System.out.println (" properties:");
		loadedProperties.list(System.out);

		if (loadedProperties.size() > 0)
		{
			if (loadedProperties.contains("indentIncrement"))
				indentIncrement = Integer.getInteger(loadedProperties.getProperty("indentIncrement"));
			if (loadedProperties.contains("indentCharacter"))
			{
				String indentCharacterProperty = loadedProperties.getProperty("indentCharacter");
				if		(indentCharacterProperty.toUpperCase().contains("SPACE"))
						 indentCharacter = indentCharacter_SPACE;
				else if (indentCharacterProperty.toUpperCase().contains("TAB"))
						 indentCharacter = indentCharacter_SPACE;
				else if (!indentCharacterProperty.isEmpty())
					System.out.println ("Error: unrecognized property indentCharacter='" + indentCharacterProperty +
						"' (allowed values are SPACE and TAB)");
			}
			// warning: properties are case sensitive
			// https://stackoverflow.com/questions/86780/how-to-check-if-a-string-contains-another-string-in-a-case-insensitive-manner-in

			if (loadedProperties.contains("SFImagePixelOutputHexadecimal"))
				SFImagePixelOutputHexadecimal = Boolean.getBoolean(loadedProperties.getProperty("SFImagePixelOutputHexadecimal"));
            if (loadedProperties.contains("showDefaultAttributes")) // original name, convert it
            {
				stripDefaultAttributes = !Boolean.getBoolean(loadedProperties.getProperty("showDefaultAttributes"));
                System.out.println ("[warning] legacy property showDefaultAttributes='" + !stripDefaultAttributes +
                    "' found, changed to stripDefaultAttributes='" + stripDefaultAttributes + "' instead");
            }
			if (loadedProperties.contains("stripDefaultAttributes"))
				stripDefaultAttributes = Boolean.getBoolean(loadedProperties.getProperty("stripDefaultAttributes"));
			if (loadedProperties.contains("validationExceptionAllowed"))
		     validationExceptionAllowed = Boolean.getBoolean(loadedProperties.getProperty("validationExceptionAllowed"));
			if (loadedProperties.contains("deleteIntermediateFiles"))
				deleteIntermediateFiles = Boolean.getBoolean(loadedProperties.getProperty("deleteIntermediateFiles"));
			if (loadedProperties.contains("stripTrailingZeroes"))
				stripTrailingZeroes = Boolean.getBoolean(loadedProperties.getProperty("stripTrailingZeroes"));
			if (loadedProperties.contains("normalizeCommentWhitespace"))
				normalizeCommentWhitespace = Boolean.getBoolean(loadedProperties.getProperty("normalizeCommentWhitespace"));
			if (loadedProperties.contains("overwriteExistingFiles"))
				overwriteExistingFiles = Boolean.getBoolean(loadedProperties.getProperty("overwriteExistingFiles"));
			if (loadedProperties.contains("consoleOutputVerbose"))
				consoleOutputVerbose   = Boolean.getBoolean(loadedProperties.getProperty("consoleOutputVerbose"));

			if      (loadedProperties.contains("EXI_ENGINE") && loadedProperties.getProperty("EXI_ENGINE").toUpperCase().contains("EXIFICIENT"))
					setExiEngine (EXI_ENGINE_EXIFICIENT);
			else if (loadedProperties.contains("EXI_ENGINE") && loadedProperties.getProperty("EXI_ENGINE").toUpperCase().contains("OPENEXI"))
					setExiEngine (EXI_ENGINE_OPENEXI);
			else if (loadedProperties.contains("EXI_ENGINE"))
					System.out.println ("Error: unrecognized property EXI_ENGINE=" + loadedProperties.getProperty("EXI_ENGINE") +
						"' (allowed values are EXIFICIENT and OPENEXI)");

			if      (loadedProperties.contains("XSLT_ENGINE") && loadedProperties.getProperty("EXI_ENGINE").toUpperCase().contains("SAXON"))
					setXsltEngine(XSLT_ENGINE_SAXON);
			else if (loadedProperties.contains("XSLT_ENGINE") && loadedProperties.getProperty("EXI_ENGINE").toUpperCase().contains("NATIVE_JAVA"))
					setXsltEngine(XSLT_ENGINE_NATIVE_JAVA);
			else if (loadedProperties.contains("XSLT_ENGINE"))
					System.out.println ("Error: unrecognized property XSLT_ENGINE=" + loadedProperties.getProperty("XSLT_ENGINE") +
						"' (allowed values are SAXON and NATIVE_JAVA)");

			if      (loadedProperties.contains("BLENDER_PATH"))
					BlenderLauncher.setBlenderPath(loadedProperties.getProperty("BLENDER_PATH"));
			if      (loadedProperties.contains("MESHLAB_PATH"))
					MeshLabLauncher.setMeshLabPath(loadedProperties.getProperty("MESHLAB_PATH"));

		}
		System.out.println ("------------------------");
		System.out.println (getPropertiesFileName() + " loading complete.");
	}

	/**
	 * Get current system CLASSPATH value.  Note that a current version of X3DJSAIL.*.jar is expected to be in the current CLASSPATH.
	 * @see <a href="https://docs.oracle.com/javase/8/docs/technotes/tools/windows/classpath.html">Java documentation: Setting the Class Path</a>
	 * @see <a href="https://docs.oracle.com/javase/tutorial/essential/environment/paths.html">Java Tutorials: PATH and CLASSPATH</a>
	 * @return system CLASSPATH value. */
	public static String getClassPath()
	{
		return System.getProperty("java.class.path");
	}
	/**
	 * Get indentCharacter used when serializing scene output.
	 * @see #setIndentCharacter(char)
	 * @see #setIndentIncrement(int)
	 * @return indentCharacter (either indentCharacter_SPACE or indentCharacter_TAB). */
	public static char getIndentCharacter()
	{
		return indentCharacter;
	}
	/**
	 * Set indentCharacter used when serializing scene output.
	 * @param newIndentCharacter is new indent value (non-negative).
	 * @see #setX3dCanonicalForm() */
	public static void setIndentCharacter (char newIndentCharacter)
	{
		if  ((newIndentCharacter == indentCharacter_SPACE) || (newIndentCharacter == indentCharacter_TAB))
			 indentCharacter = newIndentCharacter;
		else
		{
			String errorNotice = "*** Invalid indentCharacter='" + newIndentCharacter +
								 "' provided to ConfigurationProperties, expected indentCharacter_SPACE or indentCharacter_TAB";
//			validationResult.append(errorNotice).append("\n");
			throw new InvalidFieldValueException(errorNotice);
		}
	}
	/**
	 * Get number of indent characters (space or tab characters) to indent when serializing scene output.
	 * @see #setIndentCharacter(char)
	 * @see #setIndentIncrement(int)
	 * @return number of characters (non-negative). */
	public static int getIndentIncrement()
	{
		return indentIncrement;
	}
	/**
	 * Set number of characters to indent when serializing scene output.
	 * @param newIndentIncrement is new indentIncrement value (non-negative).
	 * @see #setX3dCanonicalForm() */
	public static void setIndentIncrement (int newIndentIncrement)
	{
		if  (newIndentIncrement >= 0)
			 indentIncrement = newIndentIncrement;
		else
		{
			indentIncrement = 0;
			String errorNotice = "*** Invalid indentIncrement=" + indentIncrement + " provided to ConfigurationProperties";
//			validationResult.append(errorNotice).append("\n");
			throw new IllegalArgumentException(errorNotice);
		}
	}
	/**
	 * Indicate whether X3D Canonical Form is used for toStringX3D() XML output.
	 * @see #setIndentCharacter(char)
	 * @see #setIndentIncrement(int)
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-3/V3.3/Part03/concepts.html#X3DCanonicalForm">X3D Compressed binary encoding, 4.2.3 X3D canonical form</a>
	 * @see <a href="https://www.w3.org/TR/xml-c14n">Canonical XML</a>
	 * @see <a href="https://www.w3.org/TR/exi-c14n">Canonical EXI</a>
	 * @see <a href="https://santuario.apache.org">Apache Santuario</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-3/V3.3/Part03/concepts.html#X3DCanonicalForm" target="blank">X3D Compressed Binary Encoding: X3D Canonical Form</a>
	 * @see <a href="https://www.web3d.org/x3d/tools/canonical/doc/x3dTools.htm">X3D Canonicalization (C14N) Tool</a>
	 * @return whether X3D Canonical Form is used. */
	public static boolean isX3dCanonicalForm()
	{
		return ((indentIncrement == indentIncrement_DEFAULT) &&
			    (indentCharacter == indentCharacter_DEFAULT));
	}
	/**
	 * Ensure that X3D Canonical Form is used for XML output, resetting default values for indentation.
	 * @see #setIndentIncrement(int)
	 * @see #setIndentCharacter(char)
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-3/V3.3/Part03/concepts.html#X3DCanonicalForm">X3D Compressed binary encoding, 4.2.3 X3D canonical form</a>
	 * @see <a href="https://www.w3.org/TR/xml-c14n">Canonical XML</a>
	 * @see <a href="https://www.w3.org/TR/exi-c14n">Canonical EXI</a>
	 * @see <a href="https://santuario.apache.org">Apache Santuario</a>
	 * @see <a href="https://www.web3d.org/documents/specifications/19776-3/V3.3/Part03/concepts.html#X3DCanonicalForm" target="blank">X3D Compressed Binary Encoding: X3D Canonical Form</a>
	 * @see <a href="https://www.web3d.org/x3d/tools/canonical/doc/x3dTools.htm">X3D Canonicalization (C14N) Tool</a>
	 */
	public static void setX3dCanonicalForm()
	{
		indentIncrement = indentIncrement_DEFAULT;
		indentCharacter = indentCharacter_DEFAULT;
	}
	/**
	 * Indicate whether debug mode is active.
	 * @return whether debug mode is active. */
	public static boolean isDebugModeActive()
	{
		return debugModeActive;
	}
	/**
	 * Set whether debug mode is active.
	 * @param newDebugModeActive whether debug mode is active. */
	public static void setDebugModeActive(boolean newDebugModeActive)
	{
		debugModeActive = newDebugModeActive;
	}
	/**
	 * Indicate whether SFImage pixel output values are in hexadecimal format when serializing scene output.
	 * @return whether default attributes are shown. */
	public static boolean isSFImagePixelOutputHexadecimal()
	{
		return SFImagePixelOutputHexadecimal;
	}
	/**
	 * Set whether SFImage pixel output values are in hexadecimal format when serializing scene output.
	 * @param newSFImagePixelOutputHexadecimal whether SFImage pixel output values are in hexadecimal format. */
	public static void setSFImagePixelOutputHexadecimal(boolean newSFImagePixelOutputHexadecimal)
	{
		SFImagePixelOutputHexadecimal = newSFImagePixelOutputHexadecimal;
	}
	/**
	 * Indicate whether default attributes (and their values) are stripped when serializing scene output.
	 * @return whether default attributes are shown. */
	public static boolean getStripDefaultAttributes()
	{
		return stripDefaultAttributes;
	}
	/**
	 * Set whether default attributes (and their values) are stripped when serializing scene output.
	 * @param newStripDefaultAttributes whether default attributes are shown. */
	public static void setStripDefaultAttributes(boolean newStripDefaultAttributes)
	{
		stripDefaultAttributes = newStripDefaultAttributes;
	}
	/**
	 * Indicate whether partial results are allowed if validation exception occurs when serializing scene output.
	 * @return whether validation exceptions are allowed (and operation continues) */
	public static boolean isValidationExceptionAllowed()
	{
		return validationExceptionAllowed;
	}
	/**
	 * Set whether partial results are allowed (and operation continues) if validation exception occurs when serializing scene output.
	 * Can be useful technique for debugging, default value is <i>false</i> for strict operation.
	 * <i>Warning:</i> setting value to <i>true</i> permits creation of an invalid scene graph.
	 * @param newValidationExceptionAllowed whether validation exceptions are allowed (and operation continues) */
	public static void setValidationExceptionAllowed(boolean newValidationExceptionAllowed)
	{
		validationExceptionAllowed = newValidationExceptionAllowed;
	}
	/**
	 * Indicate whether continuation is allowed if validation exception occurs when creating an object.
	 * <i>Warning:</i> TODO experimental.
	 * @return whether validation exceptions are allowed (and operation continues) during object creation */
	public static boolean isCreationConnectionValidationExceptionAllowed()
	{
		return creationConnectionValidationExceptionAllowed;
	}
	/**
	 * Set whether partial results are allowed (and operation continues) if validation exception occurs when creating an object.
	 * Can be useful technique for debugging, default value is <i>true</i> for permissive order of object creation.
	 * <i>Warning:</i> be sure to validate() this scene once construction is complete.
	 * <i>Warning:</i> TODO experimental.
	 * @param newCreationConnectionValidationExceptionAllowed whether validation exceptions are allowed (and object creation continues) */
	public static void setCreationConnectionValidationExceptionAllowed(boolean newCreationConnectionValidationExceptionAllowed)
	{
		creationConnectionValidationExceptionAllowed = newCreationConnectionValidationExceptionAllowed;
	}
	/**
	 * Indicate whether to normalize whitespace in comments, which can aid consistency in canonicalization and security.
	 * @see ConfigurationProperties#normalizeCommentWhitespace_DEFAULT
	 * @see ConfigurationProperties#setNormalizeCommentWhitespace(boolean)
	 * @return whether to normalize whitespace in comments */
	public static boolean isNormalizeCommentWhitespace()
	{
		return normalizeCommentWhitespace;
	}
	/**
	 * Set whether to normalize whitespace in comments, which can aid consistency in canonicalization and security.
	 * @see ConfigurationProperties#normalizeCommentWhitespace_DEFAULT
	 * @see ConfigurationProperties#isNormalizeCommentWhitespace()
	 * @param newNormalizeCommentWhitespace whether to normalize whitespace in comments */
	public static void setNormalizeCommentWhitespace(boolean newNormalizeCommentWhitespace)
	{
		normalizeCommentWhitespace = newNormalizeCommentWhitespace;
	}
	/**
	 * Indicate whether to allow overwriting previously existing files.
	 * @see overwriteExistingFiles_DEFAULT
	 * @return whether creation of new files can overwrite prior versions */
	public static boolean isOverwriteExistingFiles()
	{
		return overwriteExistingFiles;
	}
	/**
	 * Set whether to allow overwriting previously existing files.
	 * @see overwriteExistingFiles_DEFAULT
	 * @param newOverwriteExistingFiles whether creation of new files can overwrite prior versions */
	public static void setOverwriteExistingFiles(boolean newOverwriteExistingFiles)
	{
		overwriteExistingFiles = newOverwriteExistingFiles;
	}

        /** Get whether to show console output while processing X3D model
          * @return whether console output is verbose */
	public static boolean isConsoleOutputVerbose()
	{
            return consoleOutputVerbose;
	}
        /** Set whether to show console output while processing X3D model
	 * @see consoleOutputVerbose_DEFAULT
	 * @param newConsoleOutputVerbose whether to show console output while processing X3D model */
	public static void setConsoleOutputVerbose(boolean newConsoleOutputVerbose)
	{
            consoleOutputVerbose = newConsoleOutputVerbose;
	}
}
