/*
Web3D Consortium Open-Source License for Models and Software

Copyright (c) 1995-2025 held by the author(s).  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer
      in the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the Web3D Consortium (https://www.web3D.org)
      nor the names of its contributors may be used to endorse or
      promote products derived from this software without specific
      prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

package org.web3d.x3d.jsail.fields;
import org.web3d.x3d.jsail.*;
import org.web3d.x3d.jsail.Core.*;
import java.util.Arrays;
import org.web3d.x3d.sai.InvalidFieldValueException;
// import org.web3d.x3d.sai.*;


/**
 * This utility class provides a concrete implementation corresponding to MFMatrix3f X3D field type.
 * 
 * <br><br>
 *
 * <i>Warning:</i> this is an abstract interface that cannot be instantiated as a concrete object.
 * Java programmers typically only need to use concrete objects provided by the <code>org.web3d.x3d.jsail</code> classes.

 * <br>
 * <i>Package hint:</i>  This specification class is defined by the X3D Java Language Binding Specification for the Scene Authoring Interface (SAI).
 * MFMatrix3f specifies zero or more 3x3 matrices of single-precision floating point numbers, organized in row-major fashion. Warning: comma characters can only appear between singleton 9-tuple values.
 * <br><br>
 * Related field object: {@link SFMatrix3f}
 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix3f">X3D Tooltips: type MFMatrix3f</a>
 * 
 * @author Don Brutzman and Roy Walmsley
 * @see <a href="https://www.web3d.org/documents/specifications/19777-2/V3.3/Part2/abstracts.html#X3DFieldTypes" target="_blank">SAI Java Specification: B.4.11 X3DFieldTypes</a>
 * @see <a href="https://www.web3d.org/documents/specifications/19775-2/V3.3/Part02/dataRef.html#SAIFieldType" target="blank">SAI Abstract Specification: 5.2.15 SAIFieldType</a>
 * @see <a href="https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/fieldsDef.html#SFMatrix3fAndMFMatrix3f" target="blank">X3D Abstract Specification: SFMatrix3fAndMFMatrix3f</a>
 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html" target="_blank">X3D Tooltips</a>
 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html#field"      target="_blank">X3D Tooltips: field</a>
 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html#fieldValue" target="_blank">X3D Tooltips: fieldValue</a>
 * @see <a href="https://www.web3d.org/x3d/content/examples/X3dSceneAuthoringHints.html" target="_blank">X3D Scene Authoring Hints</a>
 */
public class MFMatrix3f extends X3DConcreteField implements org.web3d.x3d.sai.MFMatrix3f
{
	/** String constant <i>NAME</i> provides name of this element: <i>MFMatrix3f</i> */
	public static final String NAME = "MFMatrix3f";

	/** Default value for this field type is an empty array.
	 * @see <a href="https://docs.oracle.com/javase/tutorial/java/nutsandbolts/datatypes.html" target="_blank">Java Tutorials: Primitive Data Types</a>
     */
	public static final float[] DEFAULT_VALUE = new float[0]; // initialize as empty array

	/** Default string value for this field type is "". */
	public static final String DEFAULT_VALUE_STRING = "";

	/** Whether or not this field type is an array (<i>true</i>)
      * @return true if array type */
	public static final boolean isArray()
    {
        return true;
    }

	/** Default tuple size for this field type is <i>9</i> (i.e. number of component values making up a single-field SF object). */
	public static final int TUPLE_SIZE = 9;

	/** Default regular expression (regex) pattern for this field type is <i>\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){8}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*</i>
	 * @see #PATTERN
	 * @see #matches
	 * @see #matches(String)
	 * @see #validate
	 * @see <a href="https://www.web3d.org/specifications/X3dRegularExpressions.html#MFMatrix3f" target="_blank">X3D Regular Expressions: MFMatrix3f</a>
	 */
	public static final String REGEX = "\\s*((([+-]?((0|[1-9][0-9]*)(\\.[0-9]*)?|\\.[0-9]+)([Ee][+-]?[0-9]+)?)\\s+){8}([+-]?((0|[1-9][0-9]*)(\\.[0-9]*)?|\\.[0-9]+)([Ee][+-]?[0-9]+)?)\\s*,?\\s*)*";

	/** Precompiled regular expression (regex) pattern for this field type using default <em>REGEX</em> value.
	 * @see REGEX
	 * @see #matches
	 * @see #matches(String)
	 * @see #validate
	 * @see <a href="https://docs.oracle.com/javase/8/docs/api/java/util/regex/Pattern.html">java.util.regex.Pattern</a>
	 */
	public static final java.util.regex.Pattern PATTERN = java.util.regex.Pattern.compile(REGEX);

    /** Flag to minimize excessive regex warnings for this field type, helpful if memory limitations are present. */
    private static boolean priorRegexStackOverflowFound = false;

	// Member value declaration is encapsulated and private, using preferred Java types for concretes library
	private float[] MFMatrix3f = java.util.Arrays.copyOf(DEFAULT_VALUE, DEFAULT_VALUE.length); // must be separate copy

	/**
	 * Constructor for MFMatrix3f performs value initialization.
	 */
	public MFMatrix3f()
	{
		initialize();
	}

	/**
	 * Initialization for MFMatrix3f applies default initial value.
	 * Static initializer also provided to verify that the regex pattern compiles and matches that default value.
	 * @see <a href="https://stackoverflow.com/questions/2704857/how-to-check-if-a-given-regex-is-valid">Stack Overflow: How to check if a given Regex is valid?</a>
	 * @see #DEFAULT_VALUE
	 */
	@Override
	public final void initialize()
	{
		MFMatrix3f = java.util.Arrays.copyOf(DEFAULT_VALUE, DEFAULT_VALUE.length); // must be separate copy
	}
	// static initializer
	static {
		try {
            // verify pattern is compiling OK, report exception if not
			java.util.regex.Pattern.compile(REGEX);

			// verify DEFAULT_VALUE matches pattern OK, report if not
			if ((new MFMatrix3f(DEFAULT_VALUE)).matches() == false)
			{
				System.out.println("MFMatrix3f.initialize() problem: failed to match DEFAULT_VALUE=" + java.util.Arrays.toString(DEFAULT_VALUE));
			}
        }
		catch (java.util.regex.PatternSyntaxException exception)
		{
			System.out.println("Exception in MFMatrix3f initialization testing, regex pattern compilation failure.");
			System.out.println("REGEX=\"" + REGEX + "\"");
			System.out.println(exception.getDescription());
        }
	}

	/**
	 * Determine whether two objects contain equal values.
	 * @param comparisonMFMatrix3f field type to compare
	 * @return true if equivalent, false otherwise
	 */
	public boolean equals(MFMatrix3f comparisonMFMatrix3f)
	{
            if (comparisonMFMatrix3f == null)
                 return false;
            else return (getPrimitiveValue() == comparisonMFMatrix3f.getPrimitiveValue());
	}
	/**
	 * Validate current value via get/set comparison tests
	 * @return empty string if get/set testing passes, warning otherwise
	 * @see #validate
	 * @see #validateRegex()
	 */
	public final String validate()
	{
        // validation via get/set checks and comparison
        MFMatrix3f tempHold = new MFMatrix3f(getPrimitiveValue());
        if  (!this.equals(tempHold))
             return "MFMatrix3f validate() error: failed get/set round-trip test";
        else return "";
        
    }
	/**
	 * Validate current value via regular expression (regex) check of current object's toString() value, reporting errors only if found.
	 * @see #REGEX
	 * @see #PATTERN
	 * @see #matches
	 * @see #matches(String)
	 * @see #validate
	 * @see #validate()
	 * @see <a href="https://docs.oracle.com/javase/8/docs/api/java/util/regex/Pattern.html">java.util.regex.Pattern</a>
	 * @return empty string if PATTERN matches, warning otherwise
	 */
	public final String validateRegex()
	{
        try
        {
            java.util.regex.Matcher m = PATTERN.matcher(toString());
            if (!m.matches())
            {
			String errorNotice = "*** [error] Regular expression (regex) failure, new MFMatrix3f PATTERN mismatch (\"" + toString() + "\")";
                return errorNotice;
            }
		}
        catch (java.lang.StackOverflowError soe) // regex processing required too much memory for String version of this attribute
        {
            String validationMessage = "java.lang.StackOverflow warning: regular expression (regex) unable to evaluate large corresponding String value for MFMatrix3f array";
            validationResult.append(validationMessage).append("\n");
            System.out.println ("[exception] " + validationMessage);
            return validationMessage;
		}
        return "";
	}
	/**
	 * Test PATTERN match via regular expression (regex) check of current object's toString() value.
	 * @see #REGEX
	 * @see #PATTERN
	 * @see #matches
	 * @see #matches(String)
	 * @see #validate
	 * @see <a href="https://docs.oracle.com/javase/8/docs/api/java/util/regex/Pattern.html">java.util.regex.Pattern</a>
	 * @return true if PATTERN matches, false otherwise
	 */
	public final boolean matches()
	{
        try
        {
            java.util.regex.Matcher m = PATTERN.matcher(toString());
            return m.matches();
		}
        catch (java.lang.StackOverflowError soe) // regex processing required too much memory for String version of this attribute
        {
            String validationMessage = "java.lang.StackOverflow warning: regular expression (regex) unable to evaluate large corresponding String value for MFMatrix3f array";
            validationResult.append(validationMessage).append("\n");
            System.out.println ("[exception] " + validationMessage);
            return false;
		}
	}
	/**
	 * Test PATTERN match with regular expression (regex) of provided value.
	 * @param value String to check against regex pattern for successful match
	 * @see #REGEX
	 * @see #PATTERN
	 * @see #matches
	 * @see #matches(String)
	 * @see #validate
	 * @see <a href="https://docs.oracle.com/javase/8/docs/api/java/util/regex/Pattern.html">java.util.regex.Pattern</a>
	 * @return true if PATTERN matches, false otherwise
	 */
	public static final boolean matches(String value)
	{
		java.util.regex.Matcher m = PATTERN.matcher(value);
		return m.matches();
	}

	/**
	 * Utility accessor for MFMatrix3f using String value (which must pass parsing validation checks).
	 * <br ><br >
	 * <i>Warning:</i> this method is not
	 * <i><a href="https://en.wikipedia.org/wiki/Type_safety" target="_blank">type safe</a></i>
	 * at compile time!  Best practice for X3DJSAIL programmers is to use strongly typed methods instead, in order to avoid potential for run-time errors.
	 * @see X3DLoaderDOM#toX3dModelInstance(org.w3c.dom.Node)
	 * @param newValue is new value to assign, if empty then assign DEFAULT_VALUE
	 * @return {@link MFMatrix4d} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive
method invocations on the same node object).
	 */
	public MFMatrix3f setValueByString (String newValue) throws InvalidFieldValueException
	{
		if (newValue == null)
			newValue = new String(); // Principle of Least Astonishment (POLA) #1
			// https://en.wikipedia.org/wiki/Principle_of_least_astonishment

		if (!matches(newValue)) // regex test
		{
			String errorNotice = "*** [error] Regular expression (regex) failure, new MFMatrix3f(" + newValue + ")";
			validationResult.append(errorNotice).append("\n");
		}
		if (newValue.trim().isEmpty())
		{
			MFMatrix3f = java.util.Arrays.copyOf(DEFAULT_VALUE, DEFAULT_VALUE.length); // must be separate copy
			return this;
		}
		try
		{
			String[] newValues = newValue.replace(","," ").trim().split("\\s+");
			if ((newValues.length % 9) != 0) // tupleSize modulus check
			{
				String errorNotice = "*** illegal number of values (" + newValues.length + ")" +
					" in initialization string, must be multiple of 9 when declaring new MFMatrix3f(" + newValue + ")";
				validationResult.append(errorNotice).append("\n");
				throw new InvalidFieldValueException(errorNotice);
			}
			MFMatrix3f = new float[newValues.length];  // reset array size to match
			for (int i = 0; i < newValues.length;i++)
			{
				MFMatrix3f[i] = Float.parseFloat(newValues[i]); // insert in array
			}
		}
		catch (NumberFormatException e)
		{
			String errorNotice = "*** new MFMatrix3f(" + newValue + ") " + e.getMessage();
			validationResult.append(errorNotice).append("\n");
			throw new InvalidFieldValueException (errorNotice);
		}
		return this;
	}

	/**
	 * Constructor to copy an MFMatrix3f value as initial value for this new field object.
	 * @param newValue The newValue to apply
	 */
	public MFMatrix3f(MFMatrix3f newValue)
	{
		if (newValue == null)
		{
			MFMatrix3f = java.util.Arrays.copyOf(DEFAULT_VALUE, DEFAULT_VALUE.length); // must be separate copy
			return;
		}
		MFMatrix3f = newValue.getPrimitiveValue();
	}

	/**
	 * Constructor for MFMatrix3f using a corresponding Java primitive float[] array as new initial value.
	 * @param newValue is new value to assign
	 * <i>Warning:</i> newValue array length must correspond to tuple size for base type MFMatrix3f tuple size of <i>9</i>.
	 * setContainerFieldOverride(containerFieldName); // apply checksConcreteField#getTupleSize(String)
	 */
	public MFMatrix3f (float[] newValue)
	{
		if      (newValue == null)
			     newValue = java.util.Arrays.copyOf(DEFAULT_VALUE, DEFAULT_VALUE.length); // must be separate copy
		else if (newValue.length % 9 != 0) // 9-tuple check
		{
			String errorNotice = "*** Illegal MFMatrix3f newValue array length=" + newValue.length +
				", must be multiple of 9 or else be empty (newValue=" + toString(newValue) + ")";
			validationResult.append(errorNotice).append("\n");
			throw new InvalidFieldValueException(errorNotice);
		}
		
		MFMatrix3f = newValue;
	}

	/**
	 * Utility constructor for MFMatrix3f to assign a single SFMatrix3f as new initial value.
	 * @param newValue is new value to assign
	 */
	public MFMatrix3f (SFMatrix3f newValue)
	{
		setValue(newValue);
	}

	/**
	 * Utility method for MFMatrix3f to assign a single SFMatrix3f as new initial value.
	 * @param newValue is new value to assign
	 * @return {@link SFColor} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public MFMatrix3f setValue(SFMatrix3f newValue)
	{
		setValue(newValue.getPrimitiveValue());
        return this;
	}

	/**
	 * Static utility method to provide String representation of a correctly typed input value.
	 * @param value The value to convert to a String
	 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html#type">X3D Tooltips: type</a>
	 * @see SFFloat#stripTrailingZeroes(float)
	 * @return String version of the provided value, with trailing zeroes and decimal points omitted.
	 */
	public static String toString (float[] value)
	{
		StringBuilder result = new StringBuilder();
		for (int i=0; i < value.length; i++)
		{
			if  (ConfigurationProperties.isStripTrailingZeroes())
				 result.append(org.web3d.x3d.jsail.fields.SFFloat.stripTrailingZeroes(value[i])).append(" ");
			else result.append(value[i]).append(" ");
		}
		return result.toString().trim();
	}

	/**
	 * Provides current value of the field as a Java primitive type.
	 * @return current value
	 */
	public float[] getPrimitiveValue()
	{
		return MFMatrix3f;
	}
	/**
	 * Provides current value as a String.
	 * @see <a href="https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix3f">X3D Tooltips: type MFMatrix3f</a>
	 * @see org.web3d.x3d.jsail.fields.SFFloat#stripTrailingZeroes(float)
	 * @return String version of the provided value, with trailing zeroes and decimal points omitted.
	 */
	@Override
	public String toString()
	{
		StringBuilder result = new StringBuilder();
		for (int i = 0; i < MFMatrix3f.length; i++)
		{
			if  (ConfigurationProperties.isStripTrailingZeroes())
				 result.append(org.web3d.x3d.jsail.fields.SFFloat.stripTrailingZeroes(MFMatrix3f[i]));
			else result.append(MFMatrix3f[i]);
			if (i < MFMatrix3f.length - 1)
				result.append(" ");
		}
		return result.toString();
	}
	/**
	 * Apply an array of primitive values to this field.
	 * @param newValue The newValue to apply
	 * @return {@link MFMatrix3f} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public MFMatrix3f setValue(float[] newValue)
	{
		if (newValue == null)
		{
			MFMatrix3f = java.util.Arrays.copyOf(DEFAULT_VALUE, DEFAULT_VALUE.length); // must be separate copy
			return this;
		}
        if ((newValue.length % 9) != 0) // modulo, must be a multiple of SFMatrix3 size
		{
			String errorNotice = "*** Illegal array newValue=" + newValue + " must have base length 9 for setValue() on MFMatrix3f";
			validationResult.append(errorNotice).append("\n");
			throw new InvalidFieldValueException(errorNotice);
		}
		MFMatrix3f = newValue;
		return this;
	}
	/* @Override */
	public void setIdentity()
	{
		MFMatrix3f = org.web3d.x3d.jsail.fields.SFMatrix3f.DEFAULT_VALUE;
	}

	/* @Override */
	public void set(int row, int column)
	{
		// TODO, problem with interface signature
	}

	/* @Override */
	public float get(int row, int column)
	{
		if ((row < 0) || (row > 4) || (column < 0) || (column > 4))
		{
			String errorNotice = "*** Illegal MFMatrix3f[row=" + row + ",column=" + column + "] access, " +
				"all values must be in numeric range [0..4]";
			validationResult.append(errorNotice).append("\n");
			throw new InvalidFieldValueException(errorNotice);
		}
		return MFMatrix3f[3*row + column]; // TODO typing
	}

	/* @Override */
	public void setTransform(org.web3d.x3d.sai.SFVec3f translation,
		org.web3d.x3d.sai.SFRotation rotation,
		org.web3d.x3d.sai.SFVec3f scale,
		org.web3d.x3d.sai.SFRotation scaleOrientation,
		org.web3d.x3d.sai.SFVec3f center)
	{
		// TODO
	}

	/* @Override */
	public void getTransform(org.web3d.x3d.sai.SFVec3f translation,
		org.web3d.x3d.sai.SFRotation rotation,
		org.web3d.x3d.sai.SFVec3f scale)
	{
		// TODO
	}

	/* @Override */
	public org.web3d.x3d.sai.Matrix3 inverse()
	{
		return null; // TODO
	}

	/* @Override */
	public org.web3d.x3d.sai.Matrix3 transpose()
	{
		return null; // TODO
	}

	/* @Override */
	public org.web3d.x3d.sai.Matrix3 multiplyLeft (org.web3d.x3d.sai.Matrix3 matrix3x3)
	{
		return null; // TODO
	}

	/* @Override */
	public org.web3d.x3d.sai.Matrix3 multiplyRight(org.web3d.x3d.sai.Matrix3 matrix3x3)
	{
		return null; // TODO
	}

	/* @Override */
	public org.web3d.x3d.sai.Matrix3 multiplyRowVector(org.web3d.x3d.sai.SFVec3f vector3f) // TODO SFVec4f ?
	{
		return null; // TODO
	}

	/* @Override */
	public org.web3d.x3d.sai.Matrix3 multiplyColVector(org.web3d.x3d.sai.SFVec3f vector3f) // TODO SFVec4f ?
	{
		return null; // TODO
	}

	/**
	 * Provide float array for this field type.
	 *
	 * @return Array of floats in this field array.
	 */
	public float[] toFloatArray()
	{
		return MFMatrix3f;
	}

	/**
	 * Apply an MFMatrix3f value to this field.
	 * @param newValue The newValue to apply
	 * @return {@link MFMatrix3f} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public MFMatrix3f setValue(MFMatrix3f newValue)
	{
		if (newValue == null)
		{
			MFMatrix3f = java.util.Arrays.copyOf(DEFAULT_VALUE, DEFAULT_VALUE.length); // must be separate copy
			return this;
		}
		MFMatrix3f = newValue.getPrimitiveValue();
		return this;
	}

	/**
	 * Multiply scaleFactor times all values in this field type.
	 * @param scaleFactor scalar value for vector multiplication
	 * @return {@link MFMatrix3f} - namely <i>this</i> same object to allow sequential method pipelining (i.e. consecutive method invocations on the same object).
	 */
	public MFMatrix3f multiply(float scaleFactor)
	{
		for (int i = 0; i < MFMatrix3f.length; i++)
		{
			MFMatrix3f[i] *= scaleFactor;
		}
		return this;
	}

	/**
	 * Determine whether current value matches DEFAULT_VALUE
	 * @see #DEFAULT_VALUE
	 * @return whether current value matches DEFAULT_VALUE
	 */
    public boolean isDefaultValue()
    {
        return java.util.Arrays.equals(MFMatrix3f, DEFAULT_VALUE);
    }
}
