####################################################################################################
#
# Invoking X3D model self-test:
#
#   $ python TestMetadataNodes.py
#
# Python package x3d.py package is available on PyPI for import.
#   This approach simplifies Python X3D deployment and use.
#   https://pypi.org/project/x3d
#
# Installation:
#       pip install x3d
# or
#       python -m pip install x3d
#
# Developer options for loading x3d package in other Python programs:
#
#    from x3d import *  # preferred approach, terser source that avoids x3d.* class prefixes
#
# or
#    import x3d         # traditional way to subclass x3d package, all classes require x3d.* prefix,
#                       # but python source is very verbose, for example x3d.Material x3d.Shape etc.
#                       # X3dToPython.xslt stylesheet insertPackagePrefix=true supports this option.
#
####################################################################################################

from x3d import *

newModel=X3D(profile='Interchange',version='4.0',
  head=head(
    children=[
    meta(content='TestMetadataNodes.x3d',name='title'),
    meta(content='Test parentage and type checking of MetadataNodes using X3D4 defaults.',name='description'),
    meta(content='Don Brutzman',name='creator'),
    meta(content='12 March 2005',name='created'),
    meta(content='18 May 2024',name='modified'),
    meta(content='https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/components/core.html#NodeReference',name='reference'),
    meta(content='https://www.web3d.org/x3d/content/X3dTooltips.html#MetadataDouble',name='reference'),
    meta(content='metadata nodes test',name='subject'),
    meta(content='MetadataString node within NavigationInfo is legal X3D but will cause error when translating to VRML97 due to (previously) disallowed parent node.',name='warning'),
    meta(content='https://www.web3d.org/x3d/content/examples/Basic/development/TestMetadataNodes.x3d',name='identifier'),
    meta(content='X3D-Edit 4.0, https://savage.nps.edu/X3D-Edit',name='generator'),
    meta(content='../license.html',name='license')]),
  Scene=Scene(
    children=[
    WorldInfo(title='TestMetadataNodes.x3d'),
    Background(skyColor=[(0.5,0.5,1)]),
    Anchor(description='Click to see documentation example',url=["TestMetadataNodesIndex.html","https://www.web3d.org/x3d/content/examples/Basic/development/TestMetadataNodesIndex.html"],
      children=[
      Shape(
        appearance=Appearance(
          material=Material(diffuseColor=(0.8,0.9,0.9))),
        geometry=Text(length=[0],string=["TestMetadataNodes.x3d","shows X3D4 Metadata nodes"],
          fontStyle=FontStyle(DEF='TextFont',justify=["MIDDLE","MIDDLE"],style_='BOLD')))]),
    Group(
      metadata=MetadataSet(DEF='M0',name='TestSet',
        value=[
        MetadataBoolean(DEF='M1',value=[True,False,True]),
        MetadataString(DEF='M2',value=["internal","metadata"]),
        MetadataDouble(DEF='M3',value=[1.0]),
        #  intermediate comment should not be sorted 
        MetadataFloat(DEF='M4',value=[1.0]),
        MetadataInteger(DEF='M5',value=[1]),
        MetadataSet(DEF='M6',
          #  presence of comment will suppress warning about no MetadataSet children found 
          )])),
    NavigationInfo(
      metadata=MetadataString(name='Test illegal parent under VRML97')),
    #  Metadata nodes are allowed as root nodes in X3D4, but must define containerField='metadata' 
    #  https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/concepts.html#Rootnodes 
    MetadataSet(USE='M0'),
    MetadataBoolean(USE='M1'),
    MetadataString(USE='M2'),
    MetadataDouble(USE='M3'),
    MetadataFloat(USE='M4'),
    MetadataInteger(USE='M5'),
    MetadataSet(USE='M6')]
    #  Test case: Metadata nodes are only allowed to contain other metadata nodes, test fragment follows 
    #  <MetadataSet DEF='ErroneousChild'> <DirectionalLight/> </MetadataSet> 
    )
) # X3D model complete

####################################################################################################
# Self-test diagnostics
####################################################################################################

print('Self-test diagnostics for TestMetadataNodes.py:')
if        metaDiagnostics(newModel): # built-in utility method in X3D class
    print(metaDiagnostics(newModel)) # display meta info, hint, warning, error, TODO values in this model
# print('check newModel.XML() serialization...')
newModelXML= newModel.XML() # test export method XML() for exceptions during export
newModel.XMLvalidate()
# print(newModelXML) # diagnostic

try:
#   print('check newModel.VRML() serialization...')
    newModelVRML=newModel.VRML() # test export method VRML() for exceptions during export
    # print(prependLineNumbers(newModelVRML)) # debug
    print("Python-to-VRML export of VRML output successful", flush=True)
except Exception as err: # usually BaseException
    # https://stackoverflow.com/questions/18176602/how-to-get-the-name-of-an-exception-that-was-caught-in-python
    print("*** Python-to-VRML export of VRML output failed:", type(err).__name__, err)
    if newModelVRML: # may have failed to generate
        print(prependLineNumbers(newModelVRML, err.lineno))

try:
#   print('check newModel.JSON() serialization...')
    newModelJSON=newModel.JSON() # test export method JSON() for exceptions during export
#   print(prependLineNumbers(newModelJSON)) # debug
    print("Python-to-JSON export of JSON output successful (under development)")
except Exception as err: # usually SyntaxError
    print("*** Python-to-JSON export of JSON output failed:", type(err).__name__, err)
    if newModelJSON: # may have failed to generate
        print(prependLineNumbers(newModelJSON,err.lineno))

print("python TestMetadataNodes.py load and self-test diagnostics complete.")
