####################################################################################################
#
# Invoking X3D model self-test:
#
#   $ python RenderingComponentPrototypes.py
#
# Python package x3d.py package is available on PyPI for import.
#   This approach simplifies Python X3D deployment and use.
#   https://pypi.org/project/x3d
#
# Installation:
#       pip install x3d
# or
#       python -m pip install x3d
#
# Developer options for loading x3d package in other Python programs:
#
#    from x3d import *  # preferred approach, terser source that avoids x3d.* class prefixes
#
# or
#    import x3d         # traditional way to subclass x3d package, all classes require x3d.* prefix,
#                       # but python source is very verbose, for example x3d.Material x3d.Shape etc.
#                       # X3dToPython.xslt stylesheet insertPackagePrefix=true supports this option.
#
####################################################################################################

from x3d import *

newModel=X3D(profile='Immersive',version='3.0',
  head=head(
    children=[
    meta(content='RenderingComponentPrototypes.x3d',name='title'),
    meta(content='Prototype implementations of X3D Rendering component nodes (IndexedTriangleFanSet IndexedTriangleSet IndexedTriangleStripSet LineSet TriangleFanSet TriangleSet TriangleStripSet) implemented as prototypes for backwards compatibility with VRML 97.',name='description'),
    meta(content='X3dToVrml97.xslt translation stylesheet automatically invokes these prototypes upon encountering any new Rendering nodes.',name='info'),
    meta(content='Don Brutzman, George Dabrowski, Ken Curtin, Duane Davis, Christos Kalogrias',name='creator'),
    meta(content='17 November 2003',name='created'),
    meta(content='9 October 2023',name='modified'),
    meta(content='RenderingComponentExamples.x3d',name='reference'),
    meta(content='RenderingComponentExternProtoDefinitions.x3d',name='reference'),
    meta(content='https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/components/rendering.html',name='reference'),
    meta(content='https://www.web3d.org/x3d/content/examples/Vrml2Sourcebook/Chapter13-PointsLinesFaces/Figure13.11IndexedLineSetBoxWireframe.x3d',name='reference'),
    meta(content='X3D Rendering component nodes (IndexedTriangleFanSet IndexedTriangleSet IndexedTriangleStripSet LineSet TriangleFanSet TriangleSet TriangleStripSet)',name='subject'),
    meta(content='https://www.web3d.org/x3d/content/examples/Basic/development/RenderingComponentPrototypes.x3d',name='identifier'),
    meta(content='X3D-Edit 4.0, https://savage.nps.edu/X3D-Edit',name='generator'),
    meta(content='../license.html',name='license')]),
  Scene=Scene(
    #  ==================== 
    children=[
    WorldInfo(title='RenderingComponentPrototypes.x3d'),
    ProtoDeclare(appinfo='ColorRGBA defines a set of RGBA colors. Warning: VRML 97 support does not include alpha values.',name='ColorRGBA',
      ProtoInterface=ProtoInterface(
        field=[
        field(accessType='inputOutput',appinfo='locally override MFColorRGBA type (which is not supported in VRML 97) in order to downgrade to Color RGB',name='color',type='MFRotation',
          #  The color field is a 4-tuple float array, and so we map it to an MFOrientation for backwards compatibility with VRML 97. 
          ),
        field(accessType='inputOutput',appinfo='Metadata node only',name='metadata',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          )]),
      ProtoBody=ProtoBody(
        children=[
        Color(DEF='ColorRGB',color=[(0.8,0.8,0.8)]),
        #  remaining nodes are not rendered 
        OrientationInterpolator(DEF='ColorRGBAholder',
          IS=IS(
            connect=[
            connect(nodeField='keyValue',protoField='color')])),
        Script(DEF='ConvertColorRGBAtoRGB',directOutput=True,
          field=[
          field(accessType='initializeOnly',name='colorRGBAnode',type='SFNode',
            children=[
            OrientationInterpolator(USE='ColorRGBAholder')]),
          field(accessType='initializeOnly',name='colorRGBnode',type='SFNode',
            children=[
            Color(USE='ColorRGB')])],

        sourceCode="""
ecmascript:

function initialize () 
{
//	Browser.println ('colorRGBAnode.keyValue.length=' + colorRGBAnode.keyValue.length);
	for (i=0; i<=colorRGBAnode.keyValue.length-1; i++)
	{
		// type conversion of each array element
		// specifically, colorRGBAnode.keyValue[i] is an SFRotation
		// and individual element values are then extracted from that
		nextColor = new SFColor (
			colorRGBAnode.keyValue[i].x,
			colorRGBAnode.keyValue[i].y,
			colorRGBAnode.keyValue[i].z);
		// note colorRGBAnode.keyValue[i].angle holds the alpha value; ignored 
//		Browser.println ('color[' + i + ']=' + nextColor);
		colorRGBnode.color[i] = nextColor;
	}
}
"""),
        Group(
          metadata=MetadataSet(
            IS=IS(
              connect=[
              connect(nodeField='metadata',protoField='metadata')])))])),
    #  ==================== 
    ProtoDeclare(appinfo='IndexedTriangleFanSet represents a 3D shape composed of triangles that form a fan shape around the first vertex declared in each fan.',name='IndexedTriangleFanSet',
      ProtoInterface=ProtoInterface(
        field=[
        field(accessType='initializeOnly',name='ccw',type='SFBool',value=True),
        field(accessType='initializeOnly',name='colorPerVertex',type='SFBool',value=True),
        field(accessType='initializeOnly',name='normalPerVertex',type='SFBool',value=True),
        field(accessType='initializeOnly',name='solid',type='SFBool',value=True),
        field(accessType='inputOnly',appinfo='[0 infinity] or -1',name='set_index',type='MFInt32'),
        field(accessType='initializeOnly',appinfo='[0 infinity] or -1',name='index',type='MFInt32',
          #  default initialization is NULL array [] to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Color ColorRGBA node only',name='color',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Coordinate node only',name='coord',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Metadata node only',name='metadata',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Normal node only',name='normal',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='TextureCoordinate node only',name='texCoord',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          )]),
      ProtoBody=ProtoBody(
        children=[
        IndexedFaceSet(DEF='RenderedITFS',
          IS=IS(
            connect=[
            connect(nodeField='ccw',protoField='ccw'),
            connect(nodeField='normalPerVertex',protoField='normalPerVertex'),
            connect(nodeField='solid',protoField='solid'),
            connect(nodeField='color',protoField='color'),
            connect(nodeField='coord',protoField='coord'),
            connect(nodeField='normal',protoField='normal'),
            connect(nodeField='texCoord',protoField='texCoord')])),
        Group(DEF='UnrenderedITFS',
          children=[
          Shape(
            #  is this really needed at all?? 
            geometry=IndexedFaceSet(DEF='NodesHolderITFS',
              IS=IS(
                connect=[
                connect(nodeField='color',protoField='color'),
                connect(nodeField='coord',protoField='coord'),
                connect(nodeField='normal',protoField='normal'),
                connect(nodeField='texCoord',protoField='texCoord')]))),
          Script(DEF='IndexedTriangleFanSetToIndexedFaceSet',directOutput=True,
            field=[
            field(accessType='initializeOnly',name='index',type='MFInt32'),
            field(accessType='inputOnly',name='set_index',type='MFInt32'),
            field(accessType='initializeOnly',name='renderedITFS',type='SFNode',
              children=[
              IndexedFaceSet(USE='RenderedITFS')]),
            field(accessType='initializeOnly',name='nodesHolder',type='SFNode',
              children=[
              IndexedFaceSet(USE='NodesHolderITFS')]),
            field(accessType='initializeOnly',name='localTraceEnabled',type='SFBool',value=True),
            field(accessType='initializeOnly',name='coordIndexNew',type='MFInt32',
              #  constructed during initialization 
              )],
            IS=IS(
              connect=[
              connect(nodeField='index',protoField='index'),
              connect(nodeField='set_index',protoField='set_index')]),

          sourceCode="""
ecmascript:

function initialize()
{
	// index is an array of triangle indices that form a fan shape
	// around the first vertex declared in each fan. The ordering of
	// the vertices is ccw (counter-clockwise).

	// ensure terminated by -1
	if (index[index.length-1] != -1) index[index.length] = -1;

	// ensure legal index values
	for (i=0; i <= index.length-1; i++)
	{
	  if (index[i] < -1)
	  {
		alwaysPrint ('error, index[' + i + ']=' + index[i] +
' is illegal value, treated as -1');
		index[i] = -1;
	  }
	}
	tracePrint ('index.length=' + index.length);
	tracePrint ('index=' + index);

	if (index.length < 4) 
	{
		alwaysPrint ('warning, index.length=' + index.length + 
' insufficient to construct a triangle, ITFS ignored');
		return;
	}
	j = 0; // coordIndexNew counter
	coordIndexNew = new MFInt32 ();

	// i walks through index array,
	// goal is to initialize coordIndexNew list to match triangles
	for (i=2; i <= index.length-1; i++)
	{
		if ((index[i] == index[i-1]) || (index[i] == index[i-2]) || (index[i-1] == index[i-2]))
		{
			alwaysPrint ('index=' + index);
			alwaysPrint ('error, pair of equal indices in triangle');
			return;
		}
	  if (index[i] >= 0) 
	  {
		// add another triangle from latest 3 points of fan set to IFS
		// swap order to ensure normal is ccw, i.e. in correct halfplane direction
		coordIndexNew [coordIndexNew.length] = 0;
		coordIndexNew [coordIndexNew.length] = index[i];
		coordIndexNew [coordIndexNew.length] = index[i-1];
		coordIndexNew [coordIndexNew.length] = -1; // terminate
	  }
	  else if (index[i] == -1) // finish current triangle, fan
	  {
		// ensure done, or sufficient points remain to build another triangle
		if (	(i!=index.length-1) && (index.length - i < 2))
		{
			alwaysPrint ('index=' + index);
			alwaysPrint ('error, insufficient index values after' +
	'index[' + i + ']=-1');
			return;
		}
		// ensure done, or enough legal index values remain to build another triangle
		if (	(i!=index.length-1) &&
			((index[i+1] == -1) || (index[i+2] == -1) || (index[i+3] == -1)))
		{
			alwaysPrint ('index=' + index);
			alwaysPrint ('error, insufficient non-negative-one index values after' +
	'index[' + i + ']=-1');
			return;
		}
		tracePrint ('encountered -1 in index array');
		// skip ahead to build next fan set, no effect if done
		if (i!=index.length-1) i = i + 2; 
	  }
	  // incremental trace of array being built
	  tracePrint ('coordIndexNew=' + coordIndexNew);
	}
	renderedITFS.set_coordIndex = coordIndexNew;
	tracePrint ('renderedITFS.coordIndex=' + renderedITFS.coordIndex);
	// match colorIndex if any Color node exists
	if (nodesHolder.color)
	{
	  if (nodesHolder.color.color.length > 0)
	  {
		renderedITFS.set_colorIndex = coordIndexNew;
		tracePrint ('set_colorIndex=' + coordIndexNew);
	  }
	}
}
function set_index (value, timestamp)
{
	index = value;
	initialize ();
}
function tracePrint(outputString)
{
    if (localTraceEnabled)
	Browser.println ('[IndexedTriangleFanSet]' + outputString);
}
function alwaysPrint(outputString)
{
	Browser.println ('[IndexedTriangleFanSet]' + outputString);
}
"""),
          Group(
            metadata=MetadataString(
              IS=IS(
                connect=[
                connect(nodeField='metadata',protoField='metadata')])))])])),
    #  ==================== 
    ProtoDeclare(appinfo='IndexedTriangleSet represents a 3D shape composed of a collection of individual triangles.',name='IndexedTriangleSet',
      ProtoInterface=ProtoInterface(
        field=[
        field(accessType='initializeOnly',name='ccw',type='SFBool',value=True),
        field(accessType='initializeOnly',name='colorPerVertex',type='SFBool',value=True),
        field(accessType='initializeOnly',name='normalPerVertex',type='SFBool',value=True),
        field(accessType='initializeOnly',name='solid',type='SFBool',value=True),
        field(accessType='inputOnly',appinfo='[0 infinity] or -1',name='set_index',type='MFInt32'),
        field(accessType='initializeOnly',appinfo='[0 infinity] or -1',name='index',type='MFInt32',
          #  default initialization is NULL array [] to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Color ColorRGBA node only',name='color',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Coordinate node only',name='coord',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Normal node only',name='normal',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='TextureCoordinate node only',name='texCoord',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Metadata node only',name='metadata',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          )]),
      ProtoBody=ProtoBody(
        children=[
        IndexedFaceSet(DEF='RenderedITS',
          IS=IS(
            connect=[
            connect(nodeField='ccw',protoField='ccw'),
            connect(nodeField='normalPerVertex',protoField='normalPerVertex'),
            connect(nodeField='solid',protoField='solid'),
            connect(nodeField='color',protoField='color'),
            connect(nodeField='coord',protoField='coord'),
            connect(nodeField='normal',protoField='normal'),
            connect(nodeField='texCoord',protoField='texCoord')])),
        Group(DEF='UnrenderedITS',
          children=[
          Shape(
            #  is this really needed at all?? 
            geometry=IndexedFaceSet(DEF='NodesHolderITS',
              IS=IS(
                connect=[
                connect(nodeField='color',protoField='color'),
                connect(nodeField='coord',protoField='coord'),
                connect(nodeField='normal',protoField='normal'),
                connect(nodeField='texCoord',protoField='texCoord')]))),
          Script(DEF='IndexedTriangleSetToIndexedFaceSet',directOutput=True,
            field=[
            field(accessType='initializeOnly',name='index',type='MFInt32'),
            field(accessType='inputOnly',name='set_index',type='MFInt32'),
            field(accessType='initializeOnly',name='renderedITS',type='SFNode',
              children=[
              IndexedFaceSet(USE='RenderedITS')]),
            field(accessType='initializeOnly',name='nodesHolder',type='SFNode',
              children=[
              IndexedFaceSet(USE='NodesHolderITS')]),
            field(accessType='initializeOnly',name='localTraceEnabled',type='SFBool',value=True),
            field(accessType='initializeOnly',name='coordIndexNew',type='MFInt32',
              #  constructed during initialization 
              )],
            IS=IS(
              connect=[
              connect(nodeField='index',protoField='index'),
              connect(nodeField='set_index',protoField='set_index')]),

          sourceCode="""
ecmascript:

function initialize()
{
	// index is an array of triangle indices. The ordering of
	// the vertices is ccw (counter-clockwise).

	// ensure legal index values
	for (ii=0; ii <= index.length-1; ii++)
	{
	  if (index[ii] < -1)
	  {
		alwaysPrint ('error, index[' + ii + ']=' + index[ii] +
' is illegal value');
		return;
	  }
	}
	tracePrint ('index.length=' + index.length);
	tracePrint ('index=' + index);

	if (index.length < 3) 
	{
		alwaysPrint ('warning, index.length=' + index.length + 
' insufficient to construct a triangle, ITS ignored');
		return;
	}

	coordIndexNew = new MFInt32 ();

	// ii walks through index array,
	// goal is to initialize coordIndexNew list to match triangles
	for (ii=0; ii <= index.length-1; ii+=3)
	{
		if ((index[ii] == index[ii+1]) || (index[ii] == index[ii+2]) || (index[ii+1] == index[ii+2]))
		{
			alwaysPrint ('index=' + index);
			alwaysPrint ('error, pair of equal indices in triangle');
			return;
		}
	  
		if (index[ii] >= 0) 
		{
			// add another triangle from latest 3 points of fan set to ITS
			// order is ccw, i.e. in correct halfplane direction
			coordIndexNew [coordIndexNew.length] = index[ii];
			coordIndexNew [coordIndexNew.length] = index[ii+1];
			coordIndexNew [coordIndexNew.length] = index[ii+2];
			coordIndexNew [coordIndexNew.length] = -1;
		}

		if (index.length % 3 != 0) {
			alwaysPrint ('error, index field does not contain a multiple' +
		'of three coordinate values.');
			alwaysPrint ('The remaining vertices shall be ignored');
			return;
		}


		// ensure done, or sufficient points remain to build another triangle
//		if (	(i!=index.length-1) && (index.length - i < 2))
//		{
//			alwaysPrint ('index=' + index);
//			alwaysPrint ('error, insufficient index values after' +
//	'index[' + i + ']=-1');
//			return;
//		}
		// ensure done, or enough legal index values remain to build another triangle
//		if (	(i!=index.length-1) &&
//			((index[i+1] == -1) || (index[i+2] == -1) || (index[i+3] == -1)))
//		{
//			alwaysPrint ('index=' + index);
//			alwaysPrint ('error, insufficient non-negative-one index values after' +
//	'index[' + i + ']=-1');
//			return;
//		}
	  // incremental trace of array being built
	  tracePrint ('coordIndexNew=' + coordIndexNew);
	}
	renderedITS.coordIndex = coordIndexNew;
	tracePrint ('renderedITS.coordIndex=' + renderedITS.coordIndex);

	// match colorIndex if any Color node exists
	if (nodesHolder.color)
	{
	  if (nodesHolder.color.color.length > 0)
	  {
		renderedITS.set_colorIndex = coordIndexNew;
		tracePrint ('set_colorIndex=' + coordIndexNew);
	  }
	}
}
function set_index (value, timestamp)
{
	index = value;
	initialize ();
}
function tracePrint(outputString)
{
    if (localTraceEnabled)
	  Browser.println ('[IndexedTriangleSet]' + outputString);
}
function alwaysPrint(outputString)
{
	Browser.println ('[IndexedTriangleSet]' + outputString);
}
"""),
          Group(
            metadata=MetadataString(
              IS=IS(
                connect=[
                connect(nodeField='metadata',protoField='metadata')])))])])),
    #  ==================== 
    ProtoDeclare(appinfo='IndexedTriangleStripSet represents a 3D shape composed of strips of triangles.',name='IndexedTriangleStripSet',
      ProtoInterface=ProtoInterface(
        field=[
        field(accessType='initializeOnly',name='ccw',type='SFBool',value=True),
        field(accessType='initializeOnly',name='colorPerVertex',type='SFBool',value=True),
        field(accessType='initializeOnly',name='normalPerVertex',type='SFBool',value=True),
        field(accessType='initializeOnly',name='solid',type='SFBool',value=True),
        field(accessType='inputOnly',name='set_index',type='MFInt32'),
        field(accessType='initializeOnly',name='index',type='MFInt32',
          #  default initialization is NULL array [] to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Color ColorRGBA node only',name='color',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Coordinate node only',name='coord',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Normal node only',name='normal',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='TextureCoordinate node only',name='texCoord',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Metadata node only',name='metadata',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          )]),
      ProtoBody=ProtoBody(
        children=[
        IndexedFaceSet(DEF='RenderedTSS',
          IS=IS(
            connect=[
            connect(nodeField='ccw',protoField='ccw'),
            connect(nodeField='normalPerVertex',protoField='normalPerVertex'),
            connect(nodeField='solid',protoField='solid'),
            connect(nodeField='color',protoField='color'),
            connect(nodeField='coord',protoField='coord'),
            connect(nodeField='normal',protoField='normal'),
            connect(nodeField='texCoord',protoField='texCoord')])),
        Group(
          children=[
          Shape(
            #  is this really needed at all?? 
            geometry=IndexedFaceSet(DEF='NodesHolderTSS',
              IS=IS(
                connect=[
                connect(nodeField='color',protoField='color'),
                connect(nodeField='coord',protoField='coord'),
                connect(nodeField='normal',protoField='normal'),
                connect(nodeField='texCoord',protoField='texCoord')]))),
          Script(DEF='IndexedTriangleStripSetToIndexedFaceSet',directOutput=True,
            field=[
            field(accessType='initializeOnly',name='index',type='MFInt32'),
            field(accessType='inputOnly',name='set_index',type='MFInt32'),
            field(accessType='initializeOnly',name='renderedTSS',type='SFNode',
              children=[
              IndexedFaceSet(USE='RenderedTSS')]),
            field(accessType='initializeOnly',name='nodesHolder',type='SFNode',
              children=[
              IndexedFaceSet(USE='NodesHolderTSS')]),
            field(accessType='initializeOnly',name='localTraceEnabled',type='SFBool',value=True),
            field(accessType='initializeOnly',name='coordIndexNew',type='MFInt32',
              #  constructed during initialization 
              )],
            IS=IS(
              connect=[
              connect(nodeField='index',protoField='index'),
              connect(nodeField='set_index',protoField='set_index')]),

          sourceCode="""
ecmascript:

function initialize()
{
	// index is an array of triangle indices. The ordering of
	// the vertices is ccw (counter-clockwise).

	// ensure legal index values
	for (ii=0; ii <= index.length-1; ii++)
	{
	  if (index[ii] < -1)
	  {
		alwaysPrint ('error, index[' + ii + ']=' + index[ii] +
' is illegal value');
		return;
	  }
	}
	tracePrint ('index.length=' + index.length);
	tracePrint ('index=' + index);

	if (index.length < 3) 
	{
		alwaysPrint ('warning, index.length=' + index.length + 
' insufficient to construct a triangle, ITS ignored');
		return;
	}

	coordIndexNew = new MFInt32 ();

	// ii walks through index array,
	// goal is to initialize coordIndexNew list to match triangles
	for (ii=2; ii <= index.length-1; ii++)
	{
		if ((index[ii] == index[ii-1]) || (index[ii] == index[ii-2]) || (index[ii-1] == index[ii-2]))
		{
			alwaysPrint ('index=' + index);
			alwaysPrint ('error, pair of equal indices in triangle');
			return;
		}
	  
		if (index[ii] >= 0) 
	  {
			// add another triangle from latest 3 points of fan set to ITS
			// order is ccw, i.e. in correct halfplane direction
			coordIndexNew [coordIndexNew.length] = index[ii-2];
			coordIndexNew [coordIndexNew.length] = index[ii-1];
			coordIndexNew [coordIndexNew.length] = index[ii];
			coordIndexNew [coordIndexNew.length] = -1;
	  }

		// ensure done, or sufficient points remain to build another triangle
//		if (	(i!=index.length-1) && (index.length - i < 2))
//		{
//			alwaysPrint ('index=' + index);
//			alwaysPrint ('error, insufficient index values after' +
//	'index[' + i + ']=-1');
//			return;
//		}
		// ensure done, or enough legal index values remain to build another triangle
//		if (	(i!=index.length-1) &&
//			((index[i+1] == -1) || (index[i+2] == -1) || (index[i+3] == -1)))
//		{
//			alwaysPrint ('index=' + index);
//			alwaysPrint ('error, insufficient non-negative-one index values after' +
//	'index[' + i + ']=-1');
//			return;
//		}
	  // incremental trace of array being built
	  tracePrint ('TSScoordIndexNew=' + coordIndexNew);
	}
	renderedTSS.set_coordIndex = coordIndexNew;
	tracePrint ('renderedTSS.coordIndex=' + renderedTSS.coordIndex);
	// match colorIndex if any Color node exists
	if (nodesHolder.color)
	{
	  if (nodesHolder.color.color.length > 0)
	  {
		renderedTSS.set_colorIndex = coordIndexNew;
		tracePrint ('set_colorIndex=' + coordIndexNew);
	  }
	}
}
function set_index (value, timestamp)
{
	index = value;
	initialize ();
}
function tracePrint(outputString)
{
    if (localTraceEnabled)
	  Browser.println ('[IndexedTriangleStripSet]' + outputString);
}
function alwaysPrint(outputString)
{
	  Browser.println ('[IndexedTriangleStripSet]' + outputString);
}
"""),
          Group(
            metadata=MetadataString(
              IS=IS(
                connect=[
                connect(nodeField='metadata',protoField='metadata')])))])])),
    #  ==================== 
    ProtoDeclare(appinfo='LineSet represents a 3D geometry formed by constructing polylines from 3D vertices.',name='LineSet',
      ProtoInterface=ProtoInterface(
        field=[
        field(accessType='inputOutput',name='vertexCount',type='MFInt32',
          #  default initialization is NULL array [] to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Color ColorRGBA node only',name='color',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Coordinate node only',name='coord',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Metadata node only',name='metadata',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          )]),
      ProtoBody=ProtoBody(
        children=[
        IndexedLineSet(DEF='RenderedILS',
          IS=IS(
            connect=[
            connect(nodeField='color',protoField='color')])),
        Group(
          children=[
          Shape(
            geometry=IndexedLineSet(DEF='NodesHolderILS',
              IS=IS(
                connect=[
                connect(nodeField='color',protoField='color'),
                connect(nodeField='coord',protoField='coord')])),
            appearance=Appearance(DEF='UnusedAppearance',
              material=Material(emissiveColor=(0.8,0.8,0.8)))),
          Script(DEF='LineSetToIndexedLineSet',directOutput=True,
            field=[
            field(accessType='inputOutput',name='vertexCount',type='MFInt32',
              #  default initialization is NULL array [] to match X3D specification 
              ),
            field(accessType='initializeOnly',name='renderedILS',type='SFNode',
              children=[
              IndexedLineSet(USE='RenderedILS')]),
            field(accessType='initializeOnly',name='nodesHolder',type='SFNode',
              children=[
              IndexedLineSet(USE='NodesHolderILS')]),
            field(accessType='initializeOnly',name='localTraceEnabled',type='SFBool',value=True),
            field(accessType='initializeOnly',name='coordIndexNew',type='MFInt32',
              #  constructed during initialization 
              )],
            IS=IS(
              connect=[
              connect(nodeField='vertexCount',protoField='vertexCount')]),

          sourceCode="""
ecmascript:

function initialize()
{
	vertexCountSum = 0;
	tracePrint ('vertexCount=' + vertexCount);
	for (i=0; i < vertexCount.length; i++)
	{
		if (vertexCount[i] < 2)
		{
			alwaysPrint ('error, vertexCount[' + i + ']=' + vertexCount[i] +
	' is illegal value, must be >= 2');
			return;
		}
		vertexCountSum = vertexCountSum + vertexCountSum[i];
	}
	tracePrint ('vertexCountSum=' + vertexCountSum);
	numberPoints = nodesHolder.coord.point.length;

	if (numberPoints < vertexCountSum) 
	{
		alwaysPrint ('warning, Coordinate.point.length=' + numberPoints  + 
' is less than vertexCountSum=' + vertexCountSum + ', LS ignored');
		return;
	}
	coordIndexNew = new MFInt32 ();

	numberSegments = vertexCountSum.length;  // need validity check

	// i walks through array of points to build line-segment indices
	i = 0;
	for (seg=0; seg < numberSegments; seg++)
	{
	  for (j=0; j < vertexCount[seg]; j++)
	  {
		coordIndexNew [coordIndexNew.length] = i;
		i++;
	  }
	  coordIndexNew [coordIndexNew.length] = -1; // terminate current fan

	  // incremental trace of array being built
	  tracePrint ('coordIndexNew=' + coordIndexNew);
	} // repeat for all vertices

	renderedILS.coordIndex = coordIndexNew;
	tracePrint ('renderedILS.coordIndex=' + renderedILS.coordIndex);

	// match colorIndex if any Color node exists
	if (nodesHolder.color)
	{
	  if (nodesHolder.color.color.length > 0)
	  {
		renderedILS.colorIndex = coordIndexNew;
		tracePrint ('set_colorIndex=' + coordIndexNew);
	  }
	}
}
function tracePrint(outputString)
{
    if (localTraceEnabled)
	Browser.println ('[TriangleFanSet]' + outputString);
}
function alwaysPrint(outputString)
{
	Browser.println ('[TriangleFanSet]' + outputString);
}
"""),
          Group(
            metadata=MetadataString(
              IS=IS(
                connect=[
                connect(nodeField='metadata',protoField='metadata')])))])])),
    #  ==================== 
    ProtoDeclare(appinfo='TriangleFanSet represents a 3D shape composed of triangles that form a fan shape around the first vertex declared in each fan.',name='TriangleFanSet',
      ProtoInterface=ProtoInterface(
        field=[
        field(accessType='inputOutput',name='fanCount',type='MFInt32',
          #  default initialization is NULL array [] to match X3D specification 
          ),
        field(accessType='initializeOnly',name='ccw',type='SFBool',value=True),
        field(accessType='initializeOnly',name='colorPerVertex',type='SFBool',value=True),
        field(accessType='initializeOnly',name='normalPerVertex',type='SFBool',value=True),
        field(accessType='initializeOnly',name='solid',type='SFBool',value=True),
        field(accessType='inputOutput',appinfo='Color ColorRGBA node only',name='color',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Coordinate node only',name='coord',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Normal node only',name='normal',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='TextureCoordinate node only',name='texCoord',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Metadata node only',name='metadata',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          )]),
      ProtoBody=ProtoBody(
        children=[
        IndexedFaceSet(DEF='RenderedTFS',
          IS=IS(
            connect=[
            connect(nodeField='ccw',protoField='ccw'),
            connect(nodeField='colorPerVertex',protoField='colorPerVertex'),
            connect(nodeField='normalPerVertex',protoField='normalPerVertex'),
            connect(nodeField='solid',protoField='solid'),
            connect(nodeField='color',protoField='color'),
            connect(nodeField='coord',protoField='coord'),
            connect(nodeField='normal',protoField='normal'),
            connect(nodeField='texCoord',protoField='texCoord')])),
        Group(DEF='UnrenderedTFS',
          children=[
          Shape(
            #  is this really needed at all?? 
            geometry=IndexedFaceSet(DEF='NodesHolderTFS',
              IS=IS(
                connect=[
                connect(nodeField='color',protoField='color'),
                connect(nodeField='coord',protoField='coord'),
                connect(nodeField='normal',protoField='normal'),
                connect(nodeField='texCoord',protoField='texCoord')]))),
          Script(DEF='TriangleFanSetToIndexedFaceSet',directOutput=True,
            field=[
            field(accessType='inputOutput',name='fanCount',type='MFInt32',
              #  default initialization is NULL array [] to match X3D specification 
              ),
            field(accessType='initializeOnly',name='renderedTFS',type='SFNode',
              children=[
              IndexedFaceSet(USE='RenderedTFS')]),
            field(accessType='initializeOnly',name='nodesHolder',type='SFNode',
              children=[
              IndexedFaceSet(USE='NodesHolderTFS')]),
            field(accessType='initializeOnly',name='localTraceEnabled',type='SFBool',value=True),
            field(accessType='initializeOnly',name='coordIndexNew',type='MFInt32',
              #  constructed during initialization 
              )],
            IS=IS(
              connect=[
              connect(nodeField='fanCount',protoField='fanCount')]),

          sourceCode="""
ecmascript:

function initialize()
{
	fanCountSum = 0;
	tracePrint ('fanCount=' + fanCount);
	for (i=0; i < fanCount.length; i++)
	{
		if (fanCount[i] < 3)
		{
			alwaysPrint ('error, fanCount[' + i + ']=' + fanCount[i] +
	' is illegal value, must be >= 3');
			return;
		}
		fanCountSum = fanCountSum + fanCount[i];
	}
	tracePrint ('fanCountSum=' + fanCountSum);
	numberPoints = nodesHolder.coord.point.length;

	if (numberPoints < fanCountSum) 
	{
		alwaysPrint ('warning, Coordinate.point.length=' + numberPoints  + 
' is less than fanCountSum=' + fanCountSum + ', TFS ignored');
		return;
	}
	coordIndexNew = new MFInt32 ();

	numberFans = fanCount.length;  // need validity check

	// i walks through array of points to build polygon indices
	i = 0;
	for (fan=0; fan < numberFans; fan++)
	{
	  for (j=0; j < fanCount[fan]; j++)
	  {
		coordIndexNew [coordIndexNew.length] = i;
		i++;
	  }
	  coordIndexNew [coordIndexNew.length] = -1; // terminate current fan

	  // incremental trace of array being built
	  tracePrint ('coordIndexNew=' + coordIndexNew);
	} // repeat for all fans

	renderedTFS.coordIndex = coordIndexNew;
	tracePrint ('renderedTFS.coordIndex=' + renderedTFS.coordIndex);

	// match colorIndex if any Color node exists
	if (nodesHolder.color)
	{
	  if (nodesHolder.color.color.length > 0)
	  {
		renderedTFS.set_colorIndex = coordIndexNew;
		tracePrint ('set_colorIndex=' + coordIndexNew);
	  }
	}
}
function tracePrint(outputString)
{
    if (localTraceEnabled)
	Browser.println ('[TriangleFanSet]' + outputString);
}
function alwaysPrint(outputString)
{
	Browser.println ('[TriangleFanSet]' + outputString);
}
"""),
          Group(
            metadata=MetadataString(
              IS=IS(
                connect=[
                connect(nodeField='metadata',protoField='metadata')])))])])),
    #  ==================== 
    ProtoDeclare(appinfo='TriangleSet represents a 3D shape that represents a collection of individual triangles.',name='TriangleSet',
      ProtoInterface=ProtoInterface(
        field=[
        field(accessType='initializeOnly',name='ccw',type='SFBool',value=True),
        field(accessType='initializeOnly',name='colorPerVertex',type='SFBool',value=True),
        field(accessType='initializeOnly',name='normalPerVertex',type='SFBool',value=True),
        field(accessType='initializeOnly',name='solid',type='SFBool',value=True),
        field(accessType='inputOutput',appinfo='Color ColorRGBA node only',name='color',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Coordinate node only',name='coord',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Normal node only',name='normal',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='TextureCoordinate node only',name='texCoord',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Metadata node only',name='metadata',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          )]),
      ProtoBody=ProtoBody(
        children=[
        IndexedFaceSet(DEF='RenderedTS',
          IS=IS(
            connect=[
            connect(nodeField='ccw',protoField='ccw'),
            connect(nodeField='normalPerVertex',protoField='normalPerVertex'),
            connect(nodeField='solid',protoField='solid'),
            connect(nodeField='color',protoField='color'),
            connect(nodeField='coord',protoField='coord'),
            connect(nodeField='normal',protoField='normal'),
            connect(nodeField='texCoord',protoField='texCoord')])),
        Group(DEF='UnrenderedTS',
          children=[
          Shape(
            #  is this really needed at all?? 
            geometry=IndexedFaceSet(DEF='NodesHolderTS',
              IS=IS(
                connect=[
                connect(nodeField='color',protoField='color'),
                connect(nodeField='coord',protoField='coord'),
                connect(nodeField='normal',protoField='normal'),
                connect(nodeField='texCoord',protoField='texCoord')]))),
          Script(DEF='TriangleSetToIndexedFaceSet',directOutput=True,
            field=[
            field(accessType='initializeOnly',name='index',type='MFInt32',
              #  default initialization is NULL 
              ),
            field(accessType='inputOnly',name='set_index',type='MFInt32'),
            field(accessType='initializeOnly',name='renderedTS',type='SFNode',
              children=[
              IndexedFaceSet(USE='RenderedTS')]),
            field(accessType='initializeOnly',name='nodesHolder',type='SFNode',
              children=[
              IndexedFaceSet(USE='NodesHolderTS')]),
            field(accessType='initializeOnly',name='localTraceEnabled',type='SFBool',value=True),
            field(accessType='initializeOnly',name='coordIndexNew',type='MFInt32',
              #  constructed during initialization 
              )],

          sourceCode="""
ecmascript:

function initialize()
{
	// index is an array of triangle indices. The ordering of
	// the vertices is ccw (counter-clockwise).

	// ensure legal index values
	for (ii=0; ii <= index.length-1; ii++)
	{
	  if (index[ii] < -1)
	  {
		alwaysPrint ('error, index[' + ii + ']=' + index[ii] +
' is illegal value');
		return;
	  }
	}
	tracePrint ('index.length=' + index.length);
	tracePrint ('index=' + index);

	if (index.length < 3) 
	{
		alwaysPrint ('warning, index.length=' + index.length + 
' insufficient to construct a triangle, ITS ignored');
		return;
	}

	coordIndexNew = new MFInt32 ();

	// ii walks through index array,
	// goal is to initialize coordIndexNew list to match triangles
	for (ii=0; ii <= index.length-1; ii+=3)
	{
		if ((index[ii] == index[ii+1]) || (index[ii] == index[ii+2]) || (index[ii+1] == index[ii+2]))
		{
			alwaysPrint ('index=' + index);
			alwaysPrint ('error, pair of equal indices in triangle');
			return;
		}
	  
		if (index[ii] >= 0) 
		{
			// add another triangle from latest 3 points of fan set to ITS
			// order is ccw, i.e. in correct halfplane direction
			coordIndexNew [coordIndexNew.length] = index[ii];
			coordIndexNew [coordIndexNew.length] = index[ii+1];
			coordIndexNew [coordIndexNew.length] = index[ii+2];
			coordIndexNew [coordIndexNew.length] = -1;
		}

		if (index.length % 3 != 0) {
			alwaysPrint ('error, index field does not contain a multiple' +
		'of three coordinate values.');
			alwaysPrint ('The remaining vertices shall be ignored');
			return;
		}


		// ensure done, or sufficient points remain to build another triangle
//		if (	(i!=index.length-1) && (index.length - i < 2))
//		{
//			alwaysPrint ('index=' + index);
//			alwaysPrint ('error, insufficient index values after' +
//	'index[' + i + ']=-1');
//			return;
//		}
		// ensure done, or enough legal index values remain to build another triangle
//		if (	(i!=index.length-1) &&
//			((index[i+1] == -1) || (index[i+2] == -1) || (index[i+3] == -1)))
//		{
//			alwaysPrint ('index=' + index);
//			alwaysPrint ('error, insufficient non-negative-one index values after' +
//	'index[' + i + ']=-1');
//			return;
//		}
	  // incremental trace of array being built
	  tracePrint ('coordIndexNew=' + coordIndexNew);
	}
	renderedITS.set_coordIndex = coordIndexNew;
	tracePrint ('renderedITS.coordIndex=' + renderedITS.coordIndex);
	// match colorIndex if any Color node exists
	if (nodesHolder.color)
	{
	  if (nodesHolder.color.color.length > 0)
	  {
		renderedITS.set_colorIndex = coordIndexNew;
		tracePrint ('set_colorIndex=' + coordIndexNew);
	  }
	}
}
function set_index (value, timestamp)
{
	index = value;
	initialize ();
}
function tracePrint(outputString)
{
    if (localTraceEnabled)
	Browser.println ('[IndexedTriangleSet]' + outputString);
}
function alwaysPrint(outputString)
{
	Browser.println ('[IndexedTriangleSet]' + outputString);
}
"""),
          Group(
            metadata=MetadataString(
              IS=IS(
                connect=[
                connect(nodeField='metadata',protoField='metadata')])))])])),
    #  ==================== 
    ProtoDeclare(appinfo='TriangleStripSet represents a 3D shape composed of strips of triangles.',name='TriangleStripSet',
      ProtoInterface=ProtoInterface(
        field=[
        field(accessType='inputOutput',name='stripCount',type='MFInt32',
          #  default initialization is NULL array [] to match X3D specification 
          ),
        field(accessType='initializeOnly',name='ccw',type='SFBool',value=True),
        field(accessType='initializeOnly',name='colorPerVertex',type='SFBool',value=True),
        field(accessType='initializeOnly',name='normalPerVertex',type='SFBool',value=True),
        field(accessType='initializeOnly',name='solid',type='SFBool',value=True),
        field(accessType='inputOutput',appinfo='Color ColorRGBA node only',name='color',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Coordinate node only',name='coord',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Normal node only',name='normal',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='TextureCoordinate node only',name='texCoord',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          ),
        field(accessType='inputOutput',appinfo='Metadata node only',name='metadata',type='SFNode',
          #  default initialization is NULL node to match X3D specification 
          )]),
      ProtoBody=ProtoBody(
        children=[
        IndexedFaceSet(),])),
    #  ==================== 
    #  ProtoInstance examples are found in RenderingComponentExamples.x3d 
    Background(groundColor=[(0.2,0.2,0.2)],skyColor=[(0.2,0.2,0.2)]),
    Anchor(DEF='LinkToExamples',description='link to examples',url=["RenderingComponentExamples.x3d","https://www.web3d.org/x3d/content/examples/Basic/development/RenderingComponentExamples.x3d","RenderingComponentExamples.wrl","https://www.web3d.org/x3d/content/examples/Basic/development/RenderingComponentExamples.wrl"],
      children=[
      Shape(
        appearance=Appearance(
          material=Material(diffuseColor=(1,1,1))),
        geometry=Text(string=["RenderingComponentPrototypes","is a developmental file.","Click this text to view","RenderingComponentExamples"],
          fontStyle=FontStyle(justify=["MIDDLE","MIDDLE"]))),
      #  Selectable Text has transparent Box and TouchSensor description as a tooltip 
      Shape(
        geometry=Box(size=(12,5,.001)),
        appearance=Appearance(
          material=Material(transparency=0.8)))])])
) # X3D model complete

####################################################################################################
# Self-test diagnostics
####################################################################################################

print('Self-test diagnostics for RenderingComponentPrototypes.py:')
if        metaDiagnostics(newModel): # built-in utility method in X3D class
    print(metaDiagnostics(newModel)) # display meta info, hint, warning, error, TODO values in this model
# print('check newModel.XML() serialization...')
newModelXML= newModel.XML() # test export method XML() for exceptions during export
newModel.XMLvalidate()
# print(newModelXML) # diagnostic

try:
#   print('check newModel.VRML() serialization...')
    newModelVRML=newModel.VRML() # test export method VRML() for exceptions during export
    # print(prependLineNumbers(newModelVRML)) # debug
    print("Python-to-VRML export of VRML output successful", flush=True)
except Exception as err: # usually BaseException
    # https://stackoverflow.com/questions/18176602/how-to-get-the-name-of-an-exception-that-was-caught-in-python
    print("*** Python-to-VRML export of VRML output failed:", type(err).__name__, err)
    if newModelVRML: # may have failed to generate
        print(prependLineNumbers(newModelVRML, err.lineno))

try:
#   print('check newModel.JSON() serialization...')
    newModelJSON=newModel.JSON() # test export method JSON() for exceptions during export
#   print(prependLineNumbers(newModelJSON)) # debug
    print("Python-to-JSON export of JSON output successful (under development)")
except Exception as err: # usually SyntaxError
    print("*** Python-to-JSON export of JSON output failed:", type(err).__name__, err)
    if newModelJSON: # may have failed to generate
        print(prependLineNumbers(newModelJSON,err.lineno))

print("python RenderingComponentPrototypes.py load and self-test diagnostics complete.")
