####################################################################################################
#
# Invoking X3D model self-test:
#
#   $ python IntegerSequencerPrototype.py
#
# Python package x3d.py package is available on PyPI for import.
#   This approach simplifies Python X3D deployment and use.
#   https://pypi.org/project/x3d
#
# Installation:
#       pip install x3d
# or
#       python -m pip install x3d
#
# Developer options for loading x3d package in other Python programs:
#
#    from x3d import *  # preferred approach, terser source that avoids x3d.* class prefixes
#
# or
#    import x3d         # traditional way to subclass x3d package, all classes require x3d.* prefix,
#                       # but python source is very verbose, for example x3d.Material x3d.Shape etc.
#                       # X3dToPython.xslt stylesheet insertPackagePrefix=true supports this option.
#
####################################################################################################

from x3d import *

newModel=X3D(profile='Immersive',version='3.0',
  head=head(
    children=[
    meta(content='IntegerSequencerPrototype.x3d',name='title'),
    meta(content='This proto, modeled after a ScalarInterpolator, generates an array of integer values based on the input fraction and keys.',name='description'),
    meta(content='MFInt32 keyValue accessType is listed as initializeOnly/field, since inputOutput cannot be translated to exposedField in VRML97 scripting.',name='warning'),
    meta(content='Don Brutzman, Estuko Lippi, Jeff Weekley, Jane Wu, Matthew Braun',name='creator'),
    meta(content='20 August 2001',name='created'),
    meta(content='21 January 2020',name='modified'),
    meta(content='https://www.web3d.org/technicalinfo/specifications/vrml97/part1/nodesRef.html#ScalarInterpolator',name='reference'),
    meta(content='integer sequencer',name='subject'),
    meta(content='https://www.web3d.org/x3d/content/examples/Basic/development/IntegerSequencerPrototype.x3d',name='identifier'),
    meta(content='X3D-Edit 3.3, https://savage.nps.edu/X3D-Edit',name='generator'),
    meta(content='../license.html',name='license')]),
  Scene=Scene(
    children=[
    WorldInfo(title='IntegerSequencerPrototype.x3d'),
    ProtoDeclare(name='IntegerSequencer',
      ProtoInterface=ProtoInterface(
        field=[
        field(accessType='inputOnly',appinfo='Regular interpolator-style input, range [0..1]',name='set_fraction',type='SFFloat'),
        field(accessType='inputOnly',appinfo='Array sequentially increasing typically [0..1]. Must have the same number of keys as keyValues.',name='set_key',type='MFFloat'),
        field(accessType='inputOutput',appinfo='Array sequentially increasing typically [0..1]. Must have the same number of keys as keyValues.',name='key',type='MFFloat'),
        field(accessType='outputOnly',appinfo='Array sequentially increasing typically [0..1]. Must have the same number of keys as keyValues.',name='key_changed',type='MFFloat'),
        field(accessType='inputOnly',appinfo='Array of integer values. Must have the same number of keys as keyValues.',name='set_keyValue',type='MFInt32'),
        field(accessType='initializeOnly',appinfo='Array of integer values. Must have the same number of keys as keyValues.',name='keyValue',type='MFInt32'),
        field(accessType='outputOnly',appinfo='Array of integer values. Must have the same number of keys as keyValues.',name='keyValue_changed',type='MFInt32'),
        field(accessType='outputOnly',appinfo='Regular interpolator-style input',name='value_changed',type='SFInt32'),
        field(accessType='inputOnly',appinfo='Utility method',name='previous',type='SFBool'),
        field(accessType='inputOnly',appinfo='Utility method',name='next',type='SFBool')]),
      ProtoBody=ProtoBody(
        children=[
        Group(
          children=[
          Switch(whichChoice=-1,
            children=[
            ScalarInterpolator(DEF='KeyHolder',
              IS=IS(
                connect=[
                connect(nodeField='key',protoField='key')]))]),
          Script(DEF='SequencerScript',directOutput=True,
            #  Regular interpolator-style input 
            field=[
            field(accessType='inputOnly',appinfo='range [0..1]',name='set_fraction',type='SFFloat'),
            field(accessType='inputOnly',appinfo='Array sequentially increasing [0..1]. Must have the same number of keys as keyValues.',name='set_key',type='MFFloat'),
            field(accessType='initializeOnly',name='keyHolderNode',type='SFNode',
              children=[
              ScalarInterpolator(USE='KeyHolder')]),
            field(accessType='outputOnly',appinfo='Array sequentially increasing [0..1]. Must have the same number of keys as keyValues.',name='key_changed',type='MFFloat'),
            field(accessType='inputOnly',appinfo='Array of integer values. Must have the same number of keys as keyValues.',name='set_keyValue',type='MFInt32'),
            field(accessType='initializeOnly',name='keyValue',type='MFInt32'),
            field(accessType='outputOnly',appinfo='Array of integer values. Must have the same number of keys as keyValues.',name='keyValue_changed',type='MFInt32'),
            #  Regular interpolator-style output 
            field(accessType='outputOnly',name='value_changed',type='SFInt32'),
            #  Utility methods 
            field(accessType='inputOnly',name='previous',type='SFBool'),
            field(accessType='inputOnly',name='next',type='SFBool'),
            field(accessType='initializeOnly',appinfo='For development use only not for inclusion in specification implementations.',name='traceEnabled',type='SFBool',value=True),
            #  Script-specific interfaces, not needed for node definition 
            field(accessType='initializeOnly',name='previousFraction',type='SFFloat',value=0.0),
            field(accessType='initializeOnly',name='nextIndex',type='SFInt32',value=0),
            field(accessType='initializeOnly',name='isValid',type='SFBool',value=True),
            field(accessType='initializeOnly',name='recheckValidity',type='SFBool',value=False)],
            IS=IS(
              connect=[
              connect(nodeField='set_fraction',protoField='set_fraction'),
              connect(nodeField='set_key',protoField='set_key'),
              connect(nodeField='key_changed',protoField='key_changed'),
              connect(nodeField='set_keyValue',protoField='set_keyValue'),
              connect(nodeField='keyValue',protoField='keyValue'),
              connect(nodeField='keyValue_changed',protoField='keyValue_changed'),
              connect(nodeField='value_changed',protoField='value_changed'),
              connect(nodeField='previous',protoField='previous'),
              connect(nodeField='next',protoField='next')]),

          sourceCode="""
ecmascript:

var leftToRight;

function initialize()
{
	key      = keyHolderNode.key;
	tracePrint('Initializing a new IntegerSequencer.  key.length=' + key.length + '; keyValue.length=' + keyValue.length);
	tracePrint('key =' + key);
	tracePrint('keyValue =' + keyValue);

	validityCheck();
	setHalfKeyRange();

	// assume we start at first key, going left to right
	leftToRight = true;
	previousFraction = key[0];
	nextIndex = 1;  //validityCheck ensures minimum of 2 keys exist
}

function set_fraction(newFraction, timeStamp)
{
	if (recheckValidity) validityCheck();

	if (!isValid) return; //IntegerSequencer ignored

	//Bounds checking
	if (newFraction < key[0])
	{
		tracePrint('*** warning: fraction is less than first key.  fraction set to first key ***');
		newFraction = key[0];
	}
	else if (newFraction > key[key.length-1])
	{
		tracePrint('*** warning: fraction is greater than last key.  fraction set to last key ***');
		newFraction = key[key.length -1];
	}

	//Check animation direction
	if (newFraction < previousFraction && leftToRight == true)
	{
		if ((previousFraction - newFraction) > halfKeyRange) //looped around
		{
			nextIndex = 1;
		}
		else //just changed direction
		{
			leftToRight = false;
			nextIndex = nextIndex - 1;
            }
	}
	else if (newFraction > previousFraction && leftToRight == false)
	{
		if ((newFraction - previousFraction) < halfKeyRange) //looped around
		{
			nextIndex = key.length - 2;
		}
		else //just changed direction
		{
			leftToRight = true;
			nextIndex = nextIndex + 1;
            }
	}
	else if (newFraction == previousFraction)
	{ //no change, so no processing required
		return;
	}
	previousFraction = newFraction;

	if (leftToRight) // moving left to right
	{
		while (newFraction > key[nextIndex]) nextIndex++;

		if (newFraction == key[nextIndex])
			value_changed = keyValue[nextIndex];
		else	value_changed = keyValue[nextIndex -1];

		tracePrint('forward animation, fraction =' + newFraction);
		tracePrint('value_changed eventOut is:' + value_changed);
	}
	else // moving right to left
	{
		while (newFraction < key[nextIndex]) nextIndex--;

		if (newFraction == key[nextIndex])
			value_changed = keyValue[nextIndex];
		else	value_changed = keyValue[nextIndex + 1];

		tracePrint('backward animation, fraction =' + newFraction);
		tracePrint('value_changed eventOut is:' + value_changed);
	}
}

function set_key(newKey, timeStamp)
{
	key = newKey;
	keyHolderNode.key = newKey;
	setHalfKeyWidth();
	recheckValidity = true;
}

function set_keyValue(newKeyValue, timeStamp)
{
	keyValue = newKeyValue;
	recheckValidity = true;
}

function setHalfKeyRange()
{
	halfKeyRange = (key[key.length - 1] - key[0])/2.0;
}

function previous (value, timeStamp)
{
  if (value==true) // trigger on true events only
  {
	leftToRight = true;
	nextIndex = nextIndex - 2; // reset to previous
	if (nextIndex < 0) nextIndex = nextIndex + key.length;
	value_changed = keyValue[nextIndex];
	previousFraction = key[nextIndex];
	nextIndex++; // setup for next time, leftToRight
	if (nextIndex > key.length - 1) nextIndex = 0;
  }
}
function next (value, timeStamp)
{
  if (value==true) // trigger on true events only
  {
	leftToRight = true;
	value_changed = keyValue[nextIndex];
	previousFraction = key[nextIndex];
	nextIndex++; // setup for next time,leftToRight
	if (nextIndex > key.length - 1) nextIndex = 0;
  }
}

function validityCheck()
{
	//Check if lengths of key & keyValue arrays match
	if (key.length != keyValue.length)
	{
		alwaysPrint('*** error: key and keyValue arrays must be of the same length.  IntegerSequencer ignored ***');
		isValid = false;
		return;
	}
	//check to ensure minimum of 2 keys have been specified
	if (key.length < 2)
	{
		alwaysPrint('*** error: must contain at least 2 keys.  IntegerSequencer ignored ***');
		isValid = false;
		return;
	}

	//Check if key array has values in an non-decreasing order
	for (i = 1; i < key.length; i++)
	{
		tracePrint('i=' + i);

		if (key[i] < key [i-1])
		{
			alwaysPrint('*** error: key array values must be listed in a non-decreasing order.  IntegerSequencer ignored ***');
			isValid = false;
			return;
		}
	}
	isValid = true
	recheckValidity = false;
	key_changed = key;
	keyValue_changed = keyValue;
	return;
}

function tracePrint(outputString)
{
	if (traceEnabled) Browser.println ('[IntegerSequencer]' + outputString);
}

function alwaysPrint(outputString)
{
	Browser.println ('[IntegerSequencer]' + outputString);
}
""")])])),
    #  ===============Example============== 
    Anchor(description='IntegerSequencerExample',parameter=["target=_blank"],url=["IntegerSequencerExample.x3d","https://savage.nps.edu/Savage/Tools/Animation/IntegerSequencerExample.x3d","IntegerSequencerExample.wrl","https://savage.nps.edu/Savage/Tools/Animation/IntegerSequencerExample.wrl"],
      children=[
      Shape(
        geometry=Text(string=["IntegerSequencerPrototype","defines a prototype","","Click text to see example scene","IntegerSequencerExample"],
          fontStyle=FontStyle(justify=["MIDDLE","MIDDLE"],size=0.9)),
        appearance=Appearance(
          material=Material(diffuseColor=(1,1,0.2))))])])
) # X3D model complete

####################################################################################################
# Self-test diagnostics
####################################################################################################

print('Self-test diagnostics for IntegerSequencerPrototype.py:')
if        metaDiagnostics(newModel): # built-in utility method in X3D class
    print(metaDiagnostics(newModel)) # display meta info, hint, warning, error, TODO values in this model
# print('check newModel.XML() serialization...')
newModelXML= newModel.XML() # test export method XML() for exceptions during export
newModel.XMLvalidate()
# print(newModelXML) # diagnostic

try:
#   print('check newModel.VRML() serialization...')
    newModelVRML=newModel.VRML() # test export method VRML() for exceptions during export
    # print(prependLineNumbers(newModelVRML)) # debug
    print("Python-to-VRML export of VRML output successful", flush=True)
except Exception as err: # usually BaseException
    # https://stackoverflow.com/questions/18176602/how-to-get-the-name-of-an-exception-that-was-caught-in-python
    print("*** Python-to-VRML export of VRML output failed:", type(err).__name__, err)
    if newModelVRML: # may have failed to generate
        print(prependLineNumbers(newModelVRML, err.lineno))

try:
#   print('check newModel.JSON() serialization...')
    newModelJSON=newModel.JSON() # test export method JSON() for exceptions during export
#   print(prependLineNumbers(newModelJSON)) # debug
    print("Python-to-JSON export of JSON output successful (under development)")
except Exception as err: # usually SyntaxError
    print("*** Python-to-JSON export of JSON output failed:", type(err).__name__, err)
    if newModelJSON: # may have failed to generate
        print(prependLineNumbers(newModelJSON,err.lineno))

print("python IntegerSequencerPrototype.py load and self-test diagnostics complete.")
