####################################################################################################
#
# Invoking X3D model self-test:
#
#   $ python CoordinateInterpolator2dPrototype.py
#
# Python package x3d.py package is available on PyPI for import.
#   This approach simplifies Python X3D deployment and use.
#   https://pypi.org/project/x3d
#
# Installation:
#       pip install x3d
# or
#       python -m pip install x3d
#
# Developer options for loading x3d package in other Python programs:
#
#    from x3d import *  # preferred approach, terser source that avoids x3d.* class prefixes
#
# or
#    import x3d         # traditional way to subclass x3d package, all classes require x3d.* prefix,
#                       # but python source is very verbose, for example x3d.Material x3d.Shape etc.
#                       # X3dToPython.xslt stylesheet insertPackagePrefix=true supports this option.
#
####################################################################################################

from x3d import *

newModel=X3D(profile='Immersive',version='3.0',
  head=head(
    children=[
    meta(content='CoordinateInterpolator2dPrototype.x3d',name='title'),
    meta(content='CoordinateInterpolator2D prototype declaration, to interpolate across an array of Vector2FloatArray/MFVec2f values to produce an interpolated Vector2FloatArray - click text to see example.',name='description'),
    meta(content='Don Brutzman, Jeff Weekley, Jane Wu',name='creator'),
    meta(content='28 June 2001',name='created'),
    meta(content='20 January 2020',name='modified'),
    meta(content='https://www.web3d.org/technicalinfo/specifications/vrml97/part1/concepts.html#4.6.8',name='reference'),
    meta(content='https://www.web3d.org/technicalinfo/specifications/vrml97/part1/nodesRef.html#CoordinateInterpolator',name='reference'),
    meta(content='CoordinateInterpolator2D',name='subject'),
    meta(content='https://www.web3d.org/x3d/content/examples/Basic/development/CoordinateInterpolator2dPrototype.x3d',name='identifier'),
    meta(content='X3D-Edit 3.3, https://savage.nps.edu/X3D-Edit',name='generator'),
    meta(content='../license.html',name='license')]),
  Scene=Scene(
    children=[
    WorldInfo(title='CoordinateInterpolator2dPrototype.x3d'),
    ProtoDeclare(appinfo='Provide interpolation capability for Vector2FloatArray/MFVec2f values',documentation='https://www.web3d.org/technicalinfo/specifications/vrml97/part1/concepts.html#4.6.8',name='CoordinateInterpolator2D',
      ProtoInterface=ProtoInterface(
        field=[
        field(accessType='inputOnly',appinfo='Regular interpolator-style input, the set_fraction eventIn receives an SFFloat event and causes the interpolator function to evaluate resulting in a value_changed eventOut with the same timestamp as the set_fraction event.',name='set_fraction',type='SFFloat'),
        field(accessType='inputOnly',name='set_key',type='MFFloat'),
        field(accessType='inputOutput',appinfo='keyValue holds the array of Vector2FloatArrays that match each animation key.',name='key',type='MFFloat'),
        field(accessType='outputOnly',appinfo='Array sequentially increasing typically [0..1]. Must have the same number of keys as keyValues.',name='key_changed',type='MFFloat'),
        field(accessType='inputOnly',appinfo='Array of integer values. Must have the same number of keys as keyValues.',name='set_keyValue',type='MFVec2f'),
        field(accessType='inputOutput',appinfo='keyValue holds the array of Vector2FloatArrays that match each animation key.',name='keyValue',type='MFVec2f'),
        field(accessType='outputOnly',appinfo='Array of integer values. Must have the same number of keys as keyValues.',name='keyValue_changed',type='MFVec2f'),
        field(accessType='outputOnly',appinfo='Regular interpolator-style output, the interpolator function averages between respective keyValue Vector2FloatArrays resulting in a Vector2FloatArray value_changed eventOut with the same timestamp as the set_fraction event.',name='value_changed',type='MFVec2f')]),
      ProtoBody=ProtoBody(
        children=[
        Group(
          children=[
          Switch(whichChoice=-1,
            children=[
            ScalarInterpolator(DEF='KeyHolder',
              IS=IS(
                connect=[
                connect(nodeField='key',protoField='key')])),
            Shape(
              geometry=IndexedFaceSet(
                texCoord=TextureCoordinate(DEF='KeyValueHolder',
                  IS=IS(
                    connect=[
                    connect(nodeField='point',protoField='keyValue')]))),
              appearance=Appearance(DEF='DefaultAppearance',
                material=Material(),))]),
          Script(DEF='InterpolationScript',directOutput=True,
            field=[
            field(accessType='inputOnly',name='set_fraction',type='SFFloat'),
            field(accessType='initializeOnly',appinfo='local variable',name='fraction',type='SFFloat',value=0.0),
            field(accessType='inputOnly',name='set_key',type='MFFloat'),
            field(accessType='initializeOnly',name='keyHolderNode',type='SFNode',
              children=[
              ScalarInterpolator(USE='KeyHolder')]),
            field(accessType='outputOnly',name='key_changed',type='MFFloat'),
            field(accessType='inputOnly',name='set_keyValue',type='MFVec2f'),
            field(accessType='initializeOnly',name='keyValueHolderNode',type='SFNode',
              children=[
              TextureCoordinate(USE='KeyValueHolder')]),
            field(accessType='outputOnly',name='keyValue_changed',type='MFVec2f'),
            field(accessType='outputOnly',name='value_changed',type='MFVec2f')],
            IS=IS(
              connect=[
              connect(nodeField='set_fraction',protoField='set_fraction'),
              connect(nodeField='set_key',protoField='set_key'),
              connect(nodeField='key_changed',protoField='key_changed'),
              connect(nodeField='set_keyValue',protoField='set_keyValue'),
              connect(nodeField='keyValue_changed',protoField='keyValue_changed'),
              connect(nodeField='value_changed',protoField='value_changed')]),

          sourceCode="""
ecmascript:

// internal global persistent variables
var previousFraction;
var previousFractionIndex;
var blockSize;
var outputArray;

function tracePrint (outputString)
{
	var traceEnabled = false;
	if (traceEnabled) Browser.println ('[CoordinateInterpolator2D]' + outputString);
}
function alwaysPrint (outputString)
{
	Browser.println ('[CoordinateInterpolator2D]' + outputString);
}
function initialize ()
{
	key      = keyHolderNode.key;
	keyValue = keyValueHolderNode.point;
	previousFractionIndex = -1;
	previousFraction = 0;
	// check key array ranges [0..1] and is monotonically increasing
	// check that size of keyValue array is integer multiple of size of key array
	tracePrint ('key            =' + key);
	tracePrint ('key.length= ' + key.length);
	tracePrint ('keyValue=   ' + keyValue);
	tracePrint ('keyValue.length=' + keyValue.length);
	blockSize =  keyValue.length/key.length;
	tracePrint ('blockSize=' + blockSize);
	if (blockSize != Math.round(blockSize))
	{
	  alwaysPrint ('*** warning:  blockSize not an integer multiple. check sizes of key and keyValue');
	}
	if (key[0] != 0)
	{
	  alwaysPrint ('*** warning:  key[0] != 0');
	}
	if (key[key.length-1] != 1)
	{
	  alwaysPrint ('*** warning:  key[' + (key.length - 1) + '] != 1, reset from' + key[key.length-1] + ' to 1');
	  key[key.length-1] = 1;
	}
	for (index = 0; index < blockSize; index++)
	{
		if ((key[index] < 0) || (key[index] > 1))
		{
		   alwaysPrint ('*** warning:  key[' + index + '] =' + key[index] + ', out of range [0..1]');
		}
	}
	// instantiate default array, later computations just update it
	outputArray = new MFVec2f ();
	for (index = 0; index < blockSize; index++)
	{
		// dynamically grow outputArray to match initial block
		outputArray[index] = keyValue[index];
	}
	tracePrint ('initial outputArray=' + outputArray);
}

function set_fraction (inputFloat, timestamp) {
	fraction = inputFloat;
	tracePrint ('previousFractionIndex=' + previousFractionIndex
		 + ', fraction=' + fraction + ', previousFraction=' + previousFraction);

	if (fraction < 0)
	{
		tracePrint ('*** illegal fraction' + fraction + ' set to 0');
		fraction = 0;
		previousFractionIndex = 0; // first
	}
	else if (fraction > 1)
	{
		alwaysPrint ('*** illegal fraction' + fraction + ' set to 1');
		fraction = 1;
		previousFractionIndex = blockSize - 1; // last
	}
	else if (previousFractionIndex == -1)
	{
		previousFractionIndex = 0; // first
		tracePrint ('previousFractionIndex initialized for first event');
	}
	else if ((fraction >= previousFraction) && (fraction >= key[previousFractionIndex+1]))
	{
		previousFractionIndex++;
	}
	else if (fraction < previousFraction) // regress, or loop repeat without reaching one
	{
		previousFractionIndex = 0;
		while ((fraction >= key[previousFractionIndex+1]) && (previousFractionIndex < blockSize))
		{
			previousFractionIndex++;
		}
		tracePrint ('reset/reincrement previousFractionIndex to' + previousFractionIndex);
	}

	if (fraction == 1) // use final block
	{
		tracePrint ('(fraction == 1)');
		for (index = 0; index < blockSize; index++)
		{
			// update outputArray with final four keyValues
			outputArray[4 - index] = keyValue[keyValue.length - index];
		}
		previousFractionIndex = -1; // setup for restart
		tracePrint ('finished final fraction==1 block');
	}
	// when fraction matches index, calculate value_changed from corresponding keyValue array
	else if (fraction == key[previousFractionIndex])
	{
		tracePrint ('(fraction == key[previousFractionIndex])');
		for (index = 0; index < blockSize; index++)
		{
			// update outputArray - need to interpolate next
			outputArray[index] = keyValue[blockSize * (previousFractionIndex) + index];
		}
	}
	else // calculate value_changed by interpolating between adjacent keyValue arrays
	{
		partialFraction = fraction                     - key[previousFractionIndex];
		deltaFraction   = key[previousFractionIndex+1] - key[previousFractionIndex];
		percentFraction = partialFraction / deltaFraction;
	//	tracePrint ('deltaFraction   =' + deltaFraction);
	//	tracePrint ('partialFraction =' + partialFraction);
		tracePrint ('percentFraction =' + percentFraction);
		for (index = 0; index < blockSize; index++)
		{
			// no arithmetic operators provided for SFVec2f, treat element by element
			nextKeyValue  = keyValue[blockSize * (previousFractionIndex + 1) + index];
			priorKeyValue = keyValue[blockSize * (previousFractionIndex)     + index];
			deltaKeyValue = new SFVec2f (
						nextKeyValue[0] - priorKeyValue[0],
						nextKeyValue[1] - priorKeyValue[1]);
		//	tracePrint ('deltaKeyValue =' + deltaKeyValue);
			// update outputArray
			outputArray[index][0] = keyValue[blockSize * (previousFractionIndex) + index][0]
			   + percentFraction * deltaKeyValue[0];
			outputArray[index][1] = keyValue[blockSize * (previousFractionIndex) + index][1]
			   + percentFraction * deltaKeyValue[1];
		}
	}
	value_changed = outputArray;
	previousFraction = fraction;
	tracePrint ('value_changed=' + value_changed);
}

function set_key (inputArray, timestamp) {
	key = inputArray;       // update key Vector2FloatArray
	keyHolderNode.key = key; // update holder
	initialize (timestamp); // reverify key, keyValue sizes
	key_changed = key;	// eventOut
}

function set_keyValue (inputArray, timestamp) {
	keyValue = inputArray;  	// update keyValue Vector2FloatArray
	keyValueHolderNode.point = keyValue; // update holder
	initialize (timestamp); 	// reverify key, keyValue sizes
	keyValue_changed = keyValue;	// eventOut
}
""")])])),
    #  ====================================== 
    #  Example use 
    Anchor(description='CoordinateInterpolator2dExample',parameter=["target=_blank"],url=["CoordinateInterpolator2dExample.x3d","https://savage.nps.edu/Savage/Tools/Animation/CoordinateInterpolator2dExample.x3d","CoordinateInterpolator2dExample.wrl","https://savage.nps.edu/Savage/Tools/Animation/CoordinateInterpolator2dExample.wrl"],
      children=[
      Shape(
        geometry=Text(string=["CoordinateInterpolator2dPrototype","defines a prototype","","Click on this text to see","CoordinateInterpolator2dExample"," scene"],
          fontStyle=FontStyle(justify=["MIDDLE","MIDDLE"],size=0.7)),
        appearance=Appearance(
          material=Material(diffuseColor=(1,1,0.2))))])])
) # X3D model complete

####################################################################################################
# Self-test diagnostics
####################################################################################################

print('Self-test diagnostics for CoordinateInterpolator2dPrototype.py:')
if        metaDiagnostics(newModel): # built-in utility method in X3D class
    print(metaDiagnostics(newModel)) # display meta info, hint, warning, error, TODO values in this model
# print('check newModel.XML() serialization...')
newModelXML= newModel.XML() # test export method XML() for exceptions during export
newModel.XMLvalidate()
# print(newModelXML) # diagnostic

try:
#   print('check newModel.VRML() serialization...')
    newModelVRML=newModel.VRML() # test export method VRML() for exceptions during export
    # print(prependLineNumbers(newModelVRML)) # debug
    print("Python-to-VRML export of VRML output successful", flush=True)
except Exception as err: # usually BaseException
    # https://stackoverflow.com/questions/18176602/how-to-get-the-name-of-an-exception-that-was-caught-in-python
    print("*** Python-to-VRML export of VRML output failed:", type(err).__name__, err)
    if newModelVRML: # may have failed to generate
        print(prependLineNumbers(newModelVRML, err.lineno))

try:
#   print('check newModel.JSON() serialization...')
    newModelJSON=newModel.JSON() # test export method JSON() for exceptions during export
#   print(prependLineNumbers(newModelJSON)) # debug
    print("Python-to-JSON export of JSON output successful (under development)")
except Exception as err: # usually SyntaxError
    print("*** Python-to-JSON export of JSON output failed:", type(err).__name__, err)
    if newModelJSON: # may have failed to generate
        print(prependLineNumbers(newModelJSON,err.lineno))

print("python CoordinateInterpolator2dPrototype.py load and self-test diagnostics complete.")
