####################################################################################################
#
# Invoking X3D model self-test:
#
#   $ python BooleanSequencerPrototype.py
#
# Python package x3d.py package is available on PyPI for import.
#   This approach simplifies Python X3D deployment and use.
#   https://pypi.org/project/x3d
#
# Installation:
#       pip install x3d
# or
#       python -m pip install x3d
#
# Developer options for loading x3d package in other Python programs:
#
#    from x3d import *  # preferred approach, terser source that avoids x3d.* class prefixes
#
# or
#    import x3d         # traditional way to subclass x3d package, all classes require x3d.* prefix,
#                       # but python source is very verbose, for example x3d.Material x3d.Shape etc.
#                       # X3dToPython.xslt stylesheet insertPackagePrefix=true supports this option.
#
####################################################################################################

from x3d import *

newModel=X3D(profile='Immersive',version='3.0',
  head=head(
    children=[
    meta(content='BooleanSequencerPrototype.x3d',name='title'),
    meta(content='BooleanSequencer is modeled after ScalarInterpolator and generates true or false values.',name='description'),
    meta(content='Don Brutzman, Estuko Lippi, Jeff Weekley, Jane Wu',name='creator'),
    meta(content='7 August 2001',name='created'),
    meta(content='20 January 2020',name='modified'),
    meta(content='https://www.web3d.org/technicalinfo/specifications/vrml97/part1/nodesRef.html#ScalarInterpolator',name='reference'),
    meta(content='boolean sequencer',name='subject'),
    meta(content='https://www.web3d.org/x3d/content/examples/Basic/development/BooleanSequencerPrototype.x3d',name='identifier'),
    meta(content='X3D-Edit 3.3, https://savage.nps.edu/X3D-Edit',name='generator'),
    meta(content='../license.html',name='license')]),
  Scene=Scene(
    children=[
    WorldInfo(title='BooleanSequencerPrototype.x3d'),
    ProtoDeclare(name='BooleanSequencer',
      ProtoInterface=ProtoInterface(
        field=[
        field(accessType='inputOnly',appinfo='Regular interpolator-style input, typical range [0..1]',name='set_fraction',type='SFFloat'),
        field(accessType='inputOnly',name='set_key',type='MFFloat'),
        field(accessType='inputOutput',appinfo='Array sequentially increasing typically [0..1]. Must have the same number of keys as keyValues.',name='key',type='MFFloat',
          #  NULL initialization value 
          ),
        field(accessType='outputOnly',name='key_changed',type='MFFloat'),
        field(accessType='inputOnly',name='set_keyValue',type='MFBool'),
        field(accessType='inputOutput',appinfo='Array of true|false values. Must have the same number of keys as keyValues.',name='keyValue',type='MFBool',
          #  NULL initialization value 
          ),
        field(accessType='outputOnly',appinfo='Regular interpolator-style input',name='keyValue_changed',type='MFBool'),
        field(accessType='outputOnly',appinfo='Regular interpolator-style input',name='value_changed',type='SFBool'),
        field(accessType='inputOnly',appinfo='Utility method',name='previous',type='SFBool'),
        field(accessType='inputOnly',appinfo='Utility method',name='next',type='SFBool')]),
      ProtoBody=ProtoBody(
        children=[
        Group(
          children=[
          ScalarInterpolator(DEF='KeyHolder',
            IS=IS(
              connect=[
              connect(nodeField='key',protoField='key')])),
          Anchor(DEF='KeyValueHolder',
            IS=IS(
              connect=[
              connect(nodeField='description',protoField='keyValue')])),
          Script(DEF='SequencerScript',directOutput=True,
            #  Regular interpolator-style input 
            field=[
            field(accessType='inputOnly',appinfo='typical range [0..1]',name='set_fraction',type='SFFloat'),
            field(accessType='inputOnly',appinfo='Array sequentially increasing typically [0..1]. Must have the same number of keys as keyValues.',name='set_key',type='MFFloat'),
            field(accessType='initializeOnly',name='keyHolderNode',type='SFNode',
              children=[
              ScalarInterpolator(USE='KeyHolder')]),
            field(accessType='outputOnly',name='key_changed',type='MFFloat'),
            field(accessType='inputOnly',appinfo='Array of true or false values. Must have the same number of keys as keyValues.',name='set_keyValue',type='MFBool'),
            field(accessType='initializeOnly',name='keyValueHolderNode',type='SFNode',
              children=[
              Anchor(USE='KeyValueHolder')]),
            field(accessType='outputOnly',name='keyValue_changed',type='MFBool'),
            #  Regular interpolator-style output 
            field(accessType='outputOnly',name='value_changed',type='SFBool'),
            #  Utility methods 
            field(accessType='inputOnly',name='previous',type='SFBool'),
            field(accessType='inputOnly',name='next',type='SFBool'),
            #  Script-specific interfaces, not needed for node definition 
            field(accessType='initializeOnly',name='traceEnabled',type='SFBool',value=False),
            field(accessType='initializeOnly',name='keyValueArray',type='MFInt32',
              #  NULL initialization value 
              ),
            field(accessType='initializeOnly',name='previousFraction',type='SFFloat',value=0.0),
            field(accessType='initializeOnly',name='nextIndex',type='SFInt32',value=0),
            field(accessType='outputOnly',name='valid',type='SFBool',value=True),
            field(accessType='initializeOnly',name='recheckValidity',type='SFBool',value=False),
            field(accessType='initializeOnly',appinfo='leftToRight',name='forward',type='SFBool',value=True),
            field(accessType='inputOnly',name='key',type='MFFloat',
              #  NULL initialization value 
              ),
            field(accessType='inputOutput',name='keyValue',type='MFInt32',
              #  NULL initialization value 
              )],
            IS=IS(
              connect=[
              connect(nodeField='set_fraction',protoField='set_fraction'),
              connect(nodeField='set_key',protoField='set_key'),
              connect(nodeField='key_changed',protoField='key_changed'),
              connect(nodeField='set_keyValue',protoField='set_keyValue'),
              connect(nodeField='keyValue_changed',protoField='keyValue_changed'),
              connect(nodeField='value_changed',protoField='value_changed'),
              connect(nodeField='previous',protoField='previous'),
              connect(nodeField='next',protoField='next')]),

          sourceCode="""
ecmascript:

var key, keyValue;

function initialize()
{
	key      = keyHolderNode.key;
	keyValue = keyValueHolderNode.description;
	tracePrint('key =' + key);
	tracePrint('keyValue =' + keyValue);
	keyValueToKeyValueArray ();
	tracePrint('keyValueArray =' + keyValueArray);

	forward = true;
	tracePrint('Initializing a new BooleanSequencer.  key.length=' + key.length + '; keyValueArray.length=' + keyValueArray.length);
	validityCheck();
}

function keyValueToKeyValueArray ()
{
	tracePrint('keyValueToKeyValueArray starting');
	index = 0;
	complete = false;
	nextString = keyValue.toLowerCase();
	tracePrint('initial nextString=' + nextString);
	tokenCount=0;
	while ((complete != true) && (nextString.length > 0))
	{
		tracePrint('nextString=' + nextString);
		while ((nextString.substring(0,1) == ' ') || (nextString.substring(0,1) == ','))
		       nextString = nextString.slice(1);
		tracePrint('deblanked nextString=' + nextString);
		if (nextString.length == 0)
		{
			tracePrint ('nextString.length == 0');
			complete = true;
		}
		if (nextString.length < 4)
		{
			alwaysPrint ('*** illegal keyValue input=' + nextString);
			valid = false;
			complete = true;
		}
		else if (nextString.substring(0,4) =='true')
		{
			keyValueArray[keyValueArray.length] = 1; // append
			newString = nextString.slice(4);
			nextString = newString;
			tokenCount++;
			tracePrint('found true, nextString=' + nextString + ', tokenCount=' + tokenCount);
		}
		else if (nextString.length < 5)
		{
			alwaysPrint ('*** illegal keyValue input=' + nextString);
			valid = false;
			complete = true;
		}
		else if (nextString.substring(0,5) =='false')
		{
			keyValueArray[keyValueArray.length] = 0; // append
			newString = nextString.slice(5);
			nextString = newString;
			tokenCount++;
			tracePrint('found false, nextString=' + nextString + ', tokenCount=' + tokenCount);
		}
		tracePrint('  intermediate keyValueArray=' + keyValueArray);
	}
	tracePrint('keyValueToKeyValueArray complete');
}

function set_fraction(value, timeStamp)
{
	if (recheckValidity) validityCheck();

	if (!valid) return; //BooleanSequencer ignored

	tracePrint('fraction =' + value);
	//Bounds checking
	if (value < 0)
	{
		alwaysPrint('*** warning: fraction is less than 0.  fraction reset to 0 ***');
		value = 0;
	}
	else if (value > 1)
	{
		alwaysPrint('*** warning: fraction is greater than 1.  fraction reset to 1 ***');
		value = 1;
	}

	//Check animation direction
	if (value < previousFraction && forward == true)
	{
		forward = false;
		nextIndex = nextIndex - 1;
		tracePrint('Animate backward');
	}
	else if (value > previousFraction && forward == false)
	{
		forward = true;
		//nextIndex = 0;
		tracePrint('Animate forward');
	}

	previousFraction = value;

	if (forward == true)
	{
		for (i = nextIndex; i < key.length; i++)
		{
			if (value < key[i])
				return;

			nextIndex = i + 1;
			tracePrint('nextIndex =' + nextIndex);
			if (nextIndex < key.length)
			{
				if (value <= key[nextIndex])
				{
					//Fire event
					if (keyValueArray[nextIndex-1] == 0)
						value_changed = false;
					else
						value_changed = true;
					tracePrint('value_changed eventOut is:' + value_changed);
				}
			}
			else if (nextIndex == key.length)
			{
				//Fire event
				if (keyValueArray[nextIndex-1] == 0)
					value_changed = false;
				else
					value_changed = true;
				tracePrint('value_changed eventOut is:' + value_changed);
			}
			else //nextIndex > key.length
			{
				//nextIndex = 0;
				break;
			}
		}
	}
	else //backward
	{
		for (i = nextIndex; i > 0; i--)
		{
			if (value >= key[i])
				return;

			nextIndex = i - 1;
			tracePrint('nextIndex =' + nextIndex);
			if (nextIndex >= 0)
			{
				if (value >= key[nextIndex])
				{
					//Fire event
					if (keyValueArray[nextIndex] == 0)
						value_changed = false;
					else
						value_changed = true;
					tracePrint('value_changed eventOut is:' + value_changed);
				}
			}
			else //nextIndex < 0
			{
				//nextIndex = key.length;
				break;
			}
		}
	}
}

function set_key(value, timeStamp)
{
	key = value;
	keyHolderNode.key  = key;
	recheckValidity = true;
}

function set_keyValue(value, timeStamp)
{
	keyValue = value;
	keyValueHolderNode.description = keyValue;
	recheckValidity = true;
	keyValueToKeyValueArray ();
	keyValue_changed = keyValue;
}

function validityCheck()
{
	//Check if key & keyValueArray array length matches
	if (key.length != keyValueArray.length)
	{
		alwaysPrint('*** error: key and keyValue arrays must be of the same length.  BooleanSequencer ignored ***');
		valid = false;
		return;
	}

	//Check if key array has values in the range of [0..1] in an increasing order
	if (key[0] < 0 || key[0] > 1)
	{
		alwaysPrint('*** error: key[0] value is NOT in the range of [0..1].  BooleanSequencer ignored ***');
		valid = false;
		return;
	}
	for (i = 1; i < key.length; i++)
	{
		if (key[i] < 0 || key[i] > 1)
		{
			alwaysPrint('*** error: key[' + i + '] value is NOT in the range of [0..1].  BooleanSequencer ignored ***');
			valid = false;
			return;
		}

		if (key[i] <= key [i-1])
		{
			alwaysPrint('*** error: values for key[] array must be listed in an increasing order.  BooleanSequencer ignored ***');
			valid = false;
			return;
		}
	}
	recheckValidity = false;
	key_changed = key;
	return;
}
function previous (SFBoolValue, timestamp)
{
	nextIndex = nextIndex - 1;
	if (nextIndex == 0) nextIndex = key.length - 1;
}
function next (SFBoolValue, timestamp)
{
	nextIndex = nextIndex + 1;
	if (nextIndex == key.length) nextIndex = 0;
}

function tracePrint(outputString)
{
	if (traceEnabled) Browser.println ('[ BooleanSequencer ]' + outputString);
}

function alwaysPrint(outputString)
{
	Browser.println ('[ BooleanSequencer ]' + outputString);
}
""")])])),
    #  ===============Example============== 
    Anchor(description='BooleanSequencerExample',parameter=["target=_blank"],url=["BooleanSequencerExample.x3d","https://www.web3d.org/x3d/content/examples/Basic/development/BooleanSequencerExample.x3d","BooleanSequencerExample.wrl","https://www.web3d.org/x3d/content/examples/Basic/development/BooleanSequencerExample.wrl"],
      children=[
      Shape(
        geometry=Text(string=["BooleanSequencerPrototype","defines a prototype","","Click on this text to see","BooleanSequencerExample"," scene"],
          fontStyle=FontStyle(justify=["MIDDLE","MIDDLE"])),
        appearance=Appearance(
          material=Material(diffuseColor=(1,1,0.2))))])])
) # X3D model complete

####################################################################################################
# Self-test diagnostics
####################################################################################################

print('Self-test diagnostics for BooleanSequencerPrototype.py:')
if        metaDiagnostics(newModel): # built-in utility method in X3D class
    print(metaDiagnostics(newModel)) # display meta info, hint, warning, error, TODO values in this model
# print('check newModel.XML() serialization...')
newModelXML= newModel.XML() # test export method XML() for exceptions during export
newModel.XMLvalidate()
# print(newModelXML) # diagnostic

try:
#   print('check newModel.VRML() serialization...')
    newModelVRML=newModel.VRML() # test export method VRML() for exceptions during export
    # print(prependLineNumbers(newModelVRML)) # debug
    print("Python-to-VRML export of VRML output successful", flush=True)
except Exception as err: # usually BaseException
    # https://stackoverflow.com/questions/18176602/how-to-get-the-name-of-an-exception-that-was-caught-in-python
    print("*** Python-to-VRML export of VRML output failed:", type(err).__name__, err)
    if newModelVRML: # may have failed to generate
        print(prependLineNumbers(newModelVRML, err.lineno))

try:
#   print('check newModel.JSON() serialization...')
    newModelJSON=newModel.JSON() # test export method JSON() for exceptions during export
#   print(prependLineNumbers(newModelJSON)) # debug
    print("Python-to-JSON export of JSON output successful (under development)")
except Exception as err: # usually SyntaxError
    print("*** Python-to-JSON export of JSON output failed:", type(err).__name__, err)
    if newModelJSON: # may have failed to generate
        print(prependLineNumbers(newModelJSON,err.lineno))

print("python BooleanSequencerPrototype.py load and self-test diagnostics complete.")
