####################################################################################################
#
# Invoking X3D model self-test:
#
#   $ python CreateVrmlFromStringRandomBoxes.py
#
# Python package x3d.py package is available on PyPI for import.
#   This approach simplifies Python X3D deployment and use.
#   https://pypi.org/project/x3d
#
# Installation:
#       pip install x3d
# or
#       python -m pip install x3d
#
# Developer options for loading x3d package in other Python programs:
#
#    from x3d import *  # preferred approach, terser source that avoids x3d.* class prefixes
#
# or
#    import x3d         # traditional way to subclass x3d package, all classes require x3d.* prefix,
#                       # but python source is very verbose, for example x3d.Material x3d.Shape etc.
#                       # X3dToPython.xslt stylesheet insertPackagePrefix=true supports this option.
#
####################################################################################################

from x3d import *

newModel=X3D(profile='Immersive',version='3.1',
  head=head(
    children=[
    meta(content='CreateVrmlFromStringRandomBoxes.x3d',name='title'),
    meta(content='Sample world constructed in a Script node using the EcmaScript Browser call createVrmlFromString.',name='description'),
    meta(content='Vincent Gardet',name='creator'),
    meta(content='Curt Blais',name='translator'),
    meta(content='Don Brutzman',name='contributor'),
    meta(content='23 April 2002',name='created'),
    meta(content='12 October 2023',name='modified'),
    meta(content='CreateX3dFromStringRandomSpheres.x3d',name='reference'),
    meta(content='CreateVrmlFromStringRandomBoxes.png',name='Image'),
    meta(content='See CreateVrmlFromString tutorial at',name='reference'),
    meta(content='http://www.gardet.nom.fr/vrmlcity/tut_cvfs.htm',name='reference'),
    meta(content='create VRML from string, javascript',name='subject'),
    meta(content='https://www.web3d.org/x3d/content/examples/Basic/course/CreateVrmlFromStringRandomBoxes.x3d',name='identifier'),
    meta(content='X3D-Edit 4.0, https://savage.nps.edu/X3D-Edit',name='generator'),
    meta(content='../license.html',name='license')]),
  Scene=Scene(
    children=[
    WorldInfo(title='CreateVrmlFromStringRandomBoxes.x3d'),
    NavigationInfo(avatarSize=[0.1,1.6,2.0],headlight=False,speed=5,type='"EXAMINE" "WALK" "ANY"'),
    Background(skyAngle=[1.1,1.57],skyColor=[(.3,.3,1),(.4,.4,1),(.7,.7,.9)]),
    DirectionalLight(direction=(0,-.5,-1)),
    DirectionalLight(direction=(-1,-.5,.5),intensity=.9),
    DirectionalLight(direction=(1,-.5,.5),intensity=.8),
    Viewpoint(description='Interaction view',position=(50,1.6,150)),
    Viewpoint(description='Plan (top-down) view',orientation=(1,0,0,-1.54),position=(50,200,50)),
    Shape(
      appearance=Appearance(
        material=Material(diffuseColor=(.3,.3,1),transparency=0.5)),
      geometry=Box(size=(1000,0.01,1000))),
    Transform(translation=(50,1.6,140),
      children=[
      Billboard(
        children=[
        Shape(
          appearance=Appearance(
            material=Material(diffuseColor=(0.1,0.3,0.2),emissiveColor=(0.3,0.6,0.9))),
          geometry=Text(string=["Ten random boxes","computed at runtime","(select text to recompute)"],
            fontStyle=FontStyle(justify=["MIDDLE","MIDDLE"]))),
        TouchSensor(DEF='TouchText',description='Touch text to recompute random boxes'),
        #  Transparent Box makes the text much more accessible/clickable 
        Shape(
          geometry=Box(size=(8,2,0.1)),
          appearance=Appearance(
            material=Material(transparency=1)))])]),
    #  ROOT is used at initialize time as parent for randomly generated Shape children 
    Group(DEF='HoldsAutogeneratedContent',
      #  content added here at run time 
      ),
    Script(DEF='BoxGeneratorScript',directOutput=True,
      field=[
      field(accessType='initializeOnly',name='ModifiableExternalNode',type='SFNode',
        children=[
        Group(USE='HoldsAutogeneratedContent')]),
      field(accessType='inputOnly',name='recompute',type='SFBool')],

    sourceCode="""
ecmascript:

function R () {
	return Math.random();
}
function recompute (isActive) {
	if (isActive==true) initialize();
}

function initialize() {
  for (i=0; i < 10; i++) {
    rand1 = 100*R();
    rand2 = 100*R();
    rand3 = 20*R();
    rand4 = 40*R();
    rand5 = 20*R();
    sceneString ='Transform {                                                    \n' +
                      '   translation ' + rand1 + ' 0 ' + rand2 + ' \n' +
                      '   children [                                            \n' +
                      '           Shape {                                       \n' +
                      '                   appearance Appearance {               \n' +
                      '                      material Material {                \n' +
                      '                         diffuseColor ' + R() + ' ' + R() + ' ' + R() + '\n' +
                      '                      }                                  \n' +
                      '                   }                                     \n' +
                      '                   geometry Box {                        \n' +
                      '                      size ' + rand3 + ' ' + rand4 + ' ' + rand5 + '\n' +
                      '                   }                                     \n' +
                      '           }                                             \n' +
                      '   ]                                                     \n' +
                      '}                                                        \n' +
                      '#########################################################\n';
       Browser.println (sceneString);
       newNode = Browser.createVrmlFromString(sceneString);
//     newNode = Browser.createX3dFromString(sceneString);
       ModifiableExternalNode.children[i] = newNode[0];
  }
}
"""),
    ROUTE(fromField='isActive',fromNode='TouchText',toField='recompute',toNode='BoxGeneratorScript')])
) # X3D model complete

####################################################################################################
# Self-test diagnostics
####################################################################################################

print('Self-test diagnostics for CreateVrmlFromStringRandomBoxes.py:')
if        metaDiagnostics(newModel): # built-in utility method in X3D class
    print(metaDiagnostics(newModel)) # display meta info, hint, warning, error, TODO values in this model
# print('check newModel.XML() serialization...')
newModelXML= newModel.XML() # test export method XML() for exceptions during export
newModel.XMLvalidate()
# print(newModelXML) # diagnostic

try:
#   print('check newModel.VRML() serialization...')
    newModelVRML=newModel.VRML() # test export method VRML() for exceptions during export
    # print(prependLineNumbers(newModelVRML)) # debug
    print("Python-to-VRML export of VRML output successful", flush=True)
except Exception as err: # usually BaseException
    # https://stackoverflow.com/questions/18176602/how-to-get-the-name-of-an-exception-that-was-caught-in-python
    print("*** Python-to-VRML export of VRML output failed:", type(err).__name__, err)
    if newModelVRML: # may have failed to generate
        print(prependLineNumbers(newModelVRML, err.lineno))

try:
#   print('check newModel.JSON() serialization...')
    newModelJSON=newModel.JSON() # test export method JSON() for exceptions during export
#   print(prependLineNumbers(newModelJSON)) # debug
    print("Python-to-JSON export of JSON output successful (under development)")
except Exception as err: # usually SyntaxError
    print("*** Python-to-JSON export of JSON output failed:", type(err).__name__, err)
    if newModelJSON: # may have failed to generate
        print(prependLineNumbers(newModelJSON,err.lineno))

print("python CreateVrmlFromStringRandomBoxes.py load and self-test diagnostics complete.")
