####################################################################################################
#
# Invoking X3D model self-test:
#
#   $ python TextureProjectorComponentExample.py
#
# Python package x3d.py package is available on PyPI for import.
#   This approach simplifies Python X3D deployment and use.
#   https://pypi.org/project/x3d
#
# Installation:
#       pip install x3d
# or
#       python -m pip install x3d
#
# Developer options for loading x3d package in other Python programs:
#
#    from x3d import *  # preferred approach, terser source that avoids x3d.* class prefixes
#
# or
#    import x3d         # traditional way to subclass x3d package, all classes require x3d.* prefix,
#                       # but python source is very verbose, for example x3d.Material x3d.Shape etc.
#                       # X3dToPython.xslt stylesheet insertPackagePrefix=true supports this option.
#
####################################################################################################

from x3d import *

newModel=X3D(profile='Immersive',version='4.0',
  head=head(
    children=[
    component(level=2,name='TextureProjection'),
    meta(content='TextureProjectorComponentExample.x3d',name='title'),
    meta(content='Texture projector component example',name='description'),
    meta(content='Kwan Hee Yoo, Dick Puk and Don Brutzman',name='creator'),
    meta(content='1 January 2019',name='created'),
    meta(content='7 December 2024',name='modified'),
    meta(content='https://www.web3d.org/member-only/mantis/view.php?id=1255',name='reference'),
    meta(content='X3D encodings, ISO/IEC 19775-1, Part 1: Architecture and base components, Clause 42 Texture projector component, subclause 42.4.1 TextureProjector',name='specificationSection'),
    meta(content='https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/components/textureProjector.html#TextureProjector',name='specificationUrl'),
    meta(content='https://www.web3d.org/x3d/content/examples/Basic/X3dSpecifications/TextureProjectorComponentExample.x3d',name='identifier'),
    meta(content='X3D-Edit 4.0, https://savage.nps.edu/X3D-Edit',name='generator'),
    meta(content='../license.html',name='license')]),
  Scene=Scene(
    children=[
    TextureProjector(description='Project a red delicious texture',direction=(-1,0,-1),fieldOfView=0.26,location=(3,3,3),upVector=(0,1,0),ambientIntensity=0,color=(1,1,1),intensity=0.7854,
      texture=ImageTexture(DEF='RedDelicious',url=["Red_Delicious.jpg","https://www.web3d.org/x3d/content/examples/Basic/X3dSpecifications/Red_Delicious.jpg","https://upload.wikimedia.org/wikipedia/commons/f/fb/Red_Delicious.jpg"])),
    Shape(
      appearance=Appearance(
        material=Material(diffuseColor=(0.5,0.5,0.5))),
      geometry=IndexedFaceSet(coordIndex=[3,2,1,0,-1,4,5,2,3,-1,5,6,1,2,-1],solid=False,
        coord=Coordinate(point=[(1,0,1),(-1,0,1),(-1,0,-1),(1,0,-1),(1,1,-1),(-1,1,-1),(-1,1,1)]))),
    #  add additional node example for validation testing 
    TextureProjectorParallel(description='Project a red delicious texture, parallel orthographic projection',direction=(-1,0,-1),fieldOfView=(-1.5,-1.5,1.5,1.5),location=(3,3,3),ambientIntensity=0,color=(1,1,1),intensity=0.7854,upVector=(0,1,0),
      texture=ImageTexture(USE='RedDelicious'))])
) # X3D model complete

####################################################################################################
# Self-test diagnostics
####################################################################################################

print('Self-test diagnostics for TextureProjectorComponentExample.py:')
if        metaDiagnostics(newModel): # built-in utility method in X3D class
    print(metaDiagnostics(newModel)) # display meta info, hint, warning, error, TODO values in this model
# print('check newModel.XML() serialization...')
newModelXML= newModel.XML() # test export method XML() for exceptions during export
newModel.XMLvalidate()
# print(newModelXML) # diagnostic

try:
#   print('check newModel.VRML() serialization...')
    newModelVRML=newModel.VRML() # test export method VRML() for exceptions during export
    # print(prependLineNumbers(newModelVRML)) # debug
    print("Python-to-VRML export of VRML output successful", flush=True)
except Exception as err: # usually BaseException
    # https://stackoverflow.com/questions/18176602/how-to-get-the-name-of-an-exception-that-was-caught-in-python
    print("*** Python-to-VRML export of VRML output failed:", type(err).__name__, err)
    if newModelVRML: # may have failed to generate
        print(prependLineNumbers(newModelVRML, err.lineno))

try:
#   print('check newModel.JSON() serialization...')
    newModelJSON=newModel.JSON() # test export method JSON() for exceptions during export
#   print(prependLineNumbers(newModelJSON)) # debug
    print("Python-to-JSON export of JSON output successful (under development)")
except Exception as err: # usually SyntaxError
    print("*** Python-to-JSON export of JSON output failed:", type(err).__name__, err)
    if newModelJSON: # may have failed to generate
        print(prependLineNumbers(newModelJSON,err.lineno))

print("python TextureProjectorComponentExample.py load and self-test diagnostics complete.")
