####################################################################################################
#
# Invoking X3D model self-test:
#
#   $ python Robot.py
#
# Python package x3d.py package is available on PyPI for import.
#   This approach simplifies Python X3D deployment and use.
#   https://pypi.org/project/x3d
#
# Installation:
#       pip install x3d
# or
#       python -m pip install x3d
#
# Developer options for loading x3d package in other Python programs:
#
#    from x3d import *  # preferred approach, terser source that avoids x3d.* class prefixes
#
# or
#    import x3d         # traditional way to subclass x3d package, all classes require x3d.* prefix,
#                       # but python source is very verbose, for example x3d.Material x3d.Shape etc.
#                       # X3dToPython.xslt stylesheet insertPackagePrefix=true supports this option.
#
####################################################################################################

from x3d import *

newModel=X3D(profile='Immersive',version='3.3',
  head=head(
    children=[
    meta(content='Robot.x3d',name='title'),
    meta(content='X3D encodings example: Robot. Seek or zoom in towards the robot to start a simple animation sequence. This scene demonstrates nested prototype instances',name='description'),
    meta(content='Don Brutzman and Joe Williams',name='creator'),
    meta(content='1 June 2002',name='created'),
    meta(content='2 January 2025',name='modified'),
    meta(content='X3D encodings, ISO/IEC 19776-1.3, Part 1: XML encoding, Annex C.15 Robot',name='specificationSection'),
    meta(content='https://www.web3d.org/documents/specifications/19776-1/V3.3/Part01/examples.html#Robot',name='specificationUrl'),
    meta(content='https://www.web3d.org/x3d/content/examples/Basic/X3dSpecifications/Robot.x3d',name='identifier'),
    meta(content='X3D-Edit 3.3, https://savage.nps.edu/X3D-Edit',name='generator'),
    meta(content='../license.html',name='license')]),
  Scene=Scene(
    children=[
    WorldInfo(title='Robot.x3d'),
    ExternProtoDeclare(name='Shuttle',url=["ShuttlesAndPendulums.x3d#Shuttle","https://www.web3d.org/Specifications/Encodings/ShuttlesAndPendulums.x3d#Shuttle"],
      field=[
      field(accessType='initializeOnly',name='rate',type='SFTime'),
      field(accessType='initializeOnly',name='distance',type='SFFloat'),
      field(accessType='initializeOnly',name='children',type='MFNode'),
      field(accessType='inputOutput',name='startTime',type='SFTime'),
      field(accessType='inputOutput',name='stopTime',type='SFTime'),
      field(accessType='initializeOnly',name='loop',type='SFBool')]),
    ExternProtoDeclare(name='Pendulum',url=["ShuttlesAndPendulums.x3d#Pendulum","https://www.web3d.org/x3d/content/examples/Basic/X3dSpecifications/ShuttlesAndPendulums.x3d#Pendulum"],
      field=[
      field(accessType='initializeOnly',name='rate',type='SFTime'),
      field(accessType='initializeOnly',name='maxAngle',type='SFFloat'),
      field(accessType='initializeOnly',name='children',type='MFNode'),
      field(accessType='inputOutput',name='startTime',type='SFTime'),
      field(accessType='inputOutput',name='stopTime',type='SFTime'),
      field(accessType='initializeOnly',name='loop',type='SFBool')]),
    NavigationInfo(speed=20,type='"WALK" "ANY"'),
    Viewpoint(description='Watch Robot Stand',position=(0,0,150)),
    Viewpoint(description='Watch Robot Walk when close',position=(0,0,95)),
    ProximitySensor(DEF='Near',size=(200,200,200)),
    ProtoInstance(DEF='WALK',name='Shuttle',
      fieldValue=[
      fieldValue(name='stopTime',value=1),
      fieldValue(name='rate',value=10),
      fieldValue(name='distance',value=20),
      fieldValue(name='children',
        children=[
        Transform(rotation=(0,1,0,1.57),
          children=[
          Shape(
            appearance=Appearance(DEF='A',
              material=Material(diffuseColor=(0,0.5,0.7))),
            #  head 
            geometry=Box(),),
          Transform(scale=(1,5,1),translation=(0,-5,0),
            children=[
            Shape(
              appearance=Appearance(USE='A'),
              #  body 
              geometry=Sphere(),)]),
          Transform(rotation=(0,1,0,1.57),translation=(1.5,0,0),
            children=[
            ProtoInstance(DEF='Arm',name='Pendulum',
              fieldValue=[
              fieldValue(name='stopTime',value=1),
              fieldValue(name='rate',value=1),
              fieldValue(name='maxAngle',value=0.52),
              fieldValue(name='children',
                children=[
                Transform(translation=(0,-3,0),
                  children=[
                  Shape(
                    appearance=Appearance(USE='A'),
                    geometry=Cylinder(height=4,radius=0.5))])])])]),
          Transform(rotation=(0,-1,0,1.57),translation=(-1.5,0,0),
            #  name='Pendulum' no longer required for ProtoInstance USE nodes 
            children=[
            ProtoInstance(USE='Arm',name='Pendulum')])])])]),
    ROUTE(fromField='enterTime',fromNode='Near',toField='startTime',toNode='WALK'),
    ROUTE(fromField='enterTime',fromNode='Near',toField='startTime',toNode='Arm'),
    ROUTE(fromField='exitTime',fromNode='Near',toField='stopTime',toNode='WALK'),
    ROUTE(fromField='exitTime',fromNode='Near',toField='stopTime',toNode='Arm')])
) # X3D model complete

####################################################################################################
# Self-test diagnostics
####################################################################################################

print('Self-test diagnostics for Robot.py:')
if        metaDiagnostics(newModel): # built-in utility method in X3D class
    print(metaDiagnostics(newModel)) # display meta info, hint, warning, error, TODO values in this model
# print('check newModel.XML() serialization...')
newModelXML= newModel.XML() # test export method XML() for exceptions during export
newModel.XMLvalidate()
# print(newModelXML) # diagnostic

try:
#   print('check newModel.VRML() serialization...')
    newModelVRML=newModel.VRML() # test export method VRML() for exceptions during export
    # print(prependLineNumbers(newModelVRML)) # debug
    print("Python-to-VRML export of VRML output successful", flush=True)
except Exception as err: # usually BaseException
    # https://stackoverflow.com/questions/18176602/how-to-get-the-name-of-an-exception-that-was-caught-in-python
    print("*** Python-to-VRML export of VRML output failed:", type(err).__name__, err)
    if newModelVRML: # may have failed to generate
        print(prependLineNumbers(newModelVRML, err.lineno))

try:
#   print('check newModel.JSON() serialization...')
    newModelJSON=newModel.JSON() # test export method JSON() for exceptions during export
#   print(prependLineNumbers(newModelJSON)) # debug
    print("Python-to-JSON export of JSON output successful (under development)")
except Exception as err: # usually SyntaxError
    print("*** Python-to-JSON export of JSON output failed:", type(err).__name__, err)
    if newModelJSON: # may have failed to generate
        print(prependLineNumbers(newModelJSON,err.lineno))

print("python Robot.py load and self-test diagnostics complete.")
