####################################################################################################
#
# Invoking X3D model self-test:
#
#   $ python InlineExport.py
#
# Python package x3d.py package is available on PyPI for import.
#   This approach simplifies Python X3D deployment and use.
#   https://pypi.org/project/x3d
#
# Installation:
#       pip install x3d
# or
#       python -m pip install x3d
#
# Developer options for loading x3d package in other Python programs:
#
#    from x3d import *  # preferred approach, terser source that avoids x3d.* class prefixes
#
# or
#    import x3d         # traditional way to subclass x3d package, all classes require x3d.* prefix,
#                       # but python source is very verbose, for example x3d.Material x3d.Shape etc.
#                       # X3dToPython.xslt stylesheet insertPackagePrefix=true supports this option.
#
####################################################################################################

from x3d import *

newModel=X3D(profile='Immersive',version='4.0',
  head=head(
    children=[
    meta(content='InlineExport.x3d',name='title'),
    meta(content='X3D encodings example: demonstrate reuse and adaptation of content instances through Inline IMPORT, EXPORT statements. This scene does not animate, see InlineImport.x3d for selective animation of scene subgraph EXPORT localDEF=T1 AS=RootTransform',name='description'),
    meta(content='Don Brutzman',name='creator'),
    meta(content='27 May 2024',name='created'),
    meta(content='19 January 2025',name='modified'),
    meta(content='InlineImportInlineExportAnimation.png',name='Image'),
    meta(content='InlineImportInlineExportAnimation.vsdx',name='drawing'),
    meta(content='InlineExport.png',name='Image'),
    meta(content='InlineImport.x3d',name='reference'),
    meta(content='X3D Architecture version 4.0 ISO/IEC 19775-1, clause 4 Concepts, 4.4.6 Import/Export semantics',name='specificationSection'),
    meta(content='https://www.web3d.org/specifications/X3Dv4/ISO-IEC19775-1v4-IS/Part01/concepts.html#ImportExportsemantics',name='specificationUrl'),
    meta(content='X3D encodings, ISO/IEC 19776-1, Part 1: XML encoding, clause 4.3.2.6 EXPORT/IMPORT statement syntax',name='specificationSection'),
    meta(content='https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19776-1v4.0-WD1/Part01/concepts.html#EXPORT_IMPORT_StatementSyntax',name='specificationUrl'),
    meta(content='X3D encodings, ISO/IEC 19776-2, Part 1: ClassicVRML encoding, clause 4.3.2.6 EXPORT/IMPORT statement syntax',name='specificationSection'),
    meta(content='https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19776-2v4.0-WD1/Part02/concepts.html#EXPORT_IMPORT_StatementSyntax',name='specificationUrl'),
    meta(content='X3D-Edit 4.0, https://savage.nps.edu/X3D-Edit',name='generator'),
    meta(content='https://www.web3d.org/x3d/content/examples/Basic/X3dSpecifications/InlineExport.x3d',name='identifier'),
    meta(content='../license.html',name='license')]),
  Scene=Scene(
    children=[
    WorldInfo(title='InlineExport.x3d'),
    Background(skyColor=[(0.8,0.8,0.8)]),
    Transform(DEF='T1',translation=(0,1,0),
      children=[
      Shape(
        geometry=Box(),
        appearance=Appearance(
          material=Material(diffuseColor=(0.8,0.8,0.2)))),
      Transform(translation=(2,0,0),
        children=[
        Shape(DEF='SmallerSphere',
          geometry=Sphere(radius=0.2),
          appearance=Appearance(
            material=Material(diffuseColor=(0.2,0.8,0.8))))]),
      Transform(translation=(-2,0,0),
        children=[
        Shape(USE='SmallerSphere')])]),
    #  Hint: EXPORT statement must follow the DEF node of interest, since localDEF field is similar to USE (with data type IDREF). 
    #  Hint: EXPORT localDEF is DEF name within this model, AS is the name exposed to an external Inline IMPORT statement. 
    EXPORT(AS='RootTransform',localDEF='T1'),
    Transform(DEF='T2',translation=(0,-2,0),
      children=[
      Shape(
        geometry=Text(string=["Shapes can rotate if","animation events are sent","via Inline IMPORT/EXPORT"],
          fontStyle=FontStyle(family=["SANS"],justify=["MIDDLE","MIDDLE"],size=0.6,style_='BOLD')),
        appearance=Appearance(
          material=Material(diffuseColor=(0.2,0.2,0.7))))])])
) # X3D model complete

####################################################################################################
# Self-test diagnostics
####################################################################################################

print('Self-test diagnostics for InlineExport.py:')
if        metaDiagnostics(newModel): # built-in utility method in X3D class
    print(metaDiagnostics(newModel)) # display meta info, hint, warning, error, TODO values in this model
# print('check newModel.XML() serialization...')
newModelXML= newModel.XML() # test export method XML() for exceptions during export
newModel.XMLvalidate()
# print(newModelXML) # diagnostic

try:
#   print('check newModel.VRML() serialization...')
    newModelVRML=newModel.VRML() # test export method VRML() for exceptions during export
    # print(prependLineNumbers(newModelVRML)) # debug
    print("Python-to-VRML export of VRML output successful", flush=True)
except Exception as err: # usually BaseException
    # https://stackoverflow.com/questions/18176602/how-to-get-the-name-of-an-exception-that-was-caught-in-python
    print("*** Python-to-VRML export of VRML output failed:", type(err).__name__, err)
    if newModelVRML: # may have failed to generate
        print(prependLineNumbers(newModelVRML, err.lineno))

try:
#   print('check newModel.JSON() serialization...')
    newModelJSON=newModel.JSON() # test export method JSON() for exceptions during export
#   print(prependLineNumbers(newModelJSON)) # debug
    print("Python-to-JSON export of JSON output successful (under development)")
except Exception as err: # usually SyntaxError
    print("*** Python-to-JSON export of JSON output failed:", type(err).__name__, err)
    if newModelJSON: # may have failed to generate
        print(prependLineNumbers(newModelJSON,err.lineno))

print("python InlineExport.py load and self-test diagnostics complete.")
