####################################################################################################
#
# Invoking X3D model self-test:
#
#   $ python ECMAScriptSaiAddDynamicRoutes.py
#
# Python package x3d.py package is available on PyPI for import.
#   This approach simplifies Python X3D deployment and use.
#   https://pypi.org/project/x3d
#
# Installation:
#       pip install x3d
# or
#       python -m pip install x3d
#
# Developer options for loading x3d package in other Python programs:
#
#    from x3d import *  # preferred approach, terser source that avoids x3d.* class prefixes
#
# or
#    import x3d         # traditional way to subclass x3d package, all classes require x3d.* prefix,
#                       # but python source is very verbose, for example x3d.Material x3d.Shape etc.
#                       # X3dToPython.xslt stylesheet insertPackagePrefix=true supports this option.
#
####################################################################################################

from x3d import *

newModel=X3D(profile='Immersive',version='3.3',
  head=head(
    children=[
    meta(content='ECMAScriptSaiAddDynamicRoutes.x3d',name='title'),
    meta(content='Java Scene Authoring Interface (JSAI) scene and source code example that show adding dynamic ROUTEs to a scene.',name='description'),
    meta(content='X3D Working Group',name='creator'),
    meta(content='23 February 2005',name='created'),
    meta(content='28 October 2019',name='modified'),
    meta(content='https://www.web3d.org/files/specifications/19777-1/V3.3/Part1/X3D_ECMAScript.html',name='reference'),
    meta(content='X3D language bindings, ISO/IEC 19777-1, Part 1: ECMAScript, B.5 Add dynamic routes',name='specificationSection'),
    meta(content='https://www.web3d.org/files/specifications/19777-1/V3.3/Part1/examples.html#AddDynamicRoutes',name='specificationUrl'),
    meta(content='X3D-Edit 3.3, https://savage.nps.edu/X3D-Edit',name='generator'),
    meta(content='https://www.web3d.org/x3d/content/examples/Basic/X3dSpecifications/ECMAScriptSaiAddDynamicRoutes.x3d',name='identifier'),
    meta(content='../license.html',name='license')]),
  Scene=Scene(
    children=[
    WorldInfo(title='ECMAScriptSaiAddDynamicRoutes.x3d'),
    Script(DEF='SC',url=["ECMAScriptSaiAddDynamicRoutes.js","https://www.web3d.org/x3d/content/examples/Basic/X3dSpecifications/ECMAScriptSaiAddDynamicRoutes.js"],
      field=[
      field(accessType='inputOnly',name='touchTime',type='SFTime'),
      field(accessType='initializeOnly',name='Script',type='SFNode')],

    sourceCode="""
ecmascript:

function touchTime(value) {
    // Create nodes directly in the parent scene
    var scene = Browser.currentScene;

    var shape = scene.createNode('Shape');
    var box   = scene.createNode('Box');
    var touchSensor = scene.createNode('TouchSensor');
    shape.geometry = box;

    // Create a Group to hold the nodes
    var group = scene.createNode('Group');

    // Add the shape and sensor to the group
    group.children = new MFNode(shape, touchSensor);

    // Add the nodes to the scene
    scene.RootNodes[0] = group;

    // Add a route from the touchSensor to this script
    scene.addRoute(touchSensor, 'touchTime', Script, 'touchTime');
}
""")])
) # X3D model complete

####################################################################################################
# Self-test diagnostics
####################################################################################################

print('Self-test diagnostics for ECMAScriptSaiAddDynamicRoutes.py:')
if        metaDiagnostics(newModel): # built-in utility method in X3D class
    print(metaDiagnostics(newModel)) # display meta info, hint, warning, error, TODO values in this model
# print('check newModel.XML() serialization...')
newModelXML= newModel.XML() # test export method XML() for exceptions during export
newModel.XMLvalidate()
# print(newModelXML) # diagnostic

try:
#   print('check newModel.VRML() serialization...')
    newModelVRML=newModel.VRML() # test export method VRML() for exceptions during export
    # print(prependLineNumbers(newModelVRML)) # debug
    print("Python-to-VRML export of VRML output successful", flush=True)
except Exception as err: # usually BaseException
    # https://stackoverflow.com/questions/18176602/how-to-get-the-name-of-an-exception-that-was-caught-in-python
    print("*** Python-to-VRML export of VRML output failed:", type(err).__name__, err)
    if newModelVRML: # may have failed to generate
        print(prependLineNumbers(newModelVRML, err.lineno))

try:
#   print('check newModel.JSON() serialization...')
    newModelJSON=newModel.JSON() # test export method JSON() for exceptions during export
#   print(prependLineNumbers(newModelJSON)) # debug
    print("Python-to-JSON export of JSON output successful (under development)")
except Exception as err: # usually SyntaxError
    print("*** Python-to-JSON export of JSON output failed:", type(err).__name__, err)
    if newModelJSON: # may have failed to generate
        print(prependLineNumbers(newModelJSON,err.lineno))

print("python ECMAScriptSaiAddDynamicRoutes.py load and self-test diagnostics complete.")
