####################################################################################################
#
# Invoking X3D model self-test:
#
#   $ python Example14.py
#
# Python package x3d.py package is available on PyPI for import.
#   This approach simplifies Python X3D deployment and use.
#   https://pypi.org/project/x3d
#
# Installation:
#       pip install x3d
# or
#       python -m pip install x3d
#
# Developer options for loading x3d package in other Python programs:
#
#    from x3d import *  # preferred approach, terser source that avoids x3d.* class prefixes
#
# or
#    import x3d         # traditional way to subclass x3d package, all classes require x3d.* prefix,
#                       # but python source is very verbose, for example x3d.Material x3d.Shape etc.
#                       # X3dToPython.xslt stylesheet insertPackagePrefix=true supports this option.
#
####################################################################################################

from x3d import *

newModel=X3D(profile='Immersive',version='3.0',
  head=head(
    children=[
    meta(content='Example14.x3d',name='title'),
    meta(content='Don Brutzman',name='creator'),
    meta(content='22 November 2000',name='created'),
    meta(content='26 November 2015',name='modified'),
    meta(content='X3D working group',name='creator'),
    meta(content='VRML 97 specification example: animation of shuttle and pendulum prototype nodes.',name='description'),
    meta(content='originals/exampleD_14.wrl',name='reference'),
    meta(content='VRML 97, ISO/IEC 14772-1, Part 1: Functional specification and UTF-8 encoding, D.14 Shuttles and pendulums',name='specificationSection'),
    meta(content='http://www.web3d.org/documents/specifications/14772/V2.0/part1/examples.html#D.14',name='specificationUrl'),
    meta(content='http://www.web3d.org/x3d/content/examples/Basic/Vrml97Specification/Example14.x3d',name='identifier'),
    meta(content='X3D-Edit 3.3, https://savage.nps.edu/X3D-Edit',name='generator'),
    meta(content='../license.html',name='license')]),
  Scene=Scene(
    children=[
    ProtoDeclare(name='Shuttle',
      ProtoInterface=ProtoInterface(
        field=[
        field(accessType='initializeOnly',name='rate',type='SFTime',value=1),
        field(accessType='initializeOnly',name='distance',type='SFFloat',value=1),
        field(accessType='initializeOnly',name='children',type='MFNode'),
        field(accessType='inputOutput',name='startTime',type='SFTime',value=0),
        field(accessType='inputOutput',name='stopTime',type='SFTime',value=0),
        field(accessType='initializeOnly',name='loop',type='SFBool',value=True)]),
      ProtoBody=ProtoBody(
        children=[
        Transform(DEF='F',
          IS=IS(
            connect=[
            connect(nodeField='children',protoField='children')])),
        TimeSensor(DEF='T',
          IS=IS(
            connect=[
            connect(nodeField='cycleInterval',protoField='rate'),
            connect(nodeField='startTime',protoField='startTime'),
            connect(nodeField='stopTime',protoField='stopTime'),
            connect(nodeField='loop',protoField='loop')])),
        Script(DEF='S',
          field=[
          field(accessType='initializeOnly',name='distance',type='SFFloat'),
          field(accessType='outputOnly',name='position',type='MFVec3f')],
          IS=IS(
            connect=[
            connect(nodeField='distance',protoField='distance')]),

        sourceCode="""
ecmascript:

function initialize () {     // constructor:setup interpolator
	pos1 = new SFVec3f(-distance, 0, 0);
	pos2 = new SFVec3f(distance, 0, 0);
	position = new MFVec3f(pos1, pos2, pos1);
}
"""),
        PositionInterpolator(DEF='I',key=[0,0.5,1],keyValue=[(-1,0,0),(1,0,0),(-1,0,0)]),
        ROUTE(fromField='fraction_changed',fromNode='T',toField='set_fraction',toNode='I'),
        ROUTE(fromField='value_changed',fromNode='I',toField='set_translation',toNode='F'),
        ROUTE(fromField='position',fromNode='S',toField='set_keyValue',toNode='I')])),
    ProtoDeclare(name='Pendulum',
      ProtoInterface=ProtoInterface(
        field=[
        field(accessType='initializeOnly',name='rate',type='SFTime',value=1),
        field(accessType='initializeOnly',name='maxAngle',type='SFFloat',value=3.14159),
        field(accessType='initializeOnly',name='children',type='MFNode'),
        field(accessType='inputOutput',name='startTime',type='SFTime',value=0),
        field(accessType='inputOutput',name='stopTime',type='SFTime',value=0),
        field(accessType='initializeOnly',name='loop',type='SFBool',value=True)]),
      ProtoBody=ProtoBody(
        children=[
        Transform(DEF='FF',
          IS=IS(
            connect=[
            connect(nodeField='children',protoField='children')])),
        TimeSensor(DEF='TT',
          IS=IS(
            connect=[
            connect(nodeField='cycleInterval',protoField='rate'),
            connect(nodeField='startTime',protoField='startTime'),
            connect(nodeField='stopTime',protoField='stopTime'),
            connect(nodeField='loop',protoField='loop')])),
        Script(DEF='SS',
          field=[
          field(accessType='initializeOnly',name='maxAngle',type='SFFloat'),
          field(accessType='outputOnly',name='rotation',type='MFRotation')],
          IS=IS(
            connect=[
            connect(nodeField='maxAngle',protoField='maxAngle')]),

        sourceCode="""
ecmascript:

function initialize() {     // constructor: setup interpolator 
	rot1 = new SFRotation(0, 0, 1, 0);
	rot2 = new SFRotation(0, 0, 1, maxAngle/2);
	rot3 = new SFRotation(0, 0, 1, maxAngle);
	rotation = new MFRotation(rot1, rot2, rot3, rot2, rot1);
}
"""),
        OrientationInterpolator(DEF='II',key=[0,0.25,0.5,0.75,1],keyValue=[(0,0,1,0),(0,0,1,1.57),(0,0,1,3.14),(0,0,1,1.57),(0,0,1,0)]),
        ROUTE(fromField='fraction_changed',fromNode='TT',toField='set_fraction',toNode='II'),
        ROUTE(fromField='value_changed',fromNode='II',toField='set_rotation',toNode='FF'),
        ROUTE(fromField='rotation',fromNode='SS',toField='set_keyValue',toNode='II')])),
    Transform(translation=(-3,0,0),
      children=[
      ProtoInstance(name='Pendulum',
        fieldValue=[
        fieldValue(name='rate',value=3),
        fieldValue(name='maxAngle',value=6.28),
        fieldValue(name='children',
          children=[
          Shape(
            geometry=Cylinder(height=5),
            appearance=Appearance(
              material=Material(diffuseColor=(0.8,0.1,0))))])])]),
    Transform(
      children=[
      ProtoInstance(name='Shuttle',
        fieldValue=[
        fieldValue(name='rate',value=2),
        fieldValue(name='children',
          children=[
          Shape(
            geometry=Sphere(),
            appearance=Appearance(
              material=Material(diffuseColor=(0.1,0.8,0))))])])])])
) # X3D model complete

####################################################################################################
# Self-test diagnostics
####################################################################################################

print('Self-test diagnostics for Example14.py:')
if        metaDiagnostics(newModel): # built-in utility method in X3D class
    print(metaDiagnostics(newModel)) # display meta info, hint, warning, error, TODO values in this model
# print('check newModel.XML() serialization...')
newModelXML= newModel.XML() # test export method XML() for exceptions during export
newModel.XMLvalidate()
# print(newModelXML) # diagnostic

try:
#   print('check newModel.VRML() serialization...')
    newModelVRML=newModel.VRML() # test export method VRML() for exceptions during export
    # print(prependLineNumbers(newModelVRML)) # debug
    print("Python-to-VRML export of VRML output successful", flush=True)
except Exception as err: # usually BaseException
    # https://stackoverflow.com/questions/18176602/how-to-get-the-name-of-an-exception-that-was-caught-in-python
    print("*** Python-to-VRML export of VRML output failed:", type(err).__name__, err)
    if newModelVRML: # may have failed to generate
        print(prependLineNumbers(newModelVRML, err.lineno))

try:
#   print('check newModel.JSON() serialization...')
    newModelJSON=newModel.JSON() # test export method JSON() for exceptions during export
#   print(prependLineNumbers(newModelJSON)) # debug
    print("Python-to-JSON export of JSON output successful (under development)")
except Exception as err: # usually SyntaxError
    print("*** Python-to-JSON export of JSON output failed:", type(err).__name__, err)
    if newModelJSON: # may have failed to generate
        print(prependLineNumbers(newModelJSON,err.lineno))

print("python Example14.py load and self-test diagnostics complete.")
