####################################################################################################
#
# Invoking X3D model self-test:
#
#   $ python CaliforniaCampusesTour5Altitudes.py
#
# Python package x3d.py package is available on PyPI for import.
#   This approach simplifies Python X3D deployment and use.
#   https://pypi.org/project/x3d
#
# Installation:
#       pip install x3d
# or
#       python -m pip install x3d
#
# Developer options for loading x3d package in other Python programs:
#
#    from x3d import *  # preferred approach, terser source that avoids x3d.* class prefixes
#
# or
#    import x3d         # traditional way to subclass x3d package, all classes require x3d.* prefix,
#                       # but python source is very verbose, for example x3d.Material x3d.Shape etc.
#                       # X3dToPython.xslt stylesheet insertPackagePrefix=true supports this option.
#
####################################################################################################

from x3d import *

newModel=X3D(profile='Immersive',version='3.3',
  head=head(
    children=[
    component(level=1,name='Geospatial'),
    meta(content='CaliforniaCampusesTour5Altitudes.x3d',name='title'),
    meta(content='Viewpoints and tour of California campuses',name='description'),
    meta(content='Don Brutzman, Dale Tourtelotte, Mike Bailey, Don McGregor',name='creator'),
    meta(content='4 March 2010',name='created'),
    meta(content='20 October 2019',name='modified'),
    meta(content='CaliforniaCampuses.kml',name='reference'),
    meta(content='https://www.web3d.org/x3d-earth',name='reference'),
    meta(content='http://x3d-earth.nps.edu',name='reference'),
    meta(content='http://hamming.uc.nps.edu',name='reference'),
    meta(content='X3D Earth',name='subject'),
    meta(content='https://www.web3d.org/x3d/content/examples/Basic/Geospatial/CaliforniaCampusesTour5Altitudes.x3d',name='identifier'),
    meta(content='http://mmog.ern.nps.edu/California/California.x3d',name='reference'),
    meta(content='under development',name='warning'),
    meta(content='mmog.ern.nps.edu restricted to internal access within NPS firewall only',name='warning'),
    meta(content='KmlToX3dViewpoints.xslt',name='generator'),
    meta(content='X3D-Edit 3.3, https://savage.nps.edu/X3D-Edit',name='generator'),
    meta(content='https://savage.nps.edu/Savage/license.html',name='license')]),
  Scene=Scene(
    children=[
    WorldInfo(title='CaliforniaCampusesTour5Altitudes.x3d'),
    Group(DEF='LightGroup'),
    Switch(DEF='SwitchGlobes',whichChoice=-1,
      #  Select globe of interest. Warning: may need to keep alternate globes commented out to prevent browser caching. 
      #  <Inline load='true' url='"CaliforniaDemo.x3d"'/> 
      #  <Inline load='true' url='"http://x3d-earth.nps.edu/osmdemo.x3d"'/> 
      #  <Inline load='false' url='"http://mmog.ern.nps.edu/California/California.x3d"'/> 
      #  <Inline load='false' url='"http://x3d-earth.nps.edu/7_levels_plus/tiles/0/globe.x3d"'/> 
      #  <Inline load='false' url='"http://x3d-earth.nps.edu/globe/MBARI1MinuteBathy/world.x3d"'/> 
      #  <Inline load='false' url='"http://x3d-earth.nps.edu/globe/SRTM30Plus/world.x3d"'/> 
      ),
    NavigationInfo(transitionType=["ANIMATE"]),
    Group(DEF='PlacemarkGroup',
      children=[
      GeoViewpoint(DEF='View01a',description='Naval Postgraduate School 200km',orientation=(1,0,0,-1.57),position=(36.595599,-121.877148,200000)),
      GeoViewpoint(DEF='View01b',description='Naval Postgraduate School 100km',orientation=(1,0,0,-1.57),position=(36.595599,-121.877148,100000)),
      GeoViewpoint(DEF='View01c',description='Naval Postgraduate School 50km',orientation=(1,0,0,-1.57),position=(36.595599,-121.877148,50000)),
      GeoViewpoint(DEF='View01d',description='Naval Postgraduate School 25km',orientation=(1,0,0,-1.57),position=(36.595599,-121.877148,25000)),
      GeoViewpoint(DEF='View01e',description='Naval Postgraduate School 10km',orientation=(1,0,0,-1.57),position=(36.595599,-121.877148,10000)),
      GeoViewpoint(DEF='View02a',description='University of California Davis 200km',orientation=(1,0,0,-1.57),position=(38.53650615157984,-121.7489628616831,200000)),
      GeoViewpoint(DEF='View02b',description='University of California Davis 100km',orientation=(1,0,0,-1.57),position=(38.53650615157984,-121.7489628616831,100000)),
      GeoViewpoint(DEF='View02c',description='University of California Davis 50km',orientation=(1,0,0,-1.57),position=(38.53650615157984,-121.7489628616831,50000)),
      GeoViewpoint(DEF='View02d',description='University of California Davis 25km',orientation=(1,0,0,-1.57),position=(38.53650615157984,-121.7489628616831,25000)),
      GeoViewpoint(DEF='View02e',description='University of California Davis 10km',orientation=(1,0,0,-1.57),position=(38.53650615157984,-121.7489628616831,10000)),
      GeoViewpoint(DEF='View03a',description='University of California Berkeley 200km',orientation=(1,0,0,-1.57),position=(37.86963434512325,-122.2593873127355,200000)),
      GeoViewpoint(DEF='View03b',description='University of California Berkeley 100km',orientation=(1,0,0,-1.57),position=(37.86963434512325,-122.2593873127355,100000)),
      GeoViewpoint(DEF='View03c',description='University of California Berkeley 50km',orientation=(1,0,0,-1.57),position=(37.86963434512325,-122.2593873127355,50000)),
      GeoViewpoint(DEF='View03d',description='University of California Berkeley 25km',orientation=(1,0,0,-1.57),position=(37.86963434512325,-122.2593873127355,25000)),
      GeoViewpoint(DEF='View03e',description='University of California Berkeley 10km',orientation=(1,0,0,-1.57),position=(37.86963434512325,-122.2593873127355,10000)),
      GeoViewpoint(DEF='View04a',description='University of California San Francisco 200km',orientation=(1,0,0,-1.57),position=(37.76340647188392,-122.4582475377715,200000)),
      GeoViewpoint(DEF='View04b',description='University of California San Francisco 100km',orientation=(1,0,0,-1.57),position=(37.76340647188392,-122.4582475377715,100000)),
      GeoViewpoint(DEF='View04c',description='University of California San Francisco 50km',orientation=(1,0,0,-1.57),position=(37.76340647188392,-122.4582475377715,50000)),
      GeoViewpoint(DEF='View04d',description='University of California San Francisco 25km',orientation=(1,0,0,-1.57),position=(37.76340647188392,-122.4582475377715,25000)),
      GeoViewpoint(DEF='View04e',description='University of California San Francisco 10km',orientation=(1,0,0,-1.57),position=(37.76340647188392,-122.4582475377715,10000)),
      GeoViewpoint(DEF='View05a',description='University of California Santa Cruz 200km',orientation=(1,0,0,-1.57),position=(36.99628443046043,-122.0534325473795,200000)),
      GeoViewpoint(DEF='View05b',description='University of California Santa Cruz 100km',orientation=(1,0,0,-1.57),position=(36.99628443046043,-122.0534325473795,100000)),
      GeoViewpoint(DEF='View05c',description='University of California Santa Cruz 50km',orientation=(1,0,0,-1.57),position=(36.99628443046043,-122.0534325473795,50000)),
      GeoViewpoint(DEF='View05d',description='University of California Santa Cruz 25km',orientation=(1,0,0,-1.57),position=(36.99628443046043,-122.0534325473795,25000)),
      GeoViewpoint(DEF='View05e',description='University of California Santa Cruz 10km',orientation=(1,0,0,-1.57),position=(36.99628443046043,-122.0534325473795,10000)),
      GeoViewpoint(DEF='View06a',description='University of California Merced 200km',orientation=(1,0,0,-1.57),position=(37.365508575977,-120.4252597059142,200000)),
      GeoViewpoint(DEF='View06b',description='University of California Merced 100km',orientation=(1,0,0,-1.57),position=(37.365508575977,-120.4252597059142,100000)),
      GeoViewpoint(DEF='View06c',description='University of California Merced 50km',orientation=(1,0,0,-1.57),position=(37.365508575977,-120.4252597059142,50000)),
      GeoViewpoint(DEF='View06d',description='University of California Merced 25km',orientation=(1,0,0,-1.57),position=(37.365508575977,-120.4252597059142,25000)),
      GeoViewpoint(DEF='View06e',description='University of California Merced 10km',orientation=(1,0,0,-1.57),position=(37.365508575977,-120.4252597059142,10000)),
      GeoViewpoint(DEF='View07a',description='University of California Santa Barbara 200km',orientation=(1,0,0,-1.57),position=(34.41124450961521,-119.8479940053906,200000)),
      GeoViewpoint(DEF='View07b',description='University of California Santa Barbara 100km',orientation=(1,0,0,-1.57),position=(34.41124450961521,-119.8479940053906,100000)),
      GeoViewpoint(DEF='View07c',description='University of California Santa Barbara 50km',orientation=(1,0,0,-1.57),position=(34.41124450961521,-119.8479940053906,50000)),
      GeoViewpoint(DEF='View07d',description='University of California Santa Barbara 25km',orientation=(1,0,0,-1.57),position=(34.41124450961521,-119.8479940053906,25000)),
      GeoViewpoint(DEF='View07e',description='University of California Santa Barbara 10km',orientation=(1,0,0,-1.57),position=(34.41124450961521,-119.8479940053906,10000)),
      GeoViewpoint(DEF='View08a',description='University of California Los Angeles 200km',orientation=(1,0,0,-1.57),position=(34.07224474392262,-118.4408472225642,200000)),
      GeoViewpoint(DEF='View08b',description='University of California Los Angeles 100km',orientation=(1,0,0,-1.57),position=(34.07224474392262,-118.4408472225642,100000)),
      GeoViewpoint(DEF='View08c',description='University of California Los Angeles 50km',orientation=(1,0,0,-1.57),position=(34.07224474392262,-118.4408472225642,50000)),
      GeoViewpoint(DEF='View08d',description='University of California Los Angeles 25km',orientation=(1,0,0,-1.57),position=(34.07224474392262,-118.4408472225642,25000)),
      GeoViewpoint(DEF='View08e',description='University of California Los Angeles 10km',orientation=(1,0,0,-1.57),position=(34.07224474392262,-118.4408472225642,10000)),
      GeoViewpoint(DEF='View09a',description='University of California Irvine 200km',orientation=(1,0,0,-1.57),position=(33.64623283675919,-117.8427064139082,200000)),
      GeoViewpoint(DEF='View09b',description='University of California Irvine 100km',orientation=(1,0,0,-1.57),position=(33.64623283675919,-117.8427064139082,100000)),
      GeoViewpoint(DEF='View09c',description='University of California Irvine 50km',orientation=(1,0,0,-1.57),position=(33.64623283675919,-117.8427064139082,50000)),
      GeoViewpoint(DEF='View09d',description='University of California Irvine 25km',orientation=(1,0,0,-1.57),position=(33.64623283675919,-117.8427064139082,25000)),
      GeoViewpoint(DEF='View09e',description='University of California Irvine 10km',orientation=(1,0,0,-1.57),position=(33.64623283675919,-117.8427064139082,10000)),
      GeoViewpoint(DEF='View10a',description='University of California Riverside 200km',orientation=(1,0,0,-1.57),position=(33.97350567066717,-117.3281649569839,200000)),
      GeoViewpoint(DEF='View10b',description='University of California Riverside 100km',orientation=(1,0,0,-1.57),position=(33.97350567066717,-117.3281649569839,100000)),
      GeoViewpoint(DEF='View10c',description='University of California Riverside 50km',orientation=(1,0,0,-1.57),position=(33.97350567066717,-117.3281649569839,50000)),
      GeoViewpoint(DEF='View10d',description='University of California Riverside 25km',orientation=(1,0,0,-1.57),position=(33.97350567066717,-117.3281649569839,25000)),
      GeoViewpoint(DEF='View10e',description='University of California Riverside 10km',orientation=(1,0,0,-1.57),position=(33.97350567066717,-117.3281649569839,10000)),
      GeoViewpoint(DEF='View11a',description='University of California San Diego 200km',orientation=(1,0,0,-1.57),position=(32.87612136607509,-117.2367298240259,200000)),
      GeoViewpoint(DEF='View11b',description='University of California San Diego 100km',orientation=(1,0,0,-1.57),position=(32.87612136607509,-117.2367298240259,100000)),
      GeoViewpoint(DEF='View11c',description='University of California San Diego 50km',orientation=(1,0,0,-1.57),position=(32.87612136607509,-117.2367298240259,50000)),
      GeoViewpoint(DEF='View11d',description='University of California San Diego 25km',orientation=(1,0,0,-1.57),position=(32.87612136607509,-117.2367298240259,25000)),
      GeoViewpoint(DEF='View11e',description='University of California San Diego 10km',orientation=(1,0,0,-1.57),position=(32.87612136607509,-117.2367298240259,10000)),
      GeoViewpoint(DEF='View12a',description='Humboldt State University 200km',orientation=(1,0,0,-1.57),position=(40.87630235623448,-124.0785789217266,200000)),
      GeoViewpoint(DEF='View12b',description='Humboldt State University 100km',orientation=(1,0,0,-1.57),position=(40.87630235623448,-124.0785789217266,100000)),
      GeoViewpoint(DEF='View12c',description='Humboldt State University 50km',orientation=(1,0,0,-1.57),position=(40.87630235623448,-124.0785789217266,50000)),
      GeoViewpoint(DEF='View12d',description='Humboldt State University 25km',orientation=(1,0,0,-1.57),position=(40.87630235623448,-124.0785789217266,25000)),
      GeoViewpoint(DEF='View12e',description='Humboldt State University 10km',orientation=(1,0,0,-1.57),position=(40.87630235623448,-124.0785789217266,10000)),
      GeoViewpoint(DEF='View13a',description='California State University, Chico 200km',orientation=(1,0,0,-1.57),position=(39.73031527724385,-121.8453722745223,200000)),
      GeoViewpoint(DEF='View13b',description='California State University, Chico 100km',orientation=(1,0,0,-1.57),position=(39.73031527724385,-121.8453722745223,100000)),
      GeoViewpoint(DEF='View13c',description='California State University, Chico 50km',orientation=(1,0,0,-1.57),position=(39.73031527724385,-121.8453722745223,50000)),
      GeoViewpoint(DEF='View13d',description='California State University, Chico 25km',orientation=(1,0,0,-1.57),position=(39.73031527724385,-121.8453722745223,25000)),
      GeoViewpoint(DEF='View13e',description='California State University, Chico 10km',orientation=(1,0,0,-1.57),position=(39.73031527724385,-121.8453722745223,10000)),
      GeoViewpoint(DEF='View14a',description='California State University Sacramento 200km',orientation=(1,0,0,-1.57),position=(38.5626517618963,-121.4244636520556,200000)),
      GeoViewpoint(DEF='View14b',description='California State University Sacramento 100km',orientation=(1,0,0,-1.57),position=(38.5626517618963,-121.4244636520556,100000)),
      GeoViewpoint(DEF='View14c',description='California State University Sacramento 50km',orientation=(1,0,0,-1.57),position=(38.5626517618963,-121.4244636520556,50000)),
      GeoViewpoint(DEF='View14d',description='California State University Sacramento 25km',orientation=(1,0,0,-1.57),position=(38.5626517618963,-121.4244636520556,25000)),
      GeoViewpoint(DEF='View14e',description='California State University Sacramento 10km',orientation=(1,0,0,-1.57),position=(38.5626517618963,-121.4244636520556,10000)),
      GeoViewpoint(DEF='View15a',description='Sonoma State University 200km',orientation=(1,0,0,-1.57),position=(38.33922929793606,-122.6744333530031,200000)),
      GeoViewpoint(DEF='View15b',description='Sonoma State University 100km',orientation=(1,0,0,-1.57),position=(38.33922929793606,-122.6744333530031,100000)),
      GeoViewpoint(DEF='View15c',description='Sonoma State University 50km',orientation=(1,0,0,-1.57),position=(38.33922929793606,-122.6744333530031,50000)),
      GeoViewpoint(DEF='View15d',description='Sonoma State University 25km',orientation=(1,0,0,-1.57),position=(38.33922929793606,-122.6744333530031,25000)),
      GeoViewpoint(DEF='View15e',description='Sonoma State University 10km',orientation=(1,0,0,-1.57),position=(38.33922929793606,-122.6744333530031,10000)),
      GeoViewpoint(DEF='View16a',description='California Maritime Academy 200km',orientation=(1,0,0,-1.57),position=(38.06907065059484,-122.2309147135222,200000)),
      GeoViewpoint(DEF='View16b',description='California Maritime Academy 100km',orientation=(1,0,0,-1.57),position=(38.06907065059484,-122.2309147135222,100000)),
      GeoViewpoint(DEF='View16c',description='California Maritime Academy 50km',orientation=(1,0,0,-1.57),position=(38.06907065059484,-122.2309147135222,50000)),
      GeoViewpoint(DEF='View16d',description='California Maritime Academy 25km',orientation=(1,0,0,-1.57),position=(38.06907065059484,-122.2309147135222,25000)),
      GeoViewpoint(DEF='View16e',description='California Maritime Academy 10km',orientation=(1,0,0,-1.57),position=(38.06907065059484,-122.2309147135222,10000)),
      GeoViewpoint(DEF='View17a',description='San Francisco State University 200km',orientation=(1,0,0,-1.57),position=(37.722876,-122.4781512190666,200000)),
      GeoViewpoint(DEF='View17b',description='San Francisco State University 100km',orientation=(1,0,0,-1.57),position=(37.722876,-122.4781512190666,100000)),
      GeoViewpoint(DEF='View17c',description='San Francisco State University 50km',orientation=(1,0,0,-1.57),position=(37.722876,-122.4781512190666,50000)),
      GeoViewpoint(DEF='View17d',description='San Francisco State University 25km',orientation=(1,0,0,-1.57),position=(37.722876,-122.4781512190666,25000)),
      GeoViewpoint(DEF='View17e',description='San Francisco State University 10km',orientation=(1,0,0,-1.57),position=(37.722876,-122.4781512190666,10000)),
      GeoViewpoint(DEF='View18a',description='California State University, East Bay 200km',orientation=(1,0,0,-1.57),position=(37.65707914139907,-122.0569190659761,200000)),
      GeoViewpoint(DEF='View18b',description='California State University, East Bay 100km',orientation=(1,0,0,-1.57),position=(37.65707914139907,-122.0569190659761,100000)),
      GeoViewpoint(DEF='View18c',description='California State University, East Bay 50km',orientation=(1,0,0,-1.57),position=(37.65707914139907,-122.0569190659761,50000)),
      GeoViewpoint(DEF='View18d',description='California State University, East Bay 25km',orientation=(1,0,0,-1.57),position=(37.65707914139907,-122.0569190659761,25000)),
      GeoViewpoint(DEF='View18e',description='California State University, East Bay 10km',orientation=(1,0,0,-1.57),position=(37.65707914139907,-122.0569190659761,10000)),
      GeoViewpoint(DEF='View19a',description='California State University Stanislaus 200km',orientation=(1,0,0,-1.57),position=(37.52552478005697,-120.8563508560606,200000)),
      GeoViewpoint(DEF='View19b',description='California State University Stanislaus 100km',orientation=(1,0,0,-1.57),position=(37.52552478005697,-120.8563508560606,100000)),
      GeoViewpoint(DEF='View19c',description='California State University Stanislaus 50km',orientation=(1,0,0,-1.57),position=(37.52552478005697,-120.8563508560606,50000)),
      GeoViewpoint(DEF='View19d',description='California State University Stanislaus 25km',orientation=(1,0,0,-1.57),position=(37.52552478005697,-120.8563508560606,25000)),
      GeoViewpoint(DEF='View19e',description='California State University Stanislaus 10km',orientation=(1,0,0,-1.57),position=(37.52552478005697,-120.8563508560606,10000)),
      GeoViewpoint(DEF='View20a',description='San Jose State University 200km',orientation=(1,0,0,-1.57),position=(37.33580940527095,-121.8815343755784,200000)),
      GeoViewpoint(DEF='View20b',description='San Jose State University 100km',orientation=(1,0,0,-1.57),position=(37.33580940527095,-121.8815343755784,100000)),
      GeoViewpoint(DEF='View20c',description='San Jose State University 50km',orientation=(1,0,0,-1.57),position=(37.33580940527095,-121.8815343755784,50000)),
      GeoViewpoint(DEF='View20d',description='San Jose State University 25km',orientation=(1,0,0,-1.57),position=(37.33580940527095,-121.8815343755784,25000)),
      GeoViewpoint(DEF='View20e',description='San Jose State University 10km',orientation=(1,0,0,-1.57),position=(37.33580940527095,-121.8815343755784,10000)),
      GeoViewpoint(DEF='View21a',description='California State University, Monterey Bay 200km',orientation=(1,0,0,-1.57),position=(36.6538071781493,-121.7984465276711,200000)),
      GeoViewpoint(DEF='View21b',description='California State University, Monterey Bay 100km',orientation=(1,0,0,-1.57),position=(36.6538071781493,-121.7984465276711,100000)),
      GeoViewpoint(DEF='View21c',description='California State University, Monterey Bay 50km',orientation=(1,0,0,-1.57),position=(36.6538071781493,-121.7984465276711,50000)),
      GeoViewpoint(DEF='View21d',description='California State University, Monterey Bay 25km',orientation=(1,0,0,-1.57),position=(36.6538071781493,-121.7984465276711,25000)),
      GeoViewpoint(DEF='View21e',description='California State University, Monterey Bay 10km',orientation=(1,0,0,-1.57),position=(36.6538071781493,-121.7984465276711,10000)),
      GeoViewpoint(DEF='View22a',description='Fresno State University 200km',orientation=(1,0,0,-1.57),position=(36.812166,-119.7451552005852,200000)),
      GeoViewpoint(DEF='View22b',description='Fresno State University 100km',orientation=(1,0,0,-1.57),position=(36.812166,-119.7451552005852,100000)),
      GeoViewpoint(DEF='View22c',description='Fresno State University 50km',orientation=(1,0,0,-1.57),position=(36.812166,-119.7451552005852,50000)),
      GeoViewpoint(DEF='View22d',description='Fresno State University 25km',orientation=(1,0,0,-1.57),position=(36.812166,-119.7451552005852,25000)),
      GeoViewpoint(DEF='View22e',description='Fresno State University 10km',orientation=(1,0,0,-1.57),position=(36.812166,-119.7451552005852,10000)),
      GeoViewpoint(DEF='View23a',description='California State University, Bakersfield 200km',orientation=(1,0,0,-1.57),position=(35.350804,-119.1043226128032,200000)),
      GeoViewpoint(DEF='View23b',description='California State University, Bakersfield 100km',orientation=(1,0,0,-1.57),position=(35.350804,-119.1043226128032,100000)),
      GeoViewpoint(DEF='View23c',description='California State University, Bakersfield 50km',orientation=(1,0,0,-1.57),position=(35.350804,-119.1043226128032,50000)),
      GeoViewpoint(DEF='View23d',description='California State University, Bakersfield 25km',orientation=(1,0,0,-1.57),position=(35.350804,-119.1043226128032,25000)),
      GeoViewpoint(DEF='View23e',description='California State University, Bakersfield 10km',orientation=(1,0,0,-1.57),position=(35.350804,-119.1043226128032,10000)),
      GeoViewpoint(DEF='View24a',description='California Polytechnic State University 200km',orientation=(1,0,0,-1.57),position=(35.30115692599171,-120.6595761796351,200000)),
      GeoViewpoint(DEF='View24b',description='California Polytechnic State University 100km',orientation=(1,0,0,-1.57),position=(35.30115692599171,-120.6595761796351,100000)),
      GeoViewpoint(DEF='View24c',description='California Polytechnic State University 50km',orientation=(1,0,0,-1.57),position=(35.30115692599171,-120.6595761796351,50000)),
      GeoViewpoint(DEF='View24d',description='California Polytechnic State University 25km',orientation=(1,0,0,-1.57),position=(35.30115692599171,-120.6595761796351,25000)),
      GeoViewpoint(DEF='View24e',description='California Polytechnic State University 10km',orientation=(1,0,0,-1.57),position=(35.30115692599171,-120.6595761796351,10000)),
      GeoViewpoint(DEF='View25a',description='California State University, Channel Islands 200km',orientation=(1,0,0,-1.57),position=(34.1624126428852,-119.0425264841287,200000)),
      GeoViewpoint(DEF='View25b',description='California State University, Channel Islands 100km',orientation=(1,0,0,-1.57),position=(34.1624126428852,-119.0425264841287,100000)),
      GeoViewpoint(DEF='View25c',description='California State University, Channel Islands 50km',orientation=(1,0,0,-1.57),position=(34.1624126428852,-119.0425264841287,50000)),
      GeoViewpoint(DEF='View25d',description='California State University, Channel Islands 25km',orientation=(1,0,0,-1.57),position=(34.1624126428852,-119.0425264841287,25000)),
      GeoViewpoint(DEF='View25e',description='California State University, Channel Islands 10km',orientation=(1,0,0,-1.57),position=(34.1624126428852,-119.0425264841287,10000)),
      GeoViewpoint(DEF='View26a',description='California State University, Northridge 200km',orientation=(1,0,0,-1.57),position=(34.23959265263849,-118.5284272400904,200000)),
      GeoViewpoint(DEF='View26b',description='California State University, Northridge 100km',orientation=(1,0,0,-1.57),position=(34.23959265263849,-118.5284272400904,100000)),
      GeoViewpoint(DEF='View26c',description='California State University, Northridge 50km',orientation=(1,0,0,-1.57),position=(34.23959265263849,-118.5284272400904,50000)),
      GeoViewpoint(DEF='View26d',description='California State University, Northridge 25km',orientation=(1,0,0,-1.57),position=(34.23959265263849,-118.5284272400904,25000)),
      GeoViewpoint(DEF='View26e',description='California State University, Northridge 10km',orientation=(1,0,0,-1.57),position=(34.23959265263849,-118.5284272400904,10000)),
      GeoViewpoint(DEF='View27a',description='California State University, Los Angeles 200km',orientation=(1,0,0,-1.57),position=(34.06645996626264,-118.1682050902557,200000)),
      GeoViewpoint(DEF='View27b',description='California State University, Los Angeles 100km',orientation=(1,0,0,-1.57),position=(34.06645996626264,-118.1682050902557,100000)),
      GeoViewpoint(DEF='View27c',description='California State University, Los Angeles 50km',orientation=(1,0,0,-1.57),position=(34.06645996626264,-118.1682050902557,50000)),
      GeoViewpoint(DEF='View27d',description='California State University, Los Angeles 25km',orientation=(1,0,0,-1.57),position=(34.06645996626264,-118.1682050902557,25000)),
      GeoViewpoint(DEF='View27e',description='California State University, Los Angeles 10km',orientation=(1,0,0,-1.57),position=(34.06645996626264,-118.1682050902557,10000)),
      GeoViewpoint(DEF='View28a',description='California State Polytechnic University, Pomona 200km',orientation=(1,0,0,-1.57),position=(34.05727150000314,-117.8215320234812,200000)),
      GeoViewpoint(DEF='View28b',description='California State Polytechnic University, Pomona 100km',orientation=(1,0,0,-1.57),position=(34.05727150000314,-117.8215320234812,100000)),
      GeoViewpoint(DEF='View28c',description='California State Polytechnic University, Pomona 50km',orientation=(1,0,0,-1.57),position=(34.05727150000314,-117.8215320234812,50000)),
      GeoViewpoint(DEF='View28d',description='California State Polytechnic University, Pomona 25km',orientation=(1,0,0,-1.57),position=(34.05727150000314,-117.8215320234812,25000)),
      GeoViewpoint(DEF='View28e',description='California State Polytechnic University, Pomona 10km',orientation=(1,0,0,-1.57),position=(34.05727150000314,-117.8215320234812,10000)),
      GeoViewpoint(DEF='View29a',description='California State University, San Bernardino 200km',orientation=(1,0,0,-1.57),position=(34.18180116432101,-117.3243676664719,200000)),
      GeoViewpoint(DEF='View29b',description='California State University, San Bernardino 100km',orientation=(1,0,0,-1.57),position=(34.18180116432101,-117.3243676664719,100000)),
      GeoViewpoint(DEF='View29c',description='California State University, San Bernardino 50km',orientation=(1,0,0,-1.57),position=(34.18180116432101,-117.3243676664719,50000)),
      GeoViewpoint(DEF='View29d',description='California State University, San Bernardino 25km',orientation=(1,0,0,-1.57),position=(34.18180116432101,-117.3243676664719,25000)),
      GeoViewpoint(DEF='View29e',description='California State University, San Bernardino 10km',orientation=(1,0,0,-1.57),position=(34.18180116432101,-117.3243676664719,10000)),
      GeoViewpoint(DEF='View30a',description='California State University, Fullerton 200km',orientation=(1,0,0,-1.57),position=(33.882522,-117.8868367869023,200000)),
      GeoViewpoint(DEF='View30b',description='California State University, Fullerton 100km',orientation=(1,0,0,-1.57),position=(33.882522,-117.8868367869023,100000)),
      GeoViewpoint(DEF='View30c',description='California State University, Fullerton 50km',orientation=(1,0,0,-1.57),position=(33.882522,-117.8868367869023,50000)),
      GeoViewpoint(DEF='View30d',description='California State University, Fullerton 25km',orientation=(1,0,0,-1.57),position=(33.882522,-117.8868367869023,25000)),
      GeoViewpoint(DEF='View30e',description='California State University, Fullerton 10km',orientation=(1,0,0,-1.57),position=(33.882522,-117.8868367869023,10000)),
      GeoViewpoint(DEF='View31a',description='California State University Dominguez Hills 200km',orientation=(1,0,0,-1.57),position=(33.86463396857103,-118.2553596272055,200000)),
      GeoViewpoint(DEF='View31b',description='California State University Dominguez Hills 100km',orientation=(1,0,0,-1.57),position=(33.86463396857103,-118.2553596272055,100000)),
      GeoViewpoint(DEF='View31c',description='California State University Dominguez Hills 50km',orientation=(1,0,0,-1.57),position=(33.86463396857103,-118.2553596272055,50000)),
      GeoViewpoint(DEF='View31d',description='California State University Dominguez Hills 25km',orientation=(1,0,0,-1.57),position=(33.86463396857103,-118.2553596272055,25000)),
      GeoViewpoint(DEF='View31e',description='California State University Dominguez Hills 10km',orientation=(1,0,0,-1.57),position=(33.86463396857103,-118.2553596272055,10000)),
      GeoViewpoint(DEF='View32a',description='California State University, Long Beach 200km',orientation=(1,0,0,-1.57),position=(33.78196696470824,-118.112678253688,200000)),
      GeoViewpoint(DEF='View32b',description='California State University, Long Beach 100km',orientation=(1,0,0,-1.57),position=(33.78196696470824,-118.112678253688,100000)),
      GeoViewpoint(DEF='View32c',description='California State University, Long Beach 50km',orientation=(1,0,0,-1.57),position=(33.78196696470824,-118.112678253688,50000)),
      GeoViewpoint(DEF='View32d',description='California State University, Long Beach 25km',orientation=(1,0,0,-1.57),position=(33.78196696470824,-118.112678253688,25000)),
      GeoViewpoint(DEF='View32e',description='California State University, Long Beach 10km',orientation=(1,0,0,-1.57),position=(33.78196696470824,-118.112678253688,10000)),
      GeoViewpoint(DEF='View33a',description='California State University, San Marcos 200km',orientation=(1,0,0,-1.57),position=(33.12837619265174,-117.1600194071002,200000)),
      GeoViewpoint(DEF='View33b',description='California State University, San Marcos 100km',orientation=(1,0,0,-1.57),position=(33.12837619265174,-117.1600194071002,100000)),
      GeoViewpoint(DEF='View33c',description='California State University, San Marcos 50km',orientation=(1,0,0,-1.57),position=(33.12837619265174,-117.1600194071002,50000)),
      GeoViewpoint(DEF='View33d',description='California State University, San Marcos 25km',orientation=(1,0,0,-1.57),position=(33.12837619265174,-117.1600194071002,25000)),
      GeoViewpoint(DEF='View33e',description='California State University, San Marcos 10km',orientation=(1,0,0,-1.57),position=(33.12837619265174,-117.1600194071002,10000)),
      GeoViewpoint(DEF='View34a',description='San Diego State University 200km',orientation=(1,0,0,-1.57),position=(32.77593126586385,-117.0722876242553,200000)),
      GeoViewpoint(DEF='View34b',description='San Diego State University 100km',orientation=(1,0,0,-1.57),position=(32.77593126586385,-117.0722876242553,100000)),
      GeoViewpoint(DEF='View34c',description='San Diego State University 50km',orientation=(1,0,0,-1.57),position=(32.77593126586385,-117.0722876242553,50000)),
      GeoViewpoint(DEF='View34d',description='San Diego State University 25km',orientation=(1,0,0,-1.57),position=(32.77593126586385,-117.0722876242553,25000)),
      GeoViewpoint(DEF='View34e',description='San Diego State University 10km',orientation=(1,0,0,-1.57),position=(32.77593126586385,-117.0722876242553,10000)),
      GeoViewpoint(DEF='View35a',description='CENIC 200km',orientation=(1,0,0,-1.57),position=(33.879947,-118.027903,200000)),
      GeoViewpoint(DEF='View35b',description='CENIC 100km',orientation=(1,0,0,-1.57),position=(33.879947,-118.027903,100000)),
      GeoViewpoint(DEF='View35c',description='CENIC 50km',orientation=(1,0,0,-1.57),position=(33.879947,-118.027903,50000)),
      GeoViewpoint(DEF='View35d',description='CENIC 25km',orientation=(1,0,0,-1.57),position=(33.879947,-118.027903,25000)),
      GeoViewpoint(DEF='View35e',description='CENIC 10km',orientation=(1,0,0,-1.57),position=(33.879947,-118.027903,10000))]),
    #  ==================== 
    ExternProtoDeclare(appinfo='CrossHair prototype provides a heads-up display (HUD) crosshair at the view center, which is useful for assessing NavigationInfo lookAt point',name='CrossHair',url=["../../Savage/Tools/HeadsUpDisplays/CrossHairPrototype.x3d#CrossHair","https://savage.nps.edu/Savage/Tools/HeadsUpDisplays/CrossHairPrototype.x3d#CrossHair","../../Savage/Tools/HeadsUpDisplays/CrossHairPrototype.wrl#CrossHair","https://savage.nps.edu/Savage/Tools/HeadsUpDisplays/CrossHairPrototype.wrl#CrossHair"],
      field=[
      field(accessType='initializeOnly',appinfo='whether CrossHair prototype is enabled or not',name='enabled',type='SFBool'),
      field(accessType='inputOnly',appinfo='control whether enabled/disabled',name='set_enabled',type='SFBool'),
      field(accessType='inputOutput',appinfo='color of CrossHair marker',name='markerColor',type='SFColor'),
      field(accessType='inputOutput',appinfo='size of CrossHair in meters',name='scale',type='SFVec3f'),
      field(accessType='inputOutput',appinfo='distance in front of HUD viewpoint',name='positionOffsetFromCamera',type='SFVec3f')]),
    ProtoInstance(DEF='CrossHairInstance',name='CrossHair',
      fieldValue=[
      fieldValue(name='enabled',value=True),
      fieldValue(name='markerColor',value=(0.1,0.8,0.1)),
      fieldValue(name='scale',value=(0.5,0.5,0.5)),
      fieldValue(name='positionOffsetFromCamera',value=(0,0,-5))]),
    #  ==================== 
    ExternProtoDeclare(appinfo='Heads-up display (HUD) keeps child geometry aligned on screen in a consistent location',name='HeadsUpDisplay',url=["../../../X3dForWebAuthors/Chapter14Prototypes/HeadsUpDisplayPrototype.x3d#HeadsUpDisplay","https://www.web3d.org/x3d/content/examples/X3dForWebAuthors/Chapter14Prototypes/HeadsUpDisplayPrototype.x3d#HeadsUpDisplay","../../../X3dForWebAuthors/Chapter14Prototypes/HeadsUpDisplayPrototype.wrl#HeadsUpDisplay","https://www.web3d.org/x3d/content/examples/X3dForWebAuthors/Chapter14Prototypes/HeadsUpDisplayPrototype.wrl#HeadsUpDisplay"],
      field=[
      field(accessType='inputOutput',appinfo='X3D content positioned at HUD offset',name='children',type='MFNode'),
      field(accessType='inputOutput',appinfo='offset position for HUD relative to current view location, default 0 0 -5',name='screenOffset',type='SFVec3f'),
      field(accessType='outputOnly',appinfo='HUD position update (in world coordinates) relative to original location',name='position_changed',type='SFVec3f'),
      field(accessType='outputOnly',appinfo='HUD orientation update relative to original location',name='orientation_changed',type='SFRotation')]),
    #  ==================== 
    ExternProtoDeclare(appinfo='Sequentially binds each Viewpoint in a set of Viewpoint USE nodes, creating an automatic tour for a scene',name='ViewpointSequencer',url=["../../Savage/Tools/Animation/ViewpointSequencerPrototype.x3d#ViewpointSequencer","https://savage.nps.edu/Savage/Tools/Animation/ViewpointSequencerPrototype.x3d#ViewpointSequencer","../../Savage/Tools/Animation/ViewpointSequencerPrototype.wrl#ViewpointSequencer","https://savage.nps.edu/Savage/Tools/Animation/ViewpointSequencerPrototype.wrl#ViewpointSequencer"],
      field=[
      field(accessType='initializeOnly',appinfo='Viewpoint USE nodes that are sequentially bound',name='viewpoints',type='MFNode'),
      field(accessType='inputOutput',appinfo='number of seconds between viewpoint shifts',name='interval',type='SFTime'),
      field(accessType='inputOutput',appinfo='whether ViewpointSequencer is enabled or not',name='enabled',type='SFBool'),
      field(accessType='inputOnly',appinfo='whether ViewpointSequencer is enabled or not',name='set_enabled',type='SFBool'),
      field(accessType='inputOnly',appinfo='bind previous Viewpoint in list',name='previous',type='SFBool'),
      field(accessType='inputOnly',appinfo='bind next Viewpoint in list',name='next',type='SFBool'),
      field(accessType='inputOutput',appinfo='Select message to toggle ViewpointSequencer',name='toggleMessage',type='MFString'),
      field(accessType='initializeOnly',appinfo='Color for toggleMessage text',name='toggleMessageFontSize',type='SFFloat'),
      field(accessType='inputOutput',appinfo='Color for toggleMessage text',name='toggleMessageColor',type='SFColor'),
      field(accessType='inputOutput',appinfo='enable console output',name='traceEnabled',type='SFBool')]),
    ProtoInstance(DEF='ViewpointTour',name='ViewpointSequencer',
      fieldValue=[
      fieldValue(name='interval',value=10),
      #  TODO initially enabled needs to be off, and scene provides selectable text or Viewpoint to activate 
      fieldValue(name='enabled',value=True),
      fieldValue(name='toggleMessage',value=["ViewpointSequencer tour"]),
      fieldValue(name='viewpoints',
        children=[
        GeoViewpoint(USE='View01a'),
        GeoViewpoint(USE='View01b'),
        GeoViewpoint(USE='View01c'),
        GeoViewpoint(USE='View01d'),
        GeoViewpoint(USE='View01e'),
        GeoViewpoint(USE='View02a'),
        GeoViewpoint(USE='View02b'),
        GeoViewpoint(USE='View02c'),
        GeoViewpoint(USE='View02d'),
        GeoViewpoint(USE='View02e'),
        GeoViewpoint(USE='View03a'),
        GeoViewpoint(USE='View03b'),
        GeoViewpoint(USE='View03c'),
        GeoViewpoint(USE='View03d'),
        GeoViewpoint(USE='View03e'),
        GeoViewpoint(USE='View04a'),
        GeoViewpoint(USE='View04b'),
        GeoViewpoint(USE='View04c'),
        GeoViewpoint(USE='View04d'),
        GeoViewpoint(USE='View04e'),
        GeoViewpoint(USE='View05a'),
        GeoViewpoint(USE='View05b'),
        GeoViewpoint(USE='View05c'),
        GeoViewpoint(USE='View05d'),
        GeoViewpoint(USE='View05e'),
        GeoViewpoint(USE='View06a'),
        GeoViewpoint(USE='View06b'),
        GeoViewpoint(USE='View06c'),
        GeoViewpoint(USE='View06d'),
        GeoViewpoint(USE='View06e'),
        GeoViewpoint(USE='View07a'),
        GeoViewpoint(USE='View07b'),
        GeoViewpoint(USE='View07c'),
        GeoViewpoint(USE='View07d'),
        GeoViewpoint(USE='View07e'),
        GeoViewpoint(USE='View08a'),
        GeoViewpoint(USE='View08b'),
        GeoViewpoint(USE='View08c'),
        GeoViewpoint(USE='View08d'),
        GeoViewpoint(USE='View08e'),
        GeoViewpoint(USE='View09a'),
        GeoViewpoint(USE='View09b'),
        GeoViewpoint(USE='View09c'),
        GeoViewpoint(USE='View09d'),
        GeoViewpoint(USE='View09e'),
        GeoViewpoint(USE='View10a'),
        GeoViewpoint(USE='View10b'),
        GeoViewpoint(USE='View10c'),
        GeoViewpoint(USE='View10d'),
        GeoViewpoint(USE='View10e'),
        GeoViewpoint(USE='View11a'),
        GeoViewpoint(USE='View11b'),
        GeoViewpoint(USE='View11c'),
        GeoViewpoint(USE='View11d'),
        GeoViewpoint(USE='View11e'),
        GeoViewpoint(USE='View12a'),
        GeoViewpoint(USE='View12b'),
        GeoViewpoint(USE='View12c'),
        GeoViewpoint(USE='View12d'),
        GeoViewpoint(USE='View12e'),
        GeoViewpoint(USE='View13a'),
        GeoViewpoint(USE='View13b'),
        GeoViewpoint(USE='View13c'),
        GeoViewpoint(USE='View13d'),
        GeoViewpoint(USE='View13e'),
        GeoViewpoint(USE='View14a'),
        GeoViewpoint(USE='View14b'),
        GeoViewpoint(USE='View14c'),
        GeoViewpoint(USE='View14d'),
        GeoViewpoint(USE='View14e'),
        GeoViewpoint(USE='View15a'),
        GeoViewpoint(USE='View15b'),
        GeoViewpoint(USE='View15c'),
        GeoViewpoint(USE='View15d'),
        GeoViewpoint(USE='View15e'),
        GeoViewpoint(USE='View16a'),
        GeoViewpoint(USE='View16b'),
        GeoViewpoint(USE='View16c'),
        GeoViewpoint(USE='View16d'),
        GeoViewpoint(USE='View16e'),
        GeoViewpoint(USE='View17a'),
        GeoViewpoint(USE='View17b'),
        GeoViewpoint(USE='View17c'),
        GeoViewpoint(USE='View17d'),
        GeoViewpoint(USE='View17e'),
        GeoViewpoint(USE='View18a'),
        GeoViewpoint(USE='View18b'),
        GeoViewpoint(USE='View18c'),
        GeoViewpoint(USE='View18d'),
        GeoViewpoint(USE='View18e'),
        GeoViewpoint(USE='View19a'),
        GeoViewpoint(USE='View19b'),
        GeoViewpoint(USE='View19c'),
        GeoViewpoint(USE='View19d'),
        GeoViewpoint(USE='View19e'),
        GeoViewpoint(USE='View20a'),
        GeoViewpoint(USE='View20b'),
        GeoViewpoint(USE='View20c'),
        GeoViewpoint(USE='View20d'),
        GeoViewpoint(USE='View20e'),
        GeoViewpoint(USE='View21a'),
        GeoViewpoint(USE='View21b'),
        GeoViewpoint(USE='View21c'),
        GeoViewpoint(USE='View21d'),
        GeoViewpoint(USE='View21e'),
        GeoViewpoint(USE='View22a'),
        GeoViewpoint(USE='View22b'),
        GeoViewpoint(USE='View22c'),
        GeoViewpoint(USE='View22d'),
        GeoViewpoint(USE='View22e'),
        GeoViewpoint(USE='View23a'),
        GeoViewpoint(USE='View23b'),
        GeoViewpoint(USE='View23c'),
        GeoViewpoint(USE='View23d'),
        GeoViewpoint(USE='View23e'),
        GeoViewpoint(USE='View24a'),
        GeoViewpoint(USE='View24b'),
        GeoViewpoint(USE='View24c'),
        GeoViewpoint(USE='View24d'),
        GeoViewpoint(USE='View24e'),
        GeoViewpoint(USE='View25a'),
        GeoViewpoint(USE='View25b'),
        GeoViewpoint(USE='View25c'),
        GeoViewpoint(USE='View25d'),
        GeoViewpoint(USE='View25e'),
        GeoViewpoint(USE='View26a'),
        GeoViewpoint(USE='View26b'),
        GeoViewpoint(USE='View26c'),
        GeoViewpoint(USE='View26d'),
        GeoViewpoint(USE='View26e'),
        GeoViewpoint(USE='View27a'),
        GeoViewpoint(USE='View27b'),
        GeoViewpoint(USE='View27c'),
        GeoViewpoint(USE='View27d'),
        GeoViewpoint(USE='View27e'),
        GeoViewpoint(USE='View28a'),
        GeoViewpoint(USE='View28b'),
        GeoViewpoint(USE='View28c'),
        GeoViewpoint(USE='View28d'),
        GeoViewpoint(USE='View28e'),
        GeoViewpoint(USE='View29a'),
        GeoViewpoint(USE='View29b'),
        GeoViewpoint(USE='View29c'),
        GeoViewpoint(USE='View29d'),
        GeoViewpoint(USE='View29e'),
        GeoViewpoint(USE='View30a'),
        GeoViewpoint(USE='View30b'),
        GeoViewpoint(USE='View30c'),
        GeoViewpoint(USE='View30d'),
        GeoViewpoint(USE='View30e'),
        GeoViewpoint(USE='View31a'),
        GeoViewpoint(USE='View31b'),
        GeoViewpoint(USE='View31c'),
        GeoViewpoint(USE='View31d'),
        GeoViewpoint(USE='View31e'),
        GeoViewpoint(USE='View32a'),
        GeoViewpoint(USE='View32b'),
        GeoViewpoint(USE='View32c'),
        GeoViewpoint(USE='View32d'),
        GeoViewpoint(USE='View32e'),
        GeoViewpoint(USE='View33a'),
        GeoViewpoint(USE='View33b'),
        GeoViewpoint(USE='View33c'),
        GeoViewpoint(USE='View33d'),
        GeoViewpoint(USE='View33e'),
        GeoViewpoint(USE='View34a'),
        GeoViewpoint(USE='View34b'),
        GeoViewpoint(USE='View34c'),
        GeoViewpoint(USE='View34d'),
        GeoViewpoint(USE='View34e'),
        GeoViewpoint(USE='View35a'),
        GeoViewpoint(USE='View35b'),
        GeoViewpoint(USE='View35c'),
        GeoViewpoint(USE='View35d'),
        GeoViewpoint(USE='View35e'),
        Viewpoint(DEF='View4',description='View four (-X axis)',orientation=(0,1,0,-1.57),position=(-10,0,0))])]),
    #  ==================== 
    GeoLocation(geoCoords=(36.595599,-121.877148,624990),
      children=[
      Billboard(axisOfRotation=(0,0,0),
        children=[
        Shape(
          geometry=Text(string=["ViewpointSequencer tour"],
            fontStyle=FontStyle(justify=["MIDDLE","MIDDLE"])),
          appearance=Appearance(
            material=Material(ambientIntensity=0.25,diffuseColor=(0.795918,0.505869,0.093315),shininess=0.39,specularColor=(0.923469,0.428866,0.006369))))]),
      TouchSensor(DEF='TourTouch',description='Touch text to turn tour on/off'),
      BooleanToggle(DEF='TourToggle'),
      ROUTE(fromField='isActive',fromNode='TourTouch',toField='set_boolean',toNode='TourToggle'),
      ROUTE(fromField='toggle',fromNode='TourToggle',toField='set_enabled',toNode='ViewpointTour')]),
    #  ==================== 
    ExternProtoDeclare(appinfo='ViewPositionOrientation provides provides console output of local position and orientation as user navigates',name='ViewPositionOrientation',url=["../../Savage/Tools/Authoring/ViewPositionOrientationPrototype.x3d#ViewPositionOrientation","https://savage.nps.edu/Savage/Tools/Authoring/ViewPositionOrientationPrototype.x3d#ViewPositionOrientation","../../Savage/Tools/Authoring/ViewPositionOrientationPrototype.wrl#ViewPositionOrientation","https://savage.nps.edu/Savage/Tools/Authoring/ViewPositionOrientationPrototype.wrl#ViewPositionOrientation"],
      field=[
      field(accessType='inputOutput',appinfo='Whether or not ViewPositionOrientation sends output to console',name='enabled',type='SFBool'),
      field(accessType='initializeOnly',appinfo='Output internal trace messages for debugging this node, intended for developer use only',name='traceEnabled',type='SFBool'),
      field(accessType='inputOnly',appinfo='Ability to turn output tracing on/off at runtime',name='set_traceEnabled',type='SFBool'),
      field(accessType='outputOnly',appinfo='Output local position',name='position_changed',type='SFVec3f'),
      field(accessType='outputOnly',appinfo='Output local orientation',name='orientation_changed',type='SFRotation'),
      field(accessType='outputOnly',appinfo='MFString value of new Viewpoint',name='outputViewpointString',type='MFString')]),
    ProtoInstance(DEF='ExampleViewPositionOrientation',name='ViewPositionOrientation',
      fieldValue=[
      fieldValue(name='enabled',value=False)])]
    #  ==================== 
    )
) # X3D model complete

####################################################################################################
# Self-test diagnostics
####################################################################################################

print('Self-test diagnostics for CaliforniaCampusesTour5Altitudes.py:')
if        metaDiagnostics(newModel): # built-in utility method in X3D class
    print(metaDiagnostics(newModel)) # display meta info, hint, warning, error, TODO values in this model
# print('check newModel.XML() serialization...')
newModelXML= newModel.XML() # test export method XML() for exceptions during export
newModel.XMLvalidate()
# print(newModelXML) # diagnostic

try:
#   print('check newModel.VRML() serialization...')
    newModelVRML=newModel.VRML() # test export method VRML() for exceptions during export
    # print(prependLineNumbers(newModelVRML)) # debug
    print("Python-to-VRML export of VRML output successful", flush=True)
except Exception as err: # usually BaseException
    # https://stackoverflow.com/questions/18176602/how-to-get-the-name-of-an-exception-that-was-caught-in-python
    print("*** Python-to-VRML export of VRML output failed:", type(err).__name__, err)
    if newModelVRML: # may have failed to generate
        print(prependLineNumbers(newModelVRML, err.lineno))

try:
#   print('check newModel.JSON() serialization...')
    newModelJSON=newModel.JSON() # test export method JSON() for exceptions during export
#   print(prependLineNumbers(newModelJSON)) # debug
    print("Python-to-JSON export of JSON output successful (under development)")
except Exception as err: # usually SyntaxError
    print("*** Python-to-JSON export of JSON output failed:", type(err).__name__, err)
    if newModelJSON: # may have failed to generate
        print(prependLineNumbers(newModelJSON,err.lineno))

print("python CaliforniaCampusesTour5Altitudes.py load and self-test diagnostics complete.")
