/* Copyright (c) 1995-2018 held by the author(s).  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer
      in the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the Web3D Consortium (http://www.web3D.org)
      nor the names of its contributors may be used to endorse or
      promote products derived from this software without specific
      prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

package org.web3d.x3d.tests;

import java.util.Arrays;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.web3d.x3d.jsail.ConfigurationProperties;

import org.web3d.x3d.jsail.fields.*;

/**
 * FieldObjectTests.java
 *
 * Filename:     FieldObjectTests.java
 * Identifier:   
 * @author       Don Brutzman
 * Created:      23 June 2018
 * Revised:      see version control
 * Compile, run: ../build.xml
 */

/**
 * Initial test harness using junit5 for the X3D Java Scene Access Interface Library (X3DJSAIL).
 * @see <a href="http://www.web3d.org/specifications/java/X3DJSAIL.html">X3DJSAIL</a>
 * @see <a href="https://junit.org/junit5/docs/current/user-guide/#running-tests-build-ant">JUnit5 running tests with Ant</a>
 * @see <a href="https://ant.apache.org/manual/Tasks/junitlauncher.html">Ant task junitlauncher</a>
 * @see <a href="https://www.petrikainulainen.net/programming/testing/junit-5-tutorial-writing-assertions-with-junit-5-api">Tutorial: writing assertions with JUnit5</a>
 * @see <a href="../license.html">License</a>   
 */
class FieldObjectTests
{
    @Test
    @DisplayName("Test initialization of all X3D SF/MF field objects")
    void fieldObjectInitializationsTest()
	{
        System.out.println("FieldObjectTests.fieldObjectInitializationsTest() start...");

        // assert statements
        System.out.println ("Preliminary tests...");
        assertTrue (true,  "this test should explicitly pass");
        assertFalse(false, "this test should explicitly fail");
        
        // now tests for each field type
        SFBoolObjectTests();
        MFBoolObjectTests();
        
        SFImageObjectTests();
        MFImageObjectTests();
        
        SFInt32ObjectTests();
        MFInt32ObjectTests();
        
        // regex expressions for SFFloat, SFDouble and SFTime are identical but classes are unique, so test each of them!
        SFFloatObjectTests();
        SFDoubleObjectTests();
        SFTimeObjectTests();
        // regex expressions for MFFloat, MFDouble and MFTime are identical but classes are unique, so test each of them!
        MFFloatObjectTests();
        MFDoubleObjectTests();
        MFTimeObjectTests();
        
        SFVec2fObjectTests();
        SFVec2dObjectTests();
        MFVec2fObjectTests();
        MFVec2dObjectTests();
        
        SFVec3fObjectTests();
        SFVec3fBboxSizeObjectTests();
        SFVec3dObjectTests();
        MFVec3fObjectTests();
        MFVec3dObjectTests();
        
        SFVec4fObjectTests();
        SFVec4dObjectTests();
        MFVec4fObjectTests();
        MFVec4dObjectTests();
        
        SFColorObjectTests();
        MFColorObjectTests();
        SFColorRGBAObjectTests();
        MFColorRGBAObjectTests();
        
        SFRotationObjectTests();
        MFRotationObjectTests();
        
        SFMatrix3fObjectTests();
        SFMatrix3dObjectTests();
        MFMatrix3fObjectTests();
        MFMatrix3dObjectTests();
        
        SFMatrix4fObjectTests();
        SFMatrix4dObjectTests();
        MFMatrix4fObjectTests();
        MFMatrix4dObjectTests();
        
        System.out.println("FieldObjectTests.fieldObjectInitializationsTest() complete");
    }
    
    @Test
    @DisplayName("Test SFBoolObject single-field boolean")
    void SFBoolObjectTests()
	{
        System.out.println ("SFBoolObjectTests...");
        assertTrue  ((SFBoolObject.DEFAULT_VALUE == new SFBoolObject().setValueByString(SFBoolObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                       "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFBoolObject testSFBoolObject = new SFBoolObject(); // static initializer is tested, might throw exception
        assertTrue  (SFBoolObject.DEFAULT_VALUE,       "test correct default value for this field object");
        assertTrue  (testSFBoolObject.matches(),       "testSFBoolObject.matches() tests object initialization correctly matches regex");
        assertTrue  (SFBoolObject.matches(SFBoolObject.DEFAULT_VALUE_STRING),
                                                       "SFBoolObject.matches(SFBoolObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFBoolObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFBoolObject.REGEX.contains("^") && !SFBoolObject.REGEX.contains("$"), "test SFBoolObject.REGEX does not contain anchor characters ^ or $");

        testSFBoolObject.setValue(true);               // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (testSFBoolObject.getValue(),      "tests setting object value to true results in value of true");
        testSFBoolObject.setValue(false);              // returns void because it matches (overrides) Java SAI specification interface
        assertFalse (testSFBoolObject.getValue(),      "tests setting object value to false results in value of false");
        
        assertTrue  (SFBoolObject.matches("true"),     "SFBoolObject.matches(\"true\")  tests correct string value");
        assertTrue  (SFBoolObject.matches("false"),    "SFBoolObject.matches(\"false\") tests correct string value");
        assertTrue  (SFBoolObject.matches("  true "),  "SFBoolObject.matches(\"  true \") tests correct string value, including external whitespace");
        assertTrue  (SFBoolObject.matches(" false "),  "SFBoolObject.matches(\" false \") tests correct string value, including external whitespace");
        
        assertFalse (SFBoolObject.matches(""),         "SFBoolObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFBoolObject.matches(","),        "SFBoolObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (SFBoolObject.matches("true,"),    "SFBoolObject.matches(\"true,\") tests incorrect inclusion of comma as whitespace");
        assertFalse (SFBoolObject.matches(",false"),   "SFBoolObject.matches(\",false\") tests incorrect inclusion of comma as whitespace");
        assertFalse (SFBoolObject.matches("TRUE"),     "SFBoolObject.matches(\"TRUE\")  tests incorrect case of string value");
        assertFalse (SFBoolObject.matches("FALSE"),    "SFBoolObject.matches(\"FALSE\") tests incorrect case of string value");
        assertFalse (SFBoolObject.matches("rtue"),     "SFBoolObject.matches(\"rtue\") tests incorrect ordering of characters in string value");
        assertFalse (SFBoolObject.matches("aflse"),    "SFBoolObject.matches(\"aflse\") tests incorrect ordering of characters in string value");
        assertFalse (SFBoolObject.matches("blah"),     "SFBoolObject.matches(\"blah\")  tests incorrect string value");
        assertFalse (SFBoolObject.matches("NaN"),      "SFBoolObject.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test MFBoolObject multiple-field boolean")
    void MFBoolObjectTests()
	{
        System.out.println ("MFBoolObjectTests...");
        assertTrue  (Arrays.equals(MFBoolObject.DEFAULT_VALUE, new MFBoolObject().setValueByString(MFBoolObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFBoolObject testMFBoolObject = new MFBoolObject(); // static initializer is tested, might throw exception
        boolean[] emptyArray       = {};
        boolean[] singleTrueArray  = { true  };
        boolean[] singleFalseArray = { false };
        boolean[] trueTrueArray    = { true, true  };
        boolean[] trueFalseArray   = { true, false };
        boolean[] falseTrueArray   = { false, true };
        boolean[] falseFalseArray  = { false, false };
        assertTrue  (Arrays.equals(emptyArray, MFBoolObject.DEFAULT_VALUE), "test correct default value for this field object");
        assertTrue  (testMFBoolObject.matches(),       "testMFBoolObject.matches() tests object initialization correctly matches regex");
        assertTrue  (MFBoolObject.matches(MFBoolObject.DEFAULT_VALUE_STRING),
                                                       "MFBoolObject.matches(MFBoolObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFBoolObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFBoolObject.REGEX.contains("^") && !MFBoolObject.REGEX.contains("$"), "test MFBoolObject.REGEX does not contain anchor characters ^ or $");

        testMFBoolObject.setValue(true);               // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(singleTrueArray,  testMFBoolObject.getPrimitiveValue()), "tests setting object value to boolean true results in same value");
        assertTrue  (testMFBoolObject.get1Value(0),   "tests setting object value to true results in get1Value(index 0) of true");
        
        testMFBoolObject.setValue(false);               // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(singleFalseArray,  testMFBoolObject.getPrimitiveValue()), "tests setting object value to boolean false results in same value");
        assertFalse (testMFBoolObject.get1Value(0),   "tests setting object value to false results in get1Value(index 0) of false");
        
        testMFBoolObject.setValue (singleTrueArray);
        assertTrue  (Arrays.equals(singleTrueArray,  testMFBoolObject.getPrimitiveValue()), "tests setting object value to boolean[] array { true } results in same value");
        testMFBoolObject.setValue(false);              // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(singleFalseArray, testMFBoolObject.getPrimitiveValue()), "tests setting object value to boolean true results in same value");
        testMFBoolObject.setValue (singleFalseArray);
        assertTrue  (Arrays.equals(singleFalseArray, testMFBoolObject.getPrimitiveValue()), "tests setting object value to boolean[] array { false } results in same value");
        testMFBoolObject.setValue (trueTrueArray);
        assertTrue  (Arrays.equals(trueTrueArray,    testMFBoolObject.getPrimitiveValue()), "tests setting object value to boolean[] array { true, true } results in same value");
        testMFBoolObject.setValue (trueFalseArray);
        assertTrue  (Arrays.equals(trueFalseArray,   testMFBoolObject.getPrimitiveValue()), "tests setting object value to boolean[] array { true, false } results in same value");
        testMFBoolObject.setValue (falseTrueArray);
        assertTrue  (Arrays.equals(falseTrueArray,   testMFBoolObject.getPrimitiveValue()), "tests setting object value to boolean[] array { false, true } results in same value");
        testMFBoolObject.setValue (falseFalseArray);
        assertTrue  (Arrays.equals(falseFalseArray,  testMFBoolObject.getPrimitiveValue()), "tests setting object value to boolean[] array { false, false } results in same value");
        
        assertTrue  (MFBoolObject.matches(""),            "MFBoolObject.matches(\"\") tests correct empty string value");
        assertTrue  (MFBoolObject.matches("true"),        "MFBoolObject.matches(\"true\")  tests correct string value");
        assertTrue  (MFBoolObject.matches("false"),       "MFBoolObject.matches(\"false\") tests correct string value");
        assertTrue  (MFBoolObject.matches("true true"),   "MFBoolObject.matches(\"true true\")   tests correct string value");
        assertTrue  (MFBoolObject.matches("true false"),  "MFBoolObject.matches(\"true false\")  tests correct string value");
        assertTrue  (MFBoolObject.matches("false true"),  "MFBoolObject.matches(\"false true\")  tests correct string value");
        assertTrue  (MFBoolObject.matches("false false"), "MFBoolObject.matches(\"false false\") tests correct string value");
        assertTrue  (MFBoolObject.matches("  true "),  "MFBoolObject.matches(\"  true \") tests correct string value, including external whitespace");
        assertTrue  (MFBoolObject.matches(" false "),  "MFBoolObject.matches(\" false \") tests correct string value, including external whitespace");
        assertTrue  (MFBoolObject.matches("  true  true  "),  "MFBoolObject.matches(\"  true \") tests correct string value, including whitespace");
        assertTrue  (MFBoolObject.matches(" false  false "),  "MFBoolObject.matches(\" false  false \") tests correct string value, including whitespace");
        assertTrue  (MFBoolObject.matches("  false , true  "),"MFBoolObject.matches(\"  false , true \") tests correct string value, including whitespace and commas");
        assertTrue  (MFBoolObject.matches(" true,  false "),  "MFBoolObject.matches(\" true,  false \") tests correct string value, including whitespace and commas");
        assertTrue  (MFBoolObject.matches(" true,  false , "),"MFBoolObject.matches(\" true,  false , \") tests correct string value, including whitespace and commas, with allowed trailing comma");
        
        assertFalse (MFBoolObject.matches(","),           "MFBoolObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (MFBoolObject.matches("TRUE"),        "MFBoolObject.matches(\"TRUE\")  tests incorrect case of string value");
        assertFalse (MFBoolObject.matches("FALSE"),       "MFBoolObject.matches(\"FALSE\") tests incorrect case of string value");
        assertFalse (MFBoolObject.matches("rtue"),        "MFBoolObject.matches(\"rtue\") tests incorrect ordering of characters in string value");
        assertFalse (MFBoolObject.matches("aflse"),       "MFBoolObject.matches(\"aflse\") tests incorrect ordering of characters in string value");
        assertFalse (MFBoolObject.matches("blah"),        "MFBoolObject.matches(\"blah\")  tests incorrect string value");
        assertFalse (MFBoolObject.matches("NaN"),         "MFBoolObject.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
        assertFalse (MFBoolObject.matches("true TRUE"),   "MFBoolObject.matches(\"true TRUE\")  tests incorrect case of string value");
        assertFalse (MFBoolObject.matches("false FALSE"), "MFBoolObject.matches(\"false FALSE\") tests incorrect case of string value");
    }
    
    @Test
    @DisplayName("Test SFImageObject single-field variable-length integer/hexadecimal array")
    void SFImageObjectTests()
	{
        System.out.println ("SFImageObjectTests...");
        assertTrue  (Arrays.equals(SFImageObject.DEFAULT_VALUE, new SFImageObject().setValueByString(SFImageObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFImageObject testSFImageObject = new SFImageObject(); // static initializer is tested, might throw exception
        assertTrue  (testSFImageObject.matches(),       "testSFImageObject.matches() tests object initialization correctly matches regex");
        int[]  emptyArray       = {};
        int[]  defaultValueArray= { 0, 0, 0 };
        int[]  singleValueArray = { 1, 1, 1, 0 };
        int[]  doubleValueArray = { 1, 2, 2, 0xFF00,   0xFF00 };
        int[]  tripleValueArray = { 1, 3, 3, 0xFF0000, 0x00FF00, 0x0000FF };
      int[] quadrupleValueArray = { 2, 2, 2, 0xFF00,   0x00FF,   0xF0F0,  0x00FF };

        assertTrue  (Arrays.equals(defaultValueArray, SFImageObject.DEFAULT_VALUE), "test correct default value for this field object");
        assertTrue  (SFImageObject.matches(SFImageObject.DEFAULT_VALUE_STRING),
                                                        "SFImageObject.matches(SFImageObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFImageObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFImageObject.REGEX.contains("^") && !SFImageObject.REGEX.contains("$"), "test SFImageObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertFalse  (SFVec2fObject.REGEX.equals(SFImageObject.REGEX), "test SFVec2fObject.REGEX.equals(SFImageObject.REGEX) returns false");

        testSFImageObject.setValue(0, 0, 0, emptyArray); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(defaultValueArray,  testSFImageObject.getPrimitiveValue()), "tests setting object value to (0, 0, 0, emptyArray) results in default singleton array with same value");
        testSFImageObject.setValue(defaultValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(defaultValueArray,  testSFImageObject.getPrimitiveValue()), "tests setting object value to (defaultValueArray) results in default singleton array with same value");
        
        testSFImageObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(singleValueArray,testSFImageObject.getPrimitiveValue()),   "tests setting object value to (singleValueArray) results in equivalent getPrimitiveValue()");
      
        testSFImageObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(doubleValueArray,testSFImageObject.getPrimitiveValue()),   "tests setting object value to (doubleValueArray) results in equivalent getPrimitiveValue()");
      
        testSFImageObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(tripleValueArray,testSFImageObject.getPrimitiveValue()),   "tests setting object value to (tripleValueArray) results in equivalent getPrimitiveValue()");
      
        testSFImageObject.setValue(quadrupleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(quadrupleValueArray,testSFImageObject.getPrimitiveValue()),   "tests setting object value to (quadrupleValueArray) results in equivalent getPrimitiveValue()");
      
        assertTrue  (SFImageObject.matches("0 0 0"),      "SFImageObject.matches( \"0 0 0\")   tests correct string value");
        
		String valueHexadecimalInput  = " 1 2 3 0x040404 0x050505 ";
		String valueHexadecimalString = "1 2 3 0x40404 0x50505";
		String valueDecimalString     = "1 2 3 263172 328965";
		assertTrue  (SFImageObject.matches(valueHexadecimalInput),      "SFImageObject.matches( \"" + valueHexadecimalInput + "\")   tests correct string value");
        assertTrue  (testSFImageObject.setValueByString(valueHexadecimalInput).matches(), "testSFImageObject.setValueByString(\" 1 2 3 0x040404 0x050505 \").matches() tests correct string value");
        assertTrue  (testSFImageObject.toStringHexadecimal().equals(valueHexadecimalString), "testSFImageObject.toStringHexadecimal() tests correct string output");
        ConfigurationProperties.setSFImagePixelOutputHexadecimal(false);
		assertFalse (ConfigurationProperties.isSFImagePixelOutputHexadecimal(), "test ConfigurationProperties.setSFImagePixelOutputHexadecimal(false)");
        assertTrue  (testSFImageObject.toStringDecimal().equals(valueDecimalString), "testSFImageObject.toStringDecimal() tests correct string output");
        assertTrue  (testSFImageObject.toString().equals(testSFImageObject.toStringDecimal()), "testSFImageObject.toString() checks Decimal string output");
        ConfigurationProperties.setSFImagePixelOutputHexadecimal(true);
		assertTrue  (ConfigurationProperties.isSFImagePixelOutputHexadecimal(), "test ConfigurationProperties.setSFImagePixelOutputHexadecimal(true)");
		assertTrue  (testSFImageObject.toString().equals(testSFImageObject.toStringHexadecimal()), "testSFImageObject.toString() checks Hexadecimal string output");
        
		// TODO
        
        assertFalse (SFImageObject.matches(""),          "SFImageObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFImageObject.matches(","),         "SFImageObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (SFImageObject.matches("true false"),"SFImageObject.matches(\"true false\") tests incorrect boolean string value");
        assertFalse (SFImageObject.matches("blah"),      "SFImageObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFImageObject.matches("NaN NaN"),   "SFImageObject.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test MFImageObject multi-field variable-length integer/hexadecimal array")
    void MFImageObjectTests()
	{
        System.out.println ("MFImageObjectTests...");
        assertTrue  (Arrays.equals(MFImageObject.DEFAULT_VALUE, new MFImageObject().setValueByString(MFImageObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFImageObject testMFImageObject = new MFImageObject(); // static initializer is tested, might throw exception
        assertTrue  (testMFImageObject.matches(),       "testMFImageObject.matches() tests object initialization correctly matches regex");
        int[]  emptyArray       = {};
        int[]  defaultValueArray= { 0, 0, 0 };
        int[]  singleValueArray = { 1, 1, 1, 0 };
        int[]  doubleValueArray = { 1, 2, 2, 0xFF00,   0xFF00 };
        int[]  tripleValueArray = { 1, 3, 3, 0xFF0000, 0x00FF00, 0x0000FF };
      int[] quadrupleValueArray = { 2, 2, 2, 0xFF00,   0x00FF,   0xF0F0,  0x00FF };

        assertTrue  (Arrays.equals(emptyArray, MFImageObject.DEFAULT_VALUE), "test correct default value is empty array for this MF field object");
        assertTrue  (MFImageObject.matches(MFImageObject.DEFAULT_VALUE_STRING),
                                                        "MFImageObject.matches(MFImageObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFImageObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFImageObject.REGEX.contains("^") && !MFImageObject.REGEX.contains("$"), "test MFImageObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertFalse  (MFVec2fObject.REGEX.equals(MFImageObject.REGEX), "test MFVec2fObject.REGEX.equals(MFImageObject.REGEX) returns false");

        // TODO utility method needed to set singleton object using individual values
//        testMFImageObject.setValue(0.0d, 0.0d); // returns void because it matches (overrides) Java SAI specification interface
//        assertTrue  (Arrays.equals(singleValueArray,  testMFImageObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f results in singleton array with same value");
        
        testMFImageObject.setValue(emptyArray); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(emptyArray,testMFImageObject.getPrimitiveValue()),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFImageObject.matches(),       "testMFImageObject.matches() tests emptyArray initialization correctly matches regex");
        testMFImageObject.setValue(defaultValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(defaultValueArray,testMFImageObject.getPrimitiveValue()),   "tests setting object value to defaultValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFImageObject.matches(),       "testMFImageObject.matches() tests defaultValueArray correctly matches regex");
        testMFImageObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(singleValueArray,testMFImageObject.getPrimitiveValue()),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFImageObject.matches(),       "testMFImageObject.matches() tests singleValueArray correctly matches regex");
        testMFImageObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(doubleValueArray,testMFImageObject.getPrimitiveValue()),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFImageObject.matches(),       "testMFImageObject.matches() tests doubleValueArray correctly matches regex");
        testMFImageObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(tripleValueArray,testMFImageObject.getPrimitiveValue()),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFImageObject.matches(),       "testMFImageObject.matches() tests tripleValueArray correctly matches regex");
        testMFImageObject.setValue(quadrupleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(quadrupleValueArray,testMFImageObject.getPrimitiveValue()),   "tests setting object value to quadrupleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFImageObject.matches(),       "testMFImageObject.matches() tests quadrupleValueArray correctly matches regex");
        
        String valueHexadecimalInput  = " 1 2 3 0x040404 0x050505 ";
		String valueHexadecimalString = "1 2 3 0x40404 0x50505";
		String valueDecimalString     = "1 2 3 263172 328965";
		assertTrue  (SFImageObject.matches(valueHexadecimalInput),      "SFImageObject.matches( \"" + valueHexadecimalInput + "\")   tests correct string value");
        assertTrue  (testMFImageObject.setValueByString(valueHexadecimalInput).matches(), "testMFImageObject.setValueByString(\" 1 2 3 0x040404 0x050505 \").matches() tests correct string value");
        assertTrue  (testMFImageObject.toStringHexadecimal().equals(valueHexadecimalString), "testMFImageObject.toStringHexadecimal() tests correct string output");
        ConfigurationProperties.setSFImagePixelOutputHexadecimal(false);
		assertFalse (ConfigurationProperties.isSFImagePixelOutputHexadecimal(), "test ConfigurationProperties.setSFImagePixelOutputHexadecimal(false)");
        assertTrue  (testMFImageObject.toStringDecimal().equals(valueDecimalString), "testMFImageObject.toStringDecimal() tests correct string output");
        assertTrue  (testMFImageObject.toString().equals(testMFImageObject.toStringDecimal()), "testMFImageObject.toString() checks Decimal string output");
        ConfigurationProperties.setSFImagePixelOutputHexadecimal(true);
		assertTrue  (ConfigurationProperties.isSFImagePixelOutputHexadecimal(), "test ConfigurationProperties.setSFImagePixelOutputHexadecimal(true)");
		assertTrue  (testMFImageObject.toString().equals(testMFImageObject.toStringHexadecimal()), "testMFImageObject.toString() checks Hexadecimal string output");
        
		assertTrue  (MFImageObject.matches(""),            "MFImageObject.matches(\"\") tests correct empty string value");
        assertFalse (MFImageObject.matches(","),           "MFImageObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        // TODO more tests with multiple images
        
        assertFalse (MFImageObject.matches("true false"),"MFImageObject.matches(\"true false\") tests incorrect boolean string value");
        assertFalse (MFImageObject.matches("blah"),      "MFImageObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFImageObject.matches("NaN NaN"),   "MFImageObject.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFInt32Object single-field 32-bit integer")
    void SFInt32ObjectTests()
	{
        System.out.println ("SFInt32ObjectTests...");
        assertTrue  ((SFInt32Object.DEFAULT_VALUE == new SFInt32Object().setValueByString(SFInt32Object.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFInt32Object testSFInt32Object = new SFInt32Object(); // static initializer is tested, might throw exception
        assertTrue  (testSFInt32Object.matches(),       "testSFInt32Object.matches() tests object initialization correctly matches regex");
        assertEquals(0, SFInt32Object.DEFAULT_VALUE,    "test correct default value for this field object");
        assertTrue  (SFInt32Object.matches(SFInt32Object.DEFAULT_VALUE_STRING),
                                                        "SFInt32Object.matches(SFInt32Object.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFInt32Object.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFInt32Object.REGEX.contains("^") && !SFInt32Object.REGEX.contains("$"), "test SFInt32Object.REGEX does not contain anchor characters ^ or $");

		testSFInt32Object.setValue(10); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (testSFInt32Object.toStringHexadecimal().equalsIgnoreCase("0xA"), "test SFInt32Object.toStringHexadecimal().equalsIgnoreCase(\"A\") tests hexadecimal output");

        testSFInt32Object.setValue(1); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals( 1,testSFInt32Object.getValue(),   "tests setting object value to 1 results in value of 1");
        testSFInt32Object.setValue(-1); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(-1,testSFInt32Object.getValue(),   "tests setting object value to -1 results in value of -1");
        
        assertTrue  (SFInt32Object.matches( "0"),       "SFInt32Object.matches( \"0\")   tests correct string value");
        assertTrue  (SFInt32Object.matches( "1"),       "SFInt32Object.matches( \"1\")   tests correct string value");
        assertTrue  (SFInt32Object.matches("-1"),       "SFInt32Object.matches(\"-1\")   tests correct string value");
        assertTrue  (SFInt32Object.matches("  3 "),     "SFInt32Object.matches(\"  3 \") tests correct string value, including external whitespace");
        assertTrue  (SFInt32Object.matches(" -3 "),     "SFInt32Object.matches(\" -3 \") tests correct string value, including external whitespace");
        assertTrue  (SFInt32Object.matches("  12E45  "),"SFInt32Object.matches(\"  12E45  \") tests correct string value, scientific notation");
        assertTrue  (SFInt32Object.matches(" +12E+45 "),"SFInt32Object.matches(\" +12E+45 \") tests correct string value, scientific notation");
        assertTrue  (SFInt32Object.matches(" -12E-45 "),"SFInt32Object.matches(\" -12E-45 \") tests correct string value, scientific notation");
        
        assertFalse (SFInt32Object.matches(""),         "SFInt32Object.matches(\"\") tests incorrect empty string value");
        assertFalse (SFInt32Object.matches(","),        "SFInt32Object.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (SFInt32Object.matches("true"),     "SFInt32Object.matches(\"true\") tests incorrect boolean string value");
        assertFalse (SFInt32Object.matches("blah"),     "SFInt32Object.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFInt32Object.matches("0 1"),      "SFInt32Object.matches(\"0 1\")  tests incorrect array as string value");
        assertFalse (SFInt32Object.matches("NaN"),      "SFInt32Object.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test MFInt32Object multi-field 32-bit integer array")
    void MFInt32ObjectTests()
	{
        System.out.println ("MFInt32ObjectTests...");
        assertTrue  (Arrays.equals(MFInt32Object.DEFAULT_VALUE, new MFInt32Object().setValueByString(MFInt32Object.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFInt32Object testMFInt32Object = new MFInt32Object(); // static initializer is tested, might throw exception
        assertTrue  (testMFInt32Object.matches(),       "testMFInt32Object.matches() tests object initialization correctly matches regex");
        int[] emptyArray       = {};
        int[] singleValueArray = { 0 };
        int[] doubleValueArray = { -1, -2 };
        int[] tripleValueArray = {  3,  4,  5 };
        assertTrue  (Arrays.equals(emptyArray, MFInt32Object.DEFAULT_VALUE), "test correct default value for this field object");
        assertTrue  (MFInt32Object.matches(MFInt32Object.DEFAULT_VALUE_STRING),
                                                        "MFInt32Object.matches(MFInt32Object.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFInt32Object.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFInt32Object.REGEX.contains("^") && !MFInt32Object.REGEX.contains("$"), "test MFInt32Object.REGEX does not contain anchor characters ^ or $");

        testMFInt32Object.setValue(0); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals( 0,testMFInt32Object.get1Value(0),   "tests setting object value to 0 results in get1Value(index 0) of 0");
        assertTrue  (Arrays.equals(singleValueArray,  testMFInt32Object.getPrimitiveValue()), "tests setting object value to 0 results in singleton array with same value");
        
        testMFInt32Object.setValue(emptyArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(emptyArray,testMFInt32Object.getPrimitiveValue(),   "tests setting object value to emptyArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFInt32Object.matches(),       "testMFInt32Object.matches() tests emptyArray initialization correctly matches regex");
        testMFInt32Object.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testMFInt32Object.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFInt32Object.matches(),       "testMFInt32Object.matches() tests singleValueArray initialization correctly matches regex");
        testMFInt32Object.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(doubleValueArray,testMFInt32Object.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFInt32Object.matches(),       "testMFInt32Object.matches() tests doubleValueArray initialization correctly matches regex");
        testMFInt32Object.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(tripleValueArray,testMFInt32Object.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFInt32Object.matches(),       "testMFInt32Object.matches() tests tripleValueArray initialization correctly matches regex");
        
        assertTrue  (MFInt32Object.matches(""),           "MFInt32Object.matches(\"\") tests correct empty string value");
        assertFalse (MFInt32Object.matches(","),          "MFInt32Object.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (MFInt32Object.matches( "0"),         "MFInt32Object.matches( \"0\")   tests correct string value");
        assertTrue  (MFInt32Object.matches( "1"),         "MFInt32Object.matches( \"1\")   tests correct string value");
        assertTrue  (MFInt32Object.matches("-1"),         "MFInt32Object.matches(\"-1\")   tests correct string value");
        assertTrue  (MFInt32Object.matches("  3 "),       "MFInt32Object.matches(\"  3 \") tests correct string value, including external whitespace");
        assertTrue  (MFInt32Object.matches(" -3 "),       "MFInt32Object.matches(\" -3 \") tests correct string value, including external whitespace");
        assertTrue  (MFInt32Object.matches(" 0  12E45  "),"MFInt32Object.matches(\" 0  12E45  \") tests correct string value, scientific notation");
        assertTrue  (MFInt32Object.matches("+0 +12E+45 "),"MFInt32Object.matches(\"+0 +12E+45 \") tests correct string value, scientific notation");
        assertTrue  (MFInt32Object.matches("-0,-12E-45 "),"MFInt32Object.matches(\"-0,-12E-45 \") tests correct string value, scientific notation");
        assertTrue  (MFInt32Object.matches("0 1"),        "MFInt32Object.matches(\"0 1\")  tests correct array as string value");
        assertTrue  (MFInt32Object.matches("0, 1"),       "MFInt32Object.matches(\"0, 1\")  tests correct array as string value, including whitespace and comma");
        assertTrue  (MFInt32Object.matches("0 , 1 , "),   "MFInt32Object.matches(\"0 , 1 , \")  tests correct array as string value, including whitespace and commas, with allowed trailing comma");

        assertFalse (MFInt32Object.matches("true"),     "MFInt32Object.matches(\"true\") tests incorrect boolean string value");
        assertFalse (MFInt32Object.matches("blah"),     "MFInt32Object.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFInt32Object.matches("NaN"),      "MFInt32Object.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFFloatObject single-field single-precision floating-point number")
    void SFFloatObjectTests()
	{
        System.out.println ("SFFloatObjectTests...");
        assertTrue  ((SFFloatObject.DEFAULT_VALUE == new SFFloatObject().setValueByString(SFFloatObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                          "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFFloatObject testSFFloatObject = new SFFloatObject(); // static initializer is tested, might throw exception
        assertTrue  (testSFFloatObject.matches(),         "testSFFloatObject.matches() tests object initialization correctly matches regex");
        assertEquals(0.0f, SFFloatObject.DEFAULT_VALUE,   "test correct default value for this field object");
        assertTrue  (SFFloatObject.matches(SFFloatObject.DEFAULT_VALUE_STRING),
                                                          "SFFloatObject.matches(SFFloatObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFFloatObject.isDefaultValue(),  "test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFFloatObject.REGEX.contains("^") && !SFFloatObject.REGEX.contains("$"), "test SFFloatObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (SFFloatObject.REGEX.equals(SFDoubleObject.REGEX), "test SFFloatObject.REGEX.equals(SFDoubleObject.REGEX) returns true");
        assertTrue  (SFFloatObject.REGEX.equals(SFTimeObject.REGEX),   "test SFFloatObject.REGEX.equals(SFTimeObject.REGEX) returns true");
        
        testSFFloatObject.setValue(1); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals( 1.0f,testSFFloatObject.getValue(),  "tests setting object value to 1 results in value of 1.0f");
        testSFFloatObject.setValue(1.0f); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals( 1.0f,testSFFloatObject.getValue(),  "tests setting object value to 1.0f results in value of 1.0f");
        testSFFloatObject.setValue(1.0d); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals( 1.0f,testSFFloatObject.getValue(),  "tests setting object value to 1.0d results in value of 1.0f");
        testSFFloatObject.setValue(-1); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(-1.0f,testSFFloatObject.getValue(),  "tests setting object value to -1 results in value of -1.0f");
        
        assertTrue  (SFFloatObject.matches( "0"),       "SFFloatObject.matches( \"0\")   tests correct string value");
        assertTrue  (SFFloatObject.matches( "1"),       "SFFloatObject.matches( \"1\")   tests correct string value");
        assertTrue  (SFFloatObject.matches("-1"),       "SFFloatObject.matches(\"-1\")   tests correct string value");
        assertTrue  (SFFloatObject.matches( "0.0"),     "SFFloatObject.matches( \"0.0\") tests correct string value");
        assertTrue  (SFFloatObject.matches( "1.0"),     "SFFloatObject.matches( \"1.0\") tests correct string value");
        assertTrue  (SFFloatObject.matches("-1.0"),     "SFFloatObject.matches(\"-1.0\") tests correct string value");
        assertTrue  (SFFloatObject.matches("  3 "),     "SFFloatObject.matches(\"  3 \") tests correct string value, including external whitespace");
        assertTrue  (SFFloatObject.matches(" -3 "),     "SFFloatObject.matches(\" -3 \") tests correct string value, including external whitespace");
        assertTrue  (SFFloatObject.matches(" 12.3E45 "),"SFFloatObject.matches(\" 12.3E45 \") tests correct string value, scientific notation");
        assertTrue  (SFFloatObject.matches("+12.3E+45"),"SFFloatObject.matches(\"+12.3E+45\") tests correct string value, scientific notation");
        assertTrue  (SFFloatObject.matches("-12.3E-45"),"SFFloatObject.matches(\"-12.3E-45\") tests correct string value, scientific notation");
        assertTrue  (SFFloatObject.matches( ".6"),      "SFFloatObject.matches( \".6\") tests correct string value, no leading digit before decimal point");
        assertTrue  (SFFloatObject.matches("-.6"),      "SFFloatObject.matches(\"-.6\") tests correct string value, no leading digit before decimal point");
        assertTrue  (SFFloatObject.matches( ".6e7"),    "SFFloatObject.matches( \".6e7\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertTrue  (SFFloatObject.matches("-.6e-7"),   "SFFloatObject.matches(\"-.6e-7\") tests correct string value, no leading digit before decimal point, scientific notation");

        assertFalse (SFFloatObject.matches(""),        "SFFloatObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFFloatObject.matches(","),       "SFFloatObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (SFFloatObject.matches("true"),    "SFFloatObject.matches(\"true\") tests incorrect boolean string value");
        assertFalse (SFFloatObject.matches("blah"),    "SFFloatObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFFloatObject.matches("0 1"),     "SFFloatObject.matches(\"0 1\")  tests incorrect array as string value");
        assertFalse (SFFloatObject.matches("0.0 1.0"), "SFFloatObject.matches(\"0.0 1.0\") tests incorrect array as string value");
        assertFalse (SFFloatObject.matches("NaN"),     "SFFloatObject.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test MFFloatObject multi-field single-precision floating-point array")
    void MFFloatObjectTests()
	{
        System.out.println ("MFFloatObjectTests...");
        assertTrue  (Arrays.equals(MFFloatObject.DEFAULT_VALUE, new MFFloatObject().setValueByString(MFFloatObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFFloatObject testMFFloatObject = new MFFloatObject(); // static initializer is tested, might throw exception
        assertTrue  (testMFFloatObject.matches(),       "testMFFloatObject.matches() tests object initialization correctly matches regex");
        float[] emptyArray       = {};
        float[] singleValueArray = { 0.0f };
        float[] doubleValueArray = { -1.0f, -2.0f };
        float[] tripleValueArray = {  3.0f,  4.0f,  5.0f };
        assertTrue  (Arrays.equals(emptyArray, MFFloatObject.DEFAULT_VALUE), "test correct default value for this field object");
        assertTrue  (MFFloatObject.matches(MFFloatObject.DEFAULT_VALUE_STRING),
                                                        "MFFloatObject.matches(MFFloatObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFFloatObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFFloatObject.REGEX.contains("^") && !MFFloatObject.REGEX.contains("$"), "test MFFloatObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (MFFloatObject.REGEX.equals(MFDoubleObject.REGEX), "test MFFloatObject.REGEX.equals(MFDoubleObject.REGEX) returns true");
        assertTrue  (MFFloatObject.REGEX.equals(MFTimeObject.REGEX),   "test MFFloatObject.REGEX.equals(MFTimeObject.REGEX) returns true");

        testMFFloatObject.setValue(0.0f); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals( 0.0f,testMFFloatObject.get1Value(0),   "tests setting object value to 0.0f results in get1Value(index 0) of 0");
        assertTrue  (Arrays.equals(singleValueArray,  testMFFloatObject.getPrimitiveValue()), "tests setting object value to 0.0f results in singleton array with same value");
        
        testMFFloatObject.setValue(emptyArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(emptyArray,testMFFloatObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFFloatObject.matches(),       "testMFFloatObject.matches() tests emptyArray initialization correctly matches regex");
        testMFFloatObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testMFFloatObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        testMFFloatObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(doubleValueArray,testMFFloatObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        testMFFloatObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(tripleValueArray,testMFFloatObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        
        assertTrue  (MFFloatObject.matches(""),          "MFFloatObject.matches(\"\") tests correct empty string value");
        assertFalse (MFFloatObject.matches(","),         "MFFloatObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (MFFloatObject.matches( "0"),        "MFFloatObject.matches( \"0\")   tests correct string value");
        assertTrue  (MFFloatObject.matches( "1"),        "MFFloatObject.matches( \"1\")   tests correct string value");
        assertTrue  (MFFloatObject.matches("-1"),        "MFFloatObject.matches(\"-1\")   tests correct string value");
        assertTrue  (MFFloatObject.matches("  3 "),      "MFFloatObject.matches(\"  3 \") tests correct string value, including external whitespace");
        assertTrue  (MFFloatObject.matches(" -3 "),      "MFFloatObject.matches(\" -3 \") tests correct string value, including external whitespace");
        assertTrue  (MFFloatObject.matches(" 0  12E45 "),"MFFloatObject.matches(\"0  12E45 \") tests correct string value, scientific notation");
        assertTrue  (MFFloatObject.matches("+0,+12E+45"),"MFFloatObject.matches(\"+0,+12E+45\") tests correct string value, scientific notation");
        assertTrue  (MFFloatObject.matches("-0,-12E-45"),"MFFloatObject.matches(\"-0,-12E-45\") tests correct string value, scientific notation");

        assertTrue  (MFFloatObject.matches( "0.0"),        "MFFloatObject.matches( \"0.0\")   tests correct string value");
        assertTrue  (MFFloatObject.matches( "1.0"),        "MFFloatObject.matches( \"1.0\")   tests correct string value");
        assertTrue  (MFFloatObject.matches("-1.0"),        "MFFloatObject.matches(\"-1.0\")   tests correct string value");
        assertTrue  (MFFloatObject.matches("  3.0 "),      "MFFloatObject.matches(\"  3.0 \") tests correct string value, including external whitespace");
        assertTrue  (MFFloatObject.matches(" -3.0 "),      "MFFloatObject.matches(\" -3.0 \") tests correct string value, including external whitespace");
        assertTrue  (MFFloatObject.matches("0 1"),         "MFFloatObject.matches(\"0 1\")  tests correct array as string value");
        assertTrue  (MFFloatObject.matches("0.0 1.0"),     "MFFloatObject.matches(\"0.0 1.0\")  tests correct array as string value");
        assertTrue  (MFFloatObject.matches(" 0.0  12.3E45  "),    "MFFloatObject.matches(\" 0.0  12.3E45  \") tests correct string value, scientific notation");
        assertTrue  (MFFloatObject.matches("+0.0 +12.3E+45 "),    "MFFloatObject.matches(\"+0.0 +12.3E+45 \") tests correct string value, scientific notation");
        assertTrue  (MFFloatObject.matches("-0.0,-12.3E-45 "),    "MFFloatObject.matches(\"-0.0,-12.3E-45 \") tests correct string value, scientific notation");
        assertTrue  (MFFloatObject.matches( ".6 -.6 .6e7 -.6e-7"),"MFFloatObject.matches(\".6 -.6 .6e7 -.6e-7\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertTrue  (MFFloatObject.matches("0 , 1 , "),   "MFFloatObject.matches(\"0 , 1 , \")  tests correct array as string value, including whitespace and commas, with allowed trailing comma");
        
        assertFalse (MFFloatObject.matches("true"),     "MFFloatObject.matches(\"true\") tests incorrect boolean string value");
        assertFalse (MFFloatObject.matches("blah"),     "MFFloatObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFFloatObject.matches("NaN"),      "MFFloatObject.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFDoubleObject single-field double-precision floating-point number")
    void SFDoubleObjectTests()
	{
        System.out.println ("SFDoubleObjectTests...");
        assertTrue  ((SFDoubleObject.DEFAULT_VALUE == new SFDoubleObject().setValueByString(SFDoubleObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                           "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFDoubleObject testSFDoubleObject = new SFDoubleObject(); // static initializer is tested, might throw exception
        assertTrue  (testSFDoubleObject.matches(),         "testSFDoubleObject.matches() tests object initialization correctly matches regex");
        assertEquals(0.0d, SFDoubleObject.DEFAULT_VALUE,   "test correct default value for this field object");
        assertTrue  (SFDoubleObject.matches(SFDoubleObject.DEFAULT_VALUE_STRING),
                                                           "SFDoubleObject.matches(SFDoubleObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFDoubleObject.isDefaultValue(),  "test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFDoubleObject.REGEX.contains("^") && !SFDoubleObject.REGEX.contains("$"), "test SFDoubleObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (SFFloatObject.REGEX.equals(SFDoubleObject.REGEX), "test SFFloatObject.REGEX.equals(SFDoubleObject.REGEX) returns true");
        assertTrue  (SFFloatObject.REGEX.equals(SFTimeObject.REGEX),   "test SFFloatObject.REGEX.equals(SFTimeObject.REGEX) returns true");

        testSFDoubleObject.setValue(1); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals( 1.0d,testSFDoubleObject.getValue(),  "tests setting object value to 1 results in value of 1.0d");
        testSFDoubleObject.setValue(1.0d); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals( 1.0d,testSFDoubleObject.getValue(),  "tests setting object value to 1.0d results in value of 1.0d");
        testSFDoubleObject.setValue(1.0d); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals( 1.0d,testSFDoubleObject.getValue(),  "tests setting object value to 1.0d results in value of 1.0d");
        testSFDoubleObject.setValue(-1); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(-1.0d,testSFDoubleObject.getValue(),  "tests setting object value to -1 results in value of -1.0d");
        
        assertTrue  (SFDoubleObject.matches( "0"),      "SFDoubleObject.matches( \"0\")   tests correct string value");
        assertTrue  (SFDoubleObject.matches( "1"),      "SFDoubleObject.matches( \"1\")   tests correct string value");
        assertTrue  (SFDoubleObject.matches("-1"),      "SFDoubleObject.matches(\"-1\")   tests correct string value");
        assertTrue  (SFDoubleObject.matches( "0.0"),    "SFDoubleObject.matches( \"0.0\") tests correct string value");
        assertTrue  (SFDoubleObject.matches( "1.0"),    "SFDoubleObject.matches( \"1.0\") tests correct string value");
        assertTrue  (SFDoubleObject.matches("-1.0"),    "SFDoubleObject.matches(\"-1.0\") tests correct string value");
        assertTrue  (SFDoubleObject.matches("  3 "),    "SFDoubleObject.matches(\"  3 \") tests correct string value, including external whitespace");
        assertTrue  (SFDoubleObject.matches(" -3 "),    "SFDoubleObject.matches(\" -3 \") tests correct string value, including external whitespace");
        assertTrue  (SFDoubleObject.matches( ".6"),     "SFDoubleObject.matches( \".6\") tests correct string value, no leading digit before decimal point");
        assertTrue  (SFDoubleObject.matches("-.6"),     "SFDoubleObject.matches(\"-.6\") tests correct string value, no leading digit before decimal point");
        assertTrue  (SFDoubleObject.matches( ".6e7"),   "SFDoubleObject.matches( \".6e7\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertTrue  (SFDoubleObject.matches("-.6e-7"),  "SFDoubleObject.matches(\"-.6e-7\") tests correct string value, no leading digit before decimal point, scientific notation");

        assertFalse (SFDoubleObject.matches(" 0.0  12.3E45  "),"SFDoubleObject.matches(\" 0.0  12.3E45  \") tests correct string value, scientific notation");
        assertFalse (SFDoubleObject.matches("+0.0 +12.3E+45 "),"SFDoubleObject.matches(\"+0.0 +12.3E+45 \") tests correct string value, scientific notation");
        assertFalse (SFDoubleObject.matches("-0.0,-12.3E-45 "),"SFDoubleObject.matches(\"-0.0,-12.3E-45 \") tests correct string value, scientific notation");
        
        assertFalse (SFDoubleObject.matches(""),        "SFDoubleObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFDoubleObject.matches(","),       "SFDoubleObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (SFDoubleObject.matches("true"),    "SFDoubleObject.matches(\"true\") tests incorrect boolean string value");
        assertFalse (SFDoubleObject.matches("blah"),    "SFDoubleObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFDoubleObject.matches("0 1"),     "SFDoubleObject.matches(\"0 1\")  tests incorrect array as string value");
        assertFalse (SFDoubleObject.matches("0.0 1.0"), "SFDoubleObject.matches(\"0.0 1.0\") tests incorrect array as string value");
        assertFalse (SFDoubleObject.matches("NaN"),     "SFDoubleObject.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test MFDoubleObject multi-field double-precision floating-point array")
    void MFDoubleObjectTests()
	{
        System.out.println ("MFDoubleObjectTests...");
        assertTrue  (Arrays.equals(MFDoubleObject.DEFAULT_VALUE, new MFDoubleObject().setValueByString(MFDoubleObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFDoubleObject testMFDoubleObject = new MFDoubleObject(); // static initializer is tested, might throw exception
        assertTrue  (testMFDoubleObject.matches(),       "testMFDoubleObject.matches() tests object initialization correctly matches regex");
        double[] emptyArray       = {};
        double[] singleValueArray = { 0.0 };
        double[] doubleValueArray = { -1.0, -2.0 };
        double[] tripleValueArray = {  3.0,  4.0,  5.0 };
        assertTrue  (Arrays.equals(emptyArray, MFDoubleObject.DEFAULT_VALUE), "test correct default value for this field object");
        assertTrue  (MFDoubleObject.matches(MFDoubleObject.DEFAULT_VALUE_STRING),
                                                        "MFDoubleObject.matches(MFDoubleObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFDoubleObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFDoubleObject.REGEX.contains("^") && !MFDoubleObject.REGEX.contains("$"), "test MFDoubleObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (MFFloatObject.REGEX.equals(MFDoubleObject.REGEX), "test MFFloatObject.REGEX.equals(MFDoubleObject.REGEX) returns true");
        assertTrue  (MFFloatObject.REGEX.equals(MFTimeObject.REGEX),   "test MFFloatObject.REGEX.equals(MFTimeObject.REGEX) returns true");

        testMFDoubleObject.setValue(0.0); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals( 0.0,testMFDoubleObject.get1Value(0),   "tests setting object value to 0.0 results in get1Value(index 0) of 0");
        assertTrue  (Arrays.equals(singleValueArray,  testMFDoubleObject.getPrimitiveValue()), "tests setting object value to 0.0 results in singleton array with same value");
        
        testMFDoubleObject.setValue(emptyArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(emptyArray,testMFDoubleObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFDoubleObject.matches(),       "testMFDoubleObject.matches() tests emptyArray initialization correctly matches regex");
        testMFDoubleObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testMFDoubleObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        testMFDoubleObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(doubleValueArray,testMFDoubleObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        testMFDoubleObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(tripleValueArray,testMFDoubleObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        
        assertTrue  (MFDoubleObject.matches(""),          "MFDoubleObject.matches(\"\") tests correct empty string value");
        assertFalse (MFDoubleObject.matches(","),        "MFDoubleObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (MFDoubleObject.matches( "0"),        "MFDoubleObject.matches( \"0\")   tests correct string value");
        assertTrue  (MFDoubleObject.matches( "1"),        "MFDoubleObject.matches( \"1\")   tests correct string value");
        assertTrue  (MFDoubleObject.matches("-1"),        "MFDoubleObject.matches(\"-1\")   tests correct string value");
        assertTrue  (MFDoubleObject.matches("  3 "),      "MFDoubleObject.matches(\"  3 \") tests correct string value, including external whitespace");
        assertTrue  (MFDoubleObject.matches(" -3 "),      "MFDoubleObject.matches(\" -3 \") tests correct string value, including external whitespace");
        assertTrue  (MFDoubleObject.matches(" 0  12E45 "), "MFDoubleObject.matches(\" 0  12E45  \") tests correct string value, scientific notation");
        assertTrue  (MFDoubleObject.matches("+0 +12E+45 "),"MFDoubleObject.matches(\"+0 +12E+45 \") tests correct string value, scientific notation");
        assertTrue  (MFDoubleObject.matches("-0,-12E-45 "),"MFDoubleObject.matches(\"-0,-12E-45 \") tests correct string value, scientific notation");

        assertTrue  (MFDoubleObject.matches( "0.0"),        "MFDoubleObject.matches( \"0.0\")   tests correct string value");
        assertTrue  (MFDoubleObject.matches( "1.0"),        "MFDoubleObject.matches( \"1.0\")   tests correct string value");
        assertTrue  (MFDoubleObject.matches("-1.0"),        "MFDoubleObject.matches(\"-1.0\")   tests correct string value");
        assertTrue  (MFDoubleObject.matches("  3.0 "),      "MFDoubleObject.matches(\"  3.0 \") tests correct string value, including external whitespace");
        assertTrue  (MFDoubleObject.matches(" -3.0 "),      "MFDoubleObject.matches(\" -3.0 \") tests correct string value, including external whitespace");
        assertTrue  (MFDoubleObject.matches("0 1"),                "MFDoubleObject.matches(\"0 1\")  tests correct array as string value");
        assertTrue  (MFDoubleObject.matches("0.0 1.0"),            "MFDoubleObject.matches(\"0.0 1.0\")  tests correct array as string value");
        assertTrue  (MFDoubleObject.matches(" 0.0  12.3E45  "),    "MFDoubleObject.matches(\" 0.0  12.3E45 \") tests correct string value, scientific notation");
        assertTrue  (MFDoubleObject.matches("+0.0 +12.3E+45 "),    "MFDoubleObject.matches(\"+0.0 +12.3E+45 \") tests correct string value, scientific notation");
        assertTrue  (MFDoubleObject.matches("-0.0,-12.3E-45 "),    "MFDoubleObject.matches(\"-0.0,-12.3E-45 \") tests correct string value, scientific notation");
        assertTrue  (MFDoubleObject.matches( ".6 -.6 .6e7 -.6e-7"),"MFDoubleObject.matches(\".6 -.6 .6e7 -.6e-7\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertTrue  (MFDoubleObject.matches("0 , 1 , "),           "MFDoubleObject.matches(\"0 , 1 , \")  tests correct array as string value, including whitespace and commas, with allowed trailing comma");

        assertFalse (MFDoubleObject.matches("true"),     "MFDoubleObject.matches(\"true\") tests incorrect boolean string value");
        assertFalse (MFDoubleObject.matches("blah"),     "MFDoubleObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFDoubleObject.matches("NaN"),      "MFDoubleObject.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFTimeObject single-field double-precision floating-point number")
    void SFTimeObjectTests()
	{
        System.out.println ("SFTimeObjectTests...");
        assertTrue  ((SFTimeObject.DEFAULT_VALUE == new SFTimeObject().setValueByString(SFTimeObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                         "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFTimeObject testSFTimeObject = new SFTimeObject(); // static initializer is tested, might throw exception
        assertTrue  (testSFTimeObject.matches(),         "testSFTimeObject.matches() tests object initialization correctly matches regex");
        assertEquals(-1.0d, SFTimeObject.DEFAULT_VALUE,  "test correct default value for this field object");
        assertTrue  (SFTimeObject.matches(SFTimeObject.DEFAULT_VALUE_STRING),
                                                         "SFTimeObject.matches(SFTimeObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFTimeObject.isDefaultValue(),  "test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFTimeObject.REGEX.contains("^") && !SFTimeObject.REGEX.contains("$"), "test SFTimeObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (SFFloatObject.REGEX.equals(SFDoubleObject.REGEX), "test SFFloatObject.REGEX.equals(SFDoubleObject.REGEX) returns true");
        assertTrue  (SFFloatObject.REGEX.equals(SFTimeObject.REGEX),   "test SFFloatObject.REGEX.equals(SFTimeObject.REGEX) returns true");

        testSFTimeObject.setValue(1); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals( 1.0d,testSFTimeObject.getValue(),  "tests setting object value to 1 results in value of 1.0d");
        testSFTimeObject.setValue(1.0d); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals( 1.0d,testSFTimeObject.getValue(),  "tests setting object value to 1.0d results in value of 1.0d");
        testSFTimeObject.setValue(1.0d); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals( 1.0d,testSFTimeObject.getValue(),  "tests setting object value to 1.0d results in value of 1.0d");
        testSFTimeObject.setValue(-1); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(-1.0d,testSFTimeObject.getValue(),  "tests setting object value to -1 results in value of -1.0d");
        testSFTimeObject.setValue(-2); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(-2.0d,testSFTimeObject.getValue(),  "tests setting object value to -2 results in value of -2.0d, a legal negative time value");
        
        assertFalse (SFTimeObject.matches(""),        "SFTimeObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFTimeObject.matches(","),       "SFTimeObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (SFTimeObject.matches( "0"),      "SFTimeObject.matches( \"0\")   tests correct string value");
        assertTrue  (SFTimeObject.matches( "1"),      "SFTimeObject.matches( \"1\")   tests correct string value");
        assertTrue  (SFTimeObject.matches("-1"),      "SFTimeObject.matches(\"-1\")   tests correct string value");
        assertTrue  (SFTimeObject.matches( "0.0"),    "SFTimeObject.matches( \"0.0\") tests correct string value");
        assertTrue  (SFTimeObject.matches( "1.0"),    "SFTimeObject.matches( \"1.0\") tests correct string value");
        assertTrue  (SFTimeObject.matches("-1.0"),    "SFTimeObject.matches(\"-1.0\") tests correct string value");
        assertTrue  (SFTimeObject.matches("-2.0"),    "SFTimeObject.matches(\"-2.0\") tests correct string value, a legal negative time value");
        assertTrue  (SFTimeObject.matches("  3 "),    "SFTimeObject.matches(\"  3 \") tests correct string value, including external whitespace");
        assertTrue  (SFTimeObject.matches(" -3 "),    "SFTimeObject.matches(\" -3 \") tests correct string value, including external whitespace");
        assertTrue  (SFTimeObject.matches( ".6"),     "SFTimeObject.matches( \".6\") tests correct string value, no leading digit before decimal point");
        assertTrue  (SFTimeObject.matches("-.6"),     "SFTimeObject.matches(\"-.6\") tests correct string value, no leading digit before decimal point");
        assertTrue  (SFTimeObject.matches( ".6e7"),   "SFTimeObject.matches( \".6e7\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertTrue  (SFTimeObject.matches("-.6e-7"),  "SFTimeObject.matches(\"-.6e-7\") tests correct string value, no leading digit before decimal point, scientific notation");

        assertFalse (SFTimeObject.matches("true"),    "SFTimeObject.matches(\"true\") tests incorrect boolean string value");
        assertFalse (SFTimeObject.matches("blah"),    "SFTimeObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFTimeObject.matches("0 1"),     "SFTimeObject.matches(\"0 1\")  tests incorrect array as string value");
        assertFalse (SFTimeObject.matches("0.0 1.0"), "SFTimeObject.matches(\"0.0 1.0\") tests incorrect array as string value");
        assertFalse (SFTimeObject.matches("NaN"),     "SFTimeObject.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test MFTimeObject multi-field double-precision floating-point array")
    void MFTimeObjectTests()
	{
        System.out.println ("MFTimeObjectTests...");
        assertTrue  (Arrays.equals(MFTimeObject.DEFAULT_VALUE, new MFTimeObject().setValueByString(MFTimeObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFTimeObject testMFTimeObject = new MFTimeObject(); // static initializer is tested, might throw exception
        assertTrue  (testMFTimeObject.matches(),       "testMFTimeObject.matches() tests object initialization correctly matches regex");
        double[] emptyArray       = {};
        double[] singleValueArray = { 0.0 };
        double[] doubleValueArray = { -1.0, -2.0 };
        double[] tripleValueArray = {  3.0,  4.0,  5.0 };
        assertTrue  (Arrays.equals(emptyArray, MFTimeObject.DEFAULT_VALUE), "test correct default value for this field object");
        assertTrue  (MFTimeObject.matches(MFTimeObject.DEFAULT_VALUE_STRING),
                                                        "MFTimeObject.matches(MFTimeObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFTimeObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFTimeObject.REGEX.contains("^") && !MFTimeObject.REGEX.contains("$"), "test MFTimeObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (MFFloatObject.REGEX.equals(MFDoubleObject.REGEX), "test MFFloatObject.REGEX.equals(MFDoubleObject.REGEX) returns true");
        assertTrue  (MFFloatObject.REGEX.equals(MFTimeObject.REGEX),   "test MFFloatObject.REGEX.equals(MFTimeObject.REGEX) returns true");

        testMFTimeObject.setValue(0.0); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals( 0.0,testMFTimeObject.get1Value(0),   "tests setting object value to 0.0 results in get1Value(index 0) of 0");
        assertTrue  (Arrays.equals(singleValueArray,  testMFTimeObject.getPrimitiveValue()), "tests setting object value to 0.0 results in singleton array with same value");
        
        testMFTimeObject.setValue(emptyArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(emptyArray,testMFTimeObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFTimeObject.matches(),       "testMFTimeObject.matches() tests emptyArray initialization correctly matches regex");
        testMFTimeObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testMFTimeObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        testMFTimeObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(doubleValueArray,testMFTimeObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        testMFTimeObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(tripleValueArray,testMFTimeObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        
        assertTrue  (MFTimeObject.matches(""),          "MFTimeObject.matches(\"\") tests correct empty string value");
        assertFalse (MFTimeObject.matches(","),         "MFTimeObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (MFTimeObject.matches( "0"),        "MFTimeObject.matches( \"0\")   tests correct string value");
        assertTrue  (MFTimeObject.matches( "1"),        "MFTimeObject.matches( \"1\")   tests correct string value");
        assertTrue  (MFTimeObject.matches("-1"),        "MFTimeObject.matches(\"-1\")   tests correct string value");
        assertTrue  (MFTimeObject.matches("  3 "),      "MFTimeObject.matches(\"  3 \") tests correct string value, including external whitespace");
        assertTrue  (MFTimeObject.matches(" -3 "),      "MFTimeObject.matches(\" -3 \") tests correct string value, including external whitespace");
        assertTrue  (MFTimeObject.matches(" 0 12E45  "),"MFTimeObject.matches(\"0  12E45\") tests correct string value, scientific notation");
        assertTrue  (MFTimeObject.matches("+0,+12E+45"),"MFTimeObject.matches(\"+0,+12E+45\") tests correct string value, scientific notation");
        assertTrue  (MFTimeObject.matches("-0,-12E-45"),"MFTimeObject.matches(\"-0,-12E-45\") tests correct string value, scientific notation");

        assertTrue  (MFTimeObject.matches( "0.0"),        "MFTimeObject.matches( \"0\")   tests correct string value");
        assertTrue  (MFTimeObject.matches( "1.0"),        "MFTimeObject.matches( \"1\")   tests correct string value");
        assertTrue  (MFTimeObject.matches("-1.0"),        "MFTimeObject.matches(\"-1\")   tests correct string value");
        assertTrue  (MFTimeObject.matches("  3.0 "),      "MFTimeObject.matches(\"  3 \") tests correct string value, including external whitespace");
        assertTrue  (MFTimeObject.matches(" -3.0 "),      "MFTimeObject.matches(\" -3 \") tests correct string value, including external whitespace");
        assertTrue  (MFTimeObject.matches("0 1"),                "MFTimeObject.matches(\"0 1\")  tests correct array as string value");
        assertTrue  (MFTimeObject.matches("0.0 1.0"),            "MFTimeObject.matches(\"0.0 1.0\")  tests correct array as string value");
        assertTrue  (MFTimeObject.matches(" 0.0  12.3E45  "),    "MFTimeObject.matches(\" 0.0  12.3E45 \") tests correct string value, scientific notation");
        assertTrue  (MFTimeObject.matches("+0.0,+12.3E+45 "),    "MFTimeObject.matches(\"+0.0,+12.3E+45 \") tests correct string value, scientific notation");
        assertTrue  (MFTimeObject.matches("-0.0,-12.3E-45 "),    "MFTimeObject.matches(\"-0.0,-12.3E-45 \") tests correct string value, scientific notation");
        assertTrue  (MFTimeObject.matches( ".6 -.6 .6e7 -.6e-7"),"MFTimeObject.matches(\".6 -.6 .6e7 -.6e-7\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertTrue  (MFTimeObject.matches("0 , 1 , "),           "MFTimeObject.matches(\"0 , 1 , \")  tests correct array as string value, including whitespace and commas, with allowed trailing comma");
        
        assertFalse (MFTimeObject.matches("true"),     "MFTimeObject.matches(\"true\") tests incorrect boolean string value");
        assertFalse (MFTimeObject.matches("blah"),     "MFTimeObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFTimeObject.matches("NaN"),      "MFTimeObject.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFVec2fObject single-field 2-tuple single-precision floating-point array")
    void SFVec2fObjectTests()
	{
        System.out.println ("SFVec2fObjectTests...");
        assertTrue  (Arrays.equals(SFVec2fObject.DEFAULT_VALUE, new SFVec2fObject().setValueByString(SFVec2fObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFVec2fObject testSFVec2fObject = new SFVec2fObject(); // static initializer is tested, might throw exception
        assertTrue  (testSFVec2fObject.matches(),       "testSFVec2fObject.matches() tests object initialization correctly matches regex");
        float[] emptyArray       = {};
        float[] singleValueArray = { 0.0f, 0.0f };
//      float[] doubleValueArray = { 0.0f, 1.0f, -2.0f, -3.0f };
//      float[] tripleValueArray = { 0.0f, 1.0f, -2.0f, -3.0f,  4.0f,  5.0f };
        assertTrue  (Arrays.equals(singleValueArray, SFVec2fObject.DEFAULT_VALUE), "test correct default value for this field object");
        assertTrue  (SFVec2fObject.matches(SFVec2fObject.DEFAULT_VALUE_STRING),
                                                        "SFVec2fObject.matches(SFVec2fObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFVec2fObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFVec2fObject.REGEX.contains("^") && !SFVec2fObject.REGEX.contains("$"), "test SFVec2fObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (SFVec2fObject.REGEX.equals(SFVec2dObject.REGEX), "test SFVec2fObject.REGEX.equals(SFVec2dObject.REGEX) returns true");

        testSFVec2fObject.setValue(0.0f, 0.0f); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(singleValueArray,  testSFVec2fObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f results in singleton array with same value");
        
        testSFVec2fObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testSFVec2fObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
      
        assertTrue  (SFVec2fObject.matches( "0 1"),      "SFVec2fObject.matches( \"0 1\")   tests correct string value");
        assertTrue  (SFVec2fObject.matches( "2 3"),      "SFVec2fObject.matches( \"2 3\")   tests correct string value");
        assertTrue  (SFVec2fObject.matches("-1 -2"),     "SFVec2fObject.matches(\"-1 -2\")  tests correct string value");
        assertTrue  (SFVec2fObject.matches(" 0  12E45 "),"SFVec2fObject.matches(\"0  12E45 \") tests correct string value, scientific notation");
        assertTrue  (SFVec2fObject.matches("+0 +12E+45"),"SFVec2fObject.matches(\"+0,+12E+45\") tests correct string value, scientific notation");
        assertTrue  (SFVec2fObject.matches("-0 -12E-45"),"SFVec2fObject.matches(\"-0,-12E-45\") tests correct string value, scientific notation");

        assertTrue  (SFVec2fObject.matches(" 0.0  1.0 "),  "SFVec2fObject.matches(\" 0.0  1.0 \") tests correct string value");
        assertTrue  (SFVec2fObject.matches("-1.0 -3.0"),   "SFVec2fObject.matches(\"-1.0 -3.0\") tests correct string value, including external whitespace");
        assertTrue  (SFVec2fObject.matches("0 1"),         "SFVec2fObject.matches(\"0 1\")  tests correct array as string value");
        assertTrue  (SFVec2fObject.matches("0.0 1.0"),     "SFVec2fObject.matches(\"0.0 1.0\")  tests correct array as string value");
        assertTrue  (SFVec2fObject.matches(" 0.0  12.3E45  "),    "SFVec2fObject.matches(\" 0.0  12.3E45  \") tests correct string value, scientific notation");
        assertTrue  (SFVec2fObject.matches("+0.0 +12.3E+45 "),    "SFVec2fObject.matches(\"+0.0 +12.3E+45 \") tests correct string value, scientific notation");
        assertTrue  (SFVec2fObject.matches("-0.0 -12.3E-45 "),    "SFVec2fObject.matches(\"-0.0 -12.3E-45 \") tests correct string value, scientific notation");
        
        assertFalse (SFVec2fObject.matches(""),          "SFVec2fObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFVec2fObject.matches(","),         "SFVec2fObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (SFVec2fObject.matches( ".6 -.6 .6e7 -.6e-7"),"SFVec2fObject.matches(\".6 -.6 .6e7 -.6e-7\") tests incorrect string value, no leading digit before decimal point, scientific notation");
        assertFalse (SFVec2fObject.matches("true false"),"SFVec2fObject.matches(\"true false\") tests incorrect boolean string value");
        assertFalse (SFVec2fObject.matches("blah"),      "SFVec2fObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFVec2fObject.matches("NaN NaN"),   "SFVec2fObject.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test MFVec2fObject multi-field 2-tuple single-precision floating-point array")
    void MFVec2fObjectTests()
	{
        System.out.println ("MFVec2fObjectTests...");
        assertTrue  (Arrays.equals(MFVec2fObject.DEFAULT_VALUE, new MFVec2fObject().setValueByString(MFVec2fObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFVec2fObject testMFVec2fObject = new MFVec2fObject(); // static initializer is tested, might throw exception
        assertTrue  (testMFVec2fObject.matches(),       "testMFVec2fObject.matches() tests object initialization correctly matches regex");
        float[] emptyArray       = {};
        float[] singleValueArray = { 0.0f, 0.0f };
        float[] doubleValueArray = { 0.0f, 1.0f, -2.0f, -3.0f };
        float[] tripleValueArray = { 0.0f, 1.0f, -2.0f, -3.0f,  4.0f,  5.0f };
        assertTrue  (Arrays.equals(emptyArray, MFVec2fObject.DEFAULT_VALUE), "test correct default value is empty array for this MF field object");
        assertTrue  (MFVec2fObject.matches(MFVec2fObject.DEFAULT_VALUE_STRING),
                                                        "MFVec2fObject.matches(MFVec2fObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFVec2fObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFVec2fObject.REGEX.contains("^") && !MFVec2fObject.REGEX.contains("$"), "test MFVec2fObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (MFVec2fObject.REGEX.equals(MFVec2dObject.REGEX), "test MFVec2fObject.REGEX.equals(MFVec2dObject.REGEX) returns true");

        // TODO utility method needed to set singleton object using individual values
//        testMFVec2fObject.setValue(0.0f, 0.0f); // returns void because it matches (overrides) Java SAI specification interface
//        assertTrue  (Arrays.equals(singleValueArray,  testMFVec2fObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f results in singleton array with same value");
        
        testMFVec2fObject.setValue(emptyArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(emptyArray,testMFVec2fObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec2fObject.matches(),       "testMFVec2fObject.matches() tests emptyArray initialization correctly matches regex");
        testMFVec2fObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testMFVec2fObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec2fObject.matches(),       "testMFVec2fObject.matches() tests singleValueArray correctly matches regex");
        testMFVec2fObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(doubleValueArray,testMFVec2fObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec2fObject.matches(),       "testMFVec2fObject.matches() tests doubleValueArray correctly matches regex");
        testMFVec2fObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(tripleValueArray,testMFVec2fObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec2fObject.matches(),       "testMFVec2fObject.matches() tests tripleValueArray correctly matches regex");
        
        assertTrue  (MFVec2fObject.matches( "0 1  2 3"),      "MFVec2fObject.matches( \"0 1  2 3\")    tests correct string value");
        assertTrue  (MFVec2fObject.matches( "0 1, 2 3"),      "MFVec2fObject.matches( \"0 1, 2 33\")   tests correct string value");
        assertTrue  (MFVec2fObject.matches("-1 -2, -3 -4"),   "MFVec2fObject.matches(\"-1 -2, -3 -4\") tests correct string value");
        assertTrue  (MFVec2fObject.matches(" 0  12E45 "),"MFVec2fObject.matches(\"0  12E45 \") tests correct string value, scientific notation");
        assertTrue  (MFVec2fObject.matches("+0 +12E+45"),"MFVec2fObject.matches(\"+0,+12E+45\") tests correct string value, scientific notation");
        assertTrue  (MFVec2fObject.matches("-0 -12E-45"),"MFVec2fObject.matches(\"-0,-12E-45\") tests correct string value, scientific notation");

        assertTrue  (MFVec2fObject.matches(""),            "MFVec2fObject.matches(\"\") tests correct empty string value");
        assertFalse (MFVec2fObject.matches(","),           "MFVec2fObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (MFVec2fObject.matches(" 0.0  1.0 "),  "MFVec2fObject.matches(\" 0.0  1.0 \") tests correct string value");
        assertTrue  (MFVec2fObject.matches("-1.0 -3.0"),   "MFVec2fObject.matches(\"-1.0 -3.0\") tests correct string value, including external whitespace");
        assertTrue  (MFVec2fObject.matches("0 1"),         "MFVec2fObject.matches(\"0 1\")  tests correct array as string value");
        assertTrue  (MFVec2fObject.matches("0.0 1.0"),     "MFVec2fObject.matches(\"0.0 1.0\")  tests correct array as string value");
        assertTrue  (MFVec2fObject.matches(" 0.0  12.3E45  "),    "MFVec2fObject.matches(\" 0.0  12.3E45  \") tests correct string value, scientific notation");
        assertTrue  (MFVec2fObject.matches("+0.0 +12.3E+45 "),    "MFVec2fObject.matches(\"+0.0 +12.3E+45 \") tests correct string value, scientific notation");
        assertTrue  (MFVec2fObject.matches("-0.0 -12.3E-45 "),    "MFVec2fObject.matches(\"-0.0 -12.3E-45 \") tests correct string value, scientific notation");
        assertTrue  (MFVec2fObject.matches( ".6 -.6 .6e7 -.6e-7"),"MFVec2fObject.matches(\".6 -.6 .6e7 -.6e-7\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertTrue  (MFVec2fObject.matches("0 0, 1 1, "),         "MFVec2fObject.matches(\"0 0, 1 1, \")  tests correct array as string value, including whitespace and commas, with allowed trailing comma");
        
        assertFalse (MFVec2fObject.matches("true false"),"MFVec2fObject.matches(\"true false\") tests incorrect boolean string value");
        assertFalse (MFVec2fObject.matches("blah"),      "MFVec2fObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFVec2fObject.matches("NaN NaN"),   "MFVec2fObject.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFVec2dObject single-field 2-tuple double-precision floating-point array")
    void SFVec2dObjectTests()
	{
        System.out.println ("SFVec2dObjectTests...");
        assertTrue  (Arrays.equals(SFVec2dObject.DEFAULT_VALUE, new SFVec2dObject().setValueByString(SFVec2dObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFVec2dObject testSFVec2dObject = new SFVec2dObject(); // static initializer is tested, might throw exception
        assertTrue  (testSFVec2dObject.matches(),       "testSFVec2dObject.matches() tests object initialization correctly matches regex");
        double[] emptyArray       = {};
        double[] singleValueArray = { 0.0d, 0.0f };
//      double[] doubleValueArray = { 0.0d, 1.0d, -2.0d, -3.0d };
//      double[] tripleValueArray = { 0.0d, 1.0d, -2.0d, -3.0d,  4.0d,  5.0d };
        assertTrue  (Arrays.equals(singleValueArray, SFVec2dObject.DEFAULT_VALUE), "test correct default value for this field object");
        assertTrue  (SFVec2dObject.matches(SFVec2dObject.DEFAULT_VALUE_STRING),
                                                        "SFVec2dObject.matches(SFVec2dObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFVec2dObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFVec2dObject.REGEX.contains("^") && !SFVec2dObject.REGEX.contains("$"), "test SFVec2dObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (SFVec2fObject.REGEX.equals(SFVec2dObject.REGEX), "test SFVec2fObject.REGEX.equals(SFVec2dObject.REGEX) returns true");

        testSFVec2dObject.setValue(0.0f, 0.0f); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(singleValueArray,  testSFVec2dObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f results in singleton array with same value");
        
        testSFVec2dObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testSFVec2dObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
      
        assertTrue  (SFVec2dObject.matches( "0 1"),      "SFVec2dObject.matches( \"0 1\")   tests correct string value");
        assertTrue  (SFVec2dObject.matches( "2 3"),      "SFVec2dObject.matches( \"2 3\")   tests correct string value");
        assertTrue  (SFVec2dObject.matches("-1 -2"),     "SFVec2dObject.matches(\"-1 -2\")  tests correct string value");
        assertTrue  (SFVec2dObject.matches(" 0  12E45 "),"SFVec2dObject.matches(\"0  12E45 \") tests correct string value, scientific notation");
        assertTrue  (SFVec2dObject.matches("+0 +12E+45"),"SFVec2dObject.matches(\"+0,+12E+45\") tests correct string value, scientific notation");
        assertTrue  (SFVec2dObject.matches("-0 -12E-45"),"SFVec2dObject.matches(\"-0,-12E-45\") tests correct string value, scientific notation");

        assertTrue  (SFVec2dObject.matches(" 0.0  1.0 "),  "SFVec2dObject.matches(\" 0.0  1.0 \") tests correct string value");
        assertTrue  (SFVec2dObject.matches("-1.0 -3.0"),   "SFVec2dObject.matches(\"-1.0 -3.0\") tests correct string value, including external whitespace");
        assertTrue  (SFVec2dObject.matches("0 1"),         "SFVec2dObject.matches(\"0 1\")  tests correct array as string value");
        assertTrue  (SFVec2dObject.matches("0.0 1.0"),     "SFVec2dObject.matches(\"0.0 1.0\")  tests correct array as string value");
        assertTrue  (SFVec2dObject.matches(" 0.0  12.3E45  "),    "SFVec2dObject.matches(\" 0.0  12.3E45  \") tests correct string value, scientific notation");
        assertTrue  (SFVec2dObject.matches("+0.0 +12.3E+45 "),    "SFVec2dObject.matches(\"+0.0 +12.3E+45 \") tests correct string value, scientific notation");
        assertTrue  (SFVec2dObject.matches("-0.0 -12.3E-45 "),    "SFVec2dObject.matches(\"-0.0 -12.3E-45 \") tests correct string value, scientific notation");
        
        assertFalse (SFVec2dObject.matches(""),          "SFVec2dObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFVec2dObject.matches(","),         "SFVec2dObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (SFVec2dObject.matches( ".6 -.6 .6e7 -.6e-7"),"SFVec2dObject.matches(\".6 -.6 .6e7 -.6e-7\") tests incorrect string value, no leading digit before decimal point, scientific notation");
        assertFalse (SFVec2dObject.matches("true false"),"SFVec2dObject.matches(\"true false\") tests incorrect boolean string value");
        assertFalse (SFVec2dObject.matches("blah"),      "SFVec2dObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFVec2dObject.matches("NaN NaN"),   "SFVec2dObject.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test MFVec2dObject multi-field 2-tuple double-precision floating-point array")
    void MFVec2dObjectTests()
	{
        System.out.println ("MFVec2dObjectTests...");
        assertTrue  (Arrays.equals(MFVec2dObject.DEFAULT_VALUE, new MFVec2dObject().setValueByString(MFVec2dObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFVec2dObject testMFVec2dObject = new MFVec2dObject(); // static initializer is tested, might throw exception
        assertTrue  (testMFVec2dObject.matches(),       "testMFVec2dObject.matches() tests object initialization correctly matches regex");
        double[] emptyArray       = {};
        double[] singleValueArray = { 0.0d, 0.0d };
        double[] doubleValueArray = { 0.0d, 1.0d, -2.0d, -3.0d };
        double[] tripleValueArray = { 0.0d, 1.0d, -2.0d, -3.0d,  4.0d,  5.0d };
        assertTrue  (Arrays.equals(emptyArray, MFVec2dObject.DEFAULT_VALUE), "test correct default value is empty array for this MF field object");
        assertTrue  (MFVec2dObject.matches(MFVec2dObject.DEFAULT_VALUE_STRING),
                                                        "MFVec2dObject.matches(MFVec2dObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFVec2dObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFVec2dObject.REGEX.contains("^") && !MFVec2dObject.REGEX.contains("$"), "test MFVec2dObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (MFVec2fObject.REGEX.equals(MFVec2dObject.REGEX), "test MFVec2fObject.REGEX.equals(MFVec2dObject.REGEX) returns true");

        // TODO utility method needed to set singleton object using individual values
//        testMFVec2dObject.setValue(0.0d, 0.0d); // returns void because it matches (overrides) Java SAI specification interface
//        assertTrue  (Arrays.equals(singleValueArray,  testMFVec2dObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f results in singleton array with same value");
        
        testMFVec2dObject.setValue(emptyArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(emptyArray,testMFVec2dObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec2dObject.matches(),       "testMFVec2dObject.matches() tests emptyArray initialization correctly matches regex");
        testMFVec2dObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testMFVec2dObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec2dObject.matches(),       "testMFVec2dObject.matches() tests singleValueArray correctly matches regex");
        testMFVec2dObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(doubleValueArray,testMFVec2dObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec2dObject.matches(),       "testMFVec2dObject.matches() tests doubleValueArray correctly matches regex");
        testMFVec2dObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(tripleValueArray,testMFVec2dObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec2dObject.matches(),       "testMFVec2dObject.matches() tests tripleValueArray correctly matches regex");
        
        assertTrue  (MFVec2dObject.matches( "0 1  2 3"),      "MFVec2dObject.matches( \"0 1  2 3\")    tests correct string value");
        assertTrue  (MFVec2dObject.matches( "0 1, 2 3"),      "MFVec2dObject.matches( \"0 1, 2 33\")   tests correct string value");
        assertTrue  (MFVec2dObject.matches("-1 -2, -3 -4"),   "MFVec2dObject.matches(\"-1 -2, -3 -4\") tests correct string value");
        assertTrue  (MFVec2dObject.matches(" 0  12E45 "),"MFVec2dObject.matches(\"0  12E45 \") tests correct string value, scientific notation");
        assertTrue  (MFVec2dObject.matches("+0 +12E+45"),"MFVec2dObject.matches(\"+0,+12E+45\") tests correct string value, scientific notation");
        assertTrue  (MFVec2dObject.matches("-0 -12E-45"),"MFVec2dObject.matches(\"-0,-12E-45\") tests correct string value, scientific notation");

        assertTrue  (MFVec2dObject.matches(""),            "MFVec2dObject.matches(\"\") tests correct empty string value");
        assertFalse (MFVec2dObject.matches(","),           "MFVec2dObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (MFVec2dObject.matches(" 0.0  1.0 "),  "MFVec2dObject.matches(\" 0.0  1.0 \") tests correct string value");
        assertTrue  (MFVec2dObject.matches("-1.0 -3.0"),   "MFVec2dObject.matches(\"-1.0 -3.0\") tests correct string value, including external whitespace");
        assertTrue  (MFVec2dObject.matches("0 1"),         "MFVec2dObject.matches(\"0 1\")  tests correct array as string value");
        assertTrue  (MFVec2dObject.matches("0.0 1.0"),     "MFVec2dObject.matches(\"0.0 1.0\")  tests correct array as string value");
        assertTrue  (MFVec2dObject.matches(" 0.0  12.3E45  "),    "MFVec2dObject.matches(\" 0.0  12.3E45  \") tests correct string value, scientific notation");
        assertTrue  (MFVec2dObject.matches("+0.0 +12.3E+45 "),    "MFVec2dObject.matches(\"+0.0 +12.3E+45 \") tests correct string value, scientific notation");
        assertTrue  (MFVec2dObject.matches("-0.0 -12.3E-45 "),    "MFVec2dObject.matches(\"-0.0 -12.3E-45 \") tests correct string value, scientific notation");
        assertTrue  (MFVec2dObject.matches( ".6 -.6 .6e7 -.6e-7"),"MFVec2dObject.matches(\".6 -.6 .6e7 -.6e-7\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertTrue  (MFVec2dObject.matches("0 0, 1 1, "),         "MFVec2dObject.matches(\"0 0, 1 1, \")  tests correct array as string value, including whitespace and commas, with allowed trailing comma");
        
        assertFalse (MFVec2dObject.matches("true false"),"MFVec2dObject.matches(\"true false\") tests incorrect boolean string value");
        assertFalse (MFVec2dObject.matches("blah"),      "MFVec2dObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFVec2dObject.matches("NaN NaN"),   "MFVec2dObject.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFVec3fObject single-field 3-tuple single-precision floating-point array")
    void SFVec3fObjectTests()
	{
        System.out.println ("SFVec3fObjectTests...");
        assertTrue  (Arrays.equals(SFVec3fObject.DEFAULT_VALUE, new SFVec3fObject().setValueByString(SFVec3fObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFVec3fObject testSFVec3fObject = new SFVec3fObject(); // static initializer is tested, might throw exception
        assertTrue  (testSFVec3fObject.matches(),       "testSFVec3fObject.matches() tests object initialization correctly matches regex");
        float[] emptyArray       = {};
        float[] singleValueArray = { 0.0f, 0.0f, 0.0f };
//      float[] doubleValueArray = { 0.0f, 1.0f, -2.0f, -3.0f, -4.0f, -5.0f };
//      float[] tripleValueArray = { 0.0f, 1.0f, -2.0f, -3.0f,  4.0f,  5.0f, -6.0f,  7.0f,  8.0f };
        assertTrue  (Arrays.equals(singleValueArray, SFVec3fObject.DEFAULT_VALUE), "test correct default value for this field object");
        assertTrue  (SFVec3fObject.matches(SFVec3fObject.DEFAULT_VALUE_STRING),
                                                        "SFVec3fObject.matches(SFVec3fObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFVec3fObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFVec3fObject.REGEX.contains("^") && !SFVec3fObject.REGEX.contains("$"), "test SFVec3fObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (SFVec3fObject.REGEX.equals(SFVec3dObject.REGEX), "test SFVec3fObject.REGEX.equals(SFVec3dObject.REGEX) returns true");

        testSFVec3fObject.setValue(0.0f, 0.0f, 0.0f); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(singleValueArray,  testSFVec3fObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f results in singleton array with same value");
        
        testSFVec3fObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testSFVec3fObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
      
        assertTrue  (SFVec3fObject.matches( "0 1 2"),      "SFVec3fObject.matches( \"0 1 2\")   tests correct string value");
        assertTrue  (SFVec3fObject.matches( "2 3 4"),      "SFVec3fObject.matches( \"2 3 4\")   tests correct string value");
        assertTrue  (SFVec3fObject.matches("-1 -2 -3"),     "SFVec3fObject.matches(\"-1 -2 -3\")  tests correct string value");
        assertTrue  (SFVec3fObject.matches(" 0  12E45  0"),"SFVec3fObject.matches(\" 0  12E45  0\") tests correct string value, scientific notation");
        assertTrue  (SFVec3fObject.matches("+0 +12E+45 0"),"SFVec3fObject.matches(\"+0 +12E+45 0\") tests correct string value, scientific notation");
        assertTrue  (SFVec3fObject.matches("-0 -12E-45 0"),"SFVec3fObject.matches(\"-0 -12E-45 0\") tests correct string value, scientific notation");

        assertTrue  (SFVec3fObject.matches(" 0.0  1.0 2.0 "),  "SFVec3fObject.matches(\" 0.0  1.0 2.0 \") tests correct string value");
        assertTrue  (SFVec3fObject.matches("-1.0 -2.0 -3.0"),   "SFVec3fObject.matches(\"-1.0 -2.0 -3.0\") tests correct string value, including external whitespace");
        assertTrue  (SFVec3fObject.matches("0 1 2"),         "SFVec3fObject.matches(\"0 1 2\")  tests correct array as string value");
        assertTrue  (SFVec3fObject.matches("0.0 1.0 2.0"),     "SFVec3fObject.matches(\"0.0 1.0 2.0\")  tests correct array as string value");
        assertTrue  (SFVec3fObject.matches(" 0.0  12.3E45  0"),    "SFVec3fObject.matches(\" 0.0  12.3E45  0\") tests correct string value, scientific notation");
        assertTrue  (SFVec3fObject.matches("+0.0 +12.3E+45 0"),    "SFVec3fObject.matches(\"+0.0 +12.3E+45 0\") tests correct string value, scientific notation");
        assertTrue  (SFVec3fObject.matches("-0.0 -12.3E-45 0"),    "SFVec3fObject.matches(\"-0.0 -12.3E-45 0\") tests correct string value, scientific notation");
        
        assertFalse (SFVec3fObject.matches(""),                   "SFVec3fObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFVec3fObject.matches(","),                  "SFVec3fObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (SFVec3fObject.matches( ".6 -.6 .6e7 -.6e-7"),"SFVec3fObject.matches(\".6 -.6 .6e7 -.6e-7\") tests incorrect string value, no leading digit before decimal point, scientific notation");
        assertFalse (SFVec3fObject.matches("true false false"),   "SFVec3fObject.matches(\"true false false\") tests incorrect boolean string value");
        assertFalse (SFVec3fObject.matches("blah"),               "SFVec3fObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFVec3fObject.matches("NaN NaN NaN"),        "SFVec3fObject.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFVec3fObject bboxSizeType checks on single-field 3-tuple single-precision floating-point array")
    void SFVec3fBboxSizeObjectTests()
	{
        System.out.println ("SFVec3fBboxSizeObjectTests for bounding box (bbox) constraints...");
        float[] defaultBboxSizeArray = { -1.0f, -1.0f, -1.0f };
        assertTrue  (Arrays.equals(SFVec3fObject.DEFAULT_VALUE_BBOXSIZETYPE, new SFVec3fObject().setValueByString(SFVec3fObject.DEFAULT_VALUE_BBOXSIZETYPE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE_BBOXSIZETYPE matches DEFAULT_VALUE_BBOXSIZETYPE_STRING for this field object");
        SFVec3fObject testSFVec3fBboxSizeObject = new SFVec3fObject(SFVec3fObject.DEFAULT_VALUE_BBOXSIZETYPE); // static initializer is tested, might throw exception
        assertTrue  (testSFVec3fBboxSizeObject.matches(),       "testSFVec3fBboxSizeObject.matches() tests object initialization correctly matches regex");
        assertTrue  (Arrays.equals(defaultBboxSizeArray, SFVec3fObject.DEFAULT_VALUE_BBOXSIZETYPE), "test correct default value for this field object");
        assertTrue  (SFVec3fObject.matches(SFVec3fObject.DEFAULT_VALUE_BBOXSIZETYPE_STRING),
                                                        "SFVec3fObject.matches(SFVec3fObject.DEFAULT_VALUE_BBOXSIZETYPE_STRING) tests object initialization correctly matches regex");
        assertFalse (testSFVec3fBboxSizeObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFVec3fObject.REGEX_BBOXSIZETYPE.contains("^") && !SFVec3fObject.REGEX_BBOXSIZETYPE.contains("$"), "test SFVec3fObject.REGEX does not contain anchor characters ^ or $");
        // avoid unexpected equivalent regexes
        assertFalse (SFVec3fObject.REGEX.equals(SFVec3fObject.REGEX_BBOXSIZETYPE), "test SFVec3fObject.REGEX.equals(SFVec3fObject.REGEX_BBOXSIZETYPE) returns false");

        testSFVec3fBboxSizeObject.setValue(-1.0f, -1.0f, -1.0f);
        assertTrue  (Arrays.equals(defaultBboxSizeArray,  testSFVec3fBboxSizeObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f 0.0f results in singleton array with same value");
        
        testSFVec3fBboxSizeObject.setValue(defaultBboxSizeArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(defaultBboxSizeArray,testSFVec3fBboxSizeObject.getPrimitiveValue(),   "tests setting object value to default-value array results in equivalent getPrimitiveValue()");

        assertFalse  (SFVec3fObject.matches            (""), "tests empty string \"\" fails SFVec3fObject.matches(value), illegal value");
        assertTrue   (SFVec3fObject.matchesBboxSizeType(""), "tests empty string \"\" passes SFVec3fObject.matchesBboxSizeType(value), legal value");
        
        assertFalse  (testSFVec3fBboxSizeObject.setValue        ( -2.0f, -2.0f, -2.0f ).matchesBboxSizeType(), "tests setting object value to -2.0f -2.0f -2.0f fails");
        assertFalse  (testSFVec3fBboxSizeObject.setValueByString("-2.0   -2.0   -2.0" ).matchesBboxSizeType(), "tests setting object value to \"-2.0   -2.0   -2.0\" fails");
        assertFalse  (testSFVec3fBboxSizeObject.setValue        ( -2.0f, -2.0f, -2.0f ).matchesBboxSizeType(), "tests setting object value to -2.0f -2.0f -2.0f fails");
        assertTrue   (SFVec3fObject.matches            ("-2.0 -2.0 -2.0"), "tests \"-2.0 -2.0 -2.0\" passes SFVec3fObject.matches(value)");
        assertFalse  (SFVec3fObject.matchesBboxSizeType("-2.0 -2.0 -2.0"), "tests \"-2.0 -2.0 -2.0\" fails  SFVec3fObject.matchesBboxSizeType(value)");
        assertTrue   (SFVec3fObject.matches            (" 2.0  2.0  2.0"), "tests \" 2.0  2.0  2.0\" passes SFVec3fObject.matches(value)");
        assertTrue   (SFVec3fObject.matchesBboxSizeType(" 2.0  2.0  2.0"), "tests \" 2.0  2.0  2.0\" passes SFVec3fObject.matchesBboxSizeType(value)");
        assertTrue   (SFVec3fObject.matchesBboxSizeType(" -1e+00 -1e00 -1e-00 "), "tests \" -1e+00 -1e00 -1e-00 \" passes SFVec3fObject.matchesBboxSizeType(value) for sentinel with scientific notation");
        assertTrue   (SFVec3fObject.matchesBboxSizeType(" 0.0  0.0  0.0"), "tests \" 0.0  0.0  0.0\" passes SFVec3fObject.matchesBboxSizeType(value)");
        assertFalse  (SFVec3fObject.matchesBboxSizeType(" 0.0  0.0  0.0  0.0"), "tests \" 0.0  0.0  0.0  0.0\" fails SFVec3fObject.matchesBboxSizeType(value), too many values");
        assertFalse  (SFVec3fObject.matchesBboxSizeType(" 0.0  0.0"),           "tests \" 0.0  0.0\" fails SFVec3fObject.matchesBboxSizeType(value), insufficient values");
    }

    @Test
    @DisplayName("Test MFVec3fObject multi-field 3-tuple single-precision floating-point array")
    void MFVec3fObjectTests()
	{
        System.out.println ("MFVec3fObjectTests...");
        assertTrue  (Arrays.equals(MFVec3fObject.DEFAULT_VALUE, new MFVec3fObject().setValueByString(MFVec3fObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFVec3fObject testMFVec3fObject = new MFVec3fObject(); // static initializer is tested, might throw exception
        assertTrue  (testMFVec3fObject.matches(),       "testMFVec3fObject.matches() tests object initialization correctly matches regex");
        float[] emptyArray       = {};
        float[] singleValueArray = { 0.0f, 0.0f, 0.0f };
        float[] doubleValueArray = { 0.0f, 1.0f, -2.0f, -3.0f, -4.0f, -5.0f };
        float[] tripleValueArray = { 0.0f, 1.0f, -2.0f, -3.0f,  4.0f,  5.0f, 6.0f, 7.0f, 8.0f };
        assertTrue  (Arrays.equals(emptyArray, MFVec3fObject.DEFAULT_VALUE), "test correct default value is empty array for this MF field object");
        assertTrue  (MFVec3fObject.matches(MFVec3fObject.DEFAULT_VALUE_STRING),
                                                        "MFVec3fObject.matches(MFVec3fObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFVec3fObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFVec3fObject.REGEX.contains("^") && !MFVec3fObject.REGEX.contains("$"), "test MFVec3fObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (MFVec3fObject.REGEX.equals(MFVec3dObject.REGEX), "test MFVec3fObject.REGEX.equals(MFVec3dObject.REGEX) returns true");

        // TODO utility method needed to set singleton object using individual values
//        testMFVec3fObject.setValue(0.0f, 0.0f, 0.0f); // returns void because it matches (overrides) Java SAI specification interface
//        assertTrue  (Arrays.equals(singleValueArray,  testMFVec3fObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f results in singleton array with same value");
        
        testMFVec3fObject.setValue(emptyArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(emptyArray,testMFVec3fObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec3fObject.matches(),       "testMFVec3fObject.matches() tests emptyArray initialization correctly matches regex");
        testMFVec3fObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testMFVec3fObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec3fObject.matches(),       "testMFVec3fObject.matches() tests singleValueArray correctly matches regex");
        testMFVec3fObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(doubleValueArray,testMFVec3fObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec3fObject.matches(),       "testMFVec3fObject.matches() tests doubleValueArray correctly matches regex");
        testMFVec3fObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(tripleValueArray,testMFVec3fObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec3fObject.matches(),       "testMFVec3fObject.matches() tests tripleValueArray correctly matches regex");
        
        assertTrue  (MFVec3fObject.matches(" 0  1  2   3  4  5"),   "MFVec3fObject.matches( \" 0  1  2   3  4  5\")   tests correct string value");
        assertTrue  (MFVec3fObject.matches(" 0  1  2,  3  4  5"),   "MFVec3fObject.matches( \" 0  1  2,  3  4  5\")   tests correct string value");
        assertTrue  (MFVec3fObject.matches("-0 -1 -2, -3 -4 -5"),   "MFVec3fObject.matches(\"-0 -1 -2, -3 -4 -5\") tests correct string value");
        assertTrue  (MFVec3fObject.matches(" 0  12E45  0"),"MFVec3fObject.matches(\"0   12E45  0\") tests correct string value, scientific notation");
        assertTrue  (MFVec3fObject.matches("+0 +12E+45 0"),"MFVec3fObject.matches(\"+0 +12E+45 0\") tests correct string value, scientific notation");
        assertTrue  (MFVec3fObject.matches("-0 -12E-45 0"),"MFVec3fObject.matches(\"-0 -12E-45 0\") tests correct string value, scientific notation");

        assertTrue  (MFVec3fObject.matches(""),             "MFVec3fObject.matches(\"\") tests correct empty string value");
        assertFalse (MFVec3fObject.matches(","),            "MFVec3fObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (MFVec3fObject.matches(" 0.0  1.0 2.0"),"MFVec3fObject.matches(\" 0.0  1.0 2.0\") tests correct string value");
        assertTrue  (MFVec3fObject.matches("-1.0 -3.0 2.0"),"MFVec3fObject.matches(\"-1.0 -3.0 2.0\") tests correct string value, including external whitespace");
        assertTrue  (MFVec3fObject.matches("0 1 2"),        "MFVec3fObject.matches(\"0 1 2\")  tests correct array as string value");
        assertTrue  (MFVec3fObject.matches("0.0 1.0 2.0"),  "MFVec3fObject.matches(\"0.0 1.0 2.0\")  tests correct array as string value");
        assertTrue  (MFVec3fObject.matches(" 0.0  12.3E45  0"),       "MFVec3fObject.matches(\" 0.0  12.3E45  0\") tests correct string value, scientific notation");
        assertTrue  (MFVec3fObject.matches("+0.0 +12.3E+45 0"),       "MFVec3fObject.matches(\"+0.0 +12.3E+45 0\") tests correct string value, scientific notation");
        assertTrue  (MFVec3fObject.matches("-0.0 -12.3E-45 0"),       "MFVec3fObject.matches(\"-0.0 -12.3E-45 0\") tests correct string value, scientific notation");
        assertTrue  (MFVec3fObject.matches( ".6 -.6 0 .6e7 -.6e-7 0"),"MFVec3fObject.matches(\".6 -.6 0 .6e7 -.6e-7 0\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertTrue  (MFVec3fObject.matches("0 0 0, 1 1 1, "),         "MFVec3fObject.matches(\"0 0 0, 1 1 1, \")  tests correct array as string value, including whitespace and commas, with allowed trailing comma");

        assertFalse (MFVec3fObject.matches("true false"),"MFVec3fObject.matches(\"true false\") tests incorrect boolean string value");
        assertFalse (MFVec3fObject.matches("blah"),      "MFVec3fObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFVec3fObject.matches("NaN NaN"),   "MFVec3fObject.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFColorObject single-field 3-tuple single-precision floating-point array")
    void SFColorObjectTests()
	{
        System.out.println ("SFColorObjectTests...");
        assertTrue  (Arrays.equals(SFColorObject.DEFAULT_VALUE, new SFColorObject().setValueByString(SFColorObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFColorObject testSFColorObject = new SFColorObject(); // static initializer is tested, might throw exception
        assertTrue  (testSFColorObject.matches(),       "testSFColorObject.matches() tests object initialization correctly matches regex");
        float[] emptyArray       = {};
        float[] singleValueArray = { 0.0f, 0.0f, 0.0f };
//      float[] doubleValueArray = { 1.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f  };
//      float[] tripleValueArray = { 1.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 1.0f };
        assertTrue  (Arrays.equals(singleValueArray, SFColorObject.DEFAULT_VALUE), "test correct default value for this field object");
        assertTrue  (SFColorObject.matches(SFColorObject.DEFAULT_VALUE_STRING),
                                                        "SFColorObject.matches(SFColorObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFColorObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFColorObject.REGEX.contains("^") && !SFColorObject.REGEX.contains("$"), "test SFColorObject.REGEX does not contain anchor characters ^ or $");

        testSFColorObject.setValue(0.0f, 0.0f, 0.0f); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(singleValueArray,  testSFColorObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f results in singleton array with same value");
        
        testSFColorObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testSFColorObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
      
        assertTrue  (SFColorObject.matches( "1 0 0"),        "SFColorObject.matches( \"1 0 0\")   tests correct string value");
        assertTrue  (SFColorObject.matches( "0.2 0.3 0.4"),  "SFColorObject.matches( \"0.2 0.3 0.4\")   tests correct string value");
        assertTrue  (SFColorObject.matches("0 0 0"),         "SFColorObject.matches(\"-0 0 0\")  tests correct string value");
        assertTrue  (SFColorObject.matches("1 1 1"),         "SFColorObject.matches(\"1 1 1\")  tests correct string value");
        assertTrue  (SFColorObject.matches(" 0  0.12E45  0"),"SFColorObject.matches(\" 0  0.12E45  0\") tests correct string value, scientific notation");
        assertTrue  (SFColorObject.matches("+0 +0.12E+45 0"),"SFColorObject.matches(\"+0 +0.12E+45 0\") tests correct string value, scientific notation");
        assertTrue  (SFColorObject.matches(" 0  0.12E-45 0"),"SFColorObject.matches(\" 0  0.12E-45 0\") tests correct string value, scientific notation");

        assertTrue  (SFColorObject.matches(" 0.0  1.0 0.0 "),  "SFColorObject.matches(\" 0.0  1.0 0.0 \") tests correct string value");
        
        assertFalse (SFColorObject.matches(""),                   "SFColorObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFColorObject.matches(","),                  "SFColorObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (SFColorObject.matches("0 1 2"),              "SFColorObject.matches(\"0 1 2\")  tests incorrect array as string value");
        assertFalse (SFColorObject.matches("0.0 1.0 2.0"),        "SFColorObject.matches(\"0.0 1.0 2.0\")  tests incorrect array as string value");
        assertFalse (SFColorObject.matches("-1.0 -2.0 -3.0"),     "SFColorObject.matches(\"-1.0 -2.0 -3.0\") tests incorrect string value, including external whitespace");
        assertFalse (SFColorObject.matches( ".6 -.6 .6e7 -.6e-7"),"SFColorObject.matches(\".6 -.6 .6e7 -.6e-7\") tests incorrect string value, no leading digit before decimal point, scientific notation");
        assertFalse (SFColorObject.matches("true false false"),   "SFColorObject.matches(\"true false false\") tests incorrect boolean string value");
        assertFalse (SFColorObject.matches("blah"),               "SFColorObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFColorObject.matches("NaN NaN NaN"),        "SFColorObject.matches(\"NaN\") tests Not A Number (NaN) which is not an allowed string value");
    
        assertEquals(SFColorObject.toStringHTML(0.0f, 0.5f, 1.0f),"#007fff",           "SFColorObject.toStringHTML(0.0f, 0.5f, 1.0f) tests HTML export as \"#007fff\"");
        assertEquals(SFColorObject.toStringCSS (0.0f, 0.5f, 1.0f),"color:(0,127,255)", "SFColorObject.toStringCSS (0.0f, 0.5f, 1.0f) tests CSS export as \"color:(0,127,255)\"");
    }
    
    @Test
    @DisplayName("Test MFColorObject multi-field 3-tuple single-precision floating-point array")
    void MFColorObjectTests()
	{
        System.out.println ("MFColorObjectTests...");
        assertTrue  (Arrays.equals(MFColorObject.DEFAULT_VALUE, new MFColorObject().setValueByString(MFColorObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFColorObject testMFColorObject = new MFColorObject(); // static initializer is tested, might throw exception
        assertTrue  (testMFColorObject.matches(),       "testMFColorObject.matches() tests object initialization correctly matches regex");
        float[] emptyArray       = {};
        float[] singleValueArray = { 0.8f, 0.8f, 0.8f };
        float[] doubleValueArray = { 1.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f  };
        float[] tripleValueArray = { 1.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 1.0f };
        assertTrue  (Arrays.equals(emptyArray, MFColorObject.DEFAULT_VALUE), "test correct default value is empty array for this MF field object");
        assertTrue  (MFColorObject.matches(MFColorObject.DEFAULT_VALUE_STRING),
                                                        "MFColorObject.matches(MFColorObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFColorObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFColorObject.REGEX.contains("^") && !MFColorObject.REGEX.contains("$"), "test MFColorObject.REGEX does not contain anchor characters ^ or $");

        // TODO utility method needed to set singleton object using individual values
//        testMFColorObject.setValue(0.0f, 0.0f, 0.0f); // returns void because it matches (overrides) Java SAI specification interface
//        assertTrue  (Arrays.equals(singleValueArray,  testMFColorObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f results in singleton array with same value");
        
        testMFColorObject.setValue(emptyArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(emptyArray,testMFColorObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFColorObject.matches(),       "testMFColorObject.matches() tests emptyArray initialization correctly matches regex");
        testMFColorObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testMFColorObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFColorObject.matches(),       "testMFColorObject.matches() tests singleValueArray correctly matches regex");
        testMFColorObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(doubleValueArray,testMFColorObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFColorObject.matches(),       "testMFColorObject.matches() tests doubleValueArray correctly matches regex");
        testMFColorObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(tripleValueArray,testMFColorObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFColorObject.matches(),       "testMFColorObject.matches() tests tripleValueArray correctly matches regex");
        
        assertTrue  (MFColorObject.matches(" 0  1  .2  .3  .4 .5"),   "MFColorObject.matches( \" 0  1  .2  .3  .4 .5\")   tests correct string value");
        assertTrue  (MFColorObject.matches(" 0  1  .2, .3  .4 .5"),   "MFColorObject.matches( \" 0  1  .2, .3  .4 .5\")   tests correct string value");
        
        // TODO fix gnarly problem in regex, the following value should not match
//        assertFalse (MFColorObject.matches(" 0  1, .2  .3, .4 .5"),   "MFColorObject.matches( \" 0  1, .2  .3, .4 .5\")   tests incorrect comma placement in string value");
        
        assertTrue  (MFColorObject.matches(" 0  0.12E45  0"),"MFColorObject.matches(\"0   0.12E45  0\") tests correct string value, scientific notation");
        assertTrue  (MFColorObject.matches("+0 +0.12E+45 0"),"MFColorObject.matches(\"+0 +0.12E+45 0\") tests correct string value, scientific notation");
        assertTrue  (MFColorObject.matches(" 0  0.12E-45 0"),"MFColorObject.matches(\" 0  0.12E-45 0\") tests correct string value, scientific notation");

        assertTrue  (MFColorObject.matches(""),             "MFColorObject.matches(\"\") tests correct empty string value");
        assertFalse (MFColorObject.matches(","),            "MFColorObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (MFColorObject.matches(" 0.0  1.0 2.0"),"MFColorObject.matches(\" 0.0  1.0 2.0\") tests incorrect string value");
        assertFalse (MFColorObject.matches("-1.0 -3.0 2.0"),"MFColorObject.matches(\"-1.0 -3.0 2.0\") tests incorrect string value, including external whitespace");
        assertFalse (MFColorObject.matches("0 1 2"),        "MFColorObject.matches(\"0 1 2\")  tests correct array as string value");
        assertTrue  (MFColorObject.matches( ".6 -.6 0 .6e7 -.6e-7 0"),"MFColorObject.matches(\".6 -.6 0 .6e7 -.6e-7 0\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertTrue  (MFColorObject.matches("0 0 0, 1 1 1, "),         "MFColorObject.matches(\"0 0 0, 1 1 1, \")  tests correct array as string value, including whitespace and commas, with allowed trailing comma");

        assertFalse (MFColorObject.matches("true false false"),          "MFColorObject.matches(\"true false false\") tests incorrect boolean string value");
        assertFalse (MFColorObject.matches("blah"),                      "MFColorObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFColorObject.matches("NaN NaN NaN NaN NaN NaN"),   "MFColorObject.matches(\"NaN NaN NaN NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFColorRGBAObject single-field 4-tuple single-precision floating-point array")
    void SFColorRGBAObjectTests()
	{
        System.out.println ("SFColorRGBAObjectTests...");
        assertTrue  (Arrays.equals(SFColorRGBAObject.DEFAULT_VALUE, new SFColorRGBAObject().setValueByString(SFColorRGBAObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFColorRGBAObject testSFColorRGBAObject = new SFColorRGBAObject(); // static initializer is tested, might throw exception
        assertTrue  (testSFColorRGBAObject.matches(),   "testSFColorRGBAObject.matches() tests object initialization correctly matches regex");
        float[] emptyArray       = {};
        float[] singleValueArray = { 0.0f, 0.0f, 0.0f, 0.0f };
//      float[] doubleValueArray = { 1.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f  };
//      float[] tripleValueArray = { 1.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.5f, 0.6f };
        assertTrue  (Arrays.equals(singleValueArray, SFColorRGBAObject.DEFAULT_VALUE), "test correct default value for this field object");
        assertTrue  (SFColorRGBAObject.matches(SFColorRGBAObject.DEFAULT_VALUE_STRING),
                                                        "SFColorRGBAObject.matches(SFColorRGBAObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFColorRGBAObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFColorRGBAObject.REGEX.contains("^") && !SFColorRGBAObject.REGEX.contains("$"), "test SFColorRGBAObject.REGEX does not contain anchor characters ^ or $");

        testSFColorRGBAObject.setValue(0.0f, 0.0f, 0.0f, 0.0f); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(singleValueArray,  testSFColorRGBAObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f results in singleton array with same value");
        
        testSFColorRGBAObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testSFColorRGBAObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
      
        assertTrue  (SFColorRGBAObject.matches( "1 0 0 0.5"),          "SFColorRGBAObject.matches( \"1 0 0 0.5\")   tests correct string value");
        assertTrue  (SFColorRGBAObject.matches( "0.2 0.3 0.4 0.5"),    "SFColorRGBAObject.matches( \"0.2 0.3 0.4 0.5\")   tests correct string value");
        assertTrue  (SFColorRGBAObject.matches("0 0 0 0.5"),           "SFColorRGBAObject.matches(\"-0 0 0\")  tests correct string value");
        assertTrue  (SFColorRGBAObject.matches("1 1 1 0.5"),           "SFColorRGBAObject.matches(\"1 1 1\")  tests correct string value");
        assertTrue  (SFColorRGBAObject.matches(" 0  0.12E45  0 0.5e0"),"SFColorRGBAObject.matches(\" 0  0.12E45  0 0.5e0\") tests correct string value, scientific notation");
        assertTrue  (SFColorRGBAObject.matches("+0 +0.12E+45 0 0.5e0"),"SFColorRGBAObject.matches(\"+0 +0.12E+45 0\") tests correct string value, scientific notation");
        assertTrue  (SFColorRGBAObject.matches(" 0  0.12E-45 0 0.5e0"),"SFColorRGBAObject.matches(\" 0  0.12E-45 0\") tests correct string value, scientific notation");

        assertTrue  (SFColorRGBAObject.matches(" 0.0  1.0 0.0 0.5"),   "SFColorRGBAObject.matches(\" 0.0  1.0 0.0 0.5\") tests correct string value");
        assertTrue  (SFColorRGBAObject.matches( ".6 -.6 .6e7 -.6e-7"), "SFColorRGBAObject.matches(\".6 -.6 .6e7 -.6e-7\") tests incorrect string value, no leading digit before decimal point, scientific notation");
        
        assertFalse (SFColorRGBAObject.matches(""),                    "SFColorRGBAObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFColorRGBAObject.matches(","),                   "SFColorRGBAObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (SFColorRGBAObject.matches("0 1 2"),               "SFColorRGBAObject.matches(\"0 1 2\")  tests incorrect array as string value");
        assertFalse (SFColorRGBAObject.matches("0.0 1.0 2.0 0.5"),     "SFColorRGBAObject.matches(\"0.0 1.0 2.0 0.5\")  tests incorrect array as string value");
        assertFalse (SFColorRGBAObject.matches("-1.0 -2.0 -3.0 0.5"),  "SFColorRGBAObject.matches(\"-1.0 -2.0 -3.0 0.5\") tests incorrect string value, including external whitespace");
        assertFalse (SFColorRGBAObject.matches("true false false false"),"SFColorRGBAObject.matches(\"true false false false\") tests incorrect boolean string value");
        assertFalse (SFColorRGBAObject.matches("blah"),                "SFColorRGBAObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFColorRGBAObject.matches("NaN NaN NaN NaN"),     "SFColorRGBAObject.matches(\"NaN NaN NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test MFColorRGBAObject multi-field 4-tuple single-precision floating-point array")
    void MFColorRGBAObjectTests()
	{
        System.out.println ("MFColorRGBAObjectTests...");
        assertTrue  (Arrays.equals(MFColorRGBAObject.DEFAULT_VALUE, new MFColorRGBAObject().setValueByString(MFColorRGBAObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFColorRGBAObject testMFColorRGBAObject = new MFColorRGBAObject(); // static initializer is tested, might throw exception
        assertTrue  (testMFColorRGBAObject.matches(),       "testMFColorRGBAObject.matches() tests object initialization correctly matches regex");
        float[] emptyArray       = {};
        float[] singleValueArray = { 0.0f, 0.0f, 0.0f, 0.0f };
        float[] doubleValueArray = { 1.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f  };
        float[] tripleValueArray = { 1.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.5f, 0.6f };
        assertTrue  (Arrays.equals(emptyArray, MFColorRGBAObject.DEFAULT_VALUE), "test correct default value is empty array for this MF field object");
        assertTrue  (MFColorRGBAObject.matches(MFColorRGBAObject.DEFAULT_VALUE_STRING),
                                                        "MFColorRGBAObject.matches(MFColorRGBAObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFColorRGBAObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFColorRGBAObject.REGEX.contains("^") && !MFColorRGBAObject.REGEX.contains("$"), "test MFColorRGBAObject.REGEX does not contain anchor characters ^ or $");

        // TODO utility method needed to set singleton object using individual values
//        testMFColorRGBAObject.setValue(0.0f, 0.0f, 0.0f); // returns void because it matches (overrides) Java SAI specification interface
//        assertTrue  (Arrays.equals(singleValueArray,  testMFColorRGBAObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f results in singleton array with same value");
        
        testMFColorRGBAObject.setValue(emptyArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(emptyArray,testMFColorRGBAObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFColorRGBAObject.matches(),       "testMFColorRGBAObject.matches() tests emptyArray initialization correctly matches regex");
        testMFColorRGBAObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testMFColorRGBAObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFColorRGBAObject.matches(),       "testMFColorRGBAObject.matches() tests singleValueArray correctly matches regex");
        testMFColorRGBAObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(doubleValueArray,testMFColorRGBAObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFColorRGBAObject.matches(),       "testMFColorRGBAObject.matches() tests doubleValueArray correctly matches regex");
        testMFColorRGBAObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(tripleValueArray,testMFColorRGBAObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFColorRGBAObject.matches(),       "testMFColorRGBAObject.matches() tests tripleValueArray correctly matches regex");
        
        assertTrue  (MFColorRGBAObject.matches(" 0  1  .2 0.5  .3  .4 .5 0.5"),   "MFColorRGBAObject.matches( \" 0  1  .2 0.5  .3  .4 .5 0.5\")   tests correct string value");
        assertTrue  (MFColorRGBAObject.matches(" 0  1  .2 0.5, .3  .4 .5 0.5"),   "MFColorRGBAObject.matches( \" 0  1  .2 0.5, .3  .4 .5 0.5\")   tests correct string value");
        
        // TODO fix gnarly problem in regex, the following value should not match
//        assertFalse (MFColorRGBAObject.matches(" 0  1, .2 0.5  .3, .4 .5 0.5"),   "MFColorRGBAObject.matches( \" 0  1, .2 0.5  .3, .4 .5 0.5\")   tests incorrect comma placement in string value");
        
        assertTrue  (MFColorRGBAObject.matches(" 0  0.12E45  0 0.5"),"MFColorRGBAObject.matches(\"0   0.12E45  0 0.5\") tests correct string value, scientific notation");
        assertTrue  (MFColorRGBAObject.matches("+0 +0.12E+45 0 0.5"),"MFColorRGBAObject.matches(\"+0 +0.12E+45 0 0.5\") tests correct string value, scientific notation");
        assertTrue  (MFColorRGBAObject.matches(" 0  0.12E-45 0 0.5"),"MFColorRGBAObject.matches(\" 0  0.12E-45 0 0.5\") tests correct string value, scientific notation");

        assertTrue  (MFColorRGBAObject.matches(""),                  "MFColorRGBAObject.matches(\"\") tests correct empty string value");
        assertTrue  (MFColorRGBAObject.matches("0 0 0 0, 1 1 1 1, "),"MFColorRGBAObject.matches(\"0 0 0 0, 1 1 1 1, \")  tests correct array as string value, including whitespace and commas, with allowed trailing comma");

        assertFalse (MFColorRGBAObject.matches(","),            "MFColorRGBAObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (MFColorRGBAObject.matches(" 0.0  1.0 2.0 0.5"),"MFColorRGBAObject.matches(\" 0.0  1.0 2.0 0.5\") tests incorrect string value");
        assertFalse (MFColorRGBAObject.matches("-1.0 -3.0 2.0 0.5"),"MFColorRGBAObject.matches(\"-1.0 -3.0 2.0 0.5\") tests incorrect string value, including external whitespace");
        assertFalse (MFColorRGBAObject.matches("0 1 2 0.5"),        "MFColorRGBAObject.matches(\"0 1 2 0.5\")  tests incorrect array as string value");
        assertFalse (MFColorRGBAObject.matches( ".6 -.6 0 .6e7 -.6e-7 0"),"MFColorRGBAObject.matches(\".6 -.6 0 .6e7 -.6e-7 0\") tests correct string value, no leading digit before decimal point, scientific notation");
        
        assertFalse (MFColorRGBAObject.matches("true false false false"),            "MFColorRGBAObject.matches(\"true false false false\") tests incorrect boolean string value");
        assertFalse (MFColorRGBAObject.matches("blah"),                              "MFColorRGBAObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFColorRGBAObject.matches("NaN NaN NaN NaN NaN NaN NaN NaN"),   "MFColorRGBAObject.matches(\"NaN NaN NaN NaN NaN NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFVec3dObject single-field 3-tuple double-precision floating-point array")
    void SFVec3dObjectTests()
	{
        System.out.println ("SFVec3dObjectTests...");
        assertTrue  (Arrays.equals(SFVec3dObject.DEFAULT_VALUE, new SFVec3dObject().setValueByString(SFVec3dObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFVec3dObject testSFVec3dObject = new SFVec3dObject(); // static initializer is tested, might throw exception
        assertTrue  (testSFVec3dObject.matches(),       "testSFVec3dObject.matches() tests object initialization correctly matches regex");
        double[] emptyArray       = {};
        double[] singleValueArray = { 0.0d, 0.0d, 0.0d };
//      double[] doubleValueArray = { 0.0d, 1.0d, -2.0d, -3.0d, -4.0d,  5.0d };
//      double[] tripleValueArray = { 0.0d, 1.0d, -2.0d, -3.0d,  4.0d,  5.0d, -6.0d,  7.0d,  8.0d };
        assertTrue  (Arrays.equals(singleValueArray, SFVec3dObject.DEFAULT_VALUE), "test correct default value for this field object");
        assertTrue  (SFVec3dObject.matches(SFVec3dObject.DEFAULT_VALUE_STRING),
                                                        "SFVec3dObject.matches(SFVec3dObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFVec3dObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFVec3dObject.REGEX.contains("^") && !SFVec3dObject.REGEX.contains("$"), "test SFVec3dObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (SFVec3fObject.REGEX.equals(SFVec3dObject.REGEX), "test SFVec3fObject.REGEX.equals(SFVec3dObject.REGEX) returns true");

        testSFVec3dObject.setValue(0.0d, 0.0d, 0.0d); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(singleValueArray,  testSFVec3dObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f results in singleton array with same value");
        
        testSFVec3dObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testSFVec3dObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
      
        assertTrue  (SFVec3dObject.matches( "0 1 2"),      "SFVec3dObject.matches( \"0 1 2\")   tests correct string value");
        assertTrue  (SFVec3dObject.matches( "2 3 4"),      "SFVec3dObject.matches( \"2 3 4\")   tests correct string value");
        assertTrue  (SFVec3dObject.matches("-1 -2 -3"),    "SFVec3dObject.matches(\"-1 -2 -3\")  tests correct string value");
        assertTrue  (SFVec3dObject.matches(" 0  12E45  0"),"SFVec3dObject.matches(\" 0  12E45  0\") tests correct string value, scientific notation");
        assertTrue  (SFVec3dObject.matches("+0 +12E+45 0"),"SFVec3dObject.matches(\"+0 +12E+45 0\") tests correct string value, scientific notation");
        assertTrue  (SFVec3dObject.matches("-0 -12E-45 0"),"SFVec3dObject.matches(\"-0 -12E-45 0\") tests correct string value, scientific notation");

        assertTrue  (SFVec3dObject.matches(" 0.0  1.0 0"), "SFVec3dObject.matches(\" 0.0  1.0 0\") tests correct string value");
        assertTrue  (SFVec3dObject.matches("-1.0 -3.0 0"), "SFVec3dObject.matches(\"-1.0 -3.0 0\") tests correct string value, including external whitespace");
        assertTrue  (SFVec3dObject.matches("0 1 2"),       "SFVec3dObject.matches(\"0 1 2\")  tests correct array as string value");
        assertTrue  (SFVec3dObject.matches("0.0 1.0 0"),   "SFVec3dObject.matches(\"0.0 1.0 0\")  tests correct array as string value");
        assertTrue  (SFVec3dObject.matches(" 0.0  12.3E45  0"),    "SFVec3dObject.matches(\" 0.0  12.3E45  0\") tests correct string value, scientific notation");
        assertTrue  (SFVec3dObject.matches("+0.0 +12.3E+45 0"),    "SFVec3dObject.matches(\"+0.0 +12.3E+45 0\") tests correct string value, scientific notation");
        assertTrue  (SFVec3dObject.matches("-0.0 -12.3E-45 0"),    "SFVec3dObject.matches(\"-0.0 -12.3E-45 0\") tests correct string value, scientific notation");
        
        assertFalse (SFVec3dObject.matches(""),                   "SFVec3dObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFVec3dObject.matches(","),                  "SFVec3dObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (SFVec3dObject.matches( ".6 -.6 .6e7 -.6e-7"),"SFVec3dObject.matches(\".6 -.6 .6e7 -.6e-7\") tests incorrect string value, no leading digit before decimal point, scientific notation");
        assertFalse (SFVec3dObject.matches("true false false"),   "SFVec3dObject.matches(\"true false false\") tests incorrect boolean string value");
        assertFalse (SFVec3dObject.matches("blah"),               "SFVec3dObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFVec3dObject.matches("NaN NaN NaN"),        "SFVec3dObject.matches(\"NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test MFVec3dObject multi-field 3-tuple double-precision floating-point array")
    void MFVec3dObjectTests()
	{
        System.out.println ("MFVec3dObjectTests...");
        assertTrue  (Arrays.equals(MFVec3dObject.DEFAULT_VALUE, new MFVec3dObject().setValueByString(MFVec3dObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFVec3dObject testMFVec3dObject = new MFVec3dObject(); // static initializer is tested, might throw exception
        assertTrue  (testMFVec3dObject.matches(),       "testMFVec3dObject.matches() tests object initialization correctly matches regex");
        double[] emptyArray       = {};
        double[] singleValueArray = { 0.0d, 0.0d, 0.0d };
        double[] doubleValueArray = { 0.0d, 1.0d, -2.0d, -3.0d, -4.0d,  5.0d };
        double[] tripleValueArray = { 0.0d, 1.0d, -2.0d, -3.0d,  4.0d,  5.0d, -6.0d,  7.0d,  8.0d };
        assertTrue  (Arrays.equals(emptyArray, MFVec3dObject.DEFAULT_VALUE), "test correct default value is empty array for this MF field object");
        assertTrue  (MFVec3dObject.matches(MFVec3dObject.DEFAULT_VALUE_STRING),
                                                        "MFVec3dObject.matches(MFVec3dObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFVec3dObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFVec3dObject.REGEX.contains("^") && !MFVec3dObject.REGEX.contains("$"), "test MFVec3dObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (MFVec3fObject.REGEX.equals(MFVec3dObject.REGEX), "test MFVec3fObject.REGEX.equals(MFVec3dObject.REGEX) returns true");

        // TODO utility method needed to set singleton object using individual values
//        testMFVec3dObject.setValue(0.0d, 0.0d, 0.0d); // returns void because it matches (overrides) Java SAI specification interface
//        assertTrue  (Arrays.equals(singleValueArray,  testMFVec3dObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f results in singleton array with same value");
        
        testMFVec3dObject.setValue(emptyArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(emptyArray,testMFVec3dObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec3dObject.matches(),       "testMFVec3dObject.matches() tests emptyArray initialization correctly matches regex");
        testMFVec3dObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testMFVec3dObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec3dObject.matches(),       "testMFVec3dObject.matches() tests singleValueArray correctly matches regex");
        testMFVec3dObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(doubleValueArray,testMFVec3dObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec3dObject.matches(),       "testMFVec3dObject.matches() tests doubleValueArray correctly matches regex");
        testMFVec3dObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(tripleValueArray,testMFVec3dObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec3dObject.matches(),       "testMFVec3dObject.matches() tests tripleValueArray correctly matches regex");
        
        assertTrue  (MFVec3dObject.matches(" 0  1  2   3  4  5"),      "MFVec3dObject.matches( \" 0  1  2   3  4  5\")    tests correct string value");
        assertTrue  (MFVec3dObject.matches(" 0  1  2,  3  4  5"),      "MFVec3dObject.matches( \" 0  1  2,  3  4  5\")   tests correct string value");
        assertTrue  (MFVec3dObject.matches("-1 -2 -3, -4 -5 -6"),   "MFVec3dObject.matches(\"-1 -2, -3 -4\") tests correct string value");
        assertTrue  (MFVec3dObject.matches(" 0  12E45  0"),"MFVec3dObject.matches(\" 0  12E45  0\") tests correct string value, scientific notation");
        assertTrue  (MFVec3dObject.matches("+0 +12E+45 0"),"MFVec3dObject.matches(\"+0,+12E+45 0\") tests correct string value, scientific notation");
        assertTrue  (MFVec3dObject.matches("-0 -12E-45 0"),"MFVec3dObject.matches(\"-0,-12E-45 0\") tests correct string value, scientific notation");

        assertTrue  (MFVec3dObject.matches(""),             "MFVec3dObject.matches(\"\") tests correct empty string value");
        assertFalse (MFVec3dObject.matches(","),            "MFVec3dObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (MFVec3dObject.matches(" 0.0  1.0 2.0"),"MFVec3dObject.matches(\" 0.0  1.0 2.0\") tests correct string value");
        assertTrue  (MFVec3dObject.matches("-1.0 -3.0 -0"), "MFVec3dObject.matches(\"-1.0 -3.0 -0\") tests correct string value, including external whitespace");
        assertTrue  (MFVec3dObject.matches("0 1 0"),         "MFVec3dObject.matches(\"0 1 0\")  tests correct array as string value");
        assertTrue  (MFVec3dObject.matches("0.0 1.0 0"),    "MFVec3dObject.matches(\"0.0 1.0 0\")  tests correct array as string value");
        assertTrue  (MFVec3dObject.matches(" 0.0  12.3E45  0 "),      "MFVec3dObject.matches(\" 0.0  12.3E45  0\") tests correct string value, scientific notation");
        assertTrue  (MFVec3dObject.matches("+0.0 +12.3E+45 0 "),      "MFVec3dObject.matches(\"+0.0 +12.3E+45 0\") tests correct string value, scientific notation");
        assertTrue  (MFVec3dObject.matches("-0.0 -12.3E-45 0 "),      "MFVec3dObject.matches(\"-0.0 -12.3E-45 0\") tests correct string value, scientific notation");
        assertTrue  (MFVec3dObject.matches( ".6 -.6 0 .6e7 -.6e-7 0"),"MFVec3dObject.matches(\".6 -.6 0 .6e7 -.6e-7 0\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertTrue  (MFVec3dObject.matches("0 0 0, 1 1 1, "),         "MFVec3dObject.matches(\"0 0 0, 1 1 1, \")  tests correct array as string value, including whitespace and commas, with allowed trailing comma");

        assertFalse (MFVec3dObject.matches("true false false"),"MFVec3dObject.matches(\"true false false\") tests incorrect boolean string value");
        assertFalse (MFVec3dObject.matches("blah"),            "MFVec3dObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFVec3dObject.matches("NaN NaN NaN"),     "MFVec3dObject.matches(\"NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFVec4fObject single-field 4-tuple single-precision floating-point array")
    void SFVec4fObjectTests()
	{
        System.out.println ("SFVec4fObjectTests...");
        assertTrue  (Arrays.equals(SFVec4fObject.DEFAULT_VALUE, new SFVec4fObject().setValueByString(SFVec4fObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFVec4fObject testSFVec4fObject = new SFVec4fObject(); // static initializer is tested, might throw exception
        assertTrue  (testSFVec4fObject.matches(),       "testSFVec4fObject.matches() tests object initialization correctly matches regex");
        float[] emptyArray       = {};
        float[] singleValueArray = { 0.0f, 0.0f, 0.0f, 0.0f };
//      float[] doubleValueArray = { 0.0f, 1.0f, -2.0f,  3.0f, -4.0f, -5.0f, -6.0f, -7.0f };
//      float[] tripleValueArray = { 0.0f, 1.0f, -2.0f,  3.0f, -4.0f, -5.0f, -6.0f, -7.0f, 8.0f, 9.0f, 10.0f, 11.0f };
        assertTrue  (Arrays.equals(singleValueArray, SFVec4fObject.DEFAULT_VALUE), "test correct default value for this field object");
        assertTrue  (SFVec4fObject.matches(SFVec4fObject.DEFAULT_VALUE_STRING),
                                                        "SFVec4fObject.matches(SFVec4fObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFVec4fObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFVec4fObject.REGEX.contains("^") && !SFVec4fObject.REGEX.contains("$"), "test SFVec4fObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (SFVec4fObject.REGEX.equals(SFVec4dObject.REGEX),    "test SFVec4fObject.REGEX.equals(SFVec4dObject.REGEX) returns true");
        assertTrue  (SFVec4fObject.REGEX.equals(SFRotationObject.REGEX), "test SFVec4fObject.REGEX.equals(SFRotationObject.REGEX) returns true");

        testSFVec4fObject.setValue(0.0f, 0.0f, 0.0f, 0.0f); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(singleValueArray,  testSFVec4fObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f results in singleton array with same value");
        
        testSFVec4fObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testSFVec4fObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
      
        assertTrue  (SFVec4fObject.matches( "0 1 2 3"),      "SFVec4fObject.matches( \"0 1 2 3\")   tests correct string value");
        assertTrue  (SFVec4fObject.matches( "2 3 4 5"),      "SFVec4fObject.matches( \"2 3 4 5\")   tests correct string value");
        assertTrue  (SFVec4fObject.matches("-1 -2 -3 -4"),   "SFVec4fObject.matches(\"-1 -2 -3 -4\")  tests correct string value");
        assertTrue  (SFVec4fObject.matches(" 0  12E45  0 0"),"SFVec4fObject.matches(\" 0  12E45  0 0\") tests correct string value, scientific notation");
        assertTrue  (SFVec4fObject.matches("+0 +12E+45 0 0"),"SFVec4fObject.matches(\"+0 +12E+45 0 0\") tests correct string value, scientific notation");
        assertTrue  (SFVec4fObject.matches("-0 -12E-45 0 0"),"SFVec4fObject.matches(\"-0 -12E-45 0 0\") tests correct string value, scientific notation");

        assertTrue  (SFVec4fObject.matches(" 0.0  1.0 2.0 3.0"),   "SFVec4fObject.matches(\" 0.0  1.0 2.0 3.0\") tests correct string value");
        assertTrue  (SFVec4fObject.matches("-1.0 -2.0 -3.0 -4.0"), "SFVec4fObject.matches(\"-1.0 -2.0 -3.0 -4.0\") tests correct string value, including external whitespace");
        assertTrue  (SFVec4fObject.matches("0 1 2 3"),             "SFVec4fObject.matches(\"0 1 2 3\")  tests correct array as string value");
        assertTrue  (SFVec4fObject.matches("0.0 1.0 2.0 3.0"),     "SFVec4fObject.matches(\"0.0 1.0 2.0 3.0\")  tests correct array as string value");
        assertTrue  (SFVec4fObject.matches(" 0.0  12.3E45  0 0"),  "SFVec4fObject.matches(\" 0.0  12.3E45  0 0\") tests correct string value, scientific notation");
        assertTrue  (SFVec4fObject.matches("+0.0 +12.3E+45 0 0"),  "SFVec4fObject.matches(\"+0.0 +12.3E+45 0 0\") tests correct string value, scientific notation");
        assertTrue  (SFVec4fObject.matches("-0.0 -12.3E-45 0 0"),  "SFVec4fObject.matches(\"-0.0 -12.3E-45 0 0\") tests correct string value, scientific notation");
        
        assertFalse (SFVec4fObject.matches(""),                     "SFVec4fObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFVec4fObject.matches(","),                    "SFVec4fObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (SFVec4fObject.matches( ".6 -.6 .6e7 -.6e-7 0"),"SFVec4fObject.matches(\".6 -.6 .6e7 -.6e-7 0\") tests incorrect string value, no leading digit before decimal point, scientific notation");
        assertFalse (SFVec4fObject.matches("true false false true"),"SFVec4fObject.matches(\"true false false true\") tests incorrect boolean string value");
        assertFalse (SFVec4fObject.matches("blah"),                 "SFVec4fObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFVec4fObject.matches("NaN NaN NaN NaN"),      "SFVec4fObject.matches(\"NaN NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test MFVec4fObject multi-field 4-tuple single-precision floating-point array")
    void MFVec4fObjectTests()
	{
        System.out.println ("MFVec4fObjectTests...");
        assertTrue  (Arrays.equals(MFVec4fObject.DEFAULT_VALUE, new MFVec4fObject().setValueByString(MFVec4fObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFVec4fObject testMFVec4fObject = new MFVec4fObject(); // static initializer is tested, might throw exception
        assertTrue  (testMFVec4fObject.matches(),       "testMFVec4fObject.matches() tests object initialization correctly matches regex");
        float[] emptyArray       = {};
        float[] singleValueArray = { 0.0f, 0.0f, 0.0f, 0.0f };
        float[] doubleValueArray = { 0.0f, 1.0f, -2.0f,  3.0f, -4.0f, -5.0f, -6.0f, -7.0f };
        float[] tripleValueArray = { 0.0f, 1.0f, -2.0f,  3.0f, -4.0f, -5.0f, -6.0f, -7.0f, 8.0f, 9.0f, 10.0f, 11.0f };
        assertTrue  (Arrays.equals(emptyArray, MFVec4fObject.DEFAULT_VALUE), "test correct default value is empty array for this MF field object");
        assertTrue  (MFVec4fObject.matches(MFVec4fObject.DEFAULT_VALUE_STRING),
                                                        "MFVec4fObject.matches(MFVec4fObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFVec4fObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFVec4fObject.REGEX.contains("^") && !MFVec4fObject.REGEX.contains("$"), "test MFVec4fObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (MFVec4fObject.REGEX.equals(MFVec4dObject.REGEX),    "test MFVec4fObject.REGEX.equals(MFVec4dObject.REGEX) returns true");
        assertTrue  (MFVec4fObject.REGEX.equals(MFRotationObject.REGEX), "test MFVec4fObject.REGEX.equals(MFRotationObject.REGEX) returns true");

        // TODO utility method needed to set singleton object using individual values
//        testMFVec4fObject.setValue(0.0f, 0.0f, 0.0f, 0.0f); // returns void because it matches (overrides) Java SAI specification interface
//        assertTrue  (Arrays.equals(singleValueArray,  testMFVec4fObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f 0.0f 0.0f results in singleton array with same value");
        
        testMFVec4fObject.setValue(emptyArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(emptyArray,testMFVec4fObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec4fObject.matches(),       "testMFVec4fObject.matches() tests emptyArray initialization correctly matches regex");
        testMFVec4fObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testMFVec4fObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec4fObject.matches(),       "testMFVec4fObject.matches() tests singleValueArray correctly matches regex");
        testMFVec4fObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(doubleValueArray,testMFVec4fObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec4fObject.matches(),       "testMFVec4fObject.matches() tests doubleValueArray correctly matches regex");
        testMFVec4fObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(tripleValueArray,testMFVec4fObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec4fObject.matches(),       "testMFVec4fObject.matches() tests tripleValueArray correctly matches regex");
        
        assertTrue  (MFVec4fObject.matches(" 0  1  2  3   4  5  6  7"),   "MFVec4fObject.matches(\" 0  1  2  3   4  5  6  7\")   tests correct string value");
        assertTrue  (MFVec4fObject.matches(" 0  1  2  3,  4  5  6  7"),   "MFVec4fObject.matches(\" 0  1  2  3,  4  5  6  7\")   tests correct string value");
        assertTrue  (MFVec4fObject.matches("-0 -1 -2 -3, -4 -5 -6 -7"),   "MFVec4fObject.matches(\"-0 -1 -2 -3, -4 -5 -6 -7\") tests correct string value");
        assertTrue  (MFVec4fObject.matches(" 0  12E45  0 0"),"MFVec4fObject.matches(\"0   12E45  0 0\") tests correct string value, scientific notation");
        assertTrue  (MFVec4fObject.matches("+0 +12E+45 0 0"),"MFVec4fObject.matches(\"+0 +12E+45 0 0\") tests correct string value, scientific notation");
        assertTrue  (MFVec4fObject.matches("-0 -12E-45 0 0"),"MFVec4fObject.matches(\"-0 -12E-45 0 0\") tests correct string value, scientific notation");

        assertTrue  (MFVec4fObject.matches(""),               "MFVec4fObject.matches(\"\") tests correct empty string value");
        assertTrue  (MFVec4fObject.matches(" 0.0  1.0 2.0 0"),"MFVec4fObject.matches(\" 0.0  1.0 2.0 0\") tests correct string value");
        assertTrue  (MFVec4fObject.matches("-1.0 -3.0 2.0 0"),"MFVec4fObject.matches(\"-1.0 -3.0 2.0 0\") tests correct string value, including external whitespace");
        assertTrue  (MFVec4fObject.matches("0 1 2 3"),        "MFVec4fObject.matches(\"0 1 2 3\")  tests correct array as string value");
        assertTrue  (MFVec4fObject.matches("0.0 1.0 2.0 0"),  "MFVec4fObject.matches(\"0.0 1.0 2.0 0\")  tests correct array as string value");
        assertTrue  (MFVec4fObject.matches(" 0.0  12.3E45  0 0"),         "MFVec4fObject.matches(\" 0.0  12.3E45  0 0\") tests correct string value, scientific notation");
        assertTrue  (MFVec4fObject.matches("+0.0 +12.3E+45 0 0"),         "MFVec4fObject.matches(\"+0.0 +12.3E+45 0 0\") tests correct string value, scientific notation");
        assertTrue  (MFVec4fObject.matches("-0.0 -12.3E-45 0 0"),         "MFVec4fObject.matches(\"-0.0 -12.3E-45 0 0\") tests correct string value, scientific notation");
        assertTrue  (MFVec4fObject.matches( ".6 -.6 0 .6e7 -.6e-7 0 0 0"),"MFVec4fObject.matches(\".6 -.6 0 .6e7 -.6e-7 0 0 0\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertTrue  (MFVec4fObject.matches("0 0 0 0, 1 1 1 1, "),         "MFVec4fObject.matches(\"0 0 0 0, 1 1 1 1, \")  tests correct array as string value, including whitespace and commas, with allowed trailing comma");

        assertFalse (MFVec4fObject.matches(","),              "MFVec4fObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (MFVec4fObject.matches("true false false true"),"MFVec4fObject.matches(\"true false false true\") tests incorrect boolean string value");
        assertFalse (MFVec4fObject.matches("blah"),                 "MFVec4fObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFVec4fObject.matches("NaN NaN NaN NaN"),      "MFVec4fObject.matches(\"NaN NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFRotationObject single-field 4-tuple single-precision floating-point array")
    void SFRotationObjectTests()
	{
        System.out.println ("SFRotationObjectTests...");
        assertTrue  (Arrays.equals(SFRotationObject.DEFAULT_VALUE, new SFRotationObject().setValueByString(SFRotationObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFRotationObject testSFRotationObject = new SFRotationObject(); // static initializer is tested, might throw exception
        assertTrue  (testSFRotationObject.matches(),       "testSFRotationObject.matches() tests object initialization correctly matches regex");
        float[] emptyArray       = {};
        float[] singleValueArray = { 0.0f, 0.0f, 1.0f, 0.0f };
//      float[] doubleValueArray = { 0.0f, 1.0f, -2.0f,  3.0f, -4.0f, -5.0f, -6.0f, -7.0f };
//      float[] tripleValueArray = { 0.0f, 1.0f, -2.0f,  3.0f, -4.0f, -5.0f, -6.0f, -7.0f, 8.0f, 9.0f, 10.0f, 11.0f };
        assertTrue  (Arrays.equals(singleValueArray, SFRotationObject.DEFAULT_VALUE), "test correct default value for this field object");
        assertTrue  (SFRotationObject.matches(SFRotationObject.DEFAULT_VALUE_STRING),
                                                        "SFRotationObject.matches(SFRotationObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFRotationObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFRotationObject.REGEX.contains("^") && !SFRotationObject.REGEX.contains("$"), "test SFRotationObject.REGEX does not contain anchor characters ^ or $");
        testSFRotationObject.setValue(0.0f, 0.0f, 1.0f, 0.0f); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(singleValueArray,  testSFRotationObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f results in singleton array with same value");
        // confirm expected equivalent regexes
        assertTrue  (SFVec4fObject.REGEX.equals(SFVec4dObject.REGEX),    "test SFVec4fObject.REGEX.equals(SFVec4dObject.REGEX) returns true");
        assertTrue  (SFVec4fObject.REGEX.equals(SFRotationObject.REGEX), "test SFVec4fObject.REGEX.equals(SFRotationObject.REGEX) returns true");
        
        testSFRotationObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testSFRotationObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
      
        assertTrue  (SFRotationObject.matches( "0 1 2 3"),      "SFRotationObject.matches( \"0 1 2 3\")   tests correct string value");
        assertTrue  (SFRotationObject.matches( "2 3 4 5"),      "SFRotationObject.matches( \"2 3 4 5\")   tests correct string value");
        assertTrue  (SFRotationObject.matches("-1 -2 -3 -4"),   "SFRotationObject.matches(\"-1 -2 -3 -4\")  tests correct string value");
        assertTrue  (SFRotationObject.matches(" 0  12E45  0 0"),"SFRotationObject.matches(\" 0  12E45  0 0\") tests correct string value, scientific notation");
        assertTrue  (SFRotationObject.matches("+0 +12E+45 0 0"),"SFRotationObject.matches(\"+0 +12E+45 0 0\") tests correct string value, scientific notation");
        assertTrue  (SFRotationObject.matches("-0 -12E-45 0 0"),"SFRotationObject.matches(\"-0 -12E-45 0 0\") tests correct string value, scientific notation");

        assertTrue  (SFRotationObject.matches(" 0.0  1.0 2.0 3.0"),   "SFRotationObject.matches(\" 0.0  1.0 2.0 3.0\") tests correct string value");
        assertTrue  (SFRotationObject.matches("-1.0 -2.0 -3.0 -4.0"), "SFRotationObject.matches(\"-1.0 -2.0 -3.0 -4.0\") tests correct string value, including external whitespace");
        assertTrue  (SFRotationObject.matches("0 1 2 3"),             "SFRotationObject.matches(\"0 1 2 3\")  tests correct array as string value");
        assertTrue  (SFRotationObject.matches("0.0 1.0 2.0 3.0"),     "SFRotationObject.matches(\"0.0 1.0 2.0 3.0\")  tests correct array as string value");
        assertTrue  (SFRotationObject.matches(" 0.0  12.3E45  0 0"),  "SFRotationObject.matches(\" 0.0  12.3E45  0 0\") tests correct string value, scientific notation");
        assertTrue  (SFRotationObject.matches("+0.0 +12.3E+45 0 0"),  "SFRotationObject.matches(\"+0.0 +12.3E+45 0 0\") tests correct string value, scientific notation");
        assertTrue  (SFRotationObject.matches("-0.0 -12.3E-45 0 0"),  "SFRotationObject.matches(\"-0.0 -12.3E-45 0 0\") tests correct string value, scientific notation");

        assertFalse (SFRotationObject.matches(""),                     "SFRotationObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFRotationObject.matches(","),                    "SFRotationObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (SFRotationObject.matches( ".6 -.6 .6e7 -.6e-7 0"),"SFRotationObject.matches(\".6 -.6 .6e7 -.6e-7 0\") tests incorrect string value, no leading digit before decimal point, scientific notation");
        assertFalse (SFRotationObject.matches("true false false true"),"SFRotationObject.matches(\"true false false true\") tests incorrect boolean string value");
        assertFalse (SFRotationObject.matches("blah"),                 "SFRotationObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFRotationObject.matches("NaN NaN NaN NaN"),      "SFRotationObject.matches(\"NaN NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test MFRotationObject multi-field 4-tuple single-precision floating-point array")
    void MFRotationObjectTests()
	{
        System.out.println ("MFRotationObjectTests...");
        assertTrue  (Arrays.equals(MFRotationObject.DEFAULT_VALUE, new MFRotationObject().setValueByString(MFRotationObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFRotationObject testMFRotationObject = new MFRotationObject(); // static initializer is tested, might throw exception
        assertTrue  (testMFRotationObject.matches(),       "testMFRotationObject.matches() tests object initialization correctly matches regex");
        float[] emptyArray       = {};
        float[] singleValueArray = { 0.0f, 0.0f, 1.0f, 0.0f };
        float[] doubleValueArray = { 0.0f, 1.0f, -2.0f,  3.0f, -4.0f, -5.0f, -6.0f, -7.0f };
        float[] tripleValueArray = { 0.0f, 1.0f, -2.0f,  3.0f, -4.0f, -5.0f, -6.0f, -7.0f, 8.0f, 9.0f, 10.0f, 11.0f };
        assertTrue  (Arrays.equals(emptyArray, MFRotationObject.DEFAULT_VALUE), "test correct default value is empty array for this MF field object");
        assertTrue  (MFRotationObject.matches(MFRotationObject.DEFAULT_VALUE_STRING),
                                                        "MFRotationObject.matches(MFRotationObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFRotationObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFRotationObject.REGEX.contains("^") && !MFRotationObject.REGEX.contains("$"), "test MFRotationObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (MFVec4fObject.REGEX.equals(MFVec4dObject.REGEX),    "test MFVec4fObject.REGEX.equals(MFVec4dObject.REGEX) returns true");
        assertTrue  (MFVec4fObject.REGEX.equals(MFRotationObject.REGEX), "test MFVec4fObject.REGEX.equals(MFRotationObject.REGEX) returns true");

        // TODO utility method needed to set singleton object using individual values
//      // setValue(0 0 0 0) throws exception as expected due to zero-length axis
//        testMFRotationObject.setValue(0.0f, 0.0f, 0.0f, 0.0f); // returns void because it matches (overrides) Java SAI specification interface
//        assertTrue  (Arrays.equals(singleValueArray,  testMFRotationObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f 0.0f 0.0f results in singleton array with same value");
        
        testMFRotationObject.setValue(emptyArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(emptyArray,testMFRotationObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFRotationObject.matches(),       "testMFRotationObject.matches() tests emptyArray initialization correctly matches regex");
        testMFRotationObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testMFRotationObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFRotationObject.matches(),       "testMFRotationObject.matches() tests singleValueArray correctly matches regex");
        testMFRotationObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(doubleValueArray,testMFRotationObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFRotationObject.matches(),       "testMFRotationObject.matches() tests doubleValueArray correctly matches regex");
        testMFRotationObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(tripleValueArray,testMFRotationObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFRotationObject.matches(),       "testMFRotationObject.matches() tests tripleValueArray correctly matches regex");
        
        assertTrue  (MFRotationObject.matches(" 0  1  2  3   4  5  6  7"),   "MFRotationObject.matches(\" 0  1  2  3   4  5  6  7\")   tests correct string value");
        assertTrue  (MFRotationObject.matches(" 0  1  2  3,  4  5  6  7"),   "MFRotationObject.matches(\" 0  1  2  3,  4  5  6  7\")   tests correct string value");
        assertTrue  (MFRotationObject.matches("-0 -1 -2 -3, -4 -5 -6 -7"),   "MFRotationObject.matches(\"-0 -1 -2 -3, -4 -5 -6 -7\") tests correct string value");
        assertTrue  (MFRotationObject.matches(" 0  12E45  0 0"),"MFRotationObject.matches(\"0   12E45  0 0\") tests correct string value, scientific notation");
        assertTrue  (MFRotationObject.matches("+0 +12E+45 0 0"),"MFRotationObject.matches(\"+0 +12E+45 0 0\") tests correct string value, scientific notation");
        assertTrue  (MFRotationObject.matches("-0 -12E-45 0 0"),"MFRotationObject.matches(\"-0 -12E-45 0 0\") tests correct string value, scientific notation");

        assertTrue  (MFRotationObject.matches(""),               "MFRotationObject.matches(\"\") tests correct empty string value");
        assertTrue  (MFRotationObject.matches(" 0.0  1.0 2.0 0"),"MFRotationObject.matches(\" 0.0  1.0 2.0 0\") tests correct string value");
        assertTrue  (MFRotationObject.matches("-1.0 -3.0 2.0 0"),"MFRotationObject.matches(\"-1.0 -3.0 2.0 0\") tests correct string value, including external whitespace");
        assertTrue  (MFRotationObject.matches("0 1 2 3"),        "MFRotationObject.matches(\"0 1 2 3\")  tests correct array as string value");
        assertTrue  (MFRotationObject.matches("0.0 1.0 2.0 0"),  "MFRotationObject.matches(\"0.0 1.0 2.0 0\")  tests correct array as string value");
        assertTrue  (MFRotationObject.matches(" 0.0  12.3E45  0 0"),         "MFRotationObject.matches(\" 0.0  12.3E45  0 0\") tests correct string value, scientific notation");
        assertTrue  (MFRotationObject.matches("+0.0 +12.3E+45 0 0"),         "MFRotationObject.matches(\"+0.0 +12.3E+45 0 0\") tests correct string value, scientific notation");
        assertTrue  (MFRotationObject.matches("-0.0 -12.3E-45 0 0"),         "MFRotationObject.matches(\"-0.0 -12.3E-45 0 0\") tests correct string value, scientific notation");
        assertTrue  (MFRotationObject.matches( ".6 -.6 0 .6e7 -.6e-7 0 0 0"),"MFRotationObject.matches(\".6 -.6 0 .6e7 -.6e-7 0 0 0\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertTrue  (MFRotationObject.matches("0 0 0 0, 1 1 1 1, "),         "MFRotationObject.matches(\"0 0 0 0, 1 1 1 1, \")  tests correct array as string value, including whitespace and commas, with allowed trailing comma");

        assertFalse (MFRotationObject.matches(","),              "MFRotationObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (MFRotationObject.matches("true false false true"),"MFRotationObject.matches(\"true false false true\") tests incorrect boolean string value");
        assertFalse (MFRotationObject.matches("blah"),                 "MFRotationObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFRotationObject.matches("NaN NaN NaN NaN"),      "MFRotationObject.matches(\"NaN NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFVec4dObject single-field 4-tuple double-precision floating-point array")
    void SFVec4dObjectTests()
	{
        System.out.println ("SFVec4dObjectTests...");
        assertTrue  (Arrays.equals(SFVec4dObject.DEFAULT_VALUE, new SFVec4dObject().setValueByString(SFVec4dObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFVec4dObject testSFVec4dObject = new SFVec4dObject(); // static initializer is tested, might throw exception
        assertTrue  (testSFVec4dObject.matches(),       "testSFVec4dObject.matches() tests object initialization correctly matches regex");
        double[] emptyArray       = {};
        double[] singleValueArray = { 0.0d, 0.0d, 0.0d, 0.0d };
//      double[] doubleValueArray = { 0.0d, 1.0d, -2.0d,  3.0d, -4.0d, -5.0d, -6.0d, -7.0d };
//      double[] tripleValueArray = { 0.0d, 1.0d, -2.0d,  3.0d, -4.0d, -5.0d, -6.0d, -7.0d, 8.0d, 9.0d, 10.0d, 11.0d };
        assertTrue  (Arrays.equals(singleValueArray, SFVec4dObject.DEFAULT_VALUE), "test correct default value for this field object");
        assertTrue  (SFVec4dObject.matches(SFVec4dObject.DEFAULT_VALUE_STRING),
                                                        "SFVec4dObject.matches(SFVec4dObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFVec4dObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFVec4dObject.REGEX.contains("^") && !SFVec4dObject.REGEX.contains("$"), "test SFVec4dObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (SFVec4fObject.REGEX.equals(SFVec4dObject.REGEX), "test SFVec4fObject.REGEX.equals(SFVec4dObject.REGEX) returns true");

        testSFVec4dObject.setValue(0.0d, 0.0d, 0.0d, 0.0d); // returns void because it matches (overrides) Java SAI specification interface
        assertTrue  (Arrays.equals(singleValueArray,  testSFVec4dObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f results in singleton array with same value");
        
        testSFVec4dObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testSFVec4dObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
      
        assertTrue  (SFVec4dObject.matches( "0 1 2 3"),      "SFVec4dObject.matches( \"0 1 2 3\")   tests correct string value");
        assertTrue  (SFVec4dObject.matches( "2 3 4 5"),      "SFVec4dObject.matches( \"2 3 4 5\")   tests correct string value");
        assertTrue  (SFVec4dObject.matches("-1 -2 -3 -4"),    "SFVec4dObject.matches(\"-1 -2 -3 -4\")  tests correct string value");
        assertTrue  (SFVec4dObject.matches(" 0  12E45  0 0"),"SFVec4dObject.matches(\" 0  12E45  0 0\") tests correct string value, scientific notation");
        assertTrue  (SFVec4dObject.matches("+0 +12E+45 0 0"),"SFVec4dObject.matches(\"+0 +12E+45 0 0\") tests correct string value, scientific notation");
        assertTrue  (SFVec4dObject.matches("-0 -12E-45 0 0"),"SFVec4dObject.matches(\"-0 -12E-45 0 0\") tests correct string value, scientific notation");

        assertTrue  (SFVec4dObject.matches(" 0.0  1.0 0 0"), "SFVec4dObject.matches(\" 0.0  1.0 0 0\") tests correct string value");
        assertTrue  (SFVec4dObject.matches("-1.0 -3.0 0 0"), "SFVec4dObject.matches(\"-1.0 -3.0 0 0\") tests correct string value, including external whitespace");
        assertTrue  (SFVec4dObject.matches("0 1 2 3"),       "SFVec4dObject.matches(\"0 1 2 3\")  tests correct array as string value");
        assertTrue  (SFVec4dObject.matches("0.0 1.0 0 0"),   "SFVec4dObject.matches(\"0.0 1.0 0 0\")  tests correct array as string value");
        assertTrue  (SFVec4dObject.matches(" 0.0  12.3E45  0 0"),    "SFVec4dObject.matches(\" 0.0  12.3E45  0 0\") tests correct string value, scientific notation");
        assertTrue  (SFVec4dObject.matches("+0.0 +12.3E+45 0 0"),    "SFVec4dObject.matches(\"+0.0 +12.3E+45 0 0\") tests correct string value, scientific notation");
        assertTrue  (SFVec4dObject.matches("-0.0 -12.3E-45 0 0"),    "SFVec4dObject.matches(\"-0.0 -12.3E-45 0 0\") tests correct string value, scientific notation");
        
        assertFalse (SFVec4dObject.matches(""),                   "SFVec4dObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFVec4dObject.matches(","),                  "SFVec4dObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertFalse (SFVec4dObject.matches( ".6 -.6 .6e7 -.6e-7 0"),"SFVec4dObject.matches(\".6 -.6 .6e7 -.6e-7 0\") tests incorrect string value, no leading digit before decimal point, scientific notation");
        assertFalse (SFVec4dObject.matches("true false false true"),   "SFVec4dObject.matches(\"true false false true\") tests incorrect boolean string value");
        assertFalse (SFVec4dObject.matches("blah"),                    "SFVec4dObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFVec4dObject.matches("NaN NaN NaN NaN"),         "SFVec4dObject.matches(\"NaN NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test MFVec4dObject multi-field 4-tuple double-precision floating-point array")
    void MFVec4dObjectTests()
	{
        System.out.println ("MFVec4dObjectTests...");
        assertTrue  (Arrays.equals(MFVec4dObject.DEFAULT_VALUE, new MFVec4dObject().setValueByString(MFVec4dObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFVec4dObject testMFVec4dObject = new MFVec4dObject(); // static initializer is tested, might throw exception
        assertTrue  (testMFVec4dObject.matches(),       "testMFVec4dObject.matches() tests object initialization correctly matches regex");
        double[] emptyArray       = {};
        double[] singleValueArray = { 0.0d, 0.0d, 0.0d, 0.0d };
        double[] doubleValueArray = { 0.0d, 1.0d, -2.0d,  3.0d, -4.0d, -5.0d, -6.0d, -7.0d };
        double[] tripleValueArray = { 0.0d, 1.0d, -2.0d,  3.0d, -4.0d, -5.0d, -6.0d, -7.0d, 8.0d, 9.0d, 10.0d, 11.0d };
        assertTrue  (Arrays.equals(emptyArray, MFVec4dObject.DEFAULT_VALUE), "test correct default value is empty array for this MF field object");
        assertTrue  (MFVec4dObject.matches(MFVec4dObject.DEFAULT_VALUE_STRING),
                                                        "MFVec4dObject.matches(MFVec4dObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFVec4dObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFVec4dObject.REGEX.contains("^") && !MFVec4dObject.REGEX.contains("$"), "test MFVec4dObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (MFVec4fObject.REGEX.equals(MFVec4dObject.REGEX), "test MFVec4fObject.REGEX.equals(MFVec4dObject.REGEX) returns true");

        // TODO utility method needed to set singleton object using individual values
//        testMFVec4dObject.setValue(0.0d, 0.0d, 0.0d, 0.0d); // returns void because it matches (overrides) Java SAI specification interface
//        assertTrue  (Arrays.equals(singleValueArray,  testMFVec4dObject.getPrimitiveValue()), "tests setting object value to 0.0f 0.0f results in singleton array with same value");
        
        testMFVec4dObject.setValue(emptyArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(emptyArray,testMFVec4dObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec4dObject.matches(),       "testMFVec4dObject.matches() tests emptyArray initialization correctly matches regex");
        testMFVec4dObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testMFVec4dObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec4dObject.matches(),       "testMFVec4dObject.matches() tests singleValueArray correctly matches regex");
        testMFVec4dObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(doubleValueArray,testMFVec4dObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec4dObject.matches(),       "testMFVec4dObject.matches() tests doubleValueArray correctly matches regex");
        testMFVec4dObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(tripleValueArray,testMFVec4dObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFVec4dObject.matches(),       "testMFVec4dObject.matches() tests tripleValueArray correctly matches regex");
        
        assertTrue  (MFVec4dObject.matches(" 0  1  2  3   4  5  7  8"),      "MFVec4dObject.matches( \" 0  1  2  3   4  5  7  8\")    tests correct string value");
        assertTrue  (MFVec4dObject.matches(" 0  1  2  3,  4  5  7  8"),      "MFVec4dObject.matches( \" 0  1  2  3,  4  5  7  8\")   tests correct string value");
        assertTrue  (MFVec4dObject.matches("-1 -2 -3 -4, -5 -6 -7 -8"),   "MFVec4dObject.matches(\"-1 -2 -3 -4, -5 -6 -7 -8\") tests correct string value");
        assertTrue  (MFVec4dObject.matches(" 0  12E45  0 0"),"MFVec4dObject.matches(\" 0  12E45  0 0\") tests correct string value, scientific notation");
        assertTrue  (MFVec4dObject.matches("+0 +12E+45 0 0"),"MFVec4dObject.matches(\"+0,+12E+45 0 0\") tests correct string value, scientific notation");
        assertTrue  (MFVec4dObject.matches("-0 -12E-45 0 0"),"MFVec4dObject.matches(\"-0,-12E-45 0 0\") tests correct string value, scientific notation");

        assertTrue  (MFVec4dObject.matches(""),                "MFVec4dObject.matches(\"\") tests correct empty string value");
        assertFalse (MFVec4dObject.matches(","),               "MFVec4dObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (MFVec4dObject.matches(" 0.0  1.0 2.0 0"), "MFVec4dObject.matches(\" 0.0  1.0 2.0 0\") tests correct string value");
        assertTrue  (MFVec4dObject.matches("-1.0 -3.0 -0  0"), "MFVec4dObject.matches(\"-1.0 -3.0 -0  0\") tests correct string value, including external whitespace");
        assertTrue  (MFVec4dObject.matches("0 1 0 0"),         "MFVec4dObject.matches(\"0 1 0 0\")  tests correct array as string value");
        assertTrue  (MFVec4dObject.matches("0.0 1.0 0 0"),     "MFVec4dObject.matches(\"0.0 1.0 0 0\")  tests correct array as string value");
        assertTrue  (MFVec4dObject.matches(" 0.0  12.3E45  0 0 "),        "MFVec4dObject.matches(\" 0.0  12.3E45  0 0 \") tests correct string value, scientific notation");
        assertTrue  (MFVec4dObject.matches("+0.0 +12.3E+45 0 0 "),        "MFVec4dObject.matches(\"+0.0 +12.3E+45 0 0 \") tests correct string value, scientific notation");
        assertTrue  (MFVec4dObject.matches("-0.0 -12.3E-45 0 0 "),        "MFVec4dObject.matches(\"-0.0 -12.3E-45 0 0 \") tests correct string value, scientific notation");
        assertTrue  (MFVec4dObject.matches( ".6 -.6 0 .6e7 -.6e-7 0 0 0"),"MFVec4dObject.matches(\".6 -.6 0 .6e7 -.6e-7 0 0 0\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertTrue  (MFVec4dObject.matches("0 0 0 0, 1 1 1 1, "),         "MFVec4dObject.matches(\"0 0 0 0, 1 1 1 1, \")  tests correct array as string value, including whitespace and commas, with allowed trailing comma");

        assertFalse (MFVec4dObject.matches("true false false true"),"MFVec4dObject.matches(\"true false false true\") tests incorrect boolean string value");
        assertFalse (MFVec4dObject.matches("blah"),                 "MFVec4dObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFVec4dObject.matches("NaN NaN NaN"),          "MFVec4dObject.matches(\"NaN NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFMatrix3fObject single-field 9-tuple single-precision floating-point array")
    void SFMatrix3fObjectTests()
	{
        System.out.println ("SFMatrix3fObjectTests...");
        assertTrue  (Arrays.equals(SFMatrix3fObject.DEFAULT_VALUE, new SFMatrix3fObject().setValueByString(SFMatrix3fObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFMatrix3fObject testSFMatrix3fObject = new SFMatrix3fObject(); // static initializer is tested, might throw exception
        assertTrue  (testSFMatrix3fObject.matches(),       "testSFMatrix3fObject.matches() tests object initialization correctly matches regex");
        float []       emptyArray = {};
        float []    identityArray = { 1.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 1.0f };
        float [] singleValueArray = { 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f };
//      float[] doubleValueArray = { 1.0f, -2.0f, -3.0f, -4.0f,  5.0f, -6.0f,  7.0f,  8.0f, 9.0f, 11.0f, -12.0f, -13.0f, -14.0f,  15.0f, -16.0f,  17.0f,  18.0f, 19.0f  };
//      float[] tripleValueArray = { 1.0f, -2.0f, -3.0f,  4.0f,  5.0f, -6.0f,  7.0f,  8.0f, 9.0f, 11.0f, -12.0f, -13.0f, -14.0f,  15.0f, -16.0f,  17.0f,  18.0f, 19.0f, 21.0f, -22.0f, -23.0f, -24.0f,  25.0f, -26.0f,  27.0f,  28.0f, 29.0f };
        assertTrue  (Arrays.equals(identityArray, SFMatrix3fObject.DEFAULT_VALUE), "test correct default value is identity array for this MF field object");
        assertTrue  (SFMatrix3fObject.matches(SFMatrix3fObject.DEFAULT_VALUE_STRING),
                                                        "SFMatrix3fObject.matches(SFMatrix3fObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFMatrix3fObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFMatrix3fObject.REGEX.contains("^") && !SFMatrix3fObject.REGEX.contains("$"), "test SFMatrix3fObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (SFMatrix3fObject.REGEX.equals(SFMatrix3dObject.REGEX), "test SFMatrix3fObject.REGEX.equals(SFMatrix3dObject.REGEX) returns true");

        testSFMatrix3fObject = new SFMatrix3fObject();
        assertTrue  (Arrays.equals(identityArray,testSFMatrix3fObject.getPrimitiveValue()),   "tests setting object value to emptyArray results in identityArray for equivalent getPrimitiveValue()");
        assertTrue  (testSFMatrix3fObject.matches(),       "testSFMatrix3fObject.matches() tests emptyArray initialization correctly matches regex");
        testSFMatrix3fObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testSFMatrix3fObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testSFMatrix3fObject.matches(),       "testSFMatrix3fObject.matches() tests singleValueArray correctly matches regex");
//        testSFMatrix3fObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
//        assertEquals(doubleValueArray,testSFMatrix3fObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
//        assertTrue  (testSFMatrix3fObject.matches(),       "testSFMatrix3fObject.matches() tests doubleValueArray correctly matches regex");
//        testSFMatrix3fObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
//        assertEquals(tripleValueArray,testSFMatrix3fObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
//        assertTrue  (testSFMatrix3fObject.matches(),       "testSFMatrix3fObject.matches() tests tripleValueArray correctly matches regex");
        
        assertTrue  (SFMatrix3fObject.matches(" 0  1  2   3  4  5  6 7 8 "),   "SFMatrix3fObject.matches( \" 0  1  2   3  4  5  6 7 8 \") tests correct string value");
        assertFalse (SFMatrix3fObject.matches(" 0  1  2,  3  4  5, 6 7 8 "),   "SFMatrix3fObject.matches( \" 0  1  2,  3  4  5, 6 7 8 \") tests incorrect string value, no commas allowed");
        assertTrue  (SFMatrix3fObject.matches("-1 -2 -3  -4 -5 -6  7 8 9 "),   "SFMatrix3fObject.matches( \"-1 -2 -3  -4 -5 -6  7 8 9 \") tests correct string value");
        assertTrue  (SFMatrix3fObject.matches(" 0  12E45  0 4 5 6 7 8 9 "),"SFMatrix3fObject.matches(\" 0  12E45  0 4 5 6 7 8 9 \") tests correct string value, scientific notation");
        assertTrue  (SFMatrix3fObject.matches("+0 +12E+45 0 4 5 6 7 8 9 "),"SFMatrix3fObject.matches(\"+0,+12E+45 0 4 5 6 7 8 9 \") tests correct string value, scientific notation");
        assertTrue  (SFMatrix3fObject.matches("-0 -12E-45 0 4 5 6 7 8 9 "),"SFMatrix3fObject.matches(\"-0,-12E-45 0 4 5 6 7 8 9 \") tests correct string value, scientific notation");

        assertFalse (SFMatrix3fObject.matches(""),             "SFMatrix3fObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFMatrix3fObject.matches(","),            "SFMatrix3fObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (SFMatrix3fObject.matches(" 0.0  1.0 2.0 3 4 5 6 7 8"), "SFMatrix3fObject.matches(\" 0.0  1.0 2.0 3 4 5 6 7 8\") tests correct string value");
        assertTrue  (SFMatrix3fObject.matches("-1.0 -3.0 -0  3 4 5 6 7 8"), "SFMatrix3fObject.matches(\"-1.0 -3.0 -0  3 4 5 6 7 8\") tests correct string value, including external whitespace");
        assertTrue  (SFMatrix3fObject.matches("0 1 0  3 4 5 6 7 8"),        "SFMatrix3fObject.matches(\"0 1 0  3 4 5 6 7 8\")  tests correct array as string value");
        assertTrue  (SFMatrix3fObject.matches("0.0 1.0 0  3 4 5 6 7 8"),    "SFMatrix3fObject.matches(\"0.0 1.0 0  3 4 5 6 7 8\")  tests correct array as string value");
        assertTrue  (SFMatrix3fObject.matches(" 0.0  12.3E45  0  3 4 5 6 7 8 "),"SFMatrix3fObject.matches(\" 0.0  12.3E45  0  3 4 5 6 7 8\") tests correct string value, scientific notation");
        assertTrue  (SFMatrix3fObject.matches("+0.0 +12.3E+45 0  3 4 5 6 7 8 "),"SFMatrix3fObject.matches(\"+0.0 +12.3E+45 0  3 4 5 6 7 8\") tests correct string value, scientific notation");
        assertTrue  (SFMatrix3fObject.matches("-0.0 -12.3E-45 0  3 4 5 6 7 8 "),"SFMatrix3fObject.matches(\"-0.0 -12.3E-45 0  3 4 5 6 7 8\") tests correct string value, scientific notation");
        assertTrue  (SFMatrix3fObject.matches( ".6 -.6 0 .6e7 -.6e-7 0 6 7 8 "),"SFMatrix3fObject.matches(\".6 -.6 0 .6e7 -.6e-7 0 6 7 8 \") tests correct string value, no leading digit before decimal point, scientific notation");
        assertFalse (SFMatrix3fObject.matches("0 0 0, 1 1 1, 2 2 2, "),         "SFMatrix3fObject.matches(\"0 0 0, 1 1 1, 2 2 2, \")  tests incorrect array as string value, including whitespace and commas, with trailing comma");

        assertFalse (SFMatrix3fObject.matches("true false false"),"SFMatrix3fObject.matches(\"true false false\") tests incorrect boolean string value");
        assertFalse (SFMatrix3fObject.matches("blah"),            "SFMatrix3fObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFMatrix3fObject.matches("NaN NaN NaN"),     "SFMatrix3fObject.matches(\"NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFMatrix3dObject single-field 9-tuple double-precision floating-point array")
    void SFMatrix3dObjectTests()
	{
        System.out.println ("SFMatrix3dObjectTests...");
        assertTrue  (Arrays.equals(SFMatrix3dObject.DEFAULT_VALUE, new SFMatrix3dObject().setValueByString(SFMatrix3dObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFMatrix3dObject testSFMatrix3dObject = new SFMatrix3dObject(); // static initializer is tested, might throw exception
        assertTrue  (testSFMatrix3dObject.matches(),       "testSFMatrix3dObject.matches() tests object initialization correctly matches regex");
        double []       emptyArray = {};
        double []    identityArray = { 1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 1.0 };
        double [] singleValueArray = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 };
//      double [] doubleValueArray = { 1.0, -2.0, -3.0d, -4.0d,  5.0d, -6.0d,  7.0d,  8.0d, 9.0d, 11.0d, -12.0d, -13.0d, -14.0d,  15.0d, -16.0d,  17.0d,  18.0d, 19.0d  };
//      double [] tripleValueArray = { 1.0, -2.0, -3.0d,  4.0d,  5.0d, -6.0d,  7.0d,  8.0d, 9.0d, 11.0d, -12.0d, -13.0d, -14.0d,  15.0d, -16.0d,  17.0d,  18.0d, 19.0d, 21.0d, -22.0d, -23.0d, -24.0d,  25.0d, -26.0d,  27.0d,  28.0d, 29.0d };
        assertTrue  (Arrays.equals(identityArray, SFMatrix3dObject.DEFAULT_VALUE), "test correct default value is identity array for this MF field object");
        assertTrue  (SFMatrix3dObject.matches(SFMatrix3dObject.DEFAULT_VALUE_STRING),
                                                        "SFMatrix3dObject.matches(SFMatrix3dObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFMatrix3dObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFMatrix3dObject.REGEX.contains("^") && !SFMatrix3dObject.REGEX.contains("$"), "test SFMatrix3dObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (SFMatrix3fObject.REGEX.equals(SFMatrix3dObject.REGEX), "test SFMatrix3fObject.REGEX.equals(SFMatrix3dObject.REGEX) returns true");

        testSFMatrix3dObject = new SFMatrix3dObject();
        assertTrue  (Arrays.equals(identityArray,testSFMatrix3dObject.getPrimitiveValue()),   "tests setting object value to emptyArray results in identityArray for equivalent getPrimitiveValue()");
        assertTrue  (testSFMatrix3dObject.matches(),       "testSFMatrix3dObject.matches() tests emptyArray initialization correctly matches regex");
        testSFMatrix3dObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testSFMatrix3dObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testSFMatrix3dObject.matches(),       "testSFMatrix3dObject.matches() tests singleValueArray correctly matches regex");
//        testSFMatrix3dObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
//        assertEquals(doubleValueArray,testSFMatrix3dObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
//        assertTrue  (testSFMatrix3dObject.matches(),       "testSFMatrix3dObject.matches() tests doubleValueArray correctly matches regex");
//        testSFMatrix3dObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
//        assertEquals(tripleValueArray,testSFMatrix3dObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
//        assertTrue  (testSFMatrix3dObject.matches(),       "testSFMatrix3dObject.matches() tests tripleValueArray correctly matches regex");
        
        assertTrue  (SFMatrix3dObject.matches(" 0  1  2   3  4  5  6 7 8 "),   "SFMatrix3dObject.matches( \" 0  1  2   3  4  5  6 7 8 \") tests correct string value");
        assertFalse (SFMatrix3dObject.matches(" 0  1  2,  3  4  5, 6 7 8 "),   "SFMatrix3dObject.matches( \" 0  1  2,  3  4  5, 6 7 8 \") tests incorrect string value, no commas allowed");
        assertTrue  (SFMatrix3dObject.matches("-1 -2 -3  -4 -5 -6  7 8 9 "),   "SFMatrix3dObject.matches( \"-1 -2 -3  -4 -5 -6  7 8 9 \") tests correct string value");
        assertTrue  (SFMatrix3dObject.matches(" 0  12E45  0 4 5 6 7 8 9 "),"SFMatrix3dObject.matches(\" 0  12E45  0 4 5 6 7 8 9 \") tests correct string value, scientific notation");
        assertTrue  (SFMatrix3dObject.matches("+0 +12E+45 0 4 5 6 7 8 9 "),"SFMatrix3dObject.matches(\"+0,+12E+45 0 4 5 6 7 8 9 \") tests correct string value, scientific notation");
        assertTrue  (SFMatrix3dObject.matches("-0 -12E-45 0 4 5 6 7 8 9 "),"SFMatrix3dObject.matches(\"-0,-12E-45 0 4 5 6 7 8 9 \") tests correct string value, scientific notation");

        assertFalse (SFMatrix3dObject.matches(""),             "SFMatrix3dObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFMatrix3dObject.matches(","),            "SFMatrix3dObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (SFMatrix3dObject.matches(" 0.0  1.0 2.0 3 4 5 6 7 8"), "SFMatrix3dObject.matches(\" 0.0  1.0 2.0 3 4 5 6 7 8\") tests correct string value");
        assertTrue  (SFMatrix3dObject.matches("-1.0 -3.0 -0  3 4 5 6 7 8"), "SFMatrix3dObject.matches(\"-1.0 -3.0 -0  3 4 5 6 7 8\") tests correct string value, including external whitespace");
        assertTrue  (SFMatrix3dObject.matches("0 1 0  3 4 5 6 7 8"),        "SFMatrix3dObject.matches(\"0 1 0  3 4 5 6 7 8\")  tests correct array as string value");
        assertTrue  (SFMatrix3dObject.matches("0.0 1.0 0  3 4 5 6 7 8"),    "SFMatrix3dObject.matches(\"0.0 1.0 0  3 4 5 6 7 8\")  tests correct array as string value");
        assertTrue  (SFMatrix3dObject.matches(" 0.0  12.3E45  0  3 4 5 6 7 8 "),"SFMatrix3dObject.matches(\" 0.0  12.3E45  0  3 4 5 6 7 8\") tests correct string value, scientific notation");
        assertTrue  (SFMatrix3dObject.matches("+0.0 +12.3E+45 0  3 4 5 6 7 8 "),"SFMatrix3dObject.matches(\"+0.0 +12.3E+45 0  3 4 5 6 7 8\") tests correct string value, scientific notation");
        assertTrue  (SFMatrix3dObject.matches("-0.0 -12.3E-45 0  3 4 5 6 7 8 "),"SFMatrix3dObject.matches(\"-0.0 -12.3E-45 0  3 4 5 6 7 8\") tests correct string value, scientific notation");
        assertTrue  (SFMatrix3dObject.matches( ".6 -.6 0 .6e7 -.6e-7 0 6 7 8 "),"SFMatrix3dObject.matches(\".6 -.6 0 .6e7 -.6e-7 0 6 7 8 \") tests correct string value, no leading digit before decimal point, scientific notation");
        assertFalse (SFMatrix3dObject.matches("0 0 0, 1 1 1, 2 2 2, "),         "SFMatrix3dObject.matches(\"0 0 0, 1 1 1, 2 2 2, \")  tests incorrect array as string value, including whitespace and commas, with trailing comma");

        assertFalse (SFMatrix3dObject.matches("true false false"),"SFMatrix3dObject.matches(\"true false false\") tests incorrect boolean string value");
        assertFalse (SFMatrix3dObject.matches("blah"),            "SFMatrix3dObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFMatrix3dObject.matches("NaN NaN NaN"),     "SFMatrix3dObject.matches(\"NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test MFMatrix3fObject multiple-field 9-tuple single-precision floating-point array")
    void MFMatrix3fObjectTests()
	{
        System.out.println ("MFMatrix3fObjectTests...");
        assertTrue  (Arrays.equals(MFMatrix3fObject.DEFAULT_VALUE, new MFMatrix3fObject().setValueByString(MFMatrix3fObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFMatrix3fObject testMFMatrix3fObject = new MFMatrix3fObject(); // static initializer is tested, might throw exception
        assertTrue  (testMFMatrix3fObject.matches(),       "testMFMatrix3fObject.matches() tests object initialization correctly matches regex");
        float []       emptyArray = {};
//      float []    identityArray = { 1.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 1.0f };
        float [] singleValueArray = { 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f };
        float [] doubleValueArray = { 1.0f, -2.0f, -3.0f, -4.0f,  5.0f, -6.0f,  7.0f,  8.0f, 9.0f, 11.0f, -12.0f, -13.0f, -14.0f,  15.0f, -16.0f,  17.0f,  18.0f, 19.0f  };
        float [] tripleValueArray = { 1.0f, -2.0f, -3.0f,  4.0f,  5.0f, -6.0f,  7.0f,  8.0f, 9.0f, 11.0f, -12.0f, -13.0f, -14.0f,  15.0f, -16.0f,  17.0f,  18.0f, 19.0f, 21.0f, -22.0f, -23.0f, -24.0f,  25.0f, -26.0f,  27.0f,  28.0f, 29.0f };
        assertTrue  (Arrays.equals(emptyArray, MFMatrix3fObject.DEFAULT_VALUE), "test correct default value is emptyArray for this MF field object");
        assertTrue  (MFMatrix3fObject.matches(MFMatrix3fObject.DEFAULT_VALUE_STRING),
                                                        "MFMatrix3fObject.matches(MFMatrix3fObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFMatrix3fObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFMatrix3fObject.REGEX.contains("^") && !MFMatrix3fObject.REGEX.contains("$"), "test MFMatrix3fObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (MFMatrix3fObject.REGEX.equals(MFMatrix3dObject.REGEX), "test MFMatrix3fObject.REGEX.equals(MFMatrix3dObject.REGEX) returns true");

        testMFMatrix3fObject = new MFMatrix3fObject();
        assertTrue  (Arrays.equals(emptyArray,testMFMatrix3fObject.getPrimitiveValue()),   "tests default object value is emptyArray and results in emptyArray for equivalent getPrimitiveValue()");
        assertTrue  (testMFMatrix3fObject.matches(),       "testMFMatrix3fObject.matches() tests emptyArray initialization correctly matches regex");
        testMFMatrix3fObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testMFMatrix3fObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFMatrix3fObject.matches(),       "testMFMatrix3fObject.matches() tests singleValueArray correctly matches regex");
        testMFMatrix3fObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(doubleValueArray,testMFMatrix3fObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFMatrix3fObject.matches(),       "testMFMatrix3fObject.matches() tests doubleValueArray correctly matches regex");
        testMFMatrix3fObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(tripleValueArray,testMFMatrix3fObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFMatrix3fObject.matches(),       "testMFMatrix3fObject.matches() tests tripleValueArray correctly matches regex");
        
        assertTrue  (MFMatrix3fObject.matches(" 0  1  2   3  4  5  6 7 8 "),"MFMatrix3fObject.matches( \" 0  1  2   3  4  5  6 7 8 \") tests correct string value");
        assertFalse (MFMatrix3fObject.matches(" 0  1  2,  3  4  5, 6 7 8 "),"MFMatrix3fObject.matches( \" 0  1  2,  3  4  5, 6 7 8 \") tests incorrect string value, no commas allowed");
        assertTrue  (MFMatrix3fObject.matches("-1 -2 -3  -4 -5 -6  7 8 9 "),"MFMatrix3fObject.matches( \"-1 -2 -3  -4 -5 -6  7 8 9 \") tests correct string value");
        assertTrue  (MFMatrix3fObject.matches(" 0  12E45  0 4 5 6 7 8 9 "), "MFMatrix3fObject.matches(\" 0  12E45  0 4 5 6 7 8 9 \") tests correct string value, scientific notation");
        assertTrue  (MFMatrix3fObject.matches("+0 +12E+45 0 4 5 6 7 8 9 "), "MFMatrix3fObject.matches(\"+0,+12E+45 0 4 5 6 7 8 9 \") tests correct string value, scientific notation");
        assertTrue  (MFMatrix3fObject.matches("-0 -12E-45 0 4 5 6 7 8 9 ")," MFMatrix3fObject.matches(\"-0,-12E-45 0 4 5 6 7 8 9 \") tests correct string value, scientific notation");

        assertTrue  (MFMatrix3fObject.matches(""),             "MFMatrix3fObject.matches(\"\") tests correct empty string value");
        assertFalse (MFMatrix3fObject.matches(","),            "MFMatrix3fObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (MFMatrix3fObject.matches(" 0.0  1.0 2.0 3 4 5 6 7 8"), "MFMatrix3fObject.matches(\" 0.0  1.0 2.0 3 4 5 6 7 8\") tests correct string value");
        assertTrue  (MFMatrix3fObject.matches("-1.0 -3.0 -0  3 4 5 6 7 8"), "MFMatrix3fObject.matches(\"-1.0 -3.0 -0  3 4 5 6 7 8\") tests correct string value, including external whitespace");
        assertTrue  (MFMatrix3fObject.matches("0 1 0  3 4 5 6 7 8"),        "MFMatrix3fObject.matches(\"0 1 0  3 4 5 6 7 8\")  tests correct array as string value");
        assertTrue  (MFMatrix3fObject.matches("0.0 1.0 0  3 4 5 6 7 8"),    "MFMatrix3fObject.matches(\"0.0 1.0 0  3 4 5 6 7 8\")  tests correct array as string value");
        assertTrue  (MFMatrix3fObject.matches(" 0.0  12.3E45  0  3 4 5 6 7 8 "),"MFMatrix3fObject.matches(\" 0.0  12.3E45  0  3 4 5 6 7 8\") tests correct string value, scientific notation");
        assertTrue  (MFMatrix3fObject.matches("+0.0 +12.3E+45 0  3 4 5 6 7 8 "),"MFMatrix3fObject.matches(\"+0.0 +12.3E+45 0  3 4 5 6 7 8\") tests correct string value, scientific notation");
        assertTrue  (MFMatrix3fObject.matches("-0.0 -12.3E-45 0  3 4 5 6 7 8 "),"MFMatrix3fObject.matches(\"-0.0 -12.3E-45 0  3 4 5 6 7 8\") tests correct string value, scientific notation");
        assertTrue  (MFMatrix3fObject.matches( ".6 -.6 0 .6e7 -.6e-7 0 6 7 8 "),"MFMatrix3fObject.matches(\".6 -.6 0 .6e7 -.6e-7 0 6 7 8 \") tests correct string value, no leading digit before decimal point, scientific notation");
        assertFalse (MFMatrix3fObject.matches("0 0 0, 1 1 1, 2 2 2, "),         "MFMatrix3fObject.matches(\"0 0 0, 1 1 1, 2 2 2, \")  tests incorrect array as string value, including whitespace and commas, with trailing comma");

        assertFalse (MFMatrix3fObject.matches("true false false"),"MFMatrix3fObject.matches(\"true false false\") tests incorrect boolean string value");
        assertFalse (MFMatrix3fObject.matches("blah"),            "MFMatrix3fObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFMatrix3fObject.matches("NaN NaN NaN"),     "MFMatrix3fObject.matches(\"NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test MFMatrix3dObject multiple-field 9-tuple double-precision floating-point array")
    void MFMatrix3dObjectTests()
	{
        System.out.println ("MFMatrix3dObjectTests...");
        assertTrue  (Arrays.equals(MFMatrix3dObject.DEFAULT_VALUE, new MFMatrix3dObject().setValueByString(MFMatrix3dObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFMatrix3dObject testMFMatrix3dObject = new MFMatrix3dObject(); // static initializer is tested, might throw exception
        assertTrue  (testMFMatrix3dObject.matches(),       "testMFMatrix3dObject.matches() tests object initialization correctly matches regex");
        double []       emptyArray = {};
//      double []    identityArray = { 1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 1.0 };
        double [] singleValueArray = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 };
        double [] doubleValueArray = { 1.0, -2.0, -3.0d, -4.0d,  5.0d, -6.0d,  7.0d,  8.0d, 9.0d, 11.0d, -12.0d, -13.0d, -14.0d,  15.0d, -16.0d,  17.0d,  18.0d, 19.0d  };
        double [] tripleValueArray = { 1.0, -2.0, -3.0d,  4.0d,  5.0d, -6.0d,  7.0d,  8.0d, 9.0d, 11.0d, -12.0d, -13.0d, -14.0d,  15.0d, -16.0d,  17.0d,  18.0d, 19.0d, 21.0d, -22.0d, -23.0d, -24.0d,  25.0d, -26.0d,  27.0d,  28.0d, 29.0d };
        assertTrue  (Arrays.equals(emptyArray, MFMatrix3dObject.DEFAULT_VALUE), "test correct default value is emptyArray for this MF field object");
        assertTrue  (MFMatrix3dObject.matches(MFMatrix3dObject.DEFAULT_VALUE_STRING),
                                                        "MFMatrix3dObject.matches(MFMatrix3dObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFMatrix3dObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFMatrix3dObject.REGEX.contains("^") && !MFMatrix3dObject.REGEX.contains("$"), "test MFMatrix3dObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (MFMatrix3fObject.REGEX.equals(MFMatrix3dObject.REGEX), "test MFMatrix3fObject.REGEX.equals(MFMatrix3dObject.REGEX) returns true");

        testMFMatrix3dObject = new MFMatrix3dObject();
        assertTrue  (Arrays.equals(emptyArray,testMFMatrix3dObject.getPrimitiveValue()),   "tests setting object value is emptyArray and results in emptyArray for equivalent getPrimitiveValue()");
        assertTrue  (testMFMatrix3dObject.matches(),       "testMFMatrix3dObject.matches() tests emptyArray initialization correctly matches regex");
        testMFMatrix3dObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testMFMatrix3dObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFMatrix3dObject.matches(),       "testMFMatrix3dObject.matches() tests singleValueArray correctly matches regex");
        testMFMatrix3dObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(doubleValueArray,testMFMatrix3dObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFMatrix3dObject.matches(),       "testMFMatrix3dObject.matches() tests doubleValueArray correctly matches regex");
        testMFMatrix3dObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(tripleValueArray,testMFMatrix3dObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFMatrix3dObject.matches(),       "testMFMatrix3dObject.matches() tests tripleValueArray correctly matches regex");
        
        assertTrue  (MFMatrix3dObject.matches(" 0  1  2   3  4  5  6 7 8 "),"MFMatrix3dObject.matches( \" 0  1  2   3  4  5  6 7 8 \") tests correct string value");
        assertFalse (MFMatrix3dObject.matches(" 0  1  2,  3  4  5, 6 7 8 "),"MFMatrix3dObject.matches( \" 0  1  2,  3  4  5, 6 7 8 \") tests incorrect string value, no commas allowed");
        assertTrue  (MFMatrix3dObject.matches("-1 -2 -3  -4 -5 -6  7 8 9 "),"MFMatrix3dObject.matches( \"-1 -2 -3  -4 -5 -6  7 8 9 \") tests correct string value");
        assertTrue  (MFMatrix3dObject.matches(" 0  12E45  0 4 5 6 7 8 9 "), "MFMatrix3dObject.matches(\" 0  12E45  0 4 5 6 7 8 9 \") tests correct string value, scientific notation");
        assertTrue  (MFMatrix3dObject.matches("+0 +12E+45 0 4 5 6 7 8 9 "), "MFMatrix3dObject.matches(\"+0,+12E+45 0 4 5 6 7 8 9 \") tests correct string value, scientific notation");
        assertTrue  (MFMatrix3dObject.matches("-0 -12E-45 0 4 5 6 7 8 9 "), "MFMatrix3dObject.matches(\"-0,-12E-45 0 4 5 6 7 8 9 \") tests correct string value, scientific notation");

        assertTrue  (MFMatrix3dObject.matches(""),             "MFMatrix3dObject.matches(\"\") tests correct empty string value");
        assertFalse (MFMatrix3dObject.matches(","),            "MFMatrix3dObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (MFMatrix3dObject.matches(" 0.0  1.0 2.0 3 4 5 6 7 8"), "MFMatrix3dObject.matches(\" 0.0  1.0 2.0 3 4 5 6 7 8\") tests correct string value");
        assertTrue  (MFMatrix3dObject.matches("-1.0 -3.0 -0  3 4 5 6 7 8"), "MFMatrix3dObject.matches(\"-1.0 -3.0 -0  3 4 5 6 7 8\") tests correct string value, including external whitespace");
        assertTrue  (MFMatrix3dObject.matches("0 1 0  3 4 5 6 7 8"),        "MFMatrix3dObject.matches(\"0 1 0  3 4 5 6 7 8\")  tests correct array as string value");
        assertTrue  (MFMatrix3dObject.matches("0.0 1.0 0  3 4 5 6 7 8"),    "MFMatrix3dObject.matches(\"0.0 1.0 0  3 4 5 6 7 8\")  tests correct array as string value");
        assertTrue  (MFMatrix3dObject.matches(" 0.0  12.3E45  0  3 4 5 6 7 8 "),"MFMatrix3dObject.matches(\" 0.0  12.3E45  0  3 4 5 6 7 8\") tests correct string value, scientific notation");
        assertTrue  (MFMatrix3dObject.matches("+0.0 +12.3E+45 0  3 4 5 6 7 8 "),"MFMatrix3dObject.matches(\"+0.0 +12.3E+45 0  3 4 5 6 7 8\") tests correct string value, scientific notation");
        assertTrue  (MFMatrix3dObject.matches("-0.0 -12.3E-45 0  3 4 5 6 7 8 "),"MFMatrix3dObject.matches(\"-0.0 -12.3E-45 0  3 4 5 6 7 8\") tests correct string value, scientific notation");
        assertTrue  (MFMatrix3dObject.matches( ".6 -.6 0 .6e7 -.6e-7 0 6 7 8 "),"MFMatrix3dObject.matches(\".6 -.6 0 .6e7 -.6e-7 0 6 7 8 \") tests correct string value, no leading digit before decimal point, scientific notation");
        assertFalse (MFMatrix3dObject.matches("0 0 0, 1 1 1, 2 2 2, "),         "MFMatrix3dObject.matches(\"0 0 0, 1 1 1, 2 2 2, \")  tests incorrect array as string value, including whitespace and commas, with trailing comma");

        assertFalse (MFMatrix3dObject.matches("true false false"),"MFMatrix3dObject.matches(\"true false false\") tests incorrect boolean string value");
        assertFalse (MFMatrix3dObject.matches("blah"),            "MFMatrix3dObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFMatrix3dObject.matches("NaN NaN NaN"),     "MFMatrix3dObject.matches(\"NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFMatrix4fObject single-field 16-tuple single-precision floating-point array")
    void SFMatrix4fObjectTests()
	{
        System.out.println ("SFMatrix4fObjectTests...");
        assertTrue  (Arrays.equals(SFMatrix4fObject.DEFAULT_VALUE, new SFMatrix4fObject().setValueByString(SFMatrix4fObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFMatrix4fObject testSFMatrix4fObject = new SFMatrix4fObject(); // static initializer is tested, might throw exception
        assertTrue  (testSFMatrix4fObject.matches(),       "testSFMatrix4fObject.matches() tests object initialization correctly matches regex");
        float []       emptyArray = {};
        float []    identityArray = { 1.0f, 0.0f, 0.0f, 0.0f,  0.0f, 1.0f, 0.0f, 0.0f,  0.0f, 0.0f, 1.0f, 0.0f,  0.0f, 0.0f, 0.0f, 1.0f };
        float [] singleValueArray = { 0.0f, 0.0f, 0.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f };
//      float[] doubleValueArray = {  1.0f, -2.0f, -3.0f, -4.0f,  5.0f,  -6.0f,   7.0f,   8.0f,   9.0f,  10.0f, -11.0f, -12.0f, -13.0f, -14.0f, 15.0f, -16.0f, 
//                                    17.0f, 18.0f, 19.0f, 20.0f, 21.0f, -22.0f, -23.0f, -24.0f,  25.0f, -26.0f,  27.0f,  28.0f,  29.0f,  30.0f, 31.0f,  32.0f };
//      float[] tripleValueArray = {  1.0f, -2.0f, -3.0f, -4.0f,  5.0f,  -6.0f,   7.0f,   8.0f,   9.0f,  10.0f, -11.0f, -12.0f, -13.0f, -14.0f, 15.0f, -16.0f, 
//                                    17.0f, 18.0f, 19.0f, 20.0f, 21.0f, -22.0f, -23.0f, -24.0f,  25.0f, -26.0f,  27.0f,  28.0f,  29.0f,  30.0f, 31.0f,  32.0f,
//                                    33.0f, 34.0f, 35.0f, 36.0f, 37.0f, -38.0f, -39.0f, -40.0f,  41.0f, -42.0f,  43.0f,  44.0f,  45.0f,  46.0f, 47.0f,  48.0f};
        assertTrue  (Arrays.equals(identityArray, SFMatrix4fObject.DEFAULT_VALUE), "test correct default value is identity array for this MF field object");
        assertTrue  (SFMatrix4fObject.matches(SFMatrix4fObject.DEFAULT_VALUE_STRING),
                                                        "SFMatrix4fObject.matches(SFMatrix4fObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFMatrix4fObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFMatrix4fObject.REGEX.contains("^") && !SFMatrix4fObject.REGEX.contains("$"), "test SFMatrix4fObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (SFMatrix4fObject.REGEX.equals(SFMatrix4dObject.REGEX), "test SFMatrix4fObject.REGEX.equals(SFMatrix4dObject.REGEX) returns true");

        testSFMatrix4fObject = new SFMatrix4fObject();
        assertTrue  (Arrays.equals(identityArray,testSFMatrix4fObject.getPrimitiveValue()),   "tests setting default object value results in identityArray for equivalent getPrimitiveValue()");
        testSFMatrix4fObject.setIdentity();
        assertTrue  (Arrays.equals(identityArray,testSFMatrix4fObject.getPrimitiveValue()),   "tests setting object value using setIdentity() results in identityArray for equivalent getPrimitiveValue()");
        // TODO other utility methods when implemented
        assertTrue  (testSFMatrix4fObject.matches(),       "testSFMatrix4fObject.matches() tests setIdentity() initialization correctly matches regex");
        testSFMatrix4fObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testSFMatrix4fObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testSFMatrix4fObject.matches(),       "testSFMatrix4fObject.matches() tests singleValueArray correctly matches regex");
//        testSFMatrix4fObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
//        assertEquals(doubleValueArray,testSFMatrix4fObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
//        assertTrue  (testSFMatrix4fObject.matches(),       "testSFMatrix4fObject.matches() tests doubleValueArray correctly matches regex");
//        testSFMatrix4fObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
//        assertEquals(tripleValueArray,testSFMatrix4fObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
//        assertTrue  (testSFMatrix4fObject.matches(),       "testSFMatrix4fObject.matches() tests tripleValueArray correctly matches regex");
        
        assertTrue  (SFMatrix4fObject.matches(" 0  1  2   3  4  5  6 7 8  9 10 11 12 13 14 15"),"SFMatrix4fObject.matches(\" 0  1  2   3  4  5  6 7 8  9 10 11 12 13 14 15\") tests correct string value");
        assertFalse (SFMatrix4fObject.matches(" 0  1  2,  3  4  5, 6 7 8  9 10 11 12 13 14 15"),"SFMatrix4fObject.matches(\" 0  1  2,  3  4  5, 6 7 8  9 10 11 12 13 14 15\") tests incorrect string value, no commas allowed");
        assertTrue  (SFMatrix4fObject.matches("-1 -2 -3  -4 -5 -6  7 8 9 10 11 12 13 14 15 16"),"SFMatrix4fObject.matches(\"-1 -2 -3  -4 -5 -6  7 8 9 10 11 12 13 14 15 16\") tests correct string value");
        assertTrue  (SFMatrix4fObject.matches(" 0  12E45  0 4 5 6  7 8 9 10 11 12 13 14 15 16"),"SFMatrix4fObject.matches(\" 0  12E45  0 4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (SFMatrix4fObject.matches("+0 +12E+45 0 4 5 6  7 8 9 10 11 12 13 14 15 16"),"SFMatrix4fObject.matches(\"+0,+12E+45 0 4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (SFMatrix4fObject.matches("-0 -12E-45 0 4 5 6  7 8 9 10 11 12 13 14 15 16"),"SFMatrix4fObject.matches(\"-0,-12E-45 0 4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");

        assertFalse (SFMatrix4fObject.matches(""),             "SFMatrix4fObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFMatrix4fObject.matches(","),            "SFMatrix4fObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (SFMatrix4fObject.matches(" 0.0  1.0 2.0       4 5 6 7 8 9 10 11 12 13 14 15 16"), "SFMatrix4fObject.matches(\" 0.0  1.0 2.0       4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value");
        assertTrue  (SFMatrix4fObject.matches("-1.0 -3.0 -0        4 5 6 7 8 9 10 11 12 13 14 15 16"), "SFMatrix4fObject.matches(\"-1.0 -3.0 -0        4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, including external whitespace");
        assertTrue  (SFMatrix4fObject.matches("0   1   0           4 5 6 7 8 9 10 11 12 13 14 15 16"), "SFMatrix4fObject.matches(\"0   1   0           4 5 6 7 8 9 10 11 12 13 14 15 16\")  tests correct array as string value");
        assertTrue  (SFMatrix4fObject.matches("0.0 1.0 0           4 5 6 7 8 9 10 11 12 13 14 15 16"), "SFMatrix4fObject.matches(\"0.0 1.0 0           4 5 6 7 8 9 10 11 12 13 14 15 16\")  tests correct array as string value");
        assertTrue  (SFMatrix4fObject.matches(" 0.0  12.3E45  0    4 5 6 7 8 9 10 11 12 13 14 15 16"), "SFMatrix4fObject.matches(\" 0.0  12.3E45  0    4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (SFMatrix4fObject.matches("+0.0 +12.3E+45 0    4 5 6 7 8 9 10 11 12 13 14 15 16"), "SFMatrix4fObject.matches(\"+0.0 +12.3E+45 0    4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (SFMatrix4fObject.matches("-0.0 -12.3E-45 0    4 5 6 7 8 9 10 11 12 13 14 15 16"), "SFMatrix4fObject.matches(\"-0.0 -12.3E-45 0    4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (SFMatrix4fObject.matches(" .6 -.6 .6e7 -.6e-7   5 6 7 8 9 10 11 12 13 14 15 16"), "SFMatrix4fObject.matches(\" .6 -.6 .6e7 -.6e-7   5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertFalse (SFMatrix4fObject.matches("0 0 0 0, 1 1 1 1, 2 2 2 2, 3 3 3 3"),                   "SFMatrix4fObject.matches(\"0 0 0 0, 1 1 1 1, 2 2 2 2, 3 3 3 3\")  tests incorrect array as string value, including whitespace and commas, with trailing comma");

        assertFalse (SFMatrix4fObject.matches("true false false"),"SFMatrix4fObject.matches(\"true false false\") tests incorrect boolean string value");
        assertFalse (SFMatrix4fObject.matches("blah"),            "SFMatrix4fObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFMatrix4fObject.matches("NaN NaN NaN"),     "SFMatrix4fObject.matches(\"NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test SFMatrix4dObject single-field 16-tuple double-precision floating-point array")
    void SFMatrix4dObjectTests()
	{
        System.out.println ("SFMatrix4dObjectTests...");
        assertTrue  (Arrays.equals(SFMatrix4dObject.DEFAULT_VALUE, new SFMatrix4dObject().setValueByString(SFMatrix4dObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        SFMatrix4dObject testSFMatrix4dObject = new SFMatrix4dObject(); // static initializer is tested, might throw exception
        assertTrue  (testSFMatrix4dObject.matches(),       "testSFMatrix4dObject.matches() tests object initialization correctly matches regex");
        double []       emptyArray = {};
        double []    identityArray = { 1.0d, 0.0d, 0.0d, 0.0d,  0.0d, 1.0d, 0.0d, 0.0d,  0.0d, 0.0d, 1.0d, 0.0d,  0.0d, 0.0d, 0.0d, 1.0d };
        double [] singleValueArray = { 0.0d, 0.0d, 0.0d, 0.0d,  0.0d, 0.0d, 0.0d, 0.0d,  0.0d, 0.0d, 0.0d, 0.0d,  0.0d, 0.0d, 0.0d, 0.0d };
//      double [] doubleValueArray = { 1.0d, -2.0d, -3.0d, -4.0d,  5.0d,  -6.0d,   7.0d,   8.0d,   9.0d,  10.0d, -11.0d, -12.0d, -13.0d, -14.0d, 15.0d, -16.0d, 
//                                    17.0d, 18.0d, 19.0d, 20.0d, 21.0d, -22.0d, -23.0d, -24.0d,  25.0d, -26.0d,  27.0d,  28.0d,  29.0d,  30.0d, 31.0d,  32.0d };
//      double [] tripleValueArray = { 1.0d, -2.0d, -3.0d, -4.0d,  5.0d,  -6.0d,   7.0d,   8.0d,   9.0d,  10.0d, -11.0d, -12.0d, -13.0d, -14.0d, 15.0d, -16.0d, 
//                                    17.0d, 18.0d, 19.0d, 20.0d, 21.0d, -22.0d, -23.0d, -24.0d,  25.0d, -26.0d,  27.0d,  28.0d,  29.0d,  30.0d, 31.0d,  32.0d,
//                                    33.0d, 34.0d, 35.0d, 36.0d, 37.0d, -38.0d, -39.0d, -40.0d,  41.0d, -42.0d,  43.0d,  44.0d,  45.0d,  46.0d, 47.0d,  48.0d};
        assertTrue  (Arrays.equals(identityArray, SFMatrix4dObject.DEFAULT_VALUE), "test correct default value is identity array for this MF field object");
        assertTrue  (SFMatrix4dObject.matches(SFMatrix4dObject.DEFAULT_VALUE_STRING),
                                                        "SFMatrix4dObject.matches(SFMatrix4dObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testSFMatrix4dObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!SFMatrix4dObject.REGEX.contains("^") && !SFMatrix4dObject.REGEX.contains("$"), "test SFMatrix4dObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (SFMatrix4fObject.REGEX.equals(SFMatrix4dObject.REGEX), "test SFMatrix4dObject.REGEX.equals(SFMatrix4dObject.REGEX) returns true");

        testSFMatrix4dObject = new SFMatrix4dObject();
        assertTrue  (Arrays.equals(identityArray,testSFMatrix4dObject.getPrimitiveValue()),   "tests setting default object value results in identityArray for equivalent getPrimitiveValue()");
        testSFMatrix4dObject.setIdentity();
        assertTrue  (Arrays.equals(identityArray,testSFMatrix4dObject.getPrimitiveValue()),   "tests setting object value using setIdentity() results in identityArray for equivalent getPrimitiveValue()");
        assertTrue  (testSFMatrix4dObject.matches(),       "testSFMatrix4dObject.matches() tests setIdentity() initialization correctly matches regex");
        testSFMatrix4dObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testSFMatrix4dObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testSFMatrix4dObject.matches(),       "testSFMatrix4dObject.matches() tests singleValueArray correctly matches regex");
//        testSFMatrix4dObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
//        assertEquals(doubleValueArray,testSFMatrix4dObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
//        assertTrue  (testSFMatrix4dObject.matches(),       "testSFMatrix4dObject.matches() tests doubleValueArray correctly matches regex");
//        testSFMatrix4dObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
//        assertEquals(tripleValueArray,testSFMatrix4dObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
//        assertTrue  (testSFMatrix4dObject.matches(),       "testSFMatrix4dObject.matches() tests tripleValueArray correctly matches regex");
        
        assertTrue  (SFMatrix4dObject.matches(" 0  1  2   3  4  5  6 7 8  9 10 11 12 13 14 15"),"SFMatrix4dObject.matches( \" 0  1  2   3  4  5  6 7 8  9 10 11 12 13 14 15\") tests correct string value");
        assertFalse (SFMatrix4dObject.matches(" 0  1  2,  3  4  5, 6 7 8  9 10 11 12 13 14 15"),"SFMatrix4dObject.matches( \" 0  1  2,  3  4  5, 6 7 8  9 10 11 12 13 14 15\") tests incorrect string value, no commas allowed");
        assertTrue  (SFMatrix4dObject.matches("-1 -2 -3  -4 -5 -6  7 8 9 10 11 12 13 14 15 16"),"SFMatrix4dObject.matches( \"-1 -2 -3  -4 -5 -6  7 8 9 10 11 12 13 14 15 16\") tests correct string value");
        assertTrue  (SFMatrix4dObject.matches(" 0  12E45  0 4 5 6  7 8 9 10 11 12 13 14 15 16"),"SFMatrix4dObject.matches(\" 0  12E45  0 4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (SFMatrix4dObject.matches("+0 +12E+45 0 4 5 6  7 8 9 10 11 12 13 14 15 16"),"SFMatrix4dObject.matches(\"+0,+12E+45 0 4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (SFMatrix4dObject.matches("-0 -12E-45 0 4 5 6  7 8 9 10 11 12 13 14 15 16"),"SFMatrix4dObject.matches(\"-0,-12E-45 0 4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");

        assertFalse (SFMatrix4dObject.matches(""),             "SFMatrix4dObject.matches(\"\") tests incorrect empty string value");
        assertFalse (SFMatrix4dObject.matches(","),            "SFMatrix4dObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (SFMatrix4dObject.matches(" 0.0  1.0 2.0       4 5 6 7 8 9 10 11 12 13 14 15 16"), "SFMatrix4dObject.matches(\" 0.0  1.0 2.0       4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value");
        assertTrue  (SFMatrix4dObject.matches("-1.0 -3.0 -0        4 5 6 7 8 9 10 11 12 13 14 15 16"), "SFMatrix4dObject.matches(\"-1.0 -3.0 -0        4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, including external whitespace");
        assertTrue  (SFMatrix4dObject.matches("0   1   0           4 5 6 7 8 9 10 11 12 13 14 15 16"), "SFMatrix4dObject.matches(\"0   1   0           4 5 6 7 8 9 10 11 12 13 14 15 16\")  tests correct array as string value");
        assertTrue  (SFMatrix4dObject.matches("0.0 1.0 0           4 5 6 7 8 9 10 11 12 13 14 15 16"), "SFMatrix4dObject.matches(\"0.0 1.0 0           4 5 6 7 8 9 10 11 12 13 14 15 16\")  tests correct array as string value");
        assertTrue  (SFMatrix4dObject.matches(" 0.0  12.3E45  0    4 5 6 7 8 9 10 11 12 13 14 15 16"), "SFMatrix4dObject.matches(\" 0.0  12.3E45  0    4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (SFMatrix4dObject.matches("+0.0 +12.3E+45 0    4 5 6 7 8 9 10 11 12 13 14 15 16"), "SFMatrix4dObject.matches(\"+0.0 +12.3E+45 0    4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (SFMatrix4dObject.matches("-0.0 -12.3E-45 0    4 5 6 7 8 9 10 11 12 13 14 15 16"), "SFMatrix4dObject.matches(\"-0.0 -12.3E-45 0    4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (SFMatrix4dObject.matches(" .6 -.6 .6e7 -.6e-7   5 6 7 8 9 10 11 12 13 14 15 16"), "SFMatrix4dObject.matches(\" .6 -.6 .6e7 -.6e-7   5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertFalse (SFMatrix4dObject.matches("0 0 0 0, 1 1 1 1, 2 2 2 2, 3 3 3 3"),                   "SFMatrix4dObject.matches(\"0 0 0 0, 1 1 1 1, 2 2 2 2, 3 3 3 3\")  tests incorrect array as string value, including whitespace and commas, with trailing comma");

        assertFalse (SFMatrix4dObject.matches("true false false"),"SFMatrix4dObject.matches(\"true false false\") tests incorrect boolean string value");
        assertFalse (SFMatrix4dObject.matches("blah"),            "SFMatrix4dObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (SFMatrix4dObject.matches("NaN NaN NaN"),     "SFMatrix4dObject.matches(\"NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test MFMatrix4fObject multiple-field 16-tuple single-precision floating-point array")
    void MFMatrix4fObjectTests()
	{
        System.out.println ("MFMatrix4fObjectTests...");
        assertTrue  (Arrays.equals(MFMatrix4fObject.DEFAULT_VALUE, new MFMatrix4fObject().setValueByString(MFMatrix4fObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFMatrix4fObject testMFMatrix4fObject = new MFMatrix4fObject(); // static initializer is tested, might throw exception
        assertTrue  (testMFMatrix4fObject.matches(),       "testMFMatrix4fObject.matches() tests object initialization correctly matches regex");
        float []       emptyArray = {};
//      float []    identityArray = { 1.0f, 0.0f, 0.0f, 0.0f,  0.0f, 1.0f, 0.0f, 0.0f,  0.0f, 0.0f, 1.0f, 0.0f,  0.0f, 0.0f, 0.0f, 1.0f };
        float [] singleValueArray = { 0.0f, 0.0f, 0.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f };
        float [] doubleValueArray = {  1.0f, -2.0f, -3.0f, -4.0f,  5.0f,  -6.0f,   7.0f,   8.0f,   9.0f,  10.0f, -11.0f, -12.0f, -13.0f, -14.0f, 15.0f, -16.0f, 
                                      17.0f, 18.0f, 19.0f, 20.0f, 21.0f, -22.0f, -23.0f, -24.0f,  25.0f, -26.0f,  27.0f,  28.0f,  29.0f,  30.0f, 31.0f,  32.0f };
        float [] tripleValueArray = {  1.0f, -2.0f, -3.0f, -4.0f,  5.0f,  -6.0f,   7.0f,   8.0f,   9.0f,  10.0f, -11.0f, -12.0f, -13.0f, -14.0f, 15.0f, -16.0f, 
                                      17.0f, 18.0f, 19.0f, 20.0f, 21.0f, -22.0f, -23.0f, -24.0f,  25.0f, -26.0f,  27.0f,  28.0f,  29.0f,  30.0f, 31.0f,  32.0f,
                                      33.0f, 34.0f, 35.0f, 36.0f, 37.0f, -38.0f, -39.0f, -40.0f,  41.0f, -42.0f,  43.0f,  44.0f,  45.0f,  46.0f, 47.0f,  48.0f};
        assertTrue  (Arrays.equals(emptyArray, MFMatrix4fObject.DEFAULT_VALUE), "test correct default value is empty array for this MF field object");
        assertTrue  (MFMatrix4fObject.matches(MFMatrix4fObject.DEFAULT_VALUE_STRING),
                                                        "MFMatrix4fObject.matches(MFMatrix4fObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFMatrix4fObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFMatrix4fObject.REGEX.contains("^") && !MFMatrix4fObject.REGEX.contains("$"), "test MFMatrix4fObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (MFMatrix4fObject.REGEX.equals(MFMatrix4dObject.REGEX), "test MFMatrix4fObject.REGEX.equals(MFMatrix4dObject.REGEX) returns true");

        testMFMatrix4fObject = new MFMatrix4fObject();
        assertTrue  (Arrays.equals(emptyArray,testMFMatrix4fObject.getPrimitiveValue()),   "tests setting object value to default object results in emptyArray for equivalent getPrimitiveValue()");
        assertTrue  (testMFMatrix4fObject.matches(),       "testMFMatrix4fObject.matches() tests emptyArray initialization correctly matches regex");
        testMFMatrix4fObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testMFMatrix4fObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFMatrix4fObject.matches(),       "testMFMatrix4fObject.matches() tests singleValueArray correctly matches regex");
        testMFMatrix4fObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(doubleValueArray,testMFMatrix4fObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFMatrix4fObject.matches(),       "testMFMatrix4fObject.matches() tests doubleValueArray correctly matches regex");
        testMFMatrix4fObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(tripleValueArray,testMFMatrix4fObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFMatrix4fObject.matches(),       "testMFMatrix4fObject.matches() tests tripleValueArray correctly matches regex");
        
        assertTrue  (MFMatrix4fObject.matches(" 0  1  2   3  4  5  6 7 8  9 10 11 12 13 14 15"),"MFMatrix4fObject.matches( \" 0  1  2   3  4  5  6 7 8  9 10 11 12 13 14 15\") tests correct string value");
        assertFalse (MFMatrix4fObject.matches(" 0  1  2,  3  4  5, 6 7 8  9 10 11 12 13 14 15"),"MFMatrix4fObject.matches( \" 0  1  2,  3  4  5, 6 7 8  9 10 11 12 13 14 15\") tests incorrect string value, no commas allowed");
        assertTrue  (MFMatrix4fObject.matches("-1 -2 -3  -4 -5 -6  7 8 9 10 11 12 13 14 15 16"),"MFMatrix4fObject.matches( \"-1 -2 -3  -4 -5 -6  7 8 9 10 11 12 13 14 15 16\") tests correct string value");
        assertTrue  (MFMatrix4fObject.matches(" 0  12E45  0 4 5 6  7 8 9 10 11 12 13 14 15 16"),"MFMatrix4fObject.matches(\" 0  12E45  0 4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (MFMatrix4fObject.matches("+0 +12E+45 0 4 5 6  7 8 9 10 11 12 13 14 15 16"),"MFMatrix4fObject.matches(\"+0,+12E+45 0 4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (MFMatrix4fObject.matches("-0 -12E-45 0 4 5 6  7 8 9 10 11 12 13 14 15 16"),"MFMatrix4fObject.matches(\"-0,-12E-45 0 4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");

        assertTrue  (MFMatrix4fObject.matches(""),             "MFMatrix4fObject.matches(\"\") tests correct empty string value");
        assertFalse (MFMatrix4fObject.matches(","),            "MFMatrix4fObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (MFMatrix4fObject.matches(" 0.0  1.0 2.0       4 5 6 7 8 9 10 11 12 13 14 15 16"), "MFMatrix4fObject.matches(\" 0.0  1.0 2.0       4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value");
        assertTrue  (MFMatrix4fObject.matches("-1.0 -3.0 -0        4 5 6 7 8 9 10 11 12 13 14 15 16"), "MFMatrix4fObject.matches(\"-1.0 -3.0 -0        4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, including external whitespace");
        assertTrue  (MFMatrix4fObject.matches("0   1   0           4 5 6 7 8 9 10 11 12 13 14 15 16"), "MFMatrix4fObject.matches(\"0   1   0           4 5 6 7 8 9 10 11 12 13 14 15 16\")  tests correct array as string value");
        assertTrue  (MFMatrix4fObject.matches("0.0 1.0 0           4 5 6 7 8 9 10 11 12 13 14 15 16"), "MFMatrix4fObject.matches(\"0.0 1.0 0           4 5 6 7 8 9 10 11 12 13 14 15 16\")  tests correct array as string value");
        assertTrue  (MFMatrix4fObject.matches(" 0.0  12.3E45  0    4 5 6 7 8 9 10 11 12 13 14 15 16"), "MFMatrix4fObject.matches(\" 0.0  12.3E45  0    4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (MFMatrix4fObject.matches("+0.0 +12.3E+45 0    4 5 6 7 8 9 10 11 12 13 14 15 16"), "MFMatrix4fObject.matches(\"+0.0 +12.3E+45 0    4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (MFMatrix4fObject.matches("-0.0 -12.3E-45 0    4 5 6 7 8 9 10 11 12 13 14 15 16"), "MFMatrix4fObject.matches(\"-0.0 -12.3E-45 0    4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (MFMatrix4fObject.matches(" .6 -.6 .6e7 -.6e-7   5 6 7 8 9 10 11 12 13 14 15 16"), "MFMatrix4fObject.matches(\" .6 -.6 .6e7 -.6e-7   5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertFalse (MFMatrix4fObject.matches("0 0 0 0, 1 1 1 1, 2 2 2 2, 3 3 3 3"),                   "MFMatrix4fObject.matches(\"0 0 0 0, 1 1 1 1, 2 2 2 2, 3 3 3 3\")  tests incorrect array as string value, including whitespace and commas, with trailing comma");

        assertFalse (MFMatrix4fObject.matches("true false false"),"MFMatrix4fObject.matches(\"true false false\") tests incorrect boolean string value");
        assertFalse (MFMatrix4fObject.matches("blah"),            "MFMatrix4fObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFMatrix4fObject.matches("NaN NaN NaN"),     "MFMatrix4fObject.matches(\"NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }
    
    @Test
    @DisplayName("Test MFMatrix4dObject multiple-field 16-tuple double-precision floating-point array")
    void MFMatrix4dObjectTests()
	{
        System.out.println ("MFMatrix4dObjectTests...");
        assertTrue  (Arrays.equals(MFMatrix4dObject.DEFAULT_VALUE, new MFMatrix4dObject().setValueByString(MFMatrix4dObject.DEFAULT_VALUE_STRING).getPrimitiveValue()), 
                                                        "test DEFAULT_VALUE matches DEFAULT_VALUE_STRING for this field object");
        MFMatrix4dObject testMFMatrix4dObject = new MFMatrix4dObject(); // static initializer is tested, might throw exception
        assertTrue  (testMFMatrix4dObject.matches(),       "testMFMatrix4dObject.matches() tests object initialization correctly matches regex");
        double []       emptyArray = {};
//      double []    identityArray = { 1.0d, 0.0d, 0.0d, 0.0d,  0.0d, 1.0d, 0.0d, 0.0d,  0.0d, 0.0d, 1.0d, 0.0d,  0.0d, 0.0d, 0.0d, 1.0d };
        double [] singleValueArray = { 0.0d, 0.0d, 0.0d, 0.0d,  0.0d, 0.0d, 0.0d, 0.0d,  0.0d, 0.0d, 0.0d, 0.0d,  0.0d, 0.0d, 0.0d, 0.0d };
      double [] doubleValueArray = { 1.0d, -2.0d, -3.0d, -4.0d,  5.0d,  -6.0d,   7.0d,   8.0d,   9.0d,  10.0d, -11.0d, -12.0d, -13.0d, -14.0d, 15.0d, -16.0d, 
                                    17.0d, 18.0d, 19.0d, 20.0d, 21.0d, -22.0d, -23.0d, -24.0d,  25.0d, -26.0d,  27.0d,  28.0d,  29.0d,  30.0d, 31.0d,  32.0d };
      double [] tripleValueArray = { 1.0d, -2.0d, -3.0d, -4.0d,  5.0d,  -6.0d,   7.0d,   8.0d,   9.0d,  10.0d, -11.0d, -12.0d, -13.0d, -14.0d, 15.0d, -16.0d, 
                                    17.0d, 18.0d, 19.0d, 20.0d, 21.0d, -22.0d, -23.0d, -24.0d,  25.0d, -26.0d,  27.0d,  28.0d,  29.0d,  30.0d, 31.0d,  32.0d,
                                    33.0d, 34.0d, 35.0d, 36.0d, 37.0d, -38.0d, -39.0d, -40.0d,  41.0d, -42.0d,  43.0d,  44.0d,  45.0d,  46.0d, 47.0d,  48.0d};
        assertTrue  (Arrays.equals(emptyArray, MFMatrix4dObject.DEFAULT_VALUE), "test correct default value is empty array for this MF field object");
        assertTrue  (MFMatrix4dObject.matches(MFMatrix4dObject.DEFAULT_VALUE_STRING),
                                                        "MFMatrix4dObject.matches(MFMatrix4dObject.DEFAULT_VALUE_STRING) tests object initialization correctly matches regex");
        assertTrue  (testMFMatrix4dObject.isDefaultValue(),"test initialized field object isDefaultValue() returns true");
        assertTrue  (!MFMatrix4dObject.REGEX.contains("^") && !MFMatrix4dObject.REGEX.contains("$"), "test MFMatrix4dObject.REGEX does not contain anchor characters ^ or $");
        // confirm expected equivalent regexes
        assertTrue  (MFMatrix4dObject.REGEX.equals(MFMatrix4dObject.REGEX), "test MFMatrix4dObject.REGEX.equals(MFMatrix4dObject.REGEX) returns true");

        testMFMatrix4dObject = new MFMatrix4dObject();
        assertTrue  (Arrays.equals(emptyArray,testMFMatrix4dObject.getPrimitiveValue()),   "tests setting object value to default object results in emptyArray for equivalent getPrimitiveValue()");
        assertTrue  (testMFMatrix4dObject.matches(),       "testMFMatrix4dObject.matches() tests emptyArray initialization correctly matches regex");
        testMFMatrix4dObject.setValue(singleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(singleValueArray,testMFMatrix4dObject.getPrimitiveValue(),   "tests setting object value to singleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFMatrix4dObject.matches(),       "testMFMatrix4dObject.matches() tests singleValueArray correctly matches regex");
        testMFMatrix4dObject.setValue(doubleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(doubleValueArray,testMFMatrix4dObject.getPrimitiveValue(),   "tests setting object value to doubleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFMatrix4dObject.matches(),       "testMFMatrix4dObject.matches() tests doubleValueArray correctly matches regex");
        testMFMatrix4dObject.setValue(tripleValueArray); // returns void because it matches (overrides) Java SAI specification interface
        assertEquals(tripleValueArray,testMFMatrix4dObject.getPrimitiveValue(),   "tests setting object value to tripleValueArray results in equivalent getPrimitiveValue()");
        assertTrue  (testMFMatrix4dObject.matches(),       "testMFMatrix4dObject.matches() tests tripleValueArray correctly matches regex");
        
        assertTrue  (MFMatrix4dObject.matches(" 0  1  2   3  4  5  6 7 8  9 10 11 12 13 14 15"),"MFMatrix4dObject.matches(\" 0  1  2   3  4  5  6 7 8  9 10 11 12 13 14 15\") tests correct string value");
        assertFalse (MFMatrix4dObject.matches(" 0  1  2,  3  4  5, 6 7 8  9 10 11 12 13 14 15"),"MFMatrix4dObject.matches(\" 0  1  2,  3  4  5, 6 7 8  9 10 11 12 13 14 15\") tests incorrect string value, no commas allowed");
        assertTrue  (MFMatrix4dObject.matches("-1 -2 -3  -4 -5 -6  7 8 9 10 11 12 13 14 15 16"),"MFMatrix4dObject.matches(\"-1 -2 -3  -4 -5 -6  7 8 9 10 11 12 13 14 15 16\") tests correct string value");
        assertTrue  (MFMatrix4dObject.matches(" 0  12E45  0 4 5 6  7 8 9 10 11 12 13 14 15 16"),"MFMatrix4dObject.matches(\" 0  12E45  0 4 5 6  7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (MFMatrix4dObject.matches("+0 +12E+45 0 4 5 6  7 8 9 10 11 12 13 14 15 16"),"MFMatrix4dObject.matches(\"+0 +12E+45 0 4 5 6  7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (MFMatrix4dObject.matches("-0 -12E-45 0 4 5 6  7 8 9 10 11 12 13 14 15 16"),"MFMatrix4dObject.matches(\"-0 -12E-45 0 4 5 6  7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");

        assertTrue  (MFMatrix4dObject.matches(""),             "MFMatrix4dObject.matches(\"\") tests correct empty string value");
        assertFalse (MFMatrix4dObject.matches(","),            "MFMatrix4dObject.matches(\",\") tests incorrect inclusion of comma as whitespace");
        assertTrue  (MFMatrix4dObject.matches(" 0.0  1.0 2.0       4 5 6 7 8 9 10 11 12 13 14 15 16"), "MFMatrix4dObject.matches(\" 0.0  1.0 2.0       4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value");
        assertTrue  (MFMatrix4dObject.matches("-1.0 -3.0 -0        4 5 6 7 8 9 10 11 12 13 14 15 16"), "MFMatrix4dObject.matches(\"-1.0 -3.0 -0        4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, including external whitespace");
        assertTrue  (MFMatrix4dObject.matches("0   1   0           4 5 6 7 8 9 10 11 12 13 14 15 16"), "MFMatrix4dObject.matches(\"0   1   0           4 5 6 7 8 9 10 11 12 13 14 15 16\")  tests correct array as string value");
        assertTrue  (MFMatrix4dObject.matches("0.0 1.0 0           4 5 6 7 8 9 10 11 12 13 14 15 16"), "MFMatrix4dObject.matches(\"0.0 1.0 0           4 5 6 7 8 9 10 11 12 13 14 15 16\")  tests correct array as string value");
        assertTrue  (MFMatrix4dObject.matches(" 0.0  12.3E45  0    4 5 6 7 8 9 10 11 12 13 14 15 16"), "MFMatrix4dObject.matches(\" 0.0  12.3E45  0    4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (MFMatrix4dObject.matches("+0.0 +12.3E+45 0    4 5 6 7 8 9 10 11 12 13 14 15 16"), "MFMatrix4dObject.matches(\"+0.0 +12.3E+45 0    4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (MFMatrix4dObject.matches("-0.0 -12.3E-45 0    4 5 6 7 8 9 10 11 12 13 14 15 16"), "MFMatrix4dObject.matches(\"-0.0 -12.3E-45 0    4 5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, scientific notation");
        assertTrue  (MFMatrix4dObject.matches(" .6 -.6 .6e7 -.6e-7   5 6 7 8 9 10 11 12 13 14 15 16"), "MFMatrix4dObject.matches(\" .6 -.6 .6e7 -.6e-7   5 6 7 8 9 10 11 12 13 14 15 16\") tests correct string value, no leading digit before decimal point, scientific notation");
        assertFalse (MFMatrix4dObject.matches("0 0 0 0, 1 1 1 1, 2 2 2 2, 3 3 3 3"),                   "MFMatrix4dObject.matches(\"0 0 0 0, 1 1 1 1, 2 2 2 2, 3 3 3 3\")  tests incorrect array as string value, including whitespace and commas, with trailing comma");

        assertFalse (MFMatrix4dObject.matches("true false false"),"MFMatrix4dObject.matches(\"true false false\") tests incorrect boolean string value");
        assertFalse (MFMatrix4dObject.matches("blah"),            "MFMatrix4dObject.matches(\"blah\") tests incorrect alphabetic string value");
        assertFalse (MFMatrix4dObject.matches("NaN NaN NaN"),     "MFMatrix4dObject.matches(\"NaN NaN NaN\") tests Not A Number (NaN) which is not an allowed string value");
    }

    /** Default main() method provided for test purposes.
     * @param args the command line arguments
     * @see #fieldObjectInitializationsTest()
     */
    public static void main(String[] args)
    {
        System.out.println("FieldObjectTests start...");
        FieldObjectTests thisObject = new FieldObjectTests();
        thisObject.fieldObjectInitializationsTest();
        System.out.println("FieldObjectTests complete");
    }
}